//
// Literal.cs
//
// Author:
//	Cesar Lopez Nataren (cesar@ciencias.unam.mx)
//
// (C) 2003, 2004 Cesar Lopez Nataren 
// (C) 2005, Novell Inc, (http://novell.com)
//

//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

using System;
using System.Reflection;
using System.Reflection.Emit;
using System.Collections;
using Microsoft.JScript.Vsa;

namespace Microsoft.JScript {

	internal class This : AST {

		internal This (AST parent)
		{
			this.parent = parent;
		}

		internal override bool Resolve (IdentificationTable context)
		{
			return true;
		}

		internal override void Emit (EmitContext ec)
		{
			if (InFunction)
				ec.ig.Emit (OpCodes.Ldarg_0);
			else
				CodeGenerator.emit_get_default_this (ec.ig);
		}
	}

	internal class BooleanLiteral : Exp {

		internal bool val;

		internal BooleanLiteral (AST parent, bool val)
		{
			this.parent = parent;
			this.val = val;
		}

		public override string ToString ()
		{
			return val.ToString ();
		}

		internal override bool Resolve (IdentificationTable context)
		{
			return true;
		}

		internal override bool Resolve (IdentificationTable context, bool no_effect)
		{
			this.no_effect = no_effect;
			return true;
		}

		internal override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			if (val)
				ig.Emit (OpCodes.Ldc_I4_1);
			else
				ig.Emit (OpCodes.Ldc_I4_0);

			ig.Emit (OpCodes.Box, typeof (System.Boolean));

			if (no_effect)
				ig.Emit (OpCodes.Pop);
		}
	}

	internal class NumericLiteral : Exp {

		double val;

		internal NumericLiteral (AST parent, double val)
		{
			this.parent = parent;
			this.val = val;
		}

		public override string ToString ()
		{
			return val.ToString ();
		}

		internal override bool Resolve (IdentificationTable context)
		{
			return true;
		}

		internal override bool Resolve (IdentificationTable context, bool no_effect)
		{			
			this.no_effect = no_effect;
			return Resolve (context);
		}

		internal override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			if (parent is Unary && (parent as Unary).oper == JSToken.Minus)
				ig.Emit (OpCodes.Ldc_R8, (double) (val * -1));
			else
				ig.Emit (OpCodes.Ldc_R8, (double) val);
			ig.Emit (OpCodes.Box, typeof (System.Double));
			if (no_effect)
				ig.Emit (OpCodes.Pop);
		}
	}

	internal class ObjectLiteral : Exp {
		
		internal ArrayList elems;
		
		internal ObjectLiteral (ArrayList elems)
		{
			this.elems = elems;
		}

		internal ObjectLiteral (AST parent)
		{
			this.parent = parent;
			elems = new ArrayList ();
		}

		internal override bool Resolve (IdentificationTable context, bool no_effect)
		{
			this.no_effect = no_effect;
			return Resolve (context);
		}

		internal override bool Resolve (IdentificationTable context)
		{
			bool r = true;
			foreach (AST ast in elems)
				r &= ast.Resolve (context);
			return r;
		}

		internal override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;

			ig.Emit (OpCodes.Ldarg_0);
			ig.Emit (OpCodes.Ldfld, typeof (ScriptObject).GetField ("engine"));
			ig.Emit (OpCodes.Call, typeof (Microsoft.JScript.Vsa.VsaEngine).GetMethod ("GetOriginalObjectConstructor"));
			ig.Emit (OpCodes.Call, typeof (ObjectConstructor).GetMethod ("ConstructObject"));

			foreach (ObjectLiteralItem item in elems) {
				ig.Emit (OpCodes.Dup);
				item.Emit (ec);
				ig.Emit (OpCodes.Call, typeof (JSObject).GetMethod ("SetMemberValue2"));
			}
			if (no_effect)
				ig.Emit (OpCodes.Pop);
		}

		internal void Add (ObjectLiteralItem item)
		{
			elems.Add (item);
		}
	}

	internal class ObjectLiteralItem : AST {
		internal string property_name;
		internal AST exp;

		internal ObjectLiteralItem (object obj)
		{
			if (obj != null)
				property_name = obj.ToString ();
		}

		internal override bool Resolve (IdentificationTable context)
		{
			return exp.Resolve (context);
		}

		internal override void Emit (EmitContext ec)
		{
			ec.ig.Emit (OpCodes.Ldstr, property_name);
			exp.Emit (ec);
		}
	}

	internal class PropertyName {
		string name;
		internal string Name {
			get { return name; }
			set { name = value; }
		}
	}

	internal class RegExpLiteral : AST {
		internal string re;
		internal string flags;

		const char IGNORE_CASE = 'i';
		const char GLOBAL = 'g';
		const char MULTI_LINE = 'm';
		
		internal RegExpLiteral (AST parent, string re, string flags)
		{
			this.parent = parent;
			this.re = re;
			this.flags = flags;
		}

		internal override bool Resolve (IdentificationTable context)
		{
			return true;
		}

		internal override void Emit (EmitContext ec)
		{
			ILGenerator ig = ec.ig;
			TypeBuilder type = ec.type_builder;

			FieldBuilder field = type.DefineField (SemanticAnalyser.NextAnonymousRegExpObj, typeof (RegExpObject), FieldAttributes.Public | FieldAttributes.Static);

			Label label = ig.DefineLabel ();

			ig.Emit (OpCodes.Ldsfld, field);
			ig.Emit (OpCodes.Brtrue, label);

			CodeGenerator.load_engine (InFunction, ig);
			
			ig.Emit (OpCodes.Call, typeof (VsaEngine).GetMethod ("GetOriginalRegExpConstructor"));
			ig.Emit (OpCodes.Ldstr, re);

			emit_flag (ig, flags.IndexOfAny (new char [] {IGNORE_CASE}) > -1);
			emit_flag (ig, flags.IndexOfAny (new char [] {GLOBAL}) > -1);
			emit_flag (ig, flags.IndexOfAny (new char [] {MULTI_LINE}) > -1);
			
			ig.Emit (OpCodes.Call, typeof (RegExpConstructor).GetMethod ("Construct"));
			ig.Emit (OpCodes.Castclass, typeof (RegExpObject));
			ig.Emit (OpCodes.Stsfld, field);

			ig.MarkLabel (label);			
			ig.Emit (OpCodes.Ldsfld, field);
		}

		void emit_flag (ILGenerator ig, bool cond)
		{
 			if (cond)
				ig.Emit (OpCodes.Ldc_I4_1);
			else
				ig.Emit (OpCodes.Ldc_I4_0);
		}
	}		
}
