/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-favorite-view.h"

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <math.h>
#include <clutk/clutk.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>
#include <launcher/launcher.h>

#include "nl-window.h"
#include "nl-quit.h"

G_DEFINE_TYPE (NlFavoriteView, nl_favorite_view, CTK_TYPE_ACTOR);

#define NL_FAVORITE_VIEW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE \
        ((obj), NL_TYPE_FAVORITE_VIEW, NlFavoriteViewPrivate))

#define QUIT_SIZE 64

struct _NlFavoriteViewPrivate
{
  LauncherSession   *session;
  LauncherFavorites *favorites;
  NlShell           *shell;
  GSList            *children;
  GSList            *old_children;
  ClutterActor      *view;

  ClutterActor *dragged;
  gfloat        xstart;
  gfloat        ystart;
  gfloat        xdiff;
  gfloat        ydiff;

  gfloat        view_width;
};

enum
{
  PROP_0,
  PROP_SHELL
};

/* GObject stuff */
static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  NlFavoriteViewPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_VIEW (object));
  priv = NL_FAVORITE_VIEW_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      priv->shell = g_value_get_pointer (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
get_property (GObject      *object,
              guint         prop_id,
              GValue       *value,
              GParamSpec   *pspec)
{
  NlFavoriteViewPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_VIEW (object));
  priv = NL_FAVORITE_VIEW_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SHELL:
      g_value_set_pointer (value, priv->shell);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_favorite_view_finalize (GObject *object)
{
  NlFavoriteViewPrivate *priv;

  priv = NL_FAVORITE_VIEW_GET_PRIVATE (object);

  if (priv->children)
    {
      g_slist_foreach (priv->children, (GFunc)clutter_actor_destroy, NULL);
      g_slist_free (priv->children);
      priv->children = NULL;
    }
  if (priv->old_children)
    {
      g_slist_foreach (priv->old_children, (GFunc)clutter_actor_destroy, NULL);
      g_slist_free (priv->old_children);
      priv->old_children = NULL;
    }

  if (priv->favorites)
    {
      g_object_unref (priv->favorites);
      priv->favorites = NULL;
    }

  if (priv->session)
    {
      g_object_unref (priv->session);
      priv->session = NULL;
    }

  G_OBJECT_CLASS (nl_favorite_view_parent_class)->finalize (object);
}

static void
nl_favorite_view_constructed (GObject *object)
{
  if (G_OBJECT_CLASS (nl_favorite_view_parent_class)->constructed)
    G_OBJECT_CLASS (nl_favorite_view_parent_class)->constructed (object);

}

static void
paint (ClutterActor *actor)
{
  NlFavoriteViewPrivate *priv = NL_FAVORITE_VIEW (actor)->priv;
  GSList *c;

  clutter_actor_paint (priv->view);

  for (c = priv->children; c; c = c->next)
    {
      ClutterActor *child = c->data;

      if (child)
        clutter_actor_paint (child);
    }

  for (c = priv->old_children; c; c = c->next)
    {
      ClutterActor *child = c->data;

      if (child)
        clutter_actor_paint (child);
    }

}

static void
pick (ClutterActor *actor, const ClutterColor *color)
{
  NlFavoriteViewPrivate *priv = NL_FAVORITE_VIEW (actor)->priv;
  GSList *c;

  CLUTTER_ACTOR_CLASS (nl_favorite_view_parent_class)->pick (actor, color);

  clutter_actor_paint (priv->view);

  for (c = priv->children; c; c = c->next)
    {
      ClutterActor *child = c->data;

      if (child)
        clutter_actor_paint (child);
    }

}

static void
allocate (ClutterActor          *actor,
          const ClutterActorBox *box,
          ClutterAllocationFlags flags)
{
  NlFavoriteViewPrivate *priv = NL_FAVORITE_VIEW (actor)->priv;
  GSList          *c;
  ClutterActorBox  child_box;
  gfloat           vheight;
  gint             top, bottom, left, right;

  CLUTTER_ACTOR_CLASS (nl_favorite_view_parent_class)->allocate (actor,
      box, flags);

  priv->view_width = box->x2 - box->x1; 

  left = top = right = bottom = 0;
  launcher_session_get_workarea (priv->session, &left, &top, &right, &bottom);

  /* Fixed favorites */
  clutter_actor_get_preferred_height (priv->view,
                                      box->x2 - box->x1,
                                      &vheight,
                                      &vheight);
  child_box.x1 = 0;
  child_box.x2 = box->x2 - box->x1;
  child_box.y1 = 0;
  child_box.y2 = vheight;
  clutter_actor_allocate (priv->view, &child_box, flags);

  /* Floating favorites */
  for (c = priv->children; c; c = c->next)
    {
      ClutterActor *child = c->data;
      gfloat nwidth=0;
      gfloat nheight=0;

      if (!child)
        continue;

      clutter_actor_get_preferred_size (child, NULL, NULL, &nwidth, &nheight);

      child_box.x1 = clutter_actor_get_x (child);
      child_box.x2 = child_box.x1 + nwidth;
      child_box.y1 = clutter_actor_get_y (child);
      child_box.y2 = child_box.y1 + nheight;

      clutter_actor_allocate (child, &child_box, flags);
    }

}

static void
map (ClutterActor *actor)
{
  NlFavoriteViewPrivate *priv = NL_FAVORITE_VIEW (actor)->priv;
  GSList *c;

  CLUTTER_ACTOR_CLASS (nl_favorite_view_parent_class)->map (actor);

  clutter_actor_map (priv->view);

  for (c = priv->children; c; c = c->next)
    {
      ClutterActor *child = c->data;

      if (child)
        clutter_actor_map (child);
    }
  
}

static void
unmap (ClutterActor *actor)
{
  NlFavoriteViewPrivate *priv = NL_FAVORITE_VIEW (actor)->priv;
  GSList *c;

  CLUTTER_ACTOR_CLASS (nl_favorite_view_parent_class)->unmap (actor);

  for (c = priv->children; c; c = c->next)
    {
      ClutterActor *child = c->data;

      if (child)
        clutter_actor_unmap (child);
    }

  clutter_actor_unmap (priv->view);
}

static void
get_preferred_height (ClutterActor *actor,
                      gfloat        for_width,
                      gfloat       *min_height,
                      gfloat       *nat_height)
{
  NlFavoriteViewPrivate *priv = NL_FAVORITE_VIEW (actor)->priv;
  gfloat    max_height = 0;
  GSList   *c;
  
  /* Grab the view's preferred size first */
  clutter_actor_get_preferred_height (priv->view, for_width,
                                      &max_height, &max_height);

  for (c = priv->children; c; c = c->next)
    {
      ClutterActor *child = c->data;
      gfloat nheight=0;
      gfloat ny = 0;

      if (!child)
        continue;

      clutter_actor_get_anchor_point (child, NULL, &ny);
      ny = clutter_actor_get_y (child) - ny;

      clutter_actor_get_preferred_height (child, for_width, &nheight, &nheight);
      
      max_height = MAX (max_height,
                        ny + nheight);
    }
  
  if (min_height)
    *min_height = max_height;
  if (nat_height)
    *nat_height = max_height;
}


static void
change_background (GtkMenuItem *item, gpointer data)
{
#define BG_EXEC "gnome-appearance-properties --show-page=background"

  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    BG_EXEC, NULL);
}

static gboolean
button_press_event (ClutterActor *actor, ClutterButtonEvent *event)
{
  GtkWidget *menu;
  GtkWidget *item;

  if (event->button != 3)
    return FALSE;

  menu = gtk_menu_new ();

  item = gtk_menu_item_new_with_label (_("Change Desktop Background"));
  gtk_widget_show (item);
  g_signal_connect (item, "activate",
                    G_CALLBACK (change_background), NULL);

  gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);

  gtk_menu_popup (GTK_MENU (menu),
                  NULL, NULL,
                  NULL, NULL,
                  3, event->time);

  return TRUE;
}

static void
nl_favorite_view_class_init (NlFavoriteViewClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);
  GParamSpec        *pspec;

  obj_class->finalize     = nl_favorite_view_finalize;
  obj_class->constructed  = nl_favorite_view_constructed;
  obj_class->set_property = set_property;
  obj_class->get_property = get_property;

  act_class->paint = paint;
  act_class->pick = pick;
  act_class->allocate = allocate;
  act_class->map   = map;
  act_class->unmap = unmap;
  act_class->get_preferred_height = get_preferred_height;
  act_class->button_press_event = button_press_event;

  pspec = g_param_spec_pointer ("shell", "shell", "shell",
                                G_PARAM_READWRITE | G_PARAM_CONSTRUCT);
  g_object_class_install_property (obj_class, PROP_SHELL, pspec);

  g_type_class_add_private (obj_class, sizeof (NlFavoriteViewPrivate));
}

static void
nl_favorite_view_init (NlFavoriteView *self)
{
  NlFavoriteViewPrivate *priv;
  ClutterActor          *view, *bg, *bg_texture;
  CtkPadding            vpadding = { 15, 15, 15, 15 };

  priv = self->priv = NL_FAVORITE_VIEW_GET_PRIVATE (self);

  priv->session = launcher_session_get_default ();
  priv->favorites = launcher_favorites_get_default ();
  priv->children = NULL;

  clutter_actor_set_reactive (CLUTTER_ACTOR (self), TRUE);

  bg_texture = clutter_texture_new_from_file (PKGDATADIR"/iconview_normal.png",
                                              NULL);
  bg = nl_texture_frame_new (CLUTTER_TEXTURE (bg_texture), 25, 25, 25, 25);

  view = priv->view = ctk_icon_view_new ();
  ctk_actor_set_padding (CTK_ACTOR (view), &vpadding);
  ctk_actor_set_background_for_state (CTK_ACTOR (view), CTK_STATE_NORMAL, bg);
  clutter_actor_set_parent (view, CLUTTER_ACTOR (self));
  clutter_actor_set_opacity (view, 0);
  clutter_actor_show (view);
}

/*
 * Private methods
 */

static gboolean
on_motion (ClutterActor       *actor,
           ClutterMotionEvent *event,
           NlFavoriteView     *self)
{
  NlFavoriteViewPrivate *priv;
  gfloat newx=0, newy=0;

  g_return_val_if_fail (NL_IS_FAVORITE_VIEW (self), TRUE);
  priv = self->priv;

  clutter_actor_transform_stage_point (CLUTTER_ACTOR (self),
                                       event->x, event->y,
                                       &newx, &newy);
  newx -= priv->xdiff - clutter_actor_get_width (actor)/2.0;
  newy -= priv->ydiff - clutter_actor_get_width (actor)/2.0;

  newx = CLAMP (newx, priv->xdiff,
                priv->view_width - priv->xdiff);

  if (newy < priv->ydiff)
    newy = priv->ydiff;

  clutter_actor_set_position (actor, newx, newy);
  return TRUE;
}

static gboolean
start_return_opacity (ClutterActor *actor)
{
  clutter_actor_animate (actor, CLUTTER_EASE_IN_SINE, 120,
                         "opacity", 255, NULL);

  return FALSE;
}

static void
on_bounce_completed (ClutterAnimation *anim, ClutterActor *actor)
{
  clutter_threads_add_idle ((GSourceFunc)start_return_opacity, actor);
}

static gboolean
on_button_released (ClutterActor       *actor,
                    ClutterButtonEvent *event,
                    NlFavoriteView     *self)
{
  NlFavoriteViewPrivate *priv = NULL;
  gchar *uid = NULL;

  if (NL_IS_FAVORITE_VIEW (self))
    priv = self->priv;

  g_signal_handlers_disconnect_by_func (actor, on_motion, self);
  g_signal_handlers_disconnect_by_func (actor, on_button_released, self);
  clutter_ungrab_pointer ();

  if (priv)
    {
      priv->xdiff = 0;
      priv->ydiff = 0;
    }

  g_object_get (G_OBJECT (actor), "uid", &uid, NULL);
  if (uid && priv)
    {
      launcher_favorites_set_int (priv->favorites, uid, "x",
                                  clutter_actor_get_x (actor));
      launcher_favorites_set_int (priv->favorites, uid, "y",
                                  clutter_actor_get_y (actor));
    }

  clutter_actor_animate (actor, CLUTTER_EASE_OUT_ELASTIC, 420,
                         "scale-x", 1.0,
                         "scale-y", 1.0,
                         "signal::completed", on_bounce_completed, actor,
                         NULL);
  return TRUE;
}

static void
on_dragged (ClutterActor *actor, ClutterMotionEvent *event,NlFavoriteView *self)
{
  NlFavoriteViewPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_VIEW (self));
  priv = self->priv;

  priv->dragged = actor;

  clutter_actor_transform_stage_point (actor,
                                       event->x,
                                       event->y,
                                       &priv->xdiff,
                                       &priv->ydiff);
  clutter_grab_pointer (actor);
  g_signal_connect (actor, "motion-event",
                    G_CALLBACK (on_motion), self);
  g_signal_connect (actor, "button-release-event",
                    G_CALLBACK (on_button_released), self);

  clutter_actor_animate (actor, CLUTTER_EASE_OUT_SINE, 150,
                         "scale-x", 1.2,
                         "scale-y", 1.2,
                         "opacity", 200,
                         NULL);
}

static void
on_fav_removed_real (ClutterAnimation *anim,
                     NlFavoriteView   *self)
{
  NlFavoriteViewPrivate *priv;
  NlFavorite *favorite;
  
  g_return_if_fail (NL_IS_FAVORITE_VIEW (self));
  priv = self->priv;

  favorite = (NlFavorite *)clutter_animation_get_object (anim);

  if (nl_favorite_get_view_type (favorite) == NL_FAVORITE_VIEW_FIXED)
    {
      GList *children;

      nl_favorite_removed (favorite);
      clutter_container_remove_actor (CLUTTER_CONTAINER (priv->view),
                                      CLUTTER_ACTOR (favorite));

      /* If there are no more children, hide the fixed-view */
      children = clutter_container_get_children (CLUTTER_CONTAINER (priv->view));
      clutter_actor_animate (priv->view, CLUTTER_EASE_OUT_SINE, 400,
                             "opacity", g_list_length (children) ? 255 :0,
                             NULL);
      g_list_free (children);
    }
  else
    {
      priv->old_children = g_slist_remove (priv->old_children, favorite);
      nl_favorite_removed (favorite);
      clutter_actor_destroy (CLUTTER_ACTOR (favorite));
    }
}

static void
on_fav_removed (NlFavorite *favorite, NlFavoriteView *self)
{
  NlFavoriteViewPrivate *priv;

  g_return_if_fail (NL_IS_FAVORITE_VIEW (self));
  priv = self->priv;

  if (nl_favorite_get_view_type (favorite) == NL_FAVORITE_VIEW_FIXED)
    {
      clutter_actor_animate (CLUTTER_ACTOR (favorite),
                             CLUTTER_EASE_OUT_SINE, 200,
                             "opacity", 0,
                             "signal::completed", on_fav_removed_real, self,
                             NULL);
    }
  else
    {
      priv->children = g_slist_remove (priv->children, favorite);
      g_signal_handlers_disconnect_by_func (favorite, "begin-drag-move", self);
      priv->old_children = g_slist_append (priv->old_children, favorite);

      clutter_actor_animate (CLUTTER_ACTOR (favorite),
                             CLUTTER_EASE_OUT_SINE, 200,
                             "scale-x", 0.4,
                             "scale-y", 0.4,
                             "opacity", 0,
                             "signal::completed", on_fav_removed_real, self,
                             NULL);
    } 
  
  clutter_actor_set_reactive (CLUTTER_ACTOR (favorite), FALSE);
}

static void
on_fav_event_captured (ClutterActor   *fav,
                       NlFavoriteView *self)
{
  NlFavoriteViewPrivate *priv;
  GSList *c, *new_favs_order = NULL;
  GList  *children, *cc;
  
  g_return_if_fail (NL_IS_FAVORITE_VIEW (self));
  priv = self->priv;

  if (priv->children && g_slist_last (priv->children)->data == fav)
    return;

  priv->children = g_slist_remove (priv->children, fav);
  priv->children = g_slist_append (priv->children, fav);

  children = clutter_container_get_children (CLUTTER_CONTAINER (priv->view));
  for (cc = children; cc; cc = cc->next)
    {
      gchar *uid = NULL;

      g_object_get (cc->data, "uid", &uid, NULL);
      
      if (uid)
        {
          new_favs_order = g_slist_append (new_favs_order, uid);
        }
    }

  for (c = priv->children; c; c = c->next)
    {
      gchar *uid = NULL;

      g_object_get (c->data, "uid", &uid, NULL);

      if (uid)
        {
          new_favs_order = g_slist_append (new_favs_order, uid);
        }
    }
  

  launcher_favorites_set_favorites (priv->favorites, new_favs_order);
  g_slist_foreach (new_favs_order, (GFunc)g_free, NULL);
  g_slist_free (new_favs_order);
  g_list_free (children);

  clutter_actor_queue_redraw (CLUTTER_ACTOR (self));

  return;
}

/*
 * Public methods
 */
ClutterActor *
nl_favorite_view_new (NlShell *shell)

{
  ClutterActor *favorite_view;

  favorite_view = g_object_new (NL_TYPE_FAVORITE_VIEW,
                                "shell", shell,
                                NULL);

  return favorite_view;
}

void
nl_favorite_view_add_favorite (NlFavoriteView    *self,
                               NlFavorite        *favorite)
{
  NlFavoriteViewPrivate *priv;
  ClutterActor *fav_actor;
  gchar        *uid = NULL;
  gint          x, y;

  g_return_if_fail (NL_IS_FAVORITE_VIEW (self));
  g_return_if_fail (NL_IS_FAVORITE (favorite));
  priv = self->priv;

  fav_actor = CLUTTER_ACTOR (favorite);

  if (nl_favorite_get_view_type (favorite) == NL_FAVORITE_VIEW_FIXED)
    {
      clutter_container_add_actor (CLUTTER_CONTAINER (priv->view),
                                   fav_actor);
      g_signal_connect (fav_actor, "remove-me",
                        G_CALLBACK (on_fav_removed), self);

      if (clutter_actor_get_opacity (priv->view) == 0)
        clutter_actor_animate (priv->view, CLUTTER_EASE_IN_SINE, 400,
                               "opacity", 255, NULL);
      return;
    }

  clutter_actor_set_parent (fav_actor, CLUTTER_ACTOR (self));
  clutter_actor_set_opacity (fav_actor, 0);
  clutter_actor_set_anchor_point_from_gravity (fav_actor,
      CLUTTER_GRAVITY_CENTER);
  
  clutter_actor_show (fav_actor);
  g_signal_connect (fav_actor, "begin-drag-move",
                    G_CALLBACK (on_dragged), self);
  g_signal_connect (fav_actor, "remove-me",
                    G_CALLBACK (on_fav_removed), self);
  g_signal_connect (fav_actor, "active",
                    G_CALLBACK (on_fav_event_captured), self);

  g_object_get (G_OBJECT (favorite), "uid", &uid, NULL);
  if (!uid)
    {
      clutter_actor_unparent (fav_actor);
      g_warning ("Favorite does not have an associated uid :%s",
                 nl_favorite_get_name (favorite));
      return;
    }

  priv->children = g_slist_append (priv->children, favorite);
  x = launcher_favorites_get_int (priv->favorites, uid, "x");
  y = launcher_favorites_get_int (priv->favorites, uid, "y");

  if (x || y)
    {
      clutter_actor_set_position (fav_actor, x, y);
    }
  else
    {
      clutter_actor_set_position (fav_actor, 200, 200);
    }

  clutter_actor_animate (fav_actor, CLUTTER_EASE_IN_SINE, 400,
                         "opacity", 255, NULL);

  g_free (uid);
}

void
nl_favorite_view_remove_favorite (NlFavoriteView *self,
                                  NlFavorite     *favorite)
{

}
