/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-window.h"

#include <glib.h>
#include <glib/gi18n.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <launcher/launcher.h>
#include <netbook-launcher/netbook-launcher.h>
#include <clutter-gtk/clutter-gtk.h>
#include <clutk/clutk.h>

#include "nl-background.h"
#include "nl-menu-items.h"
#include "nl-places-manager.h"
#include "nl-sidebar-scrollbutton.h"
#include "nl-wm.h"

static void nl_shell_iface_init (NlShellIface *iface);

G_DEFINE_TYPE_WITH_CODE (NlWindow,
                         nl_window,
                         GTK_TYPE_WINDOW,
                         G_IMPLEMENT_INTERFACE (NL_TYPE_SHELL,
                                                nl_shell_iface_init));

#define NL_WINDOW_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_WINDOW, \
  NlWindowPrivate))

static ClutterActor *normal_texture = NULL;

struct _NlWindowPrivate
{
  GConfClient  *client;
  GtkWidget    *gtkclutter;
  ClutterActor *stage;

  LauncherSession *session;
  NlWm            *wm;

  ClutterActor *background;
  ClutterActor *toplevel;
  ClutterActor *hbox;
  ClutterActor *sidebar_scroll;
  ClutterActor *sidebar;
  ClutterActor *content;
  
  NlPlacesManager *places;

  GSList *fav_loaders;

  /* Drag favorites */
  GObject      *origin;
  ClutterActor *clone;
  NlShellGetFavoriteDataFunc data_func;
  NlShellDragFinishedFunc finished_func;
  gint startx;
  gint starty;

  GtkWindow *transient_window;
};

/* Forwards */
static gboolean on_delete_event   (GtkWidget      *widget,
                                   GdkEvent       *event);
static void     ensure_layout     (NlWindow *window);
static void     on_config_changed (NlConfig *cfg, NlWindow *window);

/* Iface impl */
ClutterActor  * nl_window_get_stage             (NlShell *shell);
NlContentView * nl_window_get_content_view      (NlShell *shell);
NlSidebar     * nl_window_get_sidebar           (NlShell *shell);
GSList        * nl_window_get_favorites_loaders (NlShell*);
void            nl_window_add_places_source     (NlShell *shell,
                                                ClutterActor  *src);
void            nl_window_add_fav_loader        (NlShell *shell,
                                                 NlFavoritesLoader *l);
void            nl_window_show_favorites	(NlShell *shell);                                                 
GtkWidget     * nl_window_get_toplevel          (NlShell *shell);

void            nl_window_begin_favorite_drag   (NlShell           *shell,
                                                 GObject          *origin,
                                                 ClutterActor      *clone,
                                                 NlShellGetFavoriteDataFunc dfunc,
                                                 NlShellDragFinishedFunc ffunc);

static void hide_launcher (NlWm *wm, WnckWindow *window, NlWindow *self);
static void show_launcher (NlWm *wm, NlWindow *self);
static void on_sidebar_scroll_allocate (ClutterActor *actor,
                                        ClutterActor *null,
                                        ClutterActor *scroll);


/* GObject stuff */
static void
nl_window_finalize (GObject *object)
{
  NlWindowPrivate *priv;

  priv = NL_WINDOW_GET_PRIVATE (object);

  if (priv->client)
    {
      g_object_unref (priv->client);
      priv->client = NULL;
    }
  if (priv->session)
    {
      g_object_unref (priv->session);
      priv->session = NULL;
    }
  if (priv->fav_loaders)
    {
      g_slist_foreach (priv->fav_loaders, (GFunc)g_object_unref, NULL);
      g_slist_free (priv->fav_loaders);
      priv->fav_loaders = NULL;
    }
  if (priv->wm)
    {
      g_object_unref (priv->wm);
      priv->wm = NULL;
    }

  if (G_IS_OBJECT (priv->toplevel))
    {
      g_object_unref (priv->toplevel);
      priv->toplevel = NULL;
    }

  G_OBJECT_CLASS (nl_window_parent_class)->finalize (object);
}

static void
nl_window_constructed (GObject *object)
{
  NlWindowPrivate *priv = NL_WINDOW (object)->priv;
  GtkWindow       *window = GTK_WINDOW (object);
  NlConfig        *cfg = nl_config_get_default ();
  ClutterActor    *vbox, *arrow, *bg;
  extern gboolean  windowed;

  priv->client = gconf_client_get_default ();

  if (windowed)
    {
      gtk_window_set_decorated (window, TRUE);
      gtk_window_set_skip_taskbar_hint (window, FALSE);
      gtk_window_set_skip_pager_hint (window, FALSE);
      gtk_window_set_type_hint (window, GDK_WINDOW_TYPE_HINT_NORMAL);
    }
  else
    {
      gtk_window_set_decorated (window, FALSE);
      gtk_window_set_skip_taskbar_hint (window, TRUE);
      gtk_window_set_skip_pager_hint (window, TRUE);
      gtk_window_set_type_hint (window, GDK_WINDOW_TYPE_HINT_DESKTOP);
    }

  gtk_window_resize (window, cfg->monitor_width, cfg->monitor_height);
  gtk_widget_realize (GTK_WIDGET (window));
  g_signal_connect (cfg, "config-changed",
                    G_CALLBACK (on_config_changed), object);

  priv->gtkclutter = gtk_clutter_embed_new ();
  gtk_container_add (GTK_CONTAINER (window), priv->gtkclutter);
  gtk_widget_realize (GTK_WIDGET (priv->gtkclutter));

  priv->stage = gtk_clutter_embed_get_stage
                (GTK_CLUTTER_EMBED (priv->gtkclutter));

  /* Set NlWidget's direction */
  ctk_actor_set_default_direction (
    gtk_widget_get_default_direction ());

  /* Background first */
  priv->background = nl_background_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->stage),
                               priv->background);
  clutter_actor_set_size (priv->background,
                          cfg->monitor_width, cfg->monitor_height);
  clutter_actor_set_position (priv->background, 0, 0);
  clutter_actor_set_opacity (priv->background, 0);
  clutter_actor_show (priv->background);
  
  gtk_widget_show_all (GTK_WIDGET (window));
  clutter_actor_show_all (priv->stage);

  clutter_actor_animate (priv->background, CLUTTER_EASE_OUT_SINE, 500,
                         "opacity", 255, NULL);

  /* Focus bin (handles the focus) */
  priv->toplevel = ctk_toplevel_get_default_for_stage (priv->stage);
  clutter_actor_set_position (priv->toplevel, 0, 0);
  clutter_actor_set_size (priv->toplevel,
                          cfg->monitor_width,
                          cfg->monitor_height);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->stage),
                               priv->toplevel);
  clutter_actor_show (priv->toplevel);

  /* Main hbox */
  priv->hbox = ctk_hbox_new (0);
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->toplevel),
                               priv->hbox);
  clutter_actor_show (priv->hbox);

  /* Sidebar vbox */
  vbox = ctk_vbox_new (0);
  ctk_box_pack (CTK_BOX (priv->hbox), vbox, FALSE, FALSE);
  clutter_actor_show (vbox);

  priv->sidebar_scroll = ctk_scroll_view_new ();

  arrow = nl_sidebar_scrollbutton_new (NL_ARROW_UP,
                                       CTK_SCROLL_VIEW (priv->sidebar_scroll));
  ctk_box_pack (CTK_BOX (vbox), arrow, FALSE, FALSE);
  clutter_actor_show (arrow);

  /* Sidebar */
  if (!CLUTTER_IS_ACTOR (normal_texture))
    {
      normal_texture = clutter_texture_new_from_file
                       (PKGDATADIR"/sidebar_bg.png", NULL);
    }
  bg = nl_texture_frame_new (CLUTTER_TEXTURE (normal_texture),
                             0, 30, 0, 0);
  ctk_actor_set_background (CTK_ACTOR (priv->sidebar_scroll), bg);
  ctk_scroll_view_set_bar_type (CTK_SCROLL_VIEW (priv->sidebar_scroll),
                                CTK_SCROLLBAR_HIDDEN);
  ctk_box_pack (CTK_BOX (vbox), priv->sidebar_scroll, TRUE, TRUE);
  clutter_actor_show (priv->sidebar_scroll);

  priv->sidebar = nl_sidebar_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->sidebar_scroll),
                               priv->sidebar);
  clutter_actor_show (priv->sidebar);
  g_signal_connect (priv->sidebar, "actor-added",
                    G_CALLBACK (on_sidebar_scroll_allocate),
                    priv->sidebar_scroll);
  g_signal_connect (priv->sidebar, "actor-removed",
                    G_CALLBACK (on_sidebar_scroll_allocate),
                    priv->sidebar_scroll);

  arrow = nl_sidebar_scrollbutton_new (NL_ARROW_DOWN,
                                       CTK_SCROLL_VIEW (priv->sidebar_scroll));
  ctk_box_pack (CTK_BOX (vbox), arrow, FALSE, FALSE);
  clutter_actor_show (arrow);

  /* Content view */
  priv->content = nl_content_view_new ();
  ctk_box_pack (CTK_BOX (priv->hbox), priv->content, TRUE, TRUE);
  clutter_actor_show (priv->content);

  /* Startup Notification */
  clutter_container_add_actor (CLUTTER_CONTAINER (priv->stage),
                               CLUTTER_ACTOR (nl_notify_get_default ()));
  clutter_actor_set_position (CLUTTER_ACTOR (nl_notify_get_default ()), 0, 0);
  clutter_actor_set_size (CLUTTER_ACTOR (nl_notify_get_default ()),
                          cfg->monitor_width, cfg->monitor_height);
  clutter_actor_show (CLUTTER_ACTOR (nl_notify_get_default ()));

  /* Places */
  priv->places = nl_places_manager_new (NL_SHELL (object));

  /* If we're windowed, a delete event should close the nl window, else
   * it should do nothing
   */
  if (G_UNLIKELY (windowed))
    g_signal_connect (window, "delete-event",
                      G_CALLBACK (gtk_main_quit), NULL);
  else
    g_signal_connect (window, "delete-event",
                      G_CALLBACK (on_delete_event), NULL);

  priv->wm = nl_wm_get_default ();
  g_signal_connect (priv->wm, "show-windec", G_CALLBACK (hide_launcher),object);
  g_signal_connect (priv->wm, "hide-windec", G_CALLBACK (show_launcher),object);

  ensure_layout (NL_WINDOW (window));

  ctk_focusable_set_focused (CTK_FOCUSABLE (priv->toplevel), TRUE);
}

static void
nl_shell_iface_init (NlShellIface *iface)
{
  iface->get_stage             = nl_window_get_stage;
  iface->get_content_view      = nl_window_get_content_view;
  iface->get_sidebar           = nl_window_get_sidebar;
  iface->get_favorites_loaders = nl_window_get_favorites_loaders;
  iface->show_favorites        = nl_window_show_favorites;
  iface->add_places_source     = nl_window_add_places_source;
  iface->add_favorites_loader  = nl_window_add_fav_loader;
  iface->get_toplevel          = nl_window_get_toplevel;
  iface->begin_favorite_drag   = nl_window_begin_favorite_drag;
}


static void
nl_window_class_init (NlWindowClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = nl_window_finalize;
  obj_class->constructed = nl_window_constructed;

  g_type_class_add_private (obj_class, sizeof (NlWindowPrivate));
}

static void
nl_window_init (NlWindow *window)
{
  NlWindowPrivate *priv;

  priv = window->priv = NL_WINDOW_GET_PRIVATE (window);

  priv->session = launcher_session_get_default ();
  g_signal_connect_swapped (priv->session, "workarea-changed",
                            G_CALLBACK (ensure_layout), window);
}

/*
 * Public methods
 */
GtkWidget *
nl_window_get_default (void)

{
  static GtkWidget *window = NULL;

  if (!window)
    window = g_object_new (NL_TYPE_WINDOW,
                           "icon-name", "distributor-logo",
                           "title", _("Home"),
                           "type", GTK_WINDOW_TOPLEVEL,
                           NULL);

  return window;
}

/*
 * Private methods
 */
static gboolean
on_delete_event (GtkWidget *widget, GdkEvent *event)
{
  return TRUE;
}

static void
ensure_layout (NlWindow *window)
{
  NlWindowPrivate *priv;
  NlConfig        *cfg = nl_config_get_default ();
  gint                   left, top, right, bottom;
  CtkPadding        padding;

  g_return_if_fail (NL_IS_WINDOW (window));
  priv = window->priv;

  left = top = right = bottom = 0;
  launcher_session_get_workarea (priv->session, &left, &top, &right, &bottom);

  padding.left = left;
  padding.top = top;
  padding.right = right;
  padding.bottom = bottom;

  clutter_actor_set_position (priv->toplevel, 0, 0);
  clutter_actor_set_size (priv->toplevel,
                          cfg->monitor_width, cfg->monitor_height);
  ctk_actor_set_padding (CTK_ACTOR (priv->hbox), &padding);
  clutter_actor_queue_relayout (priv->stage);
  clutter_actor_queue_redraw (priv->stage);
}

static void
on_config_changed (NlConfig *cfg, NlWindow *window)
{
  NlWindowPrivate *priv;

  g_return_if_fail (NL_IS_WINDOW (window));
  priv = window->priv;

  gtk_window_resize (GTK_WINDOW (window),
                     cfg->monitor_width,
                     cfg->monitor_height);
  ensure_layout (window);

  /* Background */
  
  clutter_actor_set_size (priv->background,
                          cfg->monitor_width, cfg->monitor_height);
  clutter_actor_set_position (priv->background, 0, 0);

  nl_background_ensure_layout (NL_BACKGROUND (priv->background));
}

/* Iface impl */
ClutterActor *
nl_window_get_stage (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_WINDOW (shell), NULL);
  return NL_WINDOW (shell)->priv->stage;
}

NlContentView *
nl_window_get_content_view (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_WINDOW (shell), NULL);
  return NL_CONTENT_VIEW (NL_WINDOW (shell)->priv->content);
}

NlSidebar *
nl_window_get_sidebar (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_WINDOW (shell), NULL);
  return NL_SIDEBAR (NL_WINDOW (shell)->priv->sidebar);
}

GSList *
nl_window_get_favorites_loaders (NlShell *shell)
{
  g_return_val_if_fail (NL_IS_WINDOW (shell), NULL);
  return NL_WINDOW (shell)->priv->fav_loaders;
}

void
nl_window_add_places_source (NlShell *shell,
                             ClutterActor  *source)
{
  NlWindowPrivate *priv;

  g_return_if_fail (NL_IS_WINDOW (shell));
  priv = NL_WINDOW (shell)->priv;

  nl_places_manager_append_source (priv->places, source);
}

void
nl_window_add_fav_loader (NlShell *shell,
                          NlFavoritesLoader *loader)
{
  NlWindowPrivate *priv;

  g_return_if_fail (NL_IS_WINDOW (shell));
  priv = NL_WINDOW (shell)->priv;

  g_object_ref (loader);

  priv->fav_loaders = g_slist_append (priv->fav_loaders, loader);
}

GtkWidget *
nl_window_get_toplevel (NlShell *shell)
{
  NlWindowPrivate *priv;

  g_return_val_if_fail (NL_IS_WINDOW (shell), NULL);
  priv = NL_WINDOW (shell)->priv;

  return GTK_WIDGET (shell);
}

static gboolean on_drag (ClutterActor       *stage,
                         ClutterMotionEvent *event,
                         NlWindow           *self);

static gboolean
on_drop (ClutterActor       *stage,
         ClutterButtonEvent *event,
         NlWindow  *self)
{
  NlWindowPrivate *priv;
  ClutterActor *dest = NULL;
  GList *children, *c;

  g_return_val_if_fail (NL_IS_WINDOW (self), FALSE);
  priv = self->priv;

  if (!priv->clone)
    {
      g_debug (G_STRLOC " - No context");
      return FALSE;
    }

  clutter_actor_set_position (priv->clone, event->x, event->y);

  dest = clutter_stage_get_actor_at_pos (CLUTTER_STAGE (stage), TRUE,
                                         event->x, event->y);

  children = clutter_container_get_children (CLUTTER_CONTAINER (priv->sidebar));
  if (children->data && dest == children->data)
    {
      gchar *uid = priv->data_func (priv->origin);

      if (uid)
        {

          clutter_actor_animate (priv->clone, CLUTTER_EASE_OUT_SINE, 120,
                                 "scale-x", 0.1, "scale-y", 0.1,
                                 "opacity", 0,
                                 NULL);
          nl_shell_add_favorite (NL_SHELL (self), uid);
        }
      else
        {
          clutter_actor_animate (priv->clone, CLUTTER_EASE_OUT_SINE, 120,
                                 "opacity", 0,
                                 NULL);
        }
    }
  else
    {
      clutter_actor_animate (priv->clone, CLUTTER_EASE_OUT_SINE, 120,
                             "opacity", 0,
                             NULL);
    }

  if (priv->finished_func)
    priv->finished_func (priv->origin);

  priv->clone = NULL;
  clutter_ungrab_pointer ();
  g_signal_handlers_disconnect_by_func (stage, on_drag, self);
  g_signal_handlers_disconnect_by_func (stage, on_drop, self);

  priv->origin = NULL;
  priv->clone = NULL;
  priv->data_func = NULL;
  priv->finished_func = NULL;

  for (c = children; c; c = c->next)
    {
      ClutterActor *actor = c->data;

      clutter_actor_animate (actor, CLUTTER_EASE_OUT_SINE, 250,
                             "opacity", 255, NULL);
    }
  g_list_free (children);

  return TRUE;
}

static gboolean
on_drag (ClutterActor       *stage,
         ClutterMotionEvent *event,
         NlWindow           *self)
{
  NlWindowPrivate *priv;

  g_return_val_if_fail (NL_IS_WINDOW (self), FALSE);
  priv = self->priv;

  if (!priv->clone)
    {
      return FALSE;
    }

  if (event->modifier_state & CLUTTER_BUTTON1_MASK)
    {
      clutter_actor_set_position (priv->clone, event->x, event->y);
    }
  else
    {
      GList *children, *c;

      clutter_actor_destroy (priv->clone);
      priv->clone = NULL;
      clutter_ungrab_pointer ();
      g_signal_handlers_disconnect_by_func (stage, on_drag, self);
      g_signal_handlers_disconnect_by_func (stage, on_drop, self);

      if (priv->finished_func)
        priv->finished_func (priv->origin);

      priv->origin = NULL;
      priv->clone = NULL;
      priv->data_func = NULL;
      priv->finished_func = NULL;

      children = clutter_container_get_children(CLUTTER_CONTAINER(priv->sidebar));
      for (c = children; c; c = c->next)
        {
          ClutterActor *actor = c->data;

          clutter_actor_animate (actor, CLUTTER_EASE_OUT_SINE, 250,
                                 "opacity", 255, NULL);
        }
      g_list_free (children);
      //g_signal_emit (self, _self_signals[DRAG_FINISHED], 0);
    }

  clutter_actor_queue_redraw (CLUTTER_ACTOR (priv->stage));

  return TRUE;
}

void
nl_window_begin_favorite_drag   (NlShell           *shell,
                                 GObject           *origin,
                                 ClutterActor      *clone,
                                 NlShellGetFavoriteDataFunc dfunc,
                                 NlShellDragFinishedFunc ffunc)
{
  NlWindowPrivate *priv = NL_WINDOW (shell)->priv;
  GList *children, *c;

  priv->origin = origin;
  priv->clone = clone;
  priv->data_func = dfunc;
  priv->finished_func = ffunc;

  priv->startx = clutter_actor_get_x (clone);
  priv->starty = clutter_actor_get_y (clone);

  clutter_grab_pointer (priv->stage);
  g_signal_connect (priv->stage, "motion-event",
                    G_CALLBACK (on_drag), shell);
  g_signal_connect (priv->stage, "button-release-event",
                    G_CALLBACK (on_drop), shell);

  ctk_scroll_view_set_value (CTK_SCROLL_VIEW (priv->sidebar_scroll), 0.0);

  children = clutter_container_get_children(CLUTTER_CONTAINER(priv->sidebar));
  for (c = children; c; c = c->next)
    {
      ClutterActor *actor = c->data;

      if (c->prev)
        clutter_actor_animate (actor, CLUTTER_EASE_OUT_SINE, 250,
                               "opacity", 100, NULL);
    }
  g_list_free (children);
}


/*
 * Private Methods
 */
static gboolean
eat_event (ClutterActor *stage, ClutterEvent *event, NlWindow *self)
{
  gint opacity = clutter_actor_get_opacity (self->priv->toplevel);

  if (opacity == 255)
    {
      g_signal_handlers_disconnect_by_func (self->priv->stage, eat_event, self);
      clutter_ungrab_pointer ();
      return FALSE;
    }


  return TRUE;
}

static void
ungrab (ClutterAnimation *anim, NlWindow *self)
{
  clutter_ungrab_pointer ();
  g_signal_handlers_disconnect_by_func (self->priv->stage,
                                        eat_event, self);
}

static gboolean
show_transient (GtkWindow *window)
{
  if (GTK_IS_WINDOW (window))
    gtk_window_present (GTK_WINDOW (window));

  return FALSE;
}

static void
show_launcher (NlWm *wm, NlWindow *self)
{
  NlWindowPrivate *priv;
  NlConfig *cfg = nl_config_get_default ();

  g_return_if_fail (NL_IS_WINDOW (self));
  priv = self->priv;

  if (cfg->low_graphics)
    {
      clutter_actor_set_opacity (CLUTTER_ACTOR (priv->toplevel), 255);
      ungrab (NULL, self);
    }
  else
    {
      clutter_actor_animate (priv->toplevel, CLUTTER_EASE_OUT_SINE, 300,
                             "opacity", 255,
                             "signal::completed", ungrab, self,
                             NULL);
    }

  if (GTK_IS_WINDOW (priv->transient_window))
    {
      g_idle_add ((GSourceFunc)show_transient, priv->transient_window);
      gtk_window_present (GTK_WINDOW (priv->transient_window));
    }

}

void
nl_window_show_favorites (NlShell *shell)
{
  g_return_if_fail (NL_IS_WINDOW (shell));
  
  GList           *children;
  NlWindowPrivate *priv;
  
  priv = NL_WINDOW (shell)->priv;
  
  children = clutter_container_get_children (CLUTTER_CONTAINER (priv->sidebar));
  if (children->data && CLUTTER_IS_ACTOR (children->data))
    {
      clutter_actor_grab_key_focus (children->data);
      g_signal_emit_by_name (children->data, "clicked");
    }
  g_list_free (children);

  ctk_scroll_view_set_value (CTK_SCROLL_VIEW (priv->sidebar_scroll), 0.0);
  
  wnck_screen_toggle_showing_desktop (wnck_screen_get_default (), TRUE);
}

static void
hide_launcher (NlWm *wm, WnckWindow *window, NlWindow *self)
{
  NlWindowPrivate *priv;
  NlConfig        *cfg = nl_config_get_default ();
  extern gboolean  windowed;

  g_return_if_fail (NL_IS_WINDOW (self));
  priv = self->priv;

  if (windowed)
    return;

  if (cfg->low_graphics)
    clutter_actor_set_opacity (CLUTTER_ACTOR (priv->toplevel), 50);
  else
    {
      ClutterAnimation *anim;

      anim = clutter_actor_animate (priv->toplevel, CLUTTER_EASE_OUT_SINE, 300,
                                    "opacity", 50,
                                    NULL);
    }

  

  clutter_grab_pointer (priv->stage);
  g_signal_connect (priv->stage,
                    "captured-event", G_CALLBACK (eat_event), self);
}

static void
on_sidebar_scroll_allocate (ClutterActor *actor,
                            ClutterActor *null,
                            ClutterActor *scroll)
{
  LauncherSession *session = launcher_session_get_default ();
  NlConfig        *cfg = nl_config_get_default ();
  gint             left, right, top, bottom;
  gint             scroll_height;
  gint             box_height;
    
  left = top = right = bottom = 0;
  launcher_session_get_workarea (session, &left, &top, &right, &bottom);

  scroll_height = cfg->monitor_height - top - bottom - 24 -24;
  box_height = clutter_actor_get_height (actor);

  if (box_height < scroll_height)
    {
      clutter_actor_set_clip (ctk_actor_get_background (CTK_ACTOR (scroll)),
                              0,
                              box_height,
                              clutter_actor_get_width (actor),
                              scroll_height - box_height);
    }
  else
    {
      clutter_actor_set_clip (ctk_actor_get_background (CTK_ACTOR (scroll)),
                              0, 0, 0, 0);
    }
  clutter_actor_set_opacity (ctk_actor_get_background (CTK_ACTOR (scroll)),
                             200);

  g_object_unref (session);
}

void
nl_window_set_transient (NlWindow *self, GtkWindow *window)
{
  g_return_if_fail (NL_IS_WINDOW (self));
  
  self->priv->transient_window = window;
}
