/*
 *  Copyright (c) 2008-2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "BinaryExpression.h"

#include <llvm/Instructions.h>

#include <GTLCore/Debug.h>
#include <GTLCore/LLVMBackend/CodeGenerator_p.h>
#include <GTLCore/LLVMBackend/ExpressionResult_p.h>
#include <GTLCore/Macros_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/VariableNG_p.h>
#include <GTLCore/LLVMBackend/Visitor_p.h>

#include "AccessorExpression.h"
#include "Expression.h"
#include "../LLVMBackend/ExpressionGenerationContext_p.h"
#include "GenerationVisitor.h"

using namespace GTLCore::AST;

BinaryExpression::~BinaryExpression()
{
  delete m_lhs;
  delete m_rhs;
}


const GTLCore::Type* BinaryExpression::type() const
{
  const GTLCore::Type* type1 = m_lhs->type();
  const GTLCore::Type* type2 = m_rhs->type();
  GTL_ASSERT( type1 == type2 );
  UNUSED(type2);
  return type1;
}

void BinaryExpression::markAsReturnExpression()
{
  m_lhs->markAsReturnExpression();
  m_rhs->markAsReturnExpression();
}

const GTLCore::Type* OrBinaryExpression::type() const
{
  return Type::Boolean;
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult OrBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createOrExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

ExpressionResultSP OrBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateOrExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- AndBinaryExpression -------

const GTLCore::Type* AndBinaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult AndBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createAndExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc));
}

ExpressionResultSP AndBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateAndExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult BitOrBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createBitOrExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP BitOrBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateBitOrExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult BitXorBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createBitXorExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP BitXorBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateBitXorExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult BitAndBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createBitAndExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP BitAndBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateBitAndExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

const GTLCore::Type* EqualEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult EqualEqualBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createEqualExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP EqualEqualBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateEqualExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

const GTLCore::Type* DifferentBinaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult DifferentBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createDifferentExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP DifferentBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateDifferentExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

const GTLCore::Type* InferiorEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult InferiorEqualBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createInferiorOrEqualExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP InferiorEqualBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateInferiorEqualExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

const GTLCore::Type* InferiorBinaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult InferiorBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createStrictInferiorExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP InferiorBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateInferiorExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

const GTLCore::Type* SupperiorEqualBinaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult SupperiorEqualBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createSupperiorOrEqualExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP SupperiorEqualBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateSupperiorEqualExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

const GTLCore::Type* SupperiorBinaryExpression::type() const
{
  return Type::Boolean;
}

LLVMBackend::ExpressionResult SupperiorBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createStrictSupperiorExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP SupperiorBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateSupperiorExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult RightShiftBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createRightShiftExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

ExpressionResultSP RightShiftBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateRightShiftExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult LeftShiftBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createLeftShiftExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

ExpressionResultSP LeftShiftBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateLeftShiftExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult AdditionBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createAdditionExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

ExpressionResultSP AdditionBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateAdditionExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult SubtractionBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createSubstractionExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc));
}

ExpressionResultSP SubtractionBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateSubtractionExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult MultiplicationBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createMultiplicationExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

ExpressionResultSP MultiplicationBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateMultiplicationExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult DivisionBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createDivisionExpression(_egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), rightHandSide()->generateValue( _gc, _egc) );
}

ExpressionResultSP DivisionBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateDivisionExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

LLVMBackend::ExpressionResult ModuloBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  return _gc.codeGenerator()->createModuloExpression( _egc.currentBasicBlock(), leftHandSide()->generateValue( _gc, _egc), leftHandSide()->type(), rightHandSide()->generateValue( _gc, _egc), rightHandSide()->type() );
}

ExpressionResultSP ModuloBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  return _generationVisitor->generateModuloExpresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
}

//------- OrBinaryExpression -------

AssignementBinaryExpression::AssignementBinaryExpression( AccessorExpression* lhs, Expression* rhs ) : BinaryExpression( lhs, rhs ), m_lhs(lhs)
{
  GTL_ASSERT(not lhs->isConstant() );
}

llvm::BasicBlock* AssignementBinaryExpression::generateStatement( LLVMBackend::GenerationContext& _gc, llvm::BasicBlock* _bb) const
{
  GTL_ASSERT( m_lhs);
  GTL_DEBUG( *m_lhs->type() );
  GTL_DEBUG( *rightHandSide()->type() );
  LLVMBackend::ExpressionGenerationContext egc(_bb);
  LLVMBackend::ExpressionResult rhsValue = rightHandSide()->generateValue( _gc, egc);
  return m_lhs->affect( _gc, egc, rhsValue );
}

LLVMBackend::ExpressionResult AssignementBinaryExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  GTL_ASSERT( m_lhs);
  llvm::BasicBlock* bbr = generateStatement( _gc, _egc.currentBasicBlock() );
  _egc.setCurrentBasicBlock( bbr );
  return LLVMBackend::ExpressionResult( m_lhs->generateValue( _gc, _egc) );
}

ExpressionResultSP AssignementBinaryExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  GTL_ABORT("TODO");
//   return _generationVisitor->generate Expresion( leftHandSide()->generateValue(_generationVisitor), rightHandSide()->generateValue(_generationVisitor), annotation() );
  return 0;
}
