/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENGTL_RGB_COLOR_CONVERTER_P_H_
#define _OPENGTL_RGB_COLOR_CONVERTER_P_H_
#include <iostream>
#include "AbstractColorConverter.h"
#include "StdTypes.h"
#include "RgbaF.h"

#include "ChannelMaths.h"

namespace GTLCore {
  template<typename _T_>
  struct RgbColorConverterTraits {};
  
#define GTLCORE_MAKE_COLOR_CONVERTER_TRAIT(_T_)                               \
  template<>                                                                  \
  struct RgbColorConverterTraits<_T_> {                                       \
    typedef GTLCore::GammaToLinearFloat<_T_> GammaToLinearFloat;              \
    typedef GTLCore::IntegerToFloat<_T_>     IntegerToFloat;                  \
    typedef GTLCore::LinearToGammaFloat<_T_> LinearToGammaFloat;              \
    typedef GTLCore::FloatToInteger<_T_>     FloatToInteger;                  \
    typedef FullLut< CombinedOperation<float, _T_, IntegerToFloat, GammaToLinearFloat>, float, _T_ > PixelToRgb;             \
    typedef Lut< CombinedOperation<_T_, float, LinearToGammaFloat, FloatToInteger  >, _T_, float> RgbToPixel;                 \
                                                                              \
    static PixelToRgb createPixelToRgbA(double _gamma)                        \
    {                                                                         \
      return PixelToRgb( CombinedOperation<float, _T_, IntegerToFloat, GammaToLinearFloat>( IntegerToFloat(), GammaToLinearFloat(_gamma)) );                                          \
    }                                                                         \
    static RgbToPixel createRgbAToPixel(double _gamma)                        \
    {                                                                         \
      return RgbToPixel( 0.0, 1.0, CombinedOperation<_T_, float, LinearToGammaFloat, FloatToInteger>( LinearToGammaFloat(_gamma), FloatToInteger() ) );             \
    }                                                                         \
  };
  
  GTLCORE_MAKE_COLOR_CONVERTER_TRAIT(gtl_uint8)
  GTLCORE_MAKE_COLOR_CONVERTER_TRAIT(gtl_uint16)
#undef GTLCORE_MAKE_COLOR_CONVERTER_TRAIT

  #define GTLCORE_MAKE_COLOR_CONVERTER_TRAIT(_T_)                             \
  template<>                                                                  \
  struct RgbColorConverterTraits<_T_> {                                       \
    typedef GTLCore::GammaToLinearFloat<_T_> GammaToLinearFloat;              \
    typedef GTLCore::LinearToGammaFloat<_T_> LinearToGammaFloat;              \
    typedef Lut< GammaToLinearFloat, float, _T_ > PixelToRgb;                 \
    typedef Lut< LinearToGammaFloat, _T_, float> RgbToPixel;                  \
                                                                              \
    static PixelToRgb createPixelToRgbA(double _gamma)                        \
    {                                                                         \
      return PixelToRgb( 0.0, 1.0, GammaToLinearFloat(_gamma) );              \
    }                                                                         \
    static RgbToPixel createRgbAToPixel(double _gamma)                        \
    {                                                                         \
      return RgbToPixel( 0.0, 1.0, LinearToGammaFloat(_gamma) );              \
    }                                                                         \
  };

  GTLCORE_MAKE_COLOR_CONVERTER_TRAIT(float)
#undef GTLCORE_MAKE_COLOR_CONVERTER_TRAIT

  template<typename _Channel_Type_, bool _has_alpha>
  class GTLCORE_EXPORT RgbColorConverter : public AbstractColorConverter
  {
  private:
    typedef GTLCore::GammaToLinearFloat<float> GammaFloatToLinearFloat;
    typedef GTLCore::LinearToGammaFloat<float> LinearFloatToGammaFloat;
    GTL_NO_COPY(RgbColorConverter)
  public:
    RgbColorConverter(double _gamma, const GTLCore::PixelDescription& _pixelDescription, int rIndex = 0, int gIndex = 1, int bIndex = 2, int aIndex = 3) :
        AbstractColorConverter(_pixelDescription),
        m_gamma(_gamma),
        m_pixelToRgb( RgbColorConverterTraits<_Channel_Type_>::createPixelToRgbA(_gamma)),
        m_rgbToPixel( RgbColorConverterTraits<_Channel_Type_>::createRgbAToPixel(_gamma)),
        m_vectorToRgb(_gamma),
        m_rgbToVector(_gamma),
        m_rIndex(rIndex), m_gIndex(gIndex), m_bIndex(bIndex), m_aIndex(aIndex)
    {
    }
  public:
    virtual ~RgbColorConverter()
    {
    }
    virtual void pixelToRgba(const char* _data, RgbaF* _rgba) const
    {
      const _Channel_Type_* data = reinterpret_cast<const _Channel_Type_*>(_data);
      _rgba->r = m_pixelToRgb(data[m_rIndex]);
      _rgba->g = m_pixelToRgb(data[m_gIndex]);
      _rgba->b = m_pixelToRgb(data[m_bIndex]);
      if(_has_alpha) {
        _rgba->a = ChannelMaths<_Channel_Type_, float>::scaleToA(data[m_aIndex]);
      } else {
        _rgba->a = 1.0;
      }
    }
    virtual void rgbaToPixel(const RgbaF* _rgba, char* _data) const
    {
      _Channel_Type_* data = reinterpret_cast<_Channel_Type_*>(_data);
      data[m_rIndex] = m_rgbToPixel(_rgba->r);
      data[m_gIndex] = m_rgbToPixel(_rgba->g);
      data[m_bIndex] = m_rgbToPixel(_rgba->b);
      if(_has_alpha) {
        data[m_aIndex] = ChannelMaths<float, _Channel_Type_>::scaleToA(_rgba->a);
      }
    }
    virtual void vectorToRgba(const float* _data, RgbaF* _rgba) const
    {
      _rgba->r = m_vectorToRgb(_data[m_rIndex]);
      _rgba->g = m_vectorToRgb(_data[m_gIndex]);
      _rgba->b = m_vectorToRgb(_data[m_bIndex]);
      if(_has_alpha) {
        _rgba->a = _data[m_aIndex];
      } else {
        _rgba->a = 1.0;
      }
    }
    virtual void rgbaToVector(const RgbaF* _rgba, float* _data) const
    {
      _data[m_rIndex] = m_rgbToVector(_rgba->r);
      _data[m_gIndex] = m_rgbToVector(_rgba->g);
      _data[m_bIndex] = m_rgbToVector(_rgba->b);
      if(_has_alpha) {
        _data[m_aIndex] = _rgba->a;
      }
    }
  private:
    // Those are placeholders for the future conversion to/from HSV and to/from LAB
    virtual void placeholder1() {}
    virtual void placeholder2() {}
    virtual void placeholder3() {}
    virtual void placeholder4() {}
    virtual void placeholder5() {}
    virtual void placeholder6() {}
    virtual void placeholder7() {}
    virtual void placeholder8() {}
  private:
    double m_gamma;
    typename RgbColorConverterTraits<_Channel_Type_>::PixelToRgb m_pixelToRgb;
    typename RgbColorConverterTraits<_Channel_Type_>::RgbToPixel m_rgbToPixel;
    GammaFloatToLinearFloat m_vectorToRgb;
    LinearFloatToGammaFloat m_rgbToVector;
    int m_rIndex, m_gIndex, m_bIndex, m_aIndex;
  };
};

#endif
