/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENCTL_MODULE_H_
#define _OPENCTL_MODULE_H_

#include <GTLCore/String.h>
#include <list>

namespace GTLCore {
  class ErrorMessage;
  class Function;
  class ModuleData;
  class TypeManager;
}

namespace OpenCTL {
  /**
   * A Module represent a CTL program or module.
   */
  class Module {
    public:
      /**
       * @param name the name of the module (if this name is suffixed by a .ctl, then the suffix
       *             is removed, if it's prefixed with a path, the path is removed as well);
       */
      Module(const GTLCore::String& name);
      ~Module();
      /**
       * @return the name of the module (it's usually the name of the file without the .ctl
       *         extension, and without the path)
       */
      const GTLCore::String& name() const;
      /**
       * @return the namespace used for this module
       */
      const GTLCore::String& nameSpace() const;
      /**
       * Set the code source of the module.
       */
      void setSource(const GTLCore::String& source);
      /**
       * Load the module from the given file name.
       */
      void loadFromFile(const GTLCore::String& fileName);
      /**
       * @param name the name of the function
       * @return the function with the given name
       */
      const GTLCore::Function* function(const GTLCore::String& name) const;
      /**
       * Start the compilation of the module.
       */
      void compile();
      /**
       * @return true if the module was successfully compiled.
       */
      bool isCompiled() const;
      /**
       * @return a string with the content of the compilation error.
       */
      const std::list<GTLCore::ErrorMessage>& compilationErrors() const;
      /**
       * @return a string with the content of the compilation error.
       */
      GTLCore::String compilationErrorsMessage() const;
      /**
       * @return the list of functions available in this module.
       */
      std::list<GTLCore::Function*> functions();
      /**
       * @return a pointer to the internal Data of this module (the class Module::Data
       *         is not part of the public API and therefor there is no reason for you
       *         to use that function).
       */
      const GTLCore::ModuleData* data() const;
      /**
       * @return the type manager with the types defines and use by this module (this 
       *         includes type that are defined by imported modules)
       */
      const GTLCore::TypeManager* typeManager() const;
    public:
      /**
       * @return the assembly source code, it's mostly usefull for testing purpose
       */
      GTLCore::String asmSourceCode() const;
    private:
      struct Private;
      Private* const d;
  };

}

#endif
