/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "OpenCTL/Program.h"

#include <cstring>

#include "GTLCore/FloatHalfConverter_p.h"

class TestInteger8GrayScaleProgram : public GTLTest::Case {
  public:
    TestInteger8GrayScaleProgram() : GTLTest::Case("Integer8GrayScaleProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
void testCall(int a, output int b) \n\
{ \n\
  b = a + 1; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Integer8, 1));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 4 );
      arr.rawData()[0] = 10;
      arr.rawData()[1] = 30;
      arr.rawData()[2] = 50;
      arr.rawData()[3] = 60;
      Program.apply(arr, arr);
      GTLTEST_CHECK_EQUAL( arr.rawData()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.rawData()[1], 31 );
      GTLTEST_CHECK_EQUAL( arr.rawData()[2], 51 );
      GTLTEST_CHECK_EQUAL( arr.rawData()[3], 61 );
      GTLCore::Array arrOut( 4 );
      memset(arrOut.rawData(), 0, arrOut.size());
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_EQUAL( arr.rawData()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.rawData()[1], 31 );
      GTLTEST_CHECK_EQUAL( arr.rawData()[2], 51 );
      GTLTEST_CHECK_EQUAL( arr.rawData()[3], 61 );
      GTLTEST_CHECK_EQUAL( arrOut.rawData()[0], 12 );
      GTLTEST_CHECK_EQUAL( arrOut.rawData()[1], 32 );
      GTLTEST_CHECK_EQUAL( arrOut.rawData()[2], 52 );
      GTLTEST_CHECK_EQUAL( arrOut.rawData()[3], 62 );
    }
};


class TestInteger32GrayScaleProgram : public GTLTest::Case {
  public:
    TestInteger32GrayScaleProgram() : GTLTest::Case("Integer32GrayScaleProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \n\
void testCall(int a, output int b) \n\
{ \n\
  b = a + 1; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Integer32, 1));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 4 * sizeof(int));
      arr.data<int>()[0] = 10;
      arr.data<int>()[1] = 30;
      arr.data<int>()[2] = 50;
      arr.data<int>()[3] = 60;
      Program.apply(arr, arr);
      GTLTEST_CHECK_EQUAL( arr.data<int>()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[1], 31 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[2], 51 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[3], 61 );
      GTLCore::Array arrOut( 4 * sizeof(int));
      memset(arrOut.rawData(), 0, arrOut.size());
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_EQUAL( arr.data<int>()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[1], 31 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[2], 51 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[3], 61 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[0], 12 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[1], 32 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[2], 52 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[3], 62 );
    }
};

class TestFloatGrayScaleProgram : public GTLTest::Case {
  public:
    TestFloatGrayScaleProgram() : GTLTest::Case("FloatGrayScaleProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
void testCall(float a, output float b) \n\
{ \n\
  b = a * 0.5; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Float, 1));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 4 * sizeof(float));
      arr.data<float>()[0] = 10;
      arr.data<float>()[1] = 30;
      arr.data<float>()[2] = 50;
      arr.data<float>()[3] = 60;
      Program.apply(arr, arr);
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[0], 5 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[1], 15 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[2], 25 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[3], 30 );
      GTLCore::Array arrOut( 4 * sizeof(float));
      memset(arrOut.rawData(), 0, arrOut.size());
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[0], 5 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[1], 15 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[2], 25 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[3], 30 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[0], 2.5 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[1], 7.5 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[2], 12.5 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[3], 15.0 );
    }
};



class TestInteger8RGBProgram : public GTLTest::Case {
  public:
    TestInteger8RGBProgram() : GTLTest::Case("Integer8RGBProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
void testCall(int rIn, int gIn, int bIn, output int rOut, output int gOut, output int bOut) \n\
{ \n\
  rOut = rIn + 1; \n\
  gOut = gIn - 1; \n\
  bOut = bIn / 2; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Integer8, 3));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 6 * sizeof(char));
      arr.data<char>()[0] = 10;
      arr.data<char>()[1] = 30;
      arr.data<char>()[2] = 50;
      arr.data<char>()[3] = 60;
      arr.data<char>()[4] = 20;
      arr.data<char>()[5] = 40;
      Program.apply(arr, arr);
      GTLTEST_CHECK_EQUAL( arr.data<char>()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[1], 29 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[2], 25 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[3], 61 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[4], 19 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[5], 20 );
      GTLCore::Array arrOut( 6 * sizeof(char));
      memset(arrOut.rawData(), 0, arrOut.size());
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_EQUAL( arr.data<char>()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[1], 29 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[2], 25 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[3], 61 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[4], 19 );
      GTLTEST_CHECK_EQUAL( arr.data<char>()[5], 20 );
      GTLTEST_CHECK_EQUAL( ((char*)arrOut.rawData())[0], 12 );
      GTLTEST_CHECK_EQUAL( ((char*)arrOut.rawData())[1], 28 );
      GTLTEST_CHECK_EQUAL( ((char*)arrOut.rawData())[2], 12 );
      GTLTEST_CHECK_EQUAL( ((char*)arrOut.rawData())[3], 62 );
      GTLTEST_CHECK_EQUAL( ((char*)arrOut.rawData())[4], 18 );
      GTLTEST_CHECK_EQUAL( ((char*)arrOut.rawData())[5], 10 );
    }
};



class TestInteger32RGBProgram : public GTLTest::Case {
  public:
    TestInteger32RGBProgram() : GTLTest::Case("Integer32RGBProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
void testCall(int rIn, int gIn, int bIn, output int rOut, output int gOut, output int bOut) \n\
{ \n\
  rOut = rIn + 1; \n\
  gOut = gIn - 1; \n\
  bOut = bIn * 2; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Integer32, 3));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 6 * sizeof(int));
      arr.data<int>()[0] = 10;
      arr.data<int>()[1] = 30;
      arr.data<int>()[2] = 50;
      arr.data<int>()[3] = 60;
      arr.data<int>()[4] = 20;
      arr.data<int>()[5] = 40;
      Program.apply(arr, arr);
      GTLTEST_CHECK_EQUAL( arr.data<int>()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[1], 29 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[2], 100 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[3], 61 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[4], 19 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[5], 80 );
      GTLCore::Array arrOut( 6 * sizeof(int));
      memset(arrOut.rawData(), 0, 6 * sizeof(int));
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_EQUAL( arr.data<int>()[0], 11 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[1], 29 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[2], 100 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[3], 61 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[4], 19 );
      GTLTEST_CHECK_EQUAL( arr.data<int>()[5], 80 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[0], 12 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[1], 28 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[2], 200 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[3], 62 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[4], 18 );
      GTLTEST_CHECK_EQUAL( arrOut.data<int>()[5], 160 );
    }
};

class TestHalfRGBProgram : public GTLTest::Case {
  public:
    TestHalfRGBProgram() : GTLTest::Case("FloatHalfProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
void testCall(half rIn, half gIn, half bIn, output half rOut, output half gOut, output half bOut) \n\
{ \n\
  rOut = rIn + 1; \n\
  gOut = gIn / 2.0; \n\
  bOut = bIn * 0.5; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Half, 3));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 6 * sizeof(GTLCore::half));
      arr.data<GTLCore::half>()[0] = float2half( 10 );
      arr.data<GTLCore::half>()[1] = float2half( 30 );
      arr.data<GTLCore::half>()[2] = float2half( 50 );
      arr.data<GTLCore::half>()[3] = float2half( 60 );
      arr.data<GTLCore::half>()[4] = float2half( 20 );
      arr.data<GTLCore::half>()[5] = float2half( 40 );
      Program.apply(arr, arr);
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[0]), 11 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[1]), 15 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[2]), 25 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[3]), 61 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[4]), 10 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[5]), 20 );
      GTLCore::Array arrOut( 6 * sizeof(GTLCore::half));
      memset(arrOut.rawData(), 0, arrOut.size());
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[0]), 11 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[1]), 15 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[2]), 25 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[3]), 61 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[4]), 10 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arr.data<GTLCore::half>()[5]), 20 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arrOut.data<GTLCore::half>()[0]), 12 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arrOut.data<GTLCore::half>()[1]), 7.5 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arrOut.data<GTLCore::half>()[2]), 12.5 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arrOut.data<GTLCore::half>()[3]), 62 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arrOut.data<GTLCore::half>()[4]), 5 );
      GTLTEST_CHECK_NEAR_EQUAL( half2float(arrOut.data<GTLCore::half>()[5]), 10 );
    }
};

class TestFloatRGBProgram : public GTLTest::Case {
  public:
    TestFloatRGBProgram() : GTLTest::Case("FloatRGBProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
void testCall(float rIn, float gIn, float bIn, output float rOut, output float gOut, output float bOut) \n\
{ \n\
  rOut = rIn + 1; \n\
  gOut = gIn / 2.0; \n\
  bOut = bIn * 0.5; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Float, 3));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 6 * sizeof(float));
      arr.data<float>()[0] = 10;
      arr.data<float>()[1] = 30;
      arr.data<float>()[2] = 50;
      arr.data<float>()[3] = 60;
      arr.data<float>()[4] = 20;
      arr.data<float>()[5] = 40;
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[0], 10 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[1], 30 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[2], 50 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[3], 60 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[4], 20 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[5], 40 );
      Program.apply(arr, arr);
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[0], 11 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[1], 15 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[2], 25 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[3], 61 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[4], 10 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[5], 20 );
      GTLCore::Array arrOut( 6 * sizeof(float));
      memset(arrOut.rawData(), 0, arrOut.size());
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[0], 11 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[1], 15 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[2], 25 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[3], 61 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[4], 10 );
      GTLTEST_CHECK_NEAR_EQUAL( arr.data<float>()[5], 20 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[0], 12 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[1], 7.5 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[2], 12.5 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[3], 62 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[4], 5 );
      GTLTEST_CHECK_NEAR_EQUAL( arrOut.data<float>()[5], 10 );
    }
};

class TestMixedProgram : public GTLTest::Case {
  public:
    TestMixedProgram() : GTLTest::Case("MixedProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
void testCall(float rIn, int gIn, int bIn, output float rOut, output int gOut, output int bOut) \n\
{ \n\
  rOut = rIn * 0.5; \n\
  int gOut2 = gIn + 1; \n\
  gOut = gIn + 1; \n\
  bOut = bIn / 2; \n\
  return; \n\
}");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      std::vector<const GTLCore::Type*> types;
      types.push_back( GTLCore::Type::Float );
      types.push_back( GTLCore::Type::Integer32 );
      types.push_back( GTLCore::Type::Integer8 );
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( types ));
      GTLTEST_CHECK(Program.initialised());
      GTLCore::Array arr( 2 * (sizeof(float) + sizeof(int) + sizeof(char)));
      *((float*)(arr.rawData())) = 10;
      *((int*)(arr.rawData() + 4)) = 20;
      *((char*)(arr.rawData() + 8)) = 50;
      *((float*)(arr.rawData() + 9)) = 30;
      *((int*)(arr.rawData() + 13)) = 40;
      *((char*)(arr.rawData() + 17)) = 60;
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arr.rawData())), 10 );
      GTLTEST_CHECK_EQUAL( *((int*)(arr.rawData()+4)), 20 );
      GTLTEST_CHECK_EQUAL( *((char*)(arr.rawData()+8)), 50 );
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arr.rawData()+9)), 30 );
      GTLTEST_CHECK_EQUAL( *((int*)(arr.rawData()+13)), 40 );
      GTLTEST_CHECK_EQUAL( *((char*)(arr.rawData()+17)), 60 );
      Program.apply(arr, arr);
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arr.rawData())), 5 );
      GTLTEST_CHECK_EQUAL( *((int*)(arr.rawData()+4)), 21 );
      GTLTEST_CHECK_EQUAL( *((char*)(arr.rawData()+8)), 25 );
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arr.rawData()+9)), 15.0 );
      GTLTEST_CHECK_EQUAL( *((int*)(arr.rawData()+13)), 41 );
      GTLTEST_CHECK_EQUAL( *((char*)(arr.rawData()+17)), 30 );
      GTLCore::Array arrOut( arr.size() );
      memset(arrOut.rawData(), 0, arrOut.size());
      Program.apply(arr, arrOut);
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arr.rawData())), 5 );
      GTLTEST_CHECK_EQUAL( *((int*)(arr.rawData()+4)), 21 );
      GTLTEST_CHECK_EQUAL( *((char*)(arr.rawData()+8)), 25 );
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arr.rawData()+9)), 15.0 );
      GTLTEST_CHECK_EQUAL( *((int*)(arr.rawData()+13)), 41 );
      GTLTEST_CHECK_EQUAL( *((char*)(arr.rawData()+17)), 30 );
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arrOut.rawData())), 2.5 );
      GTLTEST_CHECK_EQUAL( *((int*)(arrOut.rawData()+4)), 22 );
      GTLTEST_CHECK_EQUAL( *((char*)(arrOut.rawData()+8)), 12 );
      GTLTEST_CHECK_NEAR_EQUAL( *((float*)(arrOut.rawData()+9)), 7.5 );
      GTLTEST_CHECK_EQUAL( *((int*)(arrOut.rawData()+13)), 42 );
      GTLTEST_CHECK_EQUAL( *((char*)(arrOut.rawData()+17)), 15 );
    }
};

class TestRgb8ToRgb32Program : public GTLTest::Case {
  public:
    TestRgb8ToRgb32Program() : GTLTest::Case("TestRgb8ToRgb32Program") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
          void testCall(int rIn, int gIn, int bIn, output int rOut, output int gOut, output int bOut) \n\
      { \n\
          rOut = rIn + 1; \n\
          gOut = gIn - 1; \n\
          bOut = bIn / 2; \n\
          return; \n\
      }");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Integer8, 3), GTLCore::PixelDescription( GTLCore::Type::Integer32, 3));
      GTLCore::Array src( 2 * sizeof(char) * 3);
      *((char*)(src.rawData())) = 10;
      *((char*)(src.rawData()+1)) = 20;
      *((char*)(src.rawData()+2)) = 30;
      *((char*)(src.rawData()+3)) = 40;
      *((char*)(src.rawData()+4)) = 50;
      *((char*)(src.rawData()+5)) = 60;
      GTLCore::Array dst( 2 * sizeof(int) * 3);
      Program.apply(src, dst);
      GTLTEST_CHECK_EQUAL( *((char*)(src.rawData())), 10);
      GTLTEST_CHECK_EQUAL( *((char*)(src.rawData()+1)), 20);
      GTLTEST_CHECK_EQUAL( *((char*)(src.rawData()+2)), 30);
      GTLTEST_CHECK_EQUAL( *((char*)(src.rawData()+3)), 40);
      GTLTEST_CHECK_EQUAL( *((char*)(src.rawData()+4)), 50);
      GTLTEST_CHECK_EQUAL( *((char*)(src.rawData()+5)), 60);
      GTLTEST_CHECK_EQUAL( *((int*)(dst.rawData())), 11);
      GTLTEST_CHECK_EQUAL( *((int*)(dst.rawData()+4)), 19);
      GTLTEST_CHECK_EQUAL( *((int*)(dst.rawData()+8)), 15);
      GTLTEST_CHECK_EQUAL( *((int*)(dst.rawData()+12)), 41);
      GTLTEST_CHECK_EQUAL( *((int*)(dst.rawData()+16)), 49);
      GTLTEST_CHECK_EQUAL( *((int*)(dst.rawData()+20)), 30);
    }
};

class TestRgb32Float8ToRgb8Program : public GTLTest::Case {
  public:
    TestRgb32Float8ToRgb8Program() : GTLTest::Case("TestRgb32Float8ToRgb8Program") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
          void testCall(float rIn, float gIn, float bIn, output int rOut, output int gOut, output int bOut) \n\
      { \n\
          rOut = rIn + 1; \n\
          gOut = gIn - 1; \n\
          bOut = bIn / 2; \n\
          return; \n\
    }");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Float, 3), GTLCore::PixelDescription( GTLCore::Type::Integer8, 3));
      GTLCore::Array src( 2 * sizeof(float) * 3);
      *((float*)(src.rawData())) = 10;
      *((float*)(src.rawData()+4)) = 20;
      *((float*)(src.rawData()+8)) = 30;
      *((float*)(src.rawData()+12)) = 40;
      *((float*)(src.rawData()+16)) = 50;
      *((float*)(src.rawData()+20)) = 60;
      GTLCore::Array dst( 2 * sizeof(char) * 3);
      Program.apply(src, dst);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData())), 10);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+4)), 20);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+8)), 30);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+12)), 40);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+16)), 50);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+20)), 60);
      GTLTEST_CHECK_EQUAL( *((char*)(dst.rawData())), 11);
      GTLTEST_CHECK_EQUAL( *((char*)(dst.rawData()+1)), 19);
      GTLTEST_CHECK_EQUAL( *((char*)(dst.rawData()+2)), 15);
      GTLTEST_CHECK_EQUAL( *((char*)(dst.rawData()+3)), 41);
      GTLTEST_CHECK_EQUAL( *((char*)(dst.rawData()+4)), 49);
      GTLTEST_CHECK_EQUAL( *((char*)(dst.rawData()+5)), 30);
    }
};

class TestRgb32FloatExposureProgram : public GTLTest::Case {
  public:
    TestRgb32FloatExposureProgram() : GTLTest::Case("Rgb32FloatExposureProgram") {}
    virtual void runTest()
    {
      OpenCTL::Module p("testCall");
      p.setSource( " \
          void testCall(float rIn, float gIn, float bIn, output float rOut, output float gOut, output float bOut, varying float exposure = 1.5 ) \n\
      { \n\
          rOut = rIn * exposure; \n\
          gOut = gIn * exposure; \n\
          bOut = bIn * exposure; \n\
          return; \n\
    }");
      p.compile();
      GTLTEST_CHECK(p.isCompiled());
      OpenCTL::Program Program( "testCall", &p, GTLCore::PixelDescription( GTLCore::Type::Float, 3));
      GTLTEST_CHECK_EQUAL(Program.varying( "exposure" ).type(), GTLCore::Type::Float);
      GTLTEST_CHECK_NEAR_EQUAL( Program.varying( "exposure" ).asFloat(), 1.5);
      GTLTEST_CHECK_EQUAL(Program.varyings().size(), 1);
      GTLTEST_CHECK_EQUAL(*Program.varyings().begin(), "exposure");
      GTLCore::Array src( 2 * sizeof(float) * 3);
      *((float*)(src.rawData())) = 10;
      *((float*)(src.rawData()+4)) = 20;
      *((float*)(src.rawData()+8)) = 30;
      *((float*)(src.rawData()+12)) = 40;
      *((float*)(src.rawData()+16)) = 50;
      *((float*)(src.rawData()+20)) = 60;
      GTLCore::Array dst( 2 * sizeof(float) * 3);
      Program.apply(src, dst);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData())), 10 );
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+4)), 20);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+8)), 30);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+12)), 40);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+16)), 50);
      GTLTEST_CHECK_EQUAL( *((float*)(src.rawData()+20)), 60);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData())), 15);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+4)), 30);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+8)), 45);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+12)), 60);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+16)), 75);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+20)), 90);
      Program.setVarying( "exposure", 2 );
      GTLTEST_CHECK_EQUAL(Program.varying( "exposure" ).type(), GTLCore::Type::Float);
      GTLTEST_CHECK_NEAR_EQUAL( Program.varying( "exposure" ).asFloat(), 2.0);
      Program.apply(src, dst);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData())), 20);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+4)), 40);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+8)), 60);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+12)), 80);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+16)), 100);
      GTLTEST_CHECK_EQUAL( *((float*)(dst.rawData()+20)), 120);
    }
};

class TestProgram : public GTLTest::Suite {
  public:
    TestProgram() : GTLTest::Suite("TestProgram")
    {
      addCase(new TestInteger8GrayScaleProgram);
      addCase(new TestInteger32GrayScaleProgram);
      addCase(new TestFloatGrayScaleProgram);
      addCase(new TestInteger8RGBProgram);
      addCase(new TestInteger32RGBProgram);
      addCase(new TestFloatRGBProgram);
      addCase(new TestHalfRGBProgram);
      addCase(new TestMixedProgram);
      addCase(new TestRgb8ToRgb32Program);
      addCase(new TestRgb32Float8ToRgb8Program);
      addCase(new TestRgb32FloatExposureProgram);
    }
};
