/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Expression.h"

#include <llvm/Constant.h>
#include <llvm/Constants.h>
#include <llvm/Function.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Instructions.h>

#include "GTLCore/CodeGenerator_p.h"
#include "GTLCore/ExpressionResult_p.h"
#include "GTLCore/VariableNG_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/Utils_p.h"

#include "GTLCore/Parameter.h"
#include "GTLCore/Debug.h"
#include "GTLCore/Function.h"
#include "GTLCore/Function_p.h"

#include "AccessorExpression.h"

using namespace GTLCore::AST;

//------------------------------------------//
//-------------- Expression ----------------//
//------------------------------------------//

llvm::BasicBlock* Expression::generateStatement( GenerationContext& _context, llvm::BasicBlock* _bb) const
{
  generateValue( _context, _bb);
  return _bb;
}

//------------------------------------------//
//------------- ProxyExpression ------------//
//------------------------------------------//

GTLCore::ExpressionResult ProxyExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return m_clone->generateValue( _gc, bb);
}

GlobalDataExpression::GlobalDataExpression( Expression* expression ) : m_expression( expression )
{ }

GlobalDataExpression::~GlobalDataExpression()
{
  delete m_expression;
}

GTLCore::ExpressionResult GlobalDataExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  GTLCore::ExpressionResult value = m_expression->generateValue( _gc, 0 );
  if( type()->dataType() == Type::VECTOR )
  {
    return value;
  } else {
    return GTLCore::ExpressionResult( new llvm::GlobalVariable( value.constant()->getType(), true, llvm::GlobalValue::InternalLinkage, value.constant(), "", _gc.llvmModule() ), type() );
  }
}

//------------------------------------------//
//--------- FunctionCallExpression ---------//
//------------------------------------------//

FunctionCallExpression::~FunctionCallExpression()
{
  deleteAll( m_arguments );
}

const GTLCore::Type* FunctionCallExpression::type() const
{
  GTL_DEBUG( m_function->name() << " <=> " << *m_function->returnType() );
  return m_function->returnType();
}

GTLCore::ExpressionResult FunctionCallExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return _gc.codeGenerator()->callFunction( _gc, bb, m_function, m_arguments );
}

void FunctionCallExpression::markAsReturnExpression()
{
  // There is no easy way to know if an Array/Structure given as argument of a function
  // will or will not be returned by the function, hence the need to mark parameters
  // as part of a return expression
  for( std::list<Expression*>::iterator it = m_arguments.begin();
       it != m_arguments.end(); ++it)
  {
    (*it)->markAsReturnExpression();
  }
}

// NumberExpression

namespace GTLCore { // GCC or C++ or both sucks and it is required for the template to be defined
  namespace AST {

template<>
const GTLCore::Type*  NumberExpression<float>::type() const
{
  return GTLCore::Type::Float;
}
template<>
GTLCore::ExpressionResult NumberExpression<float>::generateValue( GenerationContext& _gc, llvm::BasicBlock* ) const
{
  return GTLCore::ExpressionResult(_gc.codeGenerator()->floatToConstant( m_val ), type());
}

template<>
const GTLCore::Type*  NumberExpression<int>::type() const
{
  return GTLCore::Type::Integer32;
}
template<>
GTLCore::ExpressionResult NumberExpression<int>::generateValue( GenerationContext& _gc, llvm::BasicBlock* ) const
{
  return GTLCore::ExpressionResult(_gc.codeGenerator()->integerToConstant( m_val ), type());
}

template<>
const GTLCore::Type*  NumberExpression<bool>::type() const
{
  return GTLCore::Type::Boolean;
}
template<>
GTLCore::ExpressionResult NumberExpression<bool>::generateValue( GenerationContext& _gc, llvm::BasicBlock* ) const
{
  return GTLCore::ExpressionResult(_gc.codeGenerator()-> boolToConstant(m_val ), type());
}
  }
}

GTLCore::ExpressionResult StringExpression::generateValue( GenerationContext& _gc, llvm::BasicBlock* bb ) const
{
  return GTLCore::ExpressionResult(llvm::ConstantArray::get(m_string, true ), Type::Integer8 ); // return a null terminated string
}
