/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _AST_STATEMENT_H_
#define _AST_STATEMENT_H_

#include <list>

#include "GTLCore/GenerationContext_p.h"

namespace llvm {
  class BasicBlock;
  class Function;
  class Module;
  class Value;
}

namespace GTLCore {
  class CodeGenerator;
  class Type;
  class VariableNG;
  namespace AST {
    class Expression;
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class Statement {
      public:
        virtual ~Statement() {}
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const = 0;
      protected:
        llvm::BasicBlock* createBlock( GenerationContext&) const;
    };
    /**
     * @internal
     * Statement that does nothing.
     * @ingroup GTLCore_AST
     */
    class DummyStatement : public Statement {
      public:
        virtual ~DummyStatement() {}
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* bb ) const
        { return bb; }
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class StatementsList : public Statement {
      public:
        StatementsList( const std::list<Statement*>& list ) : m_list( list) {}
        ~StatementsList();
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        std::list<Statement*> m_list;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class VariableDeclaration : public Statement
    {
      public:
        VariableDeclaration( const GTLCore::Type* _type, Expression* _initialiser, bool _constant, const std::list<Expression*>& _initialSizes);
        ~VariableDeclaration();
      public:
        GTLCore::VariableNG* variable() { return m_variable; }
        void setFunctionIntialiser( AST::Statement* _functionInitialiser )
        { m_functionInitialiser = _functionInitialiser; }
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        GTLCore::VariableNG* m_variable;
        Expression* m_initialiser;
        std::list<Expression*> m_initialSizes;
        Statement* m_functionInitialiser;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class IfStatement : public Statement {
      public:
        IfStatement( Expression* _expression, Statement* _ifStatement) :
          m_expression( _expression), m_ifStatement( _ifStatement )
        {
        }
        ~IfStatement();
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        Expression* m_expression;
        Statement* m_ifStatement;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class IfElseStatement : public Statement {
      public:
        IfElseStatement( Expression* _expression, Statement* _ifStatement, Statement* _elseStatement ) :
          m_expression( _expression ), m_ifStatement(_ifStatement), m_elseStatement(_elseStatement)
        {
        }
        ~IfElseStatement();
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        Expression* m_expression;
        Statement* m_ifStatement;
        Statement* m_elseStatement;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class WhileStatement : public Statement {
      public:
        WhileStatement( Expression* _expression, Statement* _whileStatement)
        : m_expression(_expression), m_whileStatement(_whileStatement)
        {
        }
        ~WhileStatement();
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        Expression* m_expression;
        Statement* m_whileStatement;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class ForStatement : public Statement {
      public:
        ForStatement( Statement* _initStatement, Expression* _testExpression, Expression* _updateExpression, Statement* _forStatement)
        : m_initStatement( _initStatement ), m_testExpression( _testExpression ), m_updateExpression(_updateExpression), m_forStatement( _forStatement )
        {}
        ~ForStatement();
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        Statement* m_initStatement;
        Expression* m_testExpression;
        Expression* m_updateExpression;
        Statement* m_forStatement;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class ReturnStatement : public Statement {
      public:
        /**
         * @param _returnExpr
         * @param _variablesToClean list of variables used so far, that will need to be
         *        cleaned by the return expression
         */
        ReturnStatement( Expression* _returnExpr, std::list<VariableNG*> _variablesToClean );
        ~ReturnStatement();
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        Expression* m_returnExpr;
        std::list<VariableNG*> m_variablesToClean;
    };
    /**
     * @internal
     * @ingroup GTLCore_AST
     */
    class PrintStatement : public Statement {
      public:
        PrintStatement( const std::list<AST::Expression*>& _expressions) : m_expressions(_expressions)
        {
        }
        ~PrintStatement();
        virtual llvm::BasicBlock* generateStatement( GenerationContext&, llvm::BasicBlock* ) const;
      private:
        std::list<AST::Expression*> m_expressions;
    };
  }
}

#endif
