/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "BufferImage.h"

#include <cstring>

#include <GTLCore/Buffer.h>

#include "Debug.h"

using namespace GTLCore;

struct BufferImage::Private
{
  GTLCore::Buffer* buffer;
  int lineWidth;
  int width;
  int height;
  char* defaultPixel;
};

BufferImage::BufferImage( int _width, int _height, GTLCore::Buffer* _buffer, const GTLCore::PixelDescription& _pixelDescription ) : AbstractImage(_pixelDescription), d(new Private)
{
  d->buffer = _buffer;
  d->lineWidth = _width * pixelSize();
  d->width = _width;
  d->height = _height;
  d->defaultPixel = new char[ pixelSize() ];
  memset( d->defaultPixel, 0, pixelSize() );
  GTL_DEBUG( d->buffer->size() << " " << d->lineWidth << " " << _height);
  GTL_ASSERT( d->buffer->size() == d->lineWidth * _height );
}

BufferImage::~BufferImage()
{
  delete[] d->defaultPixel;
  delete d->buffer;
  delete d;
}

char* BufferImage::data( int _x, int _y )
{
  if( _x >= 0 and _y >= 0 and _x < d->width and _y < d->height )
  {
    return d->buffer->rawData() + (_x * pixelSize() + _y * lineWidth());
  } else {
    return d->defaultPixel;
  }
}

const char* BufferImage::data( int _x, int _y ) const
{
  if( _x >= 0 and _y >= 0 and _x < d->width and _y < d->height )
  {
    return d->buffer->rawData() + (_x * pixelSize() + _y * lineWidth());
  } else {
    return d->defaultPixel;
  }
}

int BufferImage::lineWidth() const
{
  return d->lineWidth;
}

int BufferImage::width() const
{
  return d->width;
}

int BufferImage::height() const
{
  return d->height;
}

const GTLCore::Buffer* BufferImage::buffer() const
{
  return d->buffer;
}
