/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _OPENCTL_FUNCTION_P_H_
#define _OPENCTL_FUNCTION_P_H_

#include "Function.h"
#include "Parameter.h"
#include <vector>
#include "GTLCore/ScopedName.h"

namespace llvm {
  class Function;
  class Module;
}

namespace GTLCore {
  class FunctionCaller;
  /**
   * @internal
   * This class is not part of the public API of OpenCTL.
   * 
   */
  struct Function::Private
  {
    GTLCore::ScopedName name;
    const GTLCore::Type* returnType;
    Function::Data* data;
    std::vector<FunctionCaller*> functionsCaller;
    static GTLCore::Function* createExternalFunction(llvm::Module* _module, const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, const std::vector<GTLCore::Parameter>& arguments);
    static GTLCore::Function* createExternalFunction(llvm::Module* _module, const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, int _count, ...);
    static GTLCore::Function* createInternalFunction(llvm::Module* _module, const GTLCore::String& _name, llvm::Function* _function, const GTLCore::Type* retType, int _count, ...);
  };
  /**
   * @internal
   * This class is not part of the public API of OpenCTL.
   * 
   * This class contains specific structure coming from LLVM,
   * like a pointer to llvm::Function.
   * 
   */
  class Function::Data {
    public:
      Data( const std::vector< Parameter >& _parameters, int minimumParameters);
      void setFunctions( const std::vector<llvm::Function*>& _functions );
      void setModule( llvm::Module* module );
      /**
       * @param count the number of parameters
       * @return the llvm function corresponding to the number of parameters, or 0 if no
       *         function exist for the given number of parameters
       */
      llvm::Function* function(unsigned int count);
      /**
       * @param count the number of parameters
       * @return the llvm function corresponding to the number of parameters, or 0 if no
       *         function exist for the given number of parameters
       */
      llvm::Function* function(unsigned int count) const;
      llvm::Module* module() { return m_module; }
      unsigned int minimumParameters() const { return m_minimumParameters; }
      unsigned int maximumParameters() const { return m_maximumParameters; }
      const std::vector< Parameter >& parameters() { return m_parameters; }
      /**
       * @return the symbol name for the function with the given scoped name and parameters
       */
      static GTLCore::String symbolName( const ScopedName& _functionName, const std::vector< Parameter >& _parameters );
    private:
      std::vector< Parameter > m_parameters;
      std::vector<llvm::Function*> m_functions;
      llvm::Module* m_module;
      unsigned int m_minimumParameters;
      unsigned int m_maximumParameters;
  };
}

#endif
