/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_VARIABLENG_H_
#define _GTLCORE_VARIABLENG_H_

#include <list>

namespace llvm {
  class BasicBlock;
  class Value;
}

namespace GTLCore {
  class GenerationContext;
  class ExpressionResult;
  class Type;
  /**
   * VariableNG is the class for manipulating variables, getting and settng the values.
   * @ingroup GTLCore
   */
  class VariableNG {
    public:
      /**
       * @param _type the type of the variable
       * @param _constant set to true if the variable is a constant
       */
      VariableNG(const GTLCore::Type* _type, bool _constant );
      ~VariableNG();
      /**
       * Initialise a Variable with an initialiser. This function will call \ref Accessor::initialise .
       * @param _initialiser initial value of the variable
       * @param _initialSize the initial size of the variable
       */
      llvm::BasicBlock* initialise( GenerationContext&, llvm::BasicBlock* _bb, const ExpressionResult& _initialiser,  const std::list<llvm::Value*>& _initialSize );
      /**
       * Initialise a Variable with a given pointer. This function will not call \ref Accessor::initialise
       * @param _pointer a pointer that will be used by this variable
       */
      void initialise( GenerationContext&, llvm::Value* _pointer );
      /**
       * @return true if this variable is a constant.
       */
      bool constant() const;
      /**
       * @return the type of the variable
       */
      const GTLCore::Type* type() const;
      /**
       * Access to the value.
       * @param _ma a pointer to a structure describing the member or index accessed
       * @return the value
       */
      llvm::Value* get(GenerationContext&, llvm::BasicBlock* _currentBlock);
      /**
       * Allow to set a value. Raise an assert if called on a constant.
       * @param _ma a pointer to a structure describing the member or index accessed
       * @return the new basic block
       */
      llvm::BasicBlock* set(GenerationContext&, llvm::BasicBlock* _currentBlock, llvm::Value* _value, const GTLCore::Type* _valueType);
      /**
       * Replace the pointer of this variable by a pointer of the same type.
       * 
       * @param _pointer new pointer
       * 
       * WARNING: this VariableNG will then take ownership of the Value, and you can only
       *          replace by a pointer which was allocated in memory.
       *          This function will call cleanUp.
       */
      llvm::BasicBlock* replacePointer( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _pointer);
      /**
       * @return the pointer for this variable
       */
      llvm::Value* pointer();
      /**
       * Mark the variable to be allocated in memory
       */
      void setAllocateInMemory( bool v );
      /**
       * @return true if the variable is allocated in memory, false otherwise
       */
      bool allocatedInMemory() const;
      /**
       * @return true if the pointer is constant (which means it can't be replaced)
       */
      bool constantPointer() const;
      /**
       * Cleanup the variable, free memory if needed, but don't touch the
       * value in argument.
       * @param _donttouch variable protected in memory
       */
      llvm::BasicBlock* cleanUp( GenerationContext& _generationContext, llvm::BasicBlock* _currentBlock, llvm::Value* _donttouch );
    private:
      struct Private;
      Private* const d;
  };
  
};


#endif
