/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Kernel.h"

#include <sstream>
#include <fstream>

#include <llvm/Module.h>
#include <llvm/ModuleProvider.h>

#include "GTLCore/ErrorMessage.h"
#include "GTLCore/Function.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/PixelDescription.h"
#include "GTLCore/Region.h"
#include "GTLCore/Type.h"
#include "GTLCore/TypeManager.h"
#include "GTLCore/Value.h"
#include "GTLCore/VirtualMachine_p.h"

#include "Debug.h"
#include "CodeGenerator_p.h"
#include "Compiler_p.h"
#include "Wrapper_p.h"
#include "wrappers/ImageWrap_p.h"

using namespace OpenShiva;

struct Kernel::Private {
  GTLCore::String name;
  GTLCore::String source;
  bool enableHydraCompatibility;
  bool compiled;
  std::list<GTLCore::ErrorMessage> compilationErrors;
  llvm::ModuleProvider* moduleProvider;
  GTLCore::ModuleData* moduleData;
  llvm::Function* evaluatePixelesFunction;
  int count_channels_generic;
  Wrapper* wrapper;
};

Kernel::Kernel(const GTLCore::String& _name, int _channelsNb ) : d(new Private )
{
  d->name = _name;
  d->compiled = false;
  d->enableHydraCompatibility = false;
  d->moduleProvider = 0;
  d->moduleData = 0;
  d->count_channels_generic = _channelsNb;
  d->evaluatePixelesFunction = 0;
  d->wrapper = 0;
}

Kernel::~Kernel()
{
  cleanup();
  delete d;
}

void Kernel::cleanup()
{
  if(d->moduleProvider)
  {
    GTLCore::VirtualMachine::instance()->unregisterModule( d->moduleProvider);
    delete d->moduleProvider;
    d->moduleProvider = 0;
  }
  delete d->moduleData;
  delete d->wrapper;
  d->moduleData = 0;
  d->evaluatePixelesFunction = 0; // It's deleted by the moduleData
}


const GTLCore::String& Kernel::name() const
{
  return d->name;
}

void Kernel::setSource(const GTLCore::String& _source, bool _enableHydraCompatibility )
{
  d->enableHydraCompatibility = _enableHydraCompatibility;
  d->source = _source;
}

void Kernel::loadFromFile(const GTLCore::String& _fileName)
{
  d->source = "";
  std::ifstream in;
  in.open(_fileName.c_str() );
  if(not in)
  {
    SHIVA_DEBUG( "Impossible to open file " << _fileName );
    return;
  }
  GTLCore::String str;
  std::getline(in,str);
  while ( in ) {
    d->source += str;
    d->source += "\n";
    std::getline(in,str);
  }
}

void Kernel::compile()
{
  if(d->source.empty()) return;
  cleanup();
  d->moduleData = new GTLCore::ModuleData(new llvm::Module(d->name));
  Compiler c;
  Wrapper::fillTypeManager( d->moduleData, d->moduleData->typeManager(), c.convertCenter() , d->count_channels_generic );
  GTLCore::String nameSpace;
  bool result = c.compile( d->source, d->name, d->moduleData, nameSpace );

  if(result)
  {
    d->compiled = true;
    // Register the compiled module in the virtual machine
    d->moduleProvider = new llvm::ExistingModuleProvider( d->moduleData->llvmModule() );
    GTLCore::VirtualMachine::instance()->registerModule( d->moduleProvider );
    
    // Create a wrapper
    d->wrapper = new Wrapper(this, d->moduleData);
    // Create the generateEvaluatePixeles LLVM function
    d->evaluatePixelesFunction = CodeGenerator::generateEvaluatePixeles( std::vector<const GTLCore::Type*>(), d->moduleData->typeManager()->getStructure("image"), this, d->moduleData, d->count_channels_generic ); // TODO call with correct types
    d->name = nameSpace;
  } else {
    cleanup();
    d->compilationErrors = c.errorMessages();
  }
  
}

void Kernel::evaluatePixeles( const GTLCore::Region& _region, const std::list< GTLCore::AbstractImage* >& _inputImages, GTLCore::AbstractImage* _outputImage) const
{
  SHIVA_DEBUG( _region.x() << " " << _region.y() << " " << _region.width() << " " << _region.height());
  SHIVA_ASSERT( d->evaluatePixelesFunction );
  
  // Wrap input images
  SHIVA_ASSERT( d->wrapper );
  const void** inputImages = new const void*[ _inputImages.size() ];
  int i = 0;
  for( std::list< GTLCore::AbstractImage* >::const_iterator it = _inputImages.begin();
       it != _inputImages.end(); ++it)
  {
    inputImages[i] = (const void*)d->wrapper->wrapImage( *it );
  }
  
  // Wrap output image
  ImageWrap* owrap = d->wrapper->wrapImage( _outputImage );
  
  // Call function
  void (*func)( int, int, int, int, const void**, void*) = ( void(*)(int, int, int, int, const void**, void*))GTLCore::VirtualMachine::instance()->getPointerToFunction( d->evaluatePixelesFunction);
  SHIVA_ASSERT(func);
  (*func)( _region.x(), _region.y(), _region.width(), _region.height(), inputImages, owrap); // TODO s/200/width s/300/height
  delete[] inputImages;
  delete owrap;
  SHIVA_DEBUG( "done" );
}

int Kernel::runTest() const
{
  SHIVA_ASSERT( isCompiled() );
  const GTLCore::Function* f = d->moduleData->function( "MyKernel", "runTest");
  SHIVA_ASSERT( f );
  GTLCore::Value v = f->call( std::vector< GTLCore::Value >() );
  return v.asInt32();
}

GTLCore::String Kernel::compilationErrorsMessage() const
{
  std::ostringstream os;
  for( std::list<GTLCore::ErrorMessage>::iterator it = d->compilationErrors.begin();
       it != d->compilationErrors.end(); ++it)
  {
    os << it->fileName() << " at " << it->line() << " : " << it->errorMessage()  << std::endl;
  }
  return os.str();
}


bool Kernel::isCompiled() const
{
  return d->compiled;
}

const std::list<GTLCore::ErrorMessage>& Kernel::compilationErrors() const
{
  return d->compilationErrors;
}

GTLCore::String Kernel::asmSourceCode() const
{
  std::ostringstream os;
  os << *d->moduleData->llvmModule() << std::endl;
  return os.str();
}
