// RegistrationContextInformation.cpp: Implementierung der Klasse RegistrationContextInformation.
//
//////////////////////////////////////////////////////////////////////

#if OSL_DEBUG_LEVEL == 0
#define NDEBUG
#endif
#include <assert.h>

#ifndef _REGISTRATIONCONTEXTINFORMATION_HXX_
#include "registrationcontextinformation.hxx"
#endif

#include <tools/presys.h>
#include <windows.h>
#include <tools/postsys.h>

#define VCL_NEED_BASETSD

#include <algorithm>

#include "preg4msdoc.hrc"

#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif

#ifndef _TOOLS_RESMGR_HXX
#include <tools/resmgr.hxx>
#endif

//----------------------------------------------
//
//----------------------------------------------

namespace /* private */
{
	const int MAX_REGKEY_LENGTH_WIN9X = 16300;
}

//----------------------------------------------
/**
*/
RegistrationContextInformation::RegistrationContextInformation(ResMgr* ResourceManager, const std::wstring& OpenOfficeExecutablePath) :
	m_ResMgr(ResourceManager),
	m_IsWin9x(true),
	m_OOExecPath(OpenOfficeExecutablePath)
{
	OSVERSIONINFOA osverinfo;
    ZeroMemory(&osverinfo, sizeof(osverinfo));
    osverinfo.dwOSVersionInfoSize = sizeof(osverinfo);
    GetVersionExA(&osverinfo);

    m_IsWin9x = (osverinfo.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS);

	OSL_ASSERT(m_OOExecPath.length());

	ExtractOpenOfficeExecNameFromPath();
}

//----------------------------------------------
/**
*/
RegistrationContextInformation::~RegistrationContextInformation()
{
}

//################################
// Query
//################################

//----------------------------------------------
/** Word document information
*/
std::wstring RegistrationContextInformation::GetWordDocumentDisplayName() const
{
	std::wstring str = String(ResId(STR_MS_WORD_DOCUMENT, m_ResMgr)).GetBuffer();

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"Microsoft Word Document";

	return str;
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetWordDocumentFileExtension() const
{
	return std::wstring(L".doc");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetWordDocumentDefaultIconEntry() const
{
	return m_OOExecPath + std::wstring(L",1");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetWordDocumentDefaultShellCommand() const
{
	return std::wstring(L"open");
}

//----------------------------------------------
/** Word template information
*/
std::wstring RegistrationContextInformation::GetWordTemplateDisplayName() const
{
	std::wstring str = String(ResId(STR_MS_WORD_TEMPLATE, m_ResMgr)).GetBuffer();

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"Microsoft Word Template";

	return str;
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetWordTemplateFileExtension() const
{
	return std::wstring(L".dot");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetWordTemplateDefaultIconEntry() const
{
	return m_OOExecPath + std::wstring(L",2");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetWordTemplateDefaultShellCommand() const
{
	return std::wstring(L"new");
}

//----------------------------------------------
/** Excel sheet information
*/
std::wstring RegistrationContextInformation::GetExcelSheetDisplayName() const
{
	std::wstring str = String(ResId(STR_MS_EXCEL_WORKSHEET, m_ResMgr)).GetBuffer();

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"Microsoft Excel Worksheet";

	return str;
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetExcelSheetFileExtension() const
{
	return std::wstring(L".xls");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetExcelSheetDefaultIconEntry() const
{
	return m_OOExecPath + std::wstring(L",3");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetExcelSheetDefaultShellCommand() const
{
	return std::wstring(L"open");
}

//----------------------------------------------
/** Excel template information
*/
std::wstring RegistrationContextInformation::GetExcelTemplateDisplayName() const
{
	std::wstring str = String(ResId(STR_MS_EXCEL_TEMPLATE, m_ResMgr)).GetBuffer();

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"Microsoft Excel Template";

	return str;
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetExcelTemplateFileExtension() const
{
	return std::wstring(L".xlt");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetExcelTemplateDefaultIconEntry() const
{
	return m_OOExecPath + std::wstring(L",4");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetExcelTemplateDefaultShellCommand() const
{
	return std::wstring(L"new");
}

//----------------------------------------------
/** PowerPoint document information
*/
std::wstring RegistrationContextInformation::GetPowerPointDocumentDisplayName() const
{
	std::wstring str = String(ResId(STR_MS_POWERPOINT_PRESENTATION, m_ResMgr)).GetBuffer();

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"Microsoft PowerPoint Presentation";

	return str;
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointDocumentFileExtension() const
{
	return std::wstring(L".ppt");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointDocumentDefaultIconEntry() const
{
	return m_OOExecPath + std::wstring(L",7");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointDocumentDefaultShellCommand() const
{
	return std::wstring(L"open");
}

//----------------------------------------------
/** PowerPoint template information
*/
std::wstring RegistrationContextInformation::GetPowerPointTemplateDisplayName() const
{
	std::wstring str = String(ResId(STR_MS_POWERPOINT_TEMPLATE, m_ResMgr)).GetBuffer();

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"Microsoft PowerPoint Template";

	return str;
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointTemplateFileExtension() const
{
	return std::wstring(L".pot");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointTemplateDefaultIconEntry() const
{
	return m_OOExecPath + std::wstring(L",8");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointTemplateDefaultShellCommand() const
{
	return std::wstring(L"new");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointShowDisplayName() const
{
    std::wstring str = String(ResId(STR_MS_POWERPOINT_SHOW, m_ResMgr)).GetBuffer();

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"Microsoft PowerPoint Show";

	return str;
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointShowFileExtension() const
{
    return std::wstring(L".pps");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointShowDefaultIconEntry() const
{
    return m_OOExecPath + std::wstring(L",7");
}

//----------------------------------------------
/**
*/
std::wstring RegistrationContextInformation::GetPowerPointShowDefaultShellCommand() const
{
    return std::wstring(L"open");
}
	
//----------------------------------------------
/** The string for the "New" command that should appear
	in the Explorer context menu when someone right
	clicks a Microsoft document
*/
std::wstring RegistrationContextInformation::ShellNewCommandDisplayName() const
{
	std::wstring str = String(ResId(STR_NEW_DISPLAY_NAME, m_ResMgr)).GetBuffer();

	std::wstring::size_type idx = str.find(L"~");
	if(std::wstring::npos != idx)
		str.replace(idx, 1, L"&");

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"&New";

	return str;
}

/** The string for the "Edit" command that should
	appear in the Explorer context menu when someone
	right clicks a document
*/
std::wstring RegistrationContextInformation::ShellEditCommandDisplayName() const
{
	std::wstring str = String(ResId(STR_EDIT, m_ResMgr)).GetBuffer();

	std::wstring::size_type idx = str.find(L"~");
	if(std::wstring::npos != idx)
		str.replace(idx, 1, L"&");

	if (m_IsWin9x && !IsConvertableToAnsi(str))
		str = L"&Edit";

	return str;
}

//----------------------------------------------
/** A friendly name for the application
*/
std::wstring RegistrationContextInformation::GetOpenOfficeFriendlyAppName() const
{
	return String(ResId(STR_PRODUCTNAME, m_ResMgr)).GetBuffer();
}

//----------------------------------------------
/** The path to the StarOffice/OpenOffice executable
*/
std::wstring RegistrationContextInformation::GetOpenOfficeExecutablePath() const
{
	return m_OOExecPath;
}

//----------------------------------------------
/** The name of the executable (currently "soffice.exe"
	but may change in the future, who knows)
*/
std::wstring RegistrationContextInformation::GetOpenOfficeExecutableName() const
{
	return m_OOExecName;
}

//----------------------------------------------
/** A command line for the specified shell command
*/
std::wstring RegistrationContextInformation::GetOpenOfficeCommandline(SHELL_COMMAND ShellCommand) const
{
	// quote the path to OpenOffice, this is important
	// for Windows 9x
	std::wstring cmd_line = std::wstring(L"\"") + m_OOExecPath + std::wstring(L"\"");

	switch(ShellCommand)
	{
	case New:
		cmd_line += std::wstring(L" -n \"%1\"");
		break;

	case Open:
		cmd_line += std::wstring(L" -o \"%1\"");
		break;

	case Print:
		cmd_line += std::wstring(L" -p \"%1\"");
		break;

	case Printto:
		cmd_line += std::wstring(L" -pt \"%2\" \"%1\"");
		break;

	default:
		OSL_ASSERT(false);
	}

	return cmd_line;
}

//----------------------------------------------
/**
*/
bool RegistrationContextInformation::IsConvertableToAnsi(const std::wstring& String) const
{
	char buff[MAX_REGKEY_LENGTH_WIN9X];

	int bUsedDefChar = 0;

	if (String.length() > 0)
	{
		int rc = WideCharToMultiByte(
			CP_ACP,
			WC_COMPOSITECHECK | WC_DEFAULTCHAR,
			String.c_str(),
			String.length(),
			buff,
			sizeof(buff),
			NULL,
			&bUsedDefChar);

        //assert(rc, "WideCharToMultiByte failed");
	}

    return !bUsedDefChar;
}

//----------------------------------------------
/**
*/
void RegistrationContextInformation::ExtractOpenOfficeExecNameFromPath()
{
    std::wstring::size_type idx = m_OOExecPath.find_last_of(L'\\');

    OSL_ENSURE(idx != std::wstring::npos, "Invalid path");

	m_OOExecName = m_OOExecPath.substr(idx + 1);
}

