/*************************************************************************
 *
 *  $RCSfile: socket.icc.c,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: vg $ $Date: 2003/04/15 17:42:37 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "system.h"
#include "sockimpl.h"

#ifndef _OSL_TIME_H_
#include <osl/time.h>
#endif

#include <osl/socket.h>
#include <osl/diagnose.h>
#include <osl/module.h>

/* defines for shutdown */
#define SD_RECEIVE 0
#define SD_SEND 1
#define SD_BOTH 2

/*
  oslSocketAddr is a pointer to a Berkeley struct sockaddr.
  I refrained from using sockaddr_in because of possible further
  extensions of this socket-interface (IP-NG?).
  The intention was to hide all Berkeley data-structures from
  direct access past the osl-interface.

  The current implementation is internet (IP) centered. All
  the constructor-functions (osl_create...) take parameters
  that will probably make sense only in the IP-environment
  (e.g. because of using the dotted-Addr-format).

  If the interface will be extended to host other protocol-
  families, I expect no externally visible changes in the
  existing functions. You'll probably need only new
  constructor-functions who take the different Addr
  formats into consideration (maybe a long dotted Addr
  or whatever).
*/

/* _Note_ that I rely on the fact that oslSocketAddr and struct sockaddr */
/* are the same! I don't like it very much but see no other easy way */
/* to conceal the struct sockaddr from the eyes of the user. */

#define OSL_INVALID_SOCKET        -1
#define OSL_SOCKET_ERROR        -1

/*----------------------------------------------------------------------------*/

static HMODULE hSO32DLL = NULL;
static HMODULE hTCP32DLL = NULL;

typedef int (_System fptr_accept) (
    int, struct sockaddr *, int *);
typedef int (_System fptr_bind) (
    int, struct sockaddr *, int );
typedef int (_System fptr_bsdselect) (
    int,struct fd_set*,struct fd_set*,struct fd_set*,struct timeval*);
typedef int (_System fptr_connect) (
    int, struct sockaddr *, int );
typedef int (_System fptr_getpeername) (
    int, struct sockaddr *, int *);
typedef int (_System fptr_getsockname) (
    int, struct sockaddr *, int *);
typedef int (_System fptr_ioctl) (
    int, int, sal_Char *, int);
typedef int (_System fptr_listen) (
    int, int );
typedef int (_System fptr_recv) (
    int, sal_Char *, int, int );
typedef int (_System fptr_recvfrom) (
    int, sal_Char *, int, int, struct sockaddr *, int  *);
typedef int (_System fptr_setsockopt) (
    int, int, int, sal_Char*, int );
typedef int (_System fptr_send) (
    int, sal_Char *, int, int );
typedef int (_System fptr_sendto) (
    int, sal_Char *, int, int, struct sockaddr *, int  );
typedef int (_System fptr_socket) (
    int, int, int );
typedef int (_System fptr_soclose) (int);
typedef int (_System fptr_sock_init) (void);
typedef int (_System fptr_sock_errno) (void);

typedef int (_System fptr_getsockopt) ( int, int, int, sal_Char *, int * );
typedef int (_System fptr_shutdown) (int, int);

typedef void (_System fptr_addsockettolist) (int);
typedef void (_System fptr_removesocketfromlist) (int);


typedef sal_uInt16 (_System fptr_bswap) (sal_uInt16);
typedef sal_uInt32 (_System fptr_lswap) (sal_uInt32);

typedef sal_uInt32 (_System fptr_inet_addr) (sal_Char *);
typedef sal_Char * (_System fptr_inet_ntoa) (struct in_addr);

typedef int (_System fptr_gethostname) (sal_Char *, int);
typedef struct hostent* (_System fptr_gethostbyaddr) (sal_Char *, int, int);
typedef struct hostent* (_System fptr_gethostbyname) (sal_Char *);

typedef struct servent * (_System fptr_getservbyname) ( sal_Char *, sal_Char * );

/*----------------------------------------------------------------------------*/

static fptr_bsdselect*        pfbsdselect = 0;
static fptr_sock_init*        pfsock_init = 0;
static fptr_sock_errno*       pfsock_errno = 0;
static fptr_socket*           pfsocket = 0;
static fptr_soclose*          pfsoclose = 0;
static fptr_send*             pfsend = 0;
static fptr_bswap*            pfbswap = 0;
static fptr_lswap*            pflswap = 0;
static fptr_connect*          pfconnect = 0;
static fptr_recv*             pfrecv = 0;
static fptr_ioctl*            pfioctl = 0;
static fptr_inet_addr*        pfinet_addr = 0;
static fptr_inet_ntoa*        pfinet_ntoa = 0;
static fptr_accept*           pfaccept = 0;
static fptr_bind*             pfbind = 0;
static fptr_getpeername*      pfgetpeername = 0;
static fptr_getsockname*      pfgetsockname = 0;
static fptr_listen*           pflisten = 0;
static fptr_recvfrom*         pfrecvfrom = 0;
static fptr_sendto*           pfsendto = 0;
static fptr_setsockopt*       pfsetsockopt = 0;
static fptr_getsockopt*       pfgetsockopt = 0;
static fptr_shutdown*         pfshutdown = 0;
static fptr_addsockettolist*  pfaddsockettolist = 0;
static fptr_removesocketfromlist * pfremovesocketfromlist = 0;

static fptr_gethostname*      pfgethostname = 0;
static fptr_gethostbyaddr*    pfgethostbyaddr = 0;
static fptr_gethostbyname*    pfgethostbyname = 0;
static fptr_getservbyname*    pfgetservbyname = 0;

/*----------------------------------------------------------------------------*/

int _System bsdselect (
    int n, struct fd_set* p1, struct fd_set* p2, struct fd_set* p3,
    struct timeval* pTime)
{
    return (pfbsdselect ? pfbsdselect (n, p1, p2, p3, pTime) : -1);
}

/*----------------------------------------------------------------------------*/

int _System sock_init (void)
{
    return (pfsock_init ? pfsock_init() : -1);
}

/*----------------------------------------------------------------------------*/

int _System sock_errno (void)
{
    return (pfsock_errno ? pfsock_errno() : -1);
}

/*----------------------------------------------------------------------------*/

int _System socket (int n1, int n2, int n3)
{
    return (pfsocket ? pfsocket (n1, n2, n3) : -1);
}

/*----------------------------------------------------------------------------*/

int _System soclose (int n1)
{
    return (pfsoclose ? pfsoclose (n1) : -1);
}

/*----------------------------------------------------------------------------*/

int _System send (int n1, sal_Char* p1, int n2, int n3)
{
    return (pfsend ? pfsend (n1, p1, n2, n3) : -1);
}

/*----------------------------------------------------------------------------*/

sal_uInt16 _System bswap (sal_uInt16 n1)
{
    return (pfbswap ? pfbswap (n1) : n1);
}

/*----------------------------------------------------------------------------*/

sal_uInt32 _System lswap (sal_uInt32 n1)
{
    return (pflswap ? pflswap (n1) : n1);
}

/*----------------------------------------------------------------------------*/

int _System connect (int n1, struct sockaddr* p1, int n2)
{
    return (pfconnect ? pfconnect (n1, p1, n2) : -1);
}

/*----------------------------------------------------------------------------*/

int _System recv (int n1, sal_Char* p1, int n2, int n3)
{
    return (pfrecv ? pfrecv (n1, p1, n2, n3) : -1);
}

/*----------------------------------------------------------------------------*/

int _System ioctl(int n1, int n2, sal_Char *p1, int n3)
{
    return (pfioctl ? pfioctl (n1, n2, p1, n3) : -1);
}

/*----------------------------------------------------------------------------*/

sal_uInt32 _System inet_addr (sal_Char* p1)
{
    return (pfinet_addr ? pfinet_addr (p1) : -1);
}

/*----------------------------------------------------------------------------*/

sal_Char * _System inet_ntoa (struct in_addr a1)
{
    return (pfinet_ntoa ? pfinet_ntoa (a1) : 0);
}

/*----------------------------------------------------------------------------*/

int _System accept (int n1, struct sockaddr *p1, int *n2)
{
    return (pfaccept ? pfaccept (n1, p1, n2) : -1);
}

/*----------------------------------------------------------------------------*/

int _System bind (int n1, struct sockaddr* p1, int n2)
{
    return (pfbind ? pfbind (n1, p1, n2) : -1);
}

/*----------------------------------------------------------------------------*/

int _System getpeername (int n1, struct sockaddr* p1, int *n2)
{
    return (pfgetpeername ? pfgetpeername (n1, p1, n2) : -1);
}

/*----------------------------------------------------------------------------*/

int _System getsockname (int n1, struct sockaddr* p1, int *n2)
{
    return (pfgetsockname ? pfgetsockname (n1, p1, n2) : -1);
}

/*----------------------------------------------------------------------------*/

int _System listen( int n1, int n2)
{
    return (pflisten ? pflisten (n1, n2) : -1);
}

/*----------------------------------------------------------------------------*/

int _System recvfrom (
    int n1, sal_Char* p1, int n2, int n3, struct sockaddr* p2, int *n4)
{
    return (pfrecvfrom ? pfrecvfrom (n1, p1, n2, n3, p2, n4) : -1);
}

/*----------------------------------------------------------------------------*/

int _System sendto (
    int n1, sal_Char* p1, int n2, int n3, struct sockaddr* p2, int n4)
{
    return (pfsend ? pfsendto (n1, p1, n2, n3, p2, n4) : -1);
}

/*----------------------------------------------------------------------------*/

int _System setsockopt (int n1, int n2, int n3, sal_Char* p1, int n4)
{
    return (pfsetsockopt ? pfsetsockopt (n1, n2, n3, p1, n4) : -1);
}

/*----------------------------------------------------------------------------*/

int _System gethostname (sal_Char *name, int namelen)
{
    return (pfgethostname ? pfgethostname (name, namelen) : -1);
}

/*----------------------------------------------------------------------------*/

struct hostent* _System gethostbyaddr (sal_Char *addr, int len, int type)
{
    return (pfgethostbyaddr ? pfgethostbyaddr (addr, len, type) : 0);
}

/*----------------------------------------------------------------------------*/

struct hostent* _System gethostbyname (sal_Char *p1)
{
    return (pfgethostbyname ? pfgethostbyname (p1) : 0);
}

/*----------------------------------------------------------------------------*/

int _System getsockopt (int n1, int n2, int n3, sal_Char* p1, int * p2)
{
    return (pfgetsockopt ? pfgetsockopt (n1, n2, n3, p1, p2) : -1);
}

/*----------------------------------------------------------------------------*/

int _System shutdown (int n1, int n2)
{
    return (pfshutdown ? pfshutdown (n1, n2) : -1);
}

/*----------------------------------------------------------------------------*/

void _System addsockettolist (int n1)
{
    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return;

    if( pfaddsockettolist )
    	pfaddsockettolist( n1 );
}

/*----------------------------------------------------------------------------*/

void _System removesocketfromlist (int n1)
{
    if( pfremovesocketfromlist )
    	pfremovesocketfromlist( n1 );
}

/*----------------------------------------------------------------------------*/

struct servent * _System getservbyname (sal_Char* p1, sal_Char * p2)
{
    return (pfgetservbyname ? pfgetservbyname (p1, p2) : 0);
}

/*----------------------------------------------------------------------------*/

sal_Bool ImplLoadTCPIP( void )
{
    APIRET nRet;
    /* load first TCPIP-DLL */
    nRet = DosLoadModule ((PSZ)0, 0, "SO32DLL", &hSO32DLL);
    if( hSO32DLL == NULL )
        return sal_False;

    /* load second TCPIP-DLL */
    nRet = DosLoadModule ((PSZ)0, 0, "TCP32DLL", &hTCP32DLL);
    if( hTCP32DLL == NULL )
    {
        DosFreeModule( hSO32DLL );
        hSO32DLL = NULL;
        return sal_False;
    }

    OSL_TRACE( "TCPIP geladen!\n");

    /* Load function pointer */

    nRet  = DosQueryProcAddr(hSO32DLL, 0, "BSDSELECT", (PFN*)&pfbsdselect);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SOCK_INIT", (PFN*)&pfsock_init);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SOCK_ERRNO", (PFN*)&pfsock_errno);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SOCKET", (PFN*)&pfsocket);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SOCLOSE", (PFN*)&pfsoclose);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SEND", (PFN*)&pfsend);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SENDTO", (PFN*)&pfsendto);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "CONNECT", (PFN*)&pfconnect);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "RECV", (PFN*)&pfrecv);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "RECVFROM", (PFN*)&pfrecvfrom);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "IOCTL", (PFN*)&pfioctl);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "ACCEPT", (PFN*)&pfaccept);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "BIND", (PFN*)&pfbind);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "GETPEERNAME", (PFN*)&pfgetpeername);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "GETSOCKNAME", (PFN*)&pfgetsockname);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "LISTEN", (PFN*)&pflisten);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SETSOCKOPT", (PFN*)&pfsetsockopt);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "GETSOCKOPT", (PFN*)&pfgetsockopt);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "SHUTDOWN", (PFN*)&pfshutdown);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "ADDSOCKETTOLIST", (PFN*)&pfaddsockettolist);
    nRet += DosQueryProcAddr(hSO32DLL, 0, "REMOVESOCKETFROMLIST", (PFN*)&pfremovesocketfromlist);

    nRet += DosQueryProcAddr(hTCP32DLL, 0, "BSWAP", (PFN*)&pfbswap);
    nRet += DosQueryProcAddr(hTCP32DLL, 0, "LSWAP", (PFN*)&pflswap);
    nRet += DosQueryProcAddr(hTCP32DLL, 0, "INET_ADDR", (PFN*)&pfinet_addr);
    nRet += DosQueryProcAddr(hTCP32DLL, 0, "INET_NTOA", (PFN*)&pfinet_ntoa);
    nRet += DosQueryProcAddr(hTCP32DLL, 0, "GETHOSTNAME", (PFN*)&pfgethostname);
    nRet += DosQueryProcAddr(hTCP32DLL, 0, "GETHOSTBYADDR", (PFN*)&pfgethostbyaddr);
    nRet += DosQueryProcAddr(hTCP32DLL, 0, "GETHOSTBYNAME", (PFN*)&pfgethostbyname);
    nRet += DosQueryProcAddr(hTCP32DLL, 0, "GETSERVBYNAME", (PFN*)&pfgetservbyname);

    if (nRet)
    {
        ImplFreeTCPIP();
        return FALSE;
    }

    /* initialize Sockets */
    sock_init();

    return sal_True;
}

/*----------------------------------------------------------------------------*/

void ImplFreeTCPIP( void )
{
    if( hSO32DLL != 0 )
    {
        OSL_TRACE( "TCPIP entladen!\n");

        DosFreeModule( hSO32DLL );
        hSO32DLL = 0;
    }

    if( hTCP32DLL != 0 )
    {
        DosFreeModule( hTCP32DLL );
        hTCP32DLL = 0;
    }

}

/*****************************************************************************/
/* osl_create/destroy-SocketImpl */
/*****************************************************************************/

oslSocketImpl* __osl_createSocketImpl(SOCKET Socket)
{
	oslSocketImpl* pSockImpl;

	pSockImpl = (oslSocketImpl*)calloc(1, sizeof(oslSocketImpl));

	pSockImpl->m_Socket = Socket;

	return (pSockImpl);
}

void __osl_destroySocketImpl(oslSocketImpl *pImpl)
{
	if (pImpl != NULL)
		free(pImpl);
}

/*****************************************************************************/
/* enum oslAddrFamily */
/*****************************************************************************/

/* map */
static sal_uInt32 FamilyMap[]= {
    AF_INET,                    /* osl_Socket_FamilyInet */
    AF_IPX,                        /* osl_Socket_FamilyIpx */
    0                            /* osl_Socket_FamilyInvalid */
};

/* reverse map */
static oslAddrFamily osl_AddrFamilyFromNative(sal_uInt32 nativeType)
{
    oslAddrFamily i= 0;

    while(i != osl_Socket_FamilyInvalid)
    {
        if(FamilyMap[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define FAMILY_FROM_NATIVE(y) osl_AddrFamilyFromNative(y)
#define FAMILY_TO_NATIVE(x)    (short)FamilyMap[x]

/*****************************************************************************/
/* enum oslProtocol */
/*****************************************************************************/

/* map */
static sal_uInt32 ProtocolMap[]= {
    0,                            /* osl_Socket_FamilyInet */
    NSPROTO_IPX,                  /* osl_Socket_FamilyIpx */
    NSPROTO_SPX,                  /* osl_Socket_ProtocolSpx */
    NSPROTO_SPXII,                /* osl_Socket_ProtocolSpx_ii */
    0                             /* osl_Socket_ProtocolInvalid */
};

/* reverse map */
static oslProtocol osl_ProtocolFromNative(sal_uInt32 nativeType)
{
    oslProtocol i= 0;

    while(i != osl_Socket_ProtocolInvalid)
    {
        if(ProtocolMap[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define PROTOCOL_FROM_NATIVE(y) osl_ProtocolFromNative(y)
#define PROTOCOL_TO_NATIVE(x)    ProtocolMap[x]

/*****************************************************************************/
/* enum oslSocketType */
/*****************************************************************************/

/* map */
static sal_uInt32 TypeMap[]= {
    SOCK_STREAM,                /* osl_Socket_TypeStream */
    SOCK_DGRAM,                    /* osl_Socket_TypeDgram  */
    SOCK_RAW,                    /* osl_Socket_TypeRaw */
    SOCK_RDM,                    /* osl_Socket_TypeRdm */
    SOCK_SEQPACKET,                /* osl_Socket_TypeSeqPacket */
    0                            /* osl_Socket_TypeInvalid */
};

/* reverse map */
static oslSocketType osl_SocketTypeFromNative(sal_uInt32 nativeType)
{
    oslSocketType i= 0;

    while(i != osl_Socket_TypeInvalid)
    {
        if(TypeMap[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define TYPE_TO_NATIVE(x)        TypeMap[x]
#define TYPE_FROM_NATIVE(y)        osl_SocketTypeFromNative(y)

/*****************************************************************************/
/* enum oslSocketOption */
/*****************************************************************************/

/* map */
static sal_uInt32 OptionMap[]= {
    SO_DEBUG,                    /* osl_Socket_OptionDebug */
    SO_ACCEPTCONN,                /* osl_Socket_OptionAcceptConn */
    SO_REUSEADDR,                /* osl_Socket_OptionReuseAddr */
    SO_KEEPALIVE,                /* osl_Socket_OptionKeepAlive */
    SO_DONTROUTE,                /* osl_Socket_OptionDontRoute */
    SO_BROADCAST,                /* osl_Socket_OptionBroadcast */
    SO_USELOOPBACK,                /* osl_Socket_OptionUseLoopback */
    SO_LINGER,                    /* osl_Socket_OptionLinger */
    SO_OOBINLINE,                /* osl_Socket_OptionOOBinLine */
    SO_SNDBUF,                    /* osl_Socket_OptionSndBuf */
    SO_RCVBUF,                    /* osl_Socket_OptionRcvBuf */
    SO_SNDLOWAT,                /* osl_Socket_OptionSndLowat */
    SO_RCVLOWAT,                /* osl_Socket_OptionRcvLowat */
    SO_SNDTIMEO,                /* osl_Socket_OptionSndTimeo */
    SO_RCVTIMEO,                /* osl_Socket_OptionRcvTimeo */
    SO_ERROR,                    /* osl_Socket_OptionError */
    SO_TYPE,                    /* osl_Socket_OptionType */
    TCP_NODELAY,                /* osl_Socket_OptionTcpNoDelay */
    0                            /* osl_Socket_OptionInvalid */
};

/* reverse map */
static oslSocketOption osl_SocketOptionFromNative(sal_uInt32 nativeType)
{
    oslSocketOption i= 0;

    while(i != osl_Socket_OptionInvalid)
    {
        if(OptionMap[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define OPTION_TO_NATIVE(x)        OptionMap[x]
#define OPTION_FROM_NATIVE(y)        osl_SocketOptionFromNative(y)

/*****************************************************************************/
/* enum oslSocketOptionLevel */
/*****************************************************************************/

static sal_uInt32 OptionLevelMap[]= {
    SOL_SOCKET,                    /* osl_Socket_LevelSocket */
    IPPROTO_TCP,                /* osl_Socket_LevelTcp */
    0                            /* osl_invalid_SocketLevel */
};

/* reverse map */
static oslSocketOptionLevel osl_SocketOptionLevelFromNative(sal_uInt32 nativeType)
{
    oslSocketOptionLevel i= 0;

	while(i != osl_Socket_OptionInvalid)
    {
        if(OptionLevelMap[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define OPTION_LEVEL_TO_NATIVE(x)        OptionLevelMap[x]
#define OPTION_LEVEL_FROM_NATIVE(y)        osl_SocketOptionLevelFromNative(y)

/*****************************************************************************/
/* enum oslSocketMsgFlag */
/*****************************************************************************/

static sal_uInt32 SocketMsgFlagMap[]= {
    0,                            /* osl_Socket_MsgNormal */
    MSG_OOB,                    /* osl_Socket_MsgOOB */
    MSG_PEEK,                    /* osl_Socket_MsgPeek */
    MSG_DONTROUTE,                /* osl_Socket_MsgDontRoute */
    MSG_MAXIOVLEN                /* osl_Socket_MsgMaxIOVLen */
};

/* reverse map */
static oslSocketMsgFlag osl_SocketMsgFlagFromNative(sal_uInt32 nativeType)
{
    oslSocketMsgFlag i= 0;

    while(i != osl_Socket_MsgInvalid)
    {
        if(SocketMsgFlagMap[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define MSG_FLAG_TO_NATIVE(x)        SocketMsgFlagMap[x]
#define MSG_FLAG_FROM_NATIVE(y)        osl_SocketMsgFlagFromNative(y)

/*****************************************************************************/
/* enum oslSocketDirection */
/*****************************************************************************/

static sal_uInt32 SocketDirection[]= {
    SD_RECEIVE,                    /* osl_Socket_DirRead */
    SD_SEND,                    /* osl_Socket_DirWrite */
    SD_BOTH                        /* osl_Socket_DirReadwrite */
};

/* reverse map */
static oslSocketDirection osl_SocketDirectionFromNative(sal_uInt32 nativeType)
{
    oslSocketDirection i= 0;

    while(i != osl_Socket_DirInvalid)
    {
        if(SocketDirection[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define DIRECTION_TO_NATIVE(x)        SocketDirection[x]
#define DIRECTION_FROM_NATIVE(y)    osl_SocketDirectionFromNative(y)

/*****************************************************************************/
/* enum oslSocketError */
/*****************************************************************************/

static int SocketError[]= {

    0,                    /* no error */
    ENOTSOCK,            /* Socket operation on non-socket */
    EDESTADDRREQ,        /* Destination address required */
    EMSGSIZE,            /* Message too long */
    EPROTOTYPE,            /* Protocol wrong type for socket */
    ENOPROTOOPT,        /* Protocol not available */
    EPROTONOSUPPORT,    /* Protocol not supported */
    ESOCKTNOSUPPORT,    /* Socket type not supported */
    EOPNOTSUPP,            /* Operation not supported on socket */
    EPFNOSUPPORT,        /* Protocol family not supported */
    EAFNOSUPPORT,        /* Address family not supported by */
                            /* protocol family */
    EADDRINUSE,            /* Address already in use */
    EADDRNOTAVAIL,        /* Can't assign requested address */
    ENETDOWN,            /* Network is down */
    ENETUNREACH,        /* Network is unreachable */
    ENETRESET,            /* Network dropped connection because */
                            /* of reset */
    ECONNABORTED,        /* Software caused connection abort */
    ECONNRESET,            /* Connection reset by peer */
    ENOBUFS,            /* No buffer space available */
    EISCONN,            /* Socket is already connected */
    ENOTCONN,            /* Socket is not connected */
    ESHUTDOWN,            /* Can't send after socket shutdown */
    ETOOMANYREFS,        /* Too many references: can't splice */
    ETIMEDOUT,            /* Connection timed out */
    ECONNREFUSED,        /* Connection refused */
    EHOSTDOWN,            /* Host is down */
    EHOSTUNREACH,        /* No route to host */
    EWOULDBLOCK,        /* call would block on non-blocking socket */
    EALREADY,            /* operation already in progress */
    EINPROGRESS            /* operation now in progress */
};

/* reverse map */
static oslSocketError osl_SocketErrorFromNative(int nativeType)
{
    oslSocketError i= 0;

    while(i != osl_Socket_E_InvalidError)
    {
        if(SocketError[i] == nativeType)
            return i;
        i++;
    }

    return i;
}

/* macros */
#define ERROR_TO_NATIVE(x)      SocketError[x]
#define ERROR_FROM_NATIVE(y)    osl_SocketErrorFromNative(y)

/*****************************************************************************/
/* osl_getLastSocketError  */
/*****************************************************************************/
oslSocketError SAL_CALL osl_getLastSocketError(oslSocket Socket)
{
    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return 0;

    return ERROR_FROM_NATIVE(sock_errno());
}

/*****************************************************************************/
/* osl_createEmptySocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_createEmptySocketAddr(oslAddrFamily Family)
{
    struct sockaddr* pAddr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketAddr)NULL);

    pAddr= malloc(sizeof(struct sockaddr));

    /* is it an internet-Addr? */
    if(Family == osl_Socket_FamilyInet)
    {
        struct sockaddr_in* pInetAddr= (struct sockaddr_in*)pAddr;

        pInetAddr->sin_family= FAMILY_TO_NATIVE(osl_Socket_FamilyInet);
        pInetAddr->sin_addr.s_addr= htonl(INADDR_ANY);
        pInetAddr->sin_port= 0;
    }
    else
    {
        memset(pAddr, 0, sizeof(struct sockaddr));
        pAddr->sa_family= FAMILY_TO_NATIVE(Family);
    }

    return (oslSocketAddr)pAddr;
}

/*****************************************************************************/
/* osl_copySocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_copySocketAddr(oslSocketAddr Addr)
{
    struct sockaddr* pAddr= NULL;

    if (Addr)
    {
        pAddr= malloc(sizeof(struct sockaddr));
        OSL_ASSERT(pAddr);

        if (pAddr)
            memcpy(pAddr, Addr, sizeof(struct sockaddr));
    }

    return pAddr;
}

/*****************************************************************************/
/* osl_isEqualSocketAddr */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isEqualSocketAddr(oslSocketAddr Addr1, oslSocketAddr Addr2)
{
	struct sockaddr* pAddr1= (struct sockaddr*)Addr1;
	struct sockaddr* pAddr2= (struct sockaddr*)Addr2;

	OSL_ASSERT(pAddr1);
	OSL_ASSERT(pAddr2);

	if (pAddr1->sa_family == pAddr2->sa_family)
	{
		switch (pAddr1->sa_family)
		{
			case AF_INET:
			{
				struct sockaddr_in* pInetAddr1= (struct sockaddr_in*)pAddr1;
				struct sockaddr_in* pInetAddr2= (struct sockaddr_in*)pAddr2;

				if ((pInetAddr1->sin_family == pInetAddr2->sin_family) &&
					(pInetAddr1->sin_addr.s_addr == pInetAddr2->sin_addr.s_addr) &&
					(pInetAddr1->sin_port == pInetAddr2->sin_port))
					return (sal_True);
			}

			case AF_IPX:
			{
				struct sockaddr_ipx* pIpxAddr1= (struct sockaddr_ipx*)pAddr1;
				struct sockaddr_ipx* pIpxAddr2= (struct sockaddr_ipx*)pAddr2;

				if ((pIpxAddr1->sa_family == pIpxAddr2->sa_family) &&
					(memcmp(&pIpxAddr1->sa_netnum, &pIpxAddr2->sa_netnum, sizeof(pIpxAddr1->sa_netnum)) == 0) &&
					(memcmp(&pIpxAddr1->sa_nodenum, &pIpxAddr2->sa_nodenum, sizeof(pIpxAddr1->sa_nodenum)) == 0) &&
					(pIpxAddr1->sa_socket == pIpxAddr2->sa_socket))
					return (sal_True);
			}

			default:
			{
				return (memcmp(pAddr1, Addr2, sizeof(struct sockaddr)) == 0);
			}
		}
	}

	return (sal_False);
}

/*****************************************************************************/
/* osl_createInetSocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_createInetSocketAddr(const sal_Char* pszDottedAddr,
                                       sal_Int32 Port)
{
    sal_uInt32 Addr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketAddr)NULL);

    Addr= inet_addr((sal_Char*)pszDottedAddr);
    if(Addr != -1)
    {
        /* valid dotted addr */
        struct sockaddr_in* pInetAddr;

        pInetAddr= malloc(sizeof(struct sockaddr_in));
        OSL_ASSERT(pInetAddr);

        if (pInetAddr)
        {
            pInetAddr->sin_family=  FAMILY_TO_NATIVE(osl_Socket_FamilyInet);
            pInetAddr->sin_port= htons((sal_uInt16)Port);
            pInetAddr->sin_addr.s_addr= Addr; /* already in right order */
        }

        return (oslSocketAddr)pInetAddr;
    }

    return (oslSocketAddr)0;
}

/*****************************************************************************/
/* osl_createIpxSocketAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_createIpxSocketAddr(const sal_Char NetNumber[4],
                                      const sal_Char NodeNumber[6],
                                      sal_uInt32   SocketNumber)
{
    struct sockaddr_ipx* pIpxAddr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketAddr)NULL);

    pIpxAddr= malloc(sizeof(struct sockaddr));
    OSL_ASSERT(pIpxAddr);

    if (pIpxAddr)
    {
        pIpxAddr->sa_family= FAMILY_TO_NATIVE(osl_Socket_FamilyIpx);
        memcpy(&pIpxAddr->sa_netnum,
               NetNumber,
               sizeof(pIpxAddr->sa_netnum));
        memcpy(&pIpxAddr->sa_nodenum,
               NodeNumber,
               sizeof(pIpxAddr->sa_nodenum));
        pIpxAddr->sa_socket= (sal_uInt16)SocketNumber;
    }

    return (oslSocketAddr)pIpxAddr;
}

/*****************************************************************************/
/* oslHostAddr */
/*****************************************************************************/
typedef struct oslHostAddrImpl_t {
    sal_Char            *pHostName;
    struct sockaddr *pSockAddr;
} oslHostAddrImpl;

static oslHostAddr _osl_hostentToHostAddr (const struct hostent *he)
{
    oslHostAddrImpl *pAddr= NULL;
    struct sockaddr *sa;
    sal_Char            *cn;

    if ((he == NULL) || (he->h_name == NULL) || (he->h_addr_list[0] == NULL))
        return ((oslHostAddr)NULL);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslHostAddr)NULL);

    cn= (sal_Char *)malloc (strlen (he->h_name) + 1);
    OSL_ASSERT(cn);
    if (cn == NULL)
        return ((oslHostAddr)NULL);

    strcpy (cn, he->h_name);

    sa= (struct sockaddr *)malloc (sizeof (struct sockaddr));
    OSL_ASSERT(sa);
    if (sa == NULL)
    {
        free (cn);
        return ((oslHostAddr)NULL);
    }
    memset (sa, 0, sizeof (struct sockaddr));

    sa->sa_family= he->h_addrtype;
    if (sa->sa_family == FAMILY_TO_NATIVE(osl_Socket_FamilyInet))
    {
        struct sockaddr_in *sin= (struct sockaddr_in *)sa;
        memcpy (
            &(sin->sin_addr.s_addr),
            he->h_addr_list[0],
            he->h_length);
    }
    else if (sa->sa_family == FAMILY_TO_NATIVE(osl_Socket_FamilyIpx))
    {
        struct sockaddr_ipx *spx= (struct sockaddr_ipx *)sa;

        /* not quite sure what ipx/spx defines as a host-address:  */
        /* just the hostnumber of the combination of net- and hostnumber */
        /* I guess the later, but am prepared for anything */

        switch (he->h_length)
        {
            case sizeof(spx->sa_netnum) + sizeof(spx->sa_nodenum):
            case sizeof(spx->sa_netnum):
                memcpy (
                    &(spx->sa_netnum),
                    he->h_addr_list[0],
                    he->h_length);
                break;

            case sizeof(spx->sa_nodenum):
                memcpy (
                    &(spx->sa_nodenum),
                    he->h_addr_list[0],
                    he->h_length);
                break;

            default:
#if OSL_DEBUG_LEVEL > 1
                OSL_TRACE("_osl_hostentToHostAddr(): unknown IPX/SPX address format.\n");
#endif
                OSL_ASSERT(sal_False);

                free (sa);
                free (cn);
                return ((oslHostAddr)NULL);
        }
    }
    else
    {
        /* unknown address family */
        /* future extensions for new families might be implemented here */

#if OSL_DEBUG_LEVEL > 1
        OSL_TRACE("_osl_hostentToHostAddr(): unknown address family.\n");
#endif
        OSL_ASSERT(sal_False);

        free (sa);
        free (cn);
        return ((oslHostAddr)NULL);
    }

    pAddr= (oslHostAddrImpl *)malloc (sizeof (oslHostAddrImpl));
    OSL_ASSERT(pAddr);
    if (pAddr == NULL)
    {
        free (sa);
        free (cn);
        return ((oslHostAddr)NULL);
    }

    pAddr->pHostName= cn;
    pAddr->pSockAddr= sa;

    return ((oslHostAddr)pAddr);
}

/*****************************************************************************/
/* osl_createHostAddr */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_createHostAddr (const sal_Char *pszHostname,
                                const oslSocketAddr Addr)
{
    oslHostAddrImpl *pAddr;
    struct sockaddr *sa;
    sal_Char            *cn;

    OSL_ASSERT(pszHostname && Addr);
    if ((pszHostname == NULL) || (Addr == NULL))
        return ((oslHostAddr)NULL);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslHostAddr)NULL);

    cn= (sal_Char *)malloc (strlen (pszHostname) + 1);
    OSL_ASSERT(cn);
    if (cn == NULL)
        return ((oslHostAddr)NULL);

    strcpy (cn, pszHostname);

    sa= (struct sockaddr *)osl_copySocketAddr (Addr);
    if (sa == NULL)
    {
        free (cn);
        return ((oslHostAddr)NULL);
    }

    pAddr= (oslHostAddrImpl *)malloc (sizeof (oslHostAddrImpl));
    OSL_ASSERT(pAddr);
    if (pAddr == NULL)
    {
        free (cn);
        free (sa);
        return ((oslHostAddr)NULL);
    }

    pAddr->pHostName= cn;
    pAddr->pSockAddr= sa;

    return ((oslHostAddr)pAddr);
}

/*****************************************************************************/
/* osl_createHostAddrByName */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_createHostAddrByName (const sal_Char *pszHostname)
{
    struct hostent *he;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslHostAddr)NULL);

    he = gethostbyname ((sal_Char *)pszHostname);

    return _osl_hostentToHostAddr (he);
}

/*****************************************************************************/
/* osl_createHostAddrByAddr */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_createHostAddrByAddr (const oslSocketAddr Addr)
{
    const struct sockaddr *pAddr= (const struct sockaddr *)Addr;
    OSL_ASSERT(pAddr);

    if (pAddr == NULL)
        return ((oslHostAddr)NULL);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslHostAddr)NULL);

    if (pAddr->sa_family == FAMILY_TO_NATIVE(osl_Socket_FamilyInet))
    {
        struct sockaddr_in sin;
/*        const struct sockaddr_in *sin= (const struct sockaddr_in *)pAddr;*/
        struct hostent *he;

        memcpy(&sin, pAddr, sizeof(struct sockaddr_in));
        if (sin.sin_addr.s_addr == htonl(INADDR_ANY))
            return ((oslHostAddr)NULL);

        he= gethostbyaddr ((sal_Char *)&(sin.sin_addr),
                           sizeof (sin.sin_addr),
                           sin.sin_family);

        return _osl_hostentToHostAddr (he);
    }

    return ((oslHostAddr)NULL);
}

/*****************************************************************************/
/* osl_copyHostAddr */
/*****************************************************************************/
oslHostAddr SAL_CALL osl_copyHostAddr (const oslHostAddr Addr)
{
    oslHostAddrImpl *pAddr= (oslHostAddrImpl *)Addr;
    OSL_ASSERT(pAddr);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslHostAddr)NULL);

    if (pAddr)
        return osl_createHostAddr (pAddr->pHostName, pAddr->pSockAddr);
    else
        return ((oslHostAddr)NULL);
}

/*****************************************************************************/
/* osl_getHostnameOfHostAddr */
/*****************************************************************************/
const sal_Char* SAL_CALL osl_getHostnameOfHostAddr (const oslHostAddr Addr)
{
    const oslHostAddrImpl *pAddr= (const oslHostAddrImpl *)Addr;
    OSL_ASSERT(pAddr);

    if (pAddr)
        return pAddr->pHostName;
    else
        return NULL;
}

/*****************************************************************************/
/* osl_getSocketAddrOfHostAddr */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_getSocketAddrOfHostAddr (const oslHostAddr Addr)
{
    const oslHostAddrImpl *pAddr= (const oslHostAddrImpl *)Addr;
    OSL_ASSERT(pAddr);

    if (pAddr)
        return ((const oslSocketAddr)(pAddr->pSockAddr));
    else
        return NULL;
}

/*****************************************************************************/
/* osl_destroyHostAddr */
/*****************************************************************************/
void SAL_CALL osl_destroyHostAddr (oslHostAddr Addr)
{
    oslHostAddrImpl *pAddr= (oslHostAddrImpl *)Addr;
    if (pAddr)
    {
        if (pAddr->pHostName)
            free (pAddr->pHostName);
        if (pAddr->pSockAddr)
            osl_destroySocketAddr (pAddr->pSockAddr);
        free (pAddr);
    }
}

/*****************************************************************************/
/* osl_getLocalHostname */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getLocalHostname (sal_Char *pBuffer, sal_uInt32 nBufLen)
{
	static sal_Char LocalHostname[256] = "";

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketResult)NULL);

	if (strlen(LocalHostname) == 0)
	{
		if (gethostname(LocalHostname, sizeof(LocalHostname)) == 0)
		{
			/* check if we have an FQDN */
	    	if (strchr(LocalHostname, '.') == NULL)
	        {
	        	const sal_Char *pStr;
				oslHostAddr Addr;

				/* no, determine it via dns */
				Addr = osl_createHostAddrByName(LocalHostname);

				if ((pStr = osl_getHostnameOfHostAddr(Addr)) != NULL)
					strcpy(LocalHostname, pStr);

				osl_destroyHostAddr(Addr);
			}
		}
	}

	if (strlen(LocalHostname) > 0)
	{
		strncpy(pBuffer, LocalHostname, nBufLen);
		pBuffer[nBufLen - 1] = '\0';

		return osl_Socket_Ok;
	}

	return osl_Socket_Error;
}

/*****************************************************************************/
/* osl_resolveHostname */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_resolveHostname(const sal_Char* pszHostname)
{
    oslHostAddrImpl *pAddr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketAddr)NULL);

    pAddr= (oslHostAddrImpl *)osl_createHostAddrByName (pszHostname);

    if (pAddr)
    {
        oslSocketAddr SockAddr = osl_copySocketAddr(pAddr->pSockAddr);

        osl_destroyHostAddr(pAddr);

        return (SockAddr);
    }
    return ((oslSocketAddr)NULL);
}

/*****************************************************************************/
/* osl_getServicePort */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_getServicePort(const sal_Char* pszServicename,
                       const sal_Char* pszProtocol)
{
    struct servent* ps;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return 0;

    ps= getservbyname((sal_Char*)pszServicename, (sal_Char*)pszProtocol);

    if (ps != 0)
    {
        return ntohs(ps->s_port);
    }

    return OSL_INVALID_PORT;
}

/*****************************************************************************/
/* osl_destroySocketAddr */
/*****************************************************************************/
void SAL_CALL osl_destroySocketAddr(oslSocketAddr Addr)
{
    if(Addr)
        free(Addr);
}

/*****************************************************************************/
/* osl_getFamilyOfSocketAddr */
/*****************************************************************************/
oslAddrFamily SAL_CALL osl_getFamilyOfSocketAddr(oslSocketAddr Addr)
{
    struct sockaddr* pAddr= (struct sockaddr*)Addr;
    OSL_ASSERT(pAddr);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslAddrFamily)NULL);

    if (pAddr)
        return FAMILY_FROM_NATIVE(pAddr->sa_family);
    else
        return osl_Socket_FamilyInvalid;
}

/*****************************************************************************/
/* osl_getInetPortOfSocketAddr */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_getInetPortOfSocketAddr(oslSocketAddr Addr)
{
    struct sockaddr_in* pAddr= (struct sockaddr_in*)Addr;
    OSL_ASSERT(pAddr);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return 0;

    if (pAddr && (pAddr->sin_family == FAMILY_TO_NATIVE(osl_Socket_FamilyInet)))
        return ntohs(pAddr->sin_port);
    else
        return OSL_INVALID_PORT;
}

/*****************************************************************************/
/* osl_setInetPortOfSocketAddr */
/*****************************************************************************/
sal_Bool SAL_CALL osl_setInetPortOfSocketAddr(oslSocketAddr Addr, sal_Int32 Port)
{
    struct sockaddr_in* pAddr= (struct sockaddr_in*)Addr;
    OSL_ASSERT(pAddr);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    if (pAddr && (pAddr->sin_family == FAMILY_TO_NATIVE(osl_Socket_FamilyInet)))
    {
        pAddr->sin_port= htons((short)Port);

        return sal_True;
    }

    /* this is not a inet-addr => can't set port */
    return sal_False;
}

/*****************************************************************************/
/* osl_getHostnameOfSocketAddr */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getHostnameOfSocketAddr(oslSocketAddr Addr,
                                            sal_Char *pBuffer, sal_uInt32 BufferSize)
{
    oslHostAddrImpl *pAddr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketResult)NULL);

    pAddr= (oslHostAddrImpl *)osl_createHostAddrByAddr (Addr);

    if (pAddr)
    {
        strncpy(pBuffer, pAddr->pHostName, BufferSize);
        pBuffer[BufferSize - 1] = '\0';

        osl_destroyHostAddr(pAddr);

        return osl_Socket_Ok;
    }

    return osl_Socket_Error;
}

/*****************************************************************************/
/* osl_getDottedInetAddrOfSocketAddr */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getDottedInetAddrOfSocketAddr(
	oslSocketAddr Addr, sal_Char *pBuffer, sal_uInt32 BufferSize)
{
    struct sockaddr_in* pAddr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketResult)NULL);

    pAddr= (struct sockaddr_in*)Addr;
    OSL_ASSERT(pAddr);

    if (pAddr == NULL)
        return osl_Socket_Error;

    if (pAddr->sin_family != FAMILY_TO_NATIVE(osl_Socket_FamilyInet))
        return osl_Socket_Error;

    strncpy(pBuffer, inet_ntoa(pAddr->sin_addr), BufferSize);
    pBuffer[BufferSize - 1] = '\0';

    return osl_Socket_Ok;
}

/*****************************************************************************/
/* osl_getIpxNetNumber  */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getIpxNetNumber(oslSocketAddr Addr,
                                    oslSocketIpxNetNumber NetNumber)
{
    struct sockaddr_ipx* pAddr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketResult)NULL);

    pAddr= (struct sockaddr_ipx*)Addr;
    OSL_ASSERT(pAddr);

    if (pAddr && (pAddr->sa_family == FAMILY_TO_NATIVE(osl_Socket_FamilyIpx)))
    {
        memcpy(NetNumber, pAddr->sa_netnum, sizeof(NetNumber));

        return osl_Socket_Ok;
    }
    else
        return osl_Socket_Error;
}

/*****************************************************************************/
/* osl_getIpxNodeNumber  */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_getIpxNodeNumber(oslSocketAddr Addr,
                                     oslSocketIpxNodeNumber NodeNumber)
{
    struct sockaddr_ipx* pAddr;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketResult)NULL);

    pAddr= (struct sockaddr_ipx*)Addr;
    OSL_ASSERT(pAddr);

    if (pAddr && (pAddr->sa_family == FAMILY_TO_NATIVE(osl_Socket_FamilyIpx)))
    {
        memcpy(NodeNumber, pAddr->sa_nodenum, sizeof(NodeNumber));

        return osl_Socket_Ok;
    }
    else
        return osl_Socket_Error;
}

/*****************************************************************************/
/* osl_getIpxSocketNumber  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_getIpxSocketNumber(oslSocketAddr Addr)
{
    struct sockaddr_ipx* pAddr= (struct sockaddr_ipx*)Addr;
    OSL_ASSERT(pAddr);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return 0;

    if (pAddr && (pAddr->sa_family == FAMILY_TO_NATIVE(osl_Socket_FamilyIpx)))
        return pAddr->sa_socket;
    else
        return OSL_INVALID_IPX_SOCKET_NO;
}

#include "sockimpl.h"

/*****************************************************************************/
/* osl_createSocket  */
/*****************************************************************************/
oslSocket SAL_CALL osl_createSocket(oslAddrFamily Family,
                           oslSocketType Type,
                           oslProtocol   Protocol)
{
    oslSocketImpl* pSockImpl;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocket)NULL);

    OSL_TRACE("Creating Socket. Family: %d Type: %d Protocol: %d\n",
              FAMILY_TO_NATIVE(Family),
              TYPE_TO_NATIVE(Type),
              PROTOCOL_TO_NATIVE(Protocol));

    /* alloc memory */
	pSockImpl = __osl_createSocketImpl( 0 );
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL)
        return ((oslSocket)NULL);

    /* create socket */
    pSockImpl->m_Socket= socket(FAMILY_TO_NATIVE(Family),
                                TYPE_TO_NATIVE(Type),
                                PROTOCOL_TO_NATIVE(Protocol));

    /* creation failed => free memory */
    if(pSockImpl->m_Socket == OSL_INVALID_SOCKET)
    {
		__osl_destroySocketImpl( pSockImpl );
        pSockImpl= NULL;
    }

    return ((oslSocket)pSockImpl);
}

/*****************************************************************************/
/* osl_copySocket  */
/*****************************************************************************/
oslSocket SAL_CALL osl_copySocket(oslSocket Socket)
{
    oslSocketImpl* pSockImpl;
    oslSocketImpl* pParamSockImpl;

    pParamSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pParamSockImpl);

    if (pParamSockImpl == NULL) /* ENOTSOCK */
        return ((oslSocket)NULL);

    /* alloc memory */
	pSockImpl = __osl_createSocketImpl( 0 );
    OSL_ASSERT(pSockImpl);

    if (pSockImpl)
    {
        /* copy socket */
        pSockImpl->m_Socket= pParamSockImpl->m_Socket;
        pSockImpl->m_Flags= pParamSockImpl->m_Flags;
        pSockImpl->m_CloseCallback = pParamSockImpl->m_CloseCallback;
        pSockImpl->m_CallbackArg = pParamSockImpl->m_CallbackArg;
    }

    return ((oslSocket)pSockImpl);
}

/*****************************************************************************/
/* osl_destroySocket  */
/*****************************************************************************/
void SAL_CALL osl_destroySocket(oslSocket Socket)
{
    oslSocketImpl* pSockImpl;

    /* socket already invalid */
    if(Socket==0)
        return;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return;

    pSockImpl= (oslSocketImpl*)Socket;

    /* close */
    if( !(pSockImpl->m_Flags & OSL_SOCKET_FLAGS_SHARED) )
    {
        OSL_TRACE( "Closing Socket (%d).\n", pSockImpl->m_Socket);

        soclose(pSockImpl->m_Socket);
    }
    else
    {
	    OSL_TRACE( "Freeing Socket (%d).\n", pSockImpl->m_Socket);

	    /* call registered caback */
	    if (pSockImpl->m_CloseCallback != NULL)
	    {
	    	pSockImpl->m_CloseCallback(pSockImpl->m_CallbackArg);
	    }
	}

    /* free memory */
	__osl_destroySocketImpl( pSockImpl );
}

/*****************************************************************************/
/* osl_closeSocket  */
/*****************************************************************************/
void SAL_CALL osl_closeSocket(oslSocket Socket)
{
    oslSocketImpl* pSockImpl;

    /* socket already invalid */
    if(Socket==0)
        return;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return;

    pSockImpl= (oslSocketImpl*)Socket;

    /* close */
    if( !(pSockImpl->m_Flags & OSL_SOCKET_FLAGS_SHARED) )
    {
        OSL_TRACE( "Closing Socket (%d).\n", pSockImpl->m_Socket);

        soclose(pSockImpl->m_Socket);

		pSockImpl->m_Socket = NULL;
    }
    else
    {
	    OSL_TRACE( "Freeing Socket (%d).\n", pSockImpl->m_Socket);

	    /* call registered caback */
	    if (pSockImpl->m_CloseCallback != NULL)
	    {
	    	pSockImpl->m_CloseCallback(pSockImpl->m_CallbackArg);
	    }
	}
}

/*****************************************************************************/
/* osl_getLocalAddrOfSocket  */
/* Note that I rely on the fact that oslSocketAddr and struct sockaddr */
/* are the same! I don't like it very much but see no other easy way */
/* to conceal the struct sockaddr from the eyes of the user. */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_getLocalAddrOfSocket(oslSocket Socket)
{
    oslSocketImpl*  pSockImpl;
    struct sockaddr Addr;
    int             AddrLen;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return ((oslSocketAddr)NULL);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketAddr)NULL);

    AddrLen= sizeof(struct sockaddr);

    if (getsockname(pSockImpl->m_Socket, &Addr, &AddrLen) == OSL_SOCKET_ERROR)
        return ((oslSocketAddr)NULL);

    return osl_copySocketAddr((oslSocketAddr)&Addr);
}

/*****************************************************************************/
/* osl_getPeerAddrOfSocket  */
/*****************************************************************************/
oslSocketAddr SAL_CALL osl_getPeerAddrOfSocket(oslSocket Socket)
{
    oslSocketImpl*  pSockImpl;
    struct sockaddr Addr;
    int             AddrLen;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return ((oslSocketAddr)NULL);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketAddr)NULL);

    AddrLen= sizeof(struct sockaddr);

    if (getpeername(pSockImpl->m_Socket, &Addr, &AddrLen) == OSL_SOCKET_ERROR)
        return ((oslSocketAddr)NULL);

    return osl_copySocketAddr((oslSocketAddr)&Addr);
}

/*****************************************************************************/
/* osl_bindAddrToSocket  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_bindAddrToSocket(oslSocket Socket,
                             oslSocketAddr Addr)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return (bind(pSockImpl->m_Socket,
                 (struct sockaddr*)Addr,
                 sizeof(struct sockaddr)) != OSL_SOCKET_ERROR);
}

/*****************************************************************************/
/* osl_connectSocketTo  */
/*****************************************************************************/
oslSocketResult SAL_CALL osl_connectSocketTo(oslSocket       Socket,
                                    oslSocketAddr   Addr,
                                    const TimeValue *pTimeout)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return osl_Socket_Error;

    OSL_ASSERT(Addr);

    if (Addr == NULL) /* EDESTADDRREQ */
        return osl_Socket_Error;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocketResult)NULL);

    if (pTimeout == NULL)
    {
        if (connect(pSockImpl->m_Socket,
                    (struct sockaddr*)Addr,
                    sizeof(struct sockaddr)) == OSL_SOCKET_ERROR)
            return osl_Socket_Error;
        else
            return osl_Socket_Ok;
    }
    else
    {
        OSL_TRACE("osl_connectSocketTo(): Sorry, timeout not yet implemented.\n" );

        if (connect(pSockImpl->m_Socket,
                    (struct sockaddr*)Addr,
                    sizeof(struct sockaddr)) == OSL_SOCKET_ERROR)
            return osl_Socket_Error;
        else
            return osl_Socket_Ok;
    }
}

/*****************************************************************************/
/* osl_listenOnSocket  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_listenOnSocket(oslSocket Socket,
                           sal_uInt32 MaxPendingConnections)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return (listen(pSockImpl->m_Socket,
                   MaxPendingConnections == -1 ?
                   SOMAXCONN :
                   MaxPendingConnections) != OSL_SOCKET_ERROR);
}

/*****************************************************************************/
/* osl_acceptConnectionOnSocket  */
/*****************************************************************************/
oslSocket SAL_CALL osl_acceptConnectionOnSocket(oslSocket      Socket,
                                       oslSocketAddr* pAddr)
{
    oslSocketImpl*  pSockImpl;
    struct sockaddr Addr;
    int             AddrLen;
    SOCKET          Connection;
    oslSocketImpl*  pConnectionSockImpl;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return ((oslSocket)NULL);

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return ((oslSocket)NULL);

    AddrLen= sizeof(struct sockaddr);

    Connection= accept(pSockImpl->m_Socket,
                       (struct sockaddr*)&Addr,
                       &AddrLen);

    OSL_ASSERT(AddrLen == sizeof(struct sockaddr));

    /* accept failed? */
    if (Connection == OSL_SOCKET_ERROR)
        return ((oslSocket)NULL);

    /* copy peer-addr, if user is interested in */
    if (pAddr)
        *pAddr= osl_copySocketAddr((oslSocketAddr)&Addr);

    /* alloc memory */
	pConnectionSockImpl = __osl_createSocketImpl( Connection );
    pConnectionSockImpl->m_Flags = 0;
    pConnectionSockImpl->m_CloseCallback = NULL;
    pConnectionSockImpl->m_CallbackArg = NULL;

    return ((oslSocket)pConnectionSockImpl);
}

/*****************************************************************************/
/* osl_receiveSocket  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_receiveSocket(oslSocket        Socket,
                      void*            pBuffer,
                      sal_uInt32           BytesToRead,
                      oslSocketMsgFlag Flag)
{
    oslSocketImpl* pSockImpl;
    int erg;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return osl_Socket_Error;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return osl_Socket_Error;

    /* TCP/IP <= 4.0 is a 16bit implementation: limit buffer to 32K-1 == 32767 !*/
    if( BytesToRead > 32767 )
        BytesToRead = 32767;

    OSL_TRACE( "recv (%d): try to read %d bytes.\n", pSockImpl->m_Socket, BytesToRead );

    erg = recv(pSockImpl->m_Socket,
                (sal_Char*)pBuffer,
                BytesToRead,
                MSG_FLAG_TO_NATIVE(Flag));

    if( (erg == -1) && (sock_errno() == 10004) )
    {
    	OSL_TRACE( "recv (%d): retry.\n", pSockImpl->m_Socket );
	    erg = recv(pSockImpl->m_Socket,
    	            (sal_Char*)pBuffer,
        	        BytesToRead,
            	    MSG_FLAG_TO_NATIVE(Flag));
    }

    OSL_TRACE( "recv (%d): result: %d (SocketError: %d).\n", pSockImpl->m_Socket, erg, sock_errno());

    return erg;
}

/*****************************************************************************/
/* osl_receiveFromSocket  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_receiveFromSocket(oslSocket Socket,
                          oslSocketAddr SenderAddr,
                          void* pBuffer,
                          sal_uInt32 BufferSize,
                          oslSocketMsgFlag Flag)
{
    oslSocketImpl* pSockImpl;
    int AddrLen= SenderAddr == 0 ? 0 : sizeof(struct sockaddr);

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return osl_Socket_Error;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return osl_Socket_Error;

    /* TCP/IP <= 4.0 is a 16bit implementation: limit buffer to 32K-1 !*/
    if( BufferSize > 32767 )
        BufferSize = 32767;

    return recvfrom(pSockImpl->m_Socket,
                    (sal_Char*)pBuffer,
                    BufferSize,
                    MSG_FLAG_TO_NATIVE(Flag),
                    (struct sockaddr*)SenderAddr,
                    &AddrLen);
}

/*****************************************************************************/
/* osl_sendSocket  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_sendSocket(oslSocket Socket,
                   const void* pBuffer,
                   sal_uInt32 BytesToSend,
                   oslSocketMsgFlag Flag)
{
    oslSocketImpl* pSockImpl;
    int erg;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return osl_Socket_Error;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return osl_Socket_Error;

    /* TCP/IP <= 4.0 is a 16bit implementation: limit buffer to 32K-1 !*/
    if( BytesToSend > 32767 )
        BytesToSend = 32767;

    OSL_TRACE( "send (%d): try to write %d bytes.\n", pSockImpl->m_Socket, BytesToSend );

    erg = send(pSockImpl->m_Socket,
               (sal_Char*)pBuffer,
               BytesToSend,
               MSG_FLAG_TO_NATIVE(Flag));

    if( (erg == -1) && (sock_errno() == 10004) )
    {
    	OSL_TRACE( "send (%d): retry.\n", pSockImpl->m_Socket );
        erg = send(pSockImpl->m_Socket,
                   (sal_Char*)pBuffer,
                   BytesToSend,
                   MSG_FLAG_TO_NATIVE(Flag));
    }

    OSL_TRACE( "send (%d): result: %d (SocketError: %d).\n", pSockImpl->m_Socket, erg, sock_errno());

	return erg;
}

/*****************************************************************************/
/* osl_sendToSocket  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_sendToSocket(oslSocket Socket,
                     oslSocketAddr ReceiverAddr,
                     const void* pBuffer,
                     sal_uInt32 BytesToSend,
                     oslSocketMsgFlag Flag)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return osl_Socket_Error;

    /* ReceiverAddr might be 0 when used on a connected socket. */
    /* Then sendto should behave like send. */

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return osl_Socket_Error;

    /* TCP/IP <= 4.0 is a 16bit implementation: limit buffer to 32K-1 !*/
    if( BytesToSend > 32767 )
        BytesToSend = 32767;

    return sendto(pSockImpl->m_Socket,
                  (sal_Char*)pBuffer,
                  BytesToSend,
                  MSG_FLAG_TO_NATIVE(Flag),
                  (struct sockaddr*)ReceiverAddr,
                  ReceiverAddr == 0 ? 0 : sizeof(struct sockaddr));
}

/*****************************************************************************/
/* osl_isReceiveReady  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isReceiveReady(oslSocket Socket,
                           const TimeValue * pTimeout)
{
    fd_set         fds;
    struct timeval tv;
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    FD_ZERO(&fds);
    FD_SET(pSockImpl->m_Socket, &fds);

    if (pTimeout)
    {
        tv.tv_sec  = pTimeout->Seconds;
        tv.tv_usec = pTimeout->Nanosec / 1000;
    }

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return (bsdselect(pSockImpl->m_Socket + 1, /* highest socket to monitor */
                      &fds,                    /* check read operations */
                      NULL,                    /* check write ops */
                      NULL,                    /* ckeck for OOB */
                      (pTimeout ? &tv : NULL))==1); /* use timeout? */
}

/*****************************************************************************/
/* osl_isSendReady  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isSendReady(oslSocket Socket,
                        const TimeValue * pTimeout)
{
    fd_set         fds;
    struct timeval tv;
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    FD_ZERO(&fds);
    FD_SET(pSockImpl->m_Socket, &fds);

    if( pTimeout )
    {
        tv.tv_sec  = pTimeout->Seconds;
        tv.tv_usec = pTimeout->Nanosec / 1000;
    }

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return (bsdselect(pSockImpl->m_Socket + 1, /* highest socket to monitor */
                      NULL,                    /* check read operations */
                      &fds,                    /* check write ops */
                      NULL,                    /* ckeck for OOB */
                      (pTimeout ? &tv : NULL))==1); /* use timeout? */
}

/*****************************************************************************/
/* osl_isExceptionPending  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isExceptionPending(oslSocket Socket,
                               const TimeValue * pTimeout)
{
    fd_set         fds;
    struct timeval tv;
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    FD_ZERO(&fds);
    FD_SET(pSockImpl->m_Socket, &fds);

    if( pTimeout )
    {
        tv.tv_sec  = pTimeout->Seconds;
        tv.tv_usec = pTimeout->Nanosec / 1000;
    }

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return (bsdselect(pSockImpl->m_Socket + 1, /* highest socket to monitor */
                      NULL,                    /* check read operations */
                      NULL,                    /* check write ops */
                      &fds,                    /* ckeck for OOB */
                      (pTimeout ? &tv : NULL))==1); /* use timeout? */
}

/*****************************************************************************/
/* osl_shutdownSocket  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_shutdownSocket(oslSocket Socket,
                           oslSocketDirection Direction)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return (shutdown(pSockImpl->m_Socket, DIRECTION_TO_NATIVE(Direction))==0);
}


/*****************************************************************************/
/* osl_getSocketOption  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_getSocketOption(oslSocket            Socket,
                        oslSocketOptionLevel Level,
                        oslSocketOption      Option,
                        void*                pBuffer,
                        sal_uInt32               BufferLen)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return osl_Socket_Error;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return osl_Socket_Error;

    if (getsockopt(pSockImpl->m_Socket,
                   OPTION_LEVEL_TO_NATIVE(Level),
                   OPTION_TO_NATIVE(Option),
                   (sal_Char*)pBuffer,
                   &BufferLen) == -1)
    {
        BufferLen= -1;
    }

    return BufferLen;
}

/*****************************************************************************/
/* osl_setSocketOption  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_setSocketOption(oslSocket            Socket,
                            oslSocketOptionLevel Level,
                            oslSocketOption      Option,
                            void*                pBuffer,
                            sal_uInt32               BufferLen)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return(setsockopt(pSockImpl->m_Socket,
                      OPTION_LEVEL_TO_NATIVE(Level),
                      OPTION_TO_NATIVE(Option),
                      (sal_Char*)pBuffer,
                      BufferLen) == 0);
}

/*****************************************************************************/
/* osl_enableNonBlockingMode */
/*****************************************************************************/
sal_Bool SAL_CALL osl_enableNonBlockingMode(oslSocket Socket,
                                  sal_Bool On)
{
    oslSocketImpl* pSockImpl;
    sal_uInt32  Param= On ? 1 : 0;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    pSockImpl->m_Flags = Param ?
        (pSockImpl->m_Flags |  OSL_SOCKET_FLAGS_NONBLOCKING) :
        (pSockImpl->m_Flags & ~OSL_SOCKET_FLAGS_NONBLOCKING);

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return sal_False;

    return (ioctl(pSockImpl->m_Socket,
                  FIONBIO,
                  (sal_Char*)&Param,
                  sizeof(Param)) != OSL_SOCKET_ERROR);
}

/*****************************************************************************/
/* osl_isNonBlockingMode */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isNonBlockingMode(oslSocket Socket)
{
    oslSocketImpl* pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return sal_False;

    return((pSockImpl->m_Flags & OSL_SOCKET_FLAGS_NONBLOCKING) ? sal_True : sal_False);
}

/*****************************************************************************/
/* osl_getSocketType  */
/*****************************************************************************/
oslSocketType SAL_CALL osl_getSocketType(oslSocket Socket)
{
    oslSocketImpl* pSockImpl;
    int            Type=0;
    int            TypeSize= sizeof(Type);

    pSockImpl= (oslSocketImpl*)Socket;
    OSL_ASSERT(pSockImpl);

    if (pSockImpl == NULL) /* ENOTSOCK */
        return osl_Socket_TypeInvalid;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return osl_Socket_Error;

    if(getsockopt(pSockImpl->m_Socket,
                  OPTION_LEVEL_TO_NATIVE(osl_Socket_LevelSocket),
                  OPTION_TO_NATIVE(osl_Socket_OptionType),
                  (sal_Char*)&Type,
                  &TypeSize) == -1)
    {
        /* error */
        return osl_Socket_TypeInvalid;
    }

    return TYPE_FROM_NATIVE(Type);
}

/*****************************************************************************/
/* osl_getLastSocketErrorDescription  */
/*****************************************************************************/
void SAL_CALL osl_getLastSocketErrorDescription(oslSocket Socket, sal_Char* pBuffer, sal_uInt32 BufferSize)
{
    int error;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return;

    /* ensure pBuffer will be zero-terminated even when strncpy has to cut */
    pBuffer[BufferSize-1]= '\0';

    /* from UNIX version */
    /* @@@ DOES NOT WORK. SEE osl_getLastSocketError() @@@ */
    strncpy(pBuffer, strerror(errno), BufferSize-1);

#ifdef _FM_EXCLUDE

    switch(error = WSAGetLastError())
    {
        case WSAENOTSOCK:
            strncpy(pBuffer, "WSAENOTSOCK, Socket operation on non-socket. A socket created in one process is used by another process.", BufferSize-1);
            return;

        case WSAEDESTADDRREQ:
            strncpy(pBuffer, "WSAEDESTADDRREQ, Destination Addr required", BufferSize-1);
            return;

        case WSAEMSGSIZE:
            strncpy(pBuffer, "WSAEMSGSIZE, Message too long", BufferSize-1);
            return;

        case WSAEPROTOTYPE:
            strncpy(pBuffer, "WSAEPROTOTYPE, Protocol wrong type for socket", BufferSize-1);
            return;

        case WSAENOPROTOOPT:
            strncpy(pBuffer, "WSAENOPROTOOPT, Protocol not available", BufferSize-1);
            return;

        case WSAEPROTONOSUPPORT:
            strncpy(pBuffer, "WSAEPROTONOSUPPORT, Protocol not supported", BufferSize-1);
            return;

        case WSAESOCKTNOSUPPORT:
            strncpy(pBuffer, "WSAESOCKTNOSUPPORT, Socket type not supported", BufferSize-1);
            return;

        case WSAEOPNOTSUPP:
            strncpy(pBuffer, "WSAEOPNOTSUPP, Operation not supported on socket", BufferSize-1);
            return;

        case WSAEPFNOSUPPORT:
            strncpy(pBuffer, "WSAEPFNOSUPPORT, Protocol family not supported", BufferSize-1);
            return;

        case WSAEAFNOSUPPORT:
            strncpy(pBuffer, "WSEAFNOSUPPORT, Addr family not supported by protocol family", BufferSize-1);
            return;

        case WSAEADDRINUSE:
            strncpy(pBuffer, "WSAEADDRINUSE, Triggered by bind() because a process went down without closing a socket.", BufferSize-1);
            return;

        case WSAEADDRNOTAVAIL:
            strncpy(pBuffer, "WSAEADDRNOTAVAIL, Can't assign requested Addr", BufferSize-1);
            return;

        case WSAENETDOWN:
            strncpy(pBuffer, "WSAENETDOWN, Network is down", BufferSize-1);
            return;

        case WSAENETUNREACH:
            strncpy(pBuffer, "WSAENETUNREACH, Network is unreachable", BufferSize-1);
            return;

        case WSAENETRESET:
            strncpy(pBuffer, "WSAENETRESET, Network dropped connection or reset", BufferSize-1);
            return;

        case WSAECONNABORTED:
            strncpy(pBuffer, "WSAECONNABORTED, Software caused connection abort", BufferSize-1);
            return;

        case WSAECONNRESET:
            strncpy(pBuffer, "WSAECONNRESET, Connection reset by peer", BufferSize-1);
            return;

        case WSAENOBUFS:
            strncpy(pBuffer, "WSAENOBUFS, No buffer space available.", BufferSize-1);
            return;

        case WSAEISCONN:
            strncpy(pBuffer, "WSAEISCONN, Socket is already connected", BufferSize-1);
            return;

        case WSAENOTCONN:
            strncpy(pBuffer, "WSAENOTCONN, Socket is not connected", BufferSize-1);
            return;

        case WSAESHUTDOWN:
            strncpy(pBuffer, "WSAESHUTDOWN, Can't send after socket shutdown", BufferSize-1);
            return;

        case WSAETIMEDOUT:
            strncpy(pBuffer, "WSAETIMEDOUT, Connection timed out", BufferSize-1);
            return;

        case WSAECONNREFUSED:
            strncpy(pBuffer, "WSAECONNREFUSED, Connection refused", BufferSize-1);
            return;

        case WSAEHOSTDOWN:
            strncpy(pBuffer, "WSAEHOSTDOWN, Networking subsystem not started", BufferSize-1);
            return;

        case WSAEHOSTUNREACH:
            strncpy(pBuffer, "WSAEHOSTUNREACH, No route to host", BufferSize-1);
            return;

        case WSAEWOULDBLOCK:
            strncpy(pBuffer, "WSAEWOULDBLOCK, Operation would block", BufferSize-1);
            return;

        case WSAEINPROGRESS:

            strncpy(pBuffer, "WSAEINPROGRESS, Operation now in progress", BufferSize-1);
            return;

        case WSAEALREADY:
            strncpy(pBuffer, "WSAEALREADY, Operation already in progress", BufferSize-1);
            return;

        case WSAEINTR:
            strncpy(pBuffer, "WSAEALREADY, Operation was interrupted", BufferSize-1);
            return;

        case WSAEBADF:
            strncpy(pBuffer, "WSAEBADF, Bad file number", BufferSize-1);
            return;

        case WSAEACCES:
            strncpy(pBuffer, "WSAEACCES, Access is denied", BufferSize-1);
            return;

        case WSAEFAULT:
            strncpy(pBuffer, "WSAEFAULT, Bad memory Addr", BufferSize-1);
            return;

        case WSAEINVAL:
            strncpy(pBuffer, "WSAEINVAL, The socket has not been bound with bind() or is already connected", BufferSize-1);
            return;

        case WSAEMFILE:
            strncpy(pBuffer, "WSAEMFILE, No more file descriptors are available", BufferSize-1);
            return;

        case WSAETOOMANYREFS:
            strncpy(pBuffer, "WSAETOOMANYREFS, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAENAMETOOLONG:
            strncpy(pBuffer, "WSAENAMETOOLONG, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAENOTEMPTY:
            strncpy(pBuffer, "WSAENOTEMPTY, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEPROCLIM:

            strncpy(pBuffer, "WSAEPROCLIM, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEUSERS:
            strncpy(pBuffer, "WSAEUSERS, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEDQUOT:
            strncpy(pBuffer, "WSAEDQUOT, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAESTALE:
            strncpy(pBuffer, "WSAESTALE, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEREMOTE:
            strncpy(pBuffer, "WSAEREMOTE, Undocumented WinSock error", BufferSize-1);
            return;

        case WSAEDISCON:
            strncpy(pBuffer, "WSAEDISCON, Circuit was gracefully terminated", BufferSize-1);
            return;

        case WSASYSNOTREADY:
            strncpy(pBuffer, "WSASYSNOTREADY, The underlying network subsystem is not ready for network communication", BufferSize-1);
            return;

        case WSAVERNOTSUPPORTED:
            strncpy(pBuffer, "WSAVERNOTSUPPORTED, The version of Windows Sockets API support requested is not provided by this particular Windows Sockets implementation", BufferSize-1);
            return;

        case WSANOTINITIALISED:
            strncpy(pBuffer, "WSANOTINITIALISED, WSAStartup() has not been called", BufferSize-1);
            return;

        case WSAHOST_NOT_FOUND:
            strncpy(pBuffer, "WSAHOST_NOT_FOUND, Authoritative answer host not found", BufferSize-1);
            return;

        case WSATRY_AGAIN:
            strncpy(pBuffer, "WSATRY_AGAIN, Non-authoritative answer host not found or SERVERFAIL", BufferSize-1);
            return;

        case WSANO_RECOVERY:
            strncpy(pBuffer, "WSANO_RECOVERY, Non recoverable errors, FORMERR, REFUSED, NOTIMP", BufferSize-1);
            return;

        case WSANO_DATA:
            strncpy(pBuffer, "WSANO_DATA or WSANO_ADDRESS, Valid name, no data record of requested type", BufferSize-1);
            return;

        default:
        {
            sal_Char message[128];

            sprintf((LPTSTR)message, (LPCSTR)"Unknown WinSock Error Number %d", error);
            strncpy(pBuffer, message, BufferSize-1);
        }
        return;
    }

#endif

}

/*****************************************************************************/
/* SocketSet                                                                 */
/*****************************************************************************/
typedef struct _TSocketSetImpl
{
    fd_set m_Set;       /* the set of descriptors */
    int    m_MaxHandle; /* for select(), the largest descriptor in set */
} TSocketSetImpl;

/*****************************************************************************/
/* osl_createSocketSet  */
/*****************************************************************************/
oslSocketSet SAL_CALL osl_createSocketSet()
{
    TSocketSetImpl* pSet;

    pSet= malloc(sizeof(TSocketSetImpl));
    OSL_ASSERT(pSet);

    if(pSet)
    {
        pSet->m_MaxHandle= 0;
        FD_ZERO(&pSet->m_Set);
    }

    return (oslSocketSet)pSet;
}

/*****************************************************************************/
/* osl_destroySocketSet  */
/*****************************************************************************/
void SAL_CALL osl_destroySocketSet(oslSocketSet Set)
{
    if(Set)
        free(Set);
}

/*****************************************************************************/
/* osl_clearSocketSet  */
/*****************************************************************************/
void SAL_CALL osl_clearSocketSet(oslSocketSet Set)
{
    TSocketSetImpl* pSet;

    pSet= (TSocketSetImpl*)Set;
    OSL_ASSERT(pSet);

    if (pSet)
    {
        pSet->m_MaxHandle= 0;
        FD_ZERO(&pSet->m_Set);
    }
}

/*****************************************************************************/
/* osl_addToSocketSet  */
/*****************************************************************************/
void SAL_CALL osl_addToSocketSet(oslSocketSet Set, oslSocket Socket)
{
    TSocketSetImpl* pSet;
    oslSocketImpl*  pSockImpl;

    pSet= (TSocketSetImpl*)Set;
    pSockImpl= (oslSocketImpl*)Socket;

    OSL_ASSERT(pSet);
    OSL_ASSERT(pSockImpl);

    if (pSet && pSockImpl)
    {
        /* correct max handle */
        if (pSockImpl->m_Socket > pSet->m_MaxHandle)
            pSet->m_MaxHandle= pSockImpl->m_Socket;

        FD_SET(pSockImpl->m_Socket, &pSet->m_Set);
    }
}

/*****************************************************************************/
/* osl_removeFromSocketSet  */
/*****************************************************************************/
void SAL_CALL osl_removeFromSocketSet(oslSocketSet Set, oslSocket Socket)
{
    TSocketSetImpl* pSet;
    oslSocketImpl*  pSockImpl;

    pSet= (TSocketSetImpl*)Set;
    pSockImpl= (oslSocketImpl*)Socket;

    OSL_ASSERT(pSet);
    OSL_ASSERT(pSockImpl);

    if (pSet && pSockImpl)
    {
        /* correct max handle */
        if (pSockImpl->m_Socket == pSet->m_MaxHandle)
        {
            /* not optimal, since the next used socket may be */
            /* much smaller than m_Socket-1, but this will do */

            if (pSet->m_MaxHandle > 0) /* avoid underflow */
                pSet->m_MaxHandle--;
        }

        FD_CLR(pSockImpl->m_Socket, &pSet->m_Set);
    }
}

/*****************************************************************************/
/* osl_isInSocketSet  */
/*****************************************************************************/
sal_Bool SAL_CALL osl_isInSocketSet(oslSocketSet Set, oslSocket Socket)
{
    TSocketSetImpl* pSet;
    oslSocketImpl*  pSockImpl;

    pSockImpl= (oslSocketImpl*)Socket;
    pSet= (TSocketSetImpl*)Set;

    OSL_ASSERT(pSet);
    OSL_ASSERT(pSockImpl);

    if (pSet && pSockImpl)
        return FD_ISSET(pSockImpl->m_Socket, &pSet->m_Set);
    else
        return sal_False;
}

/*****************************************************************************/
/* osl_demultiplexSocketEvents  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_demultiplexSocketEvents(oslSocketSet IncomingSet,
                                oslSocketSet OutgoingSet,
                                oslSocketSet OutOfBandSet,
                                const TimeValue* pTimeout)
{
    int             MaxHandle= 0;
    struct timeval  tv;
    TSocketSetImpl* pInSet;
    TSocketSetImpl* pOutSet;
    TSocketSetImpl* pOOBSet;

    if(pTimeout)
    {
        tv.tv_sec  = pTimeout->Seconds;
        tv.tv_usec = pTimeout->Nanosec / 1000;
    }

    /* map opaque data to impl-types */
    pInSet= (TSocketSetImpl*)IncomingSet;
    pOutSet= (TSocketSetImpl*)OutgoingSet;
    pOOBSet= (TSocketSetImpl*)OutOfBandSet;

    /* get max handle from all sets */
    if (pInSet)
        MaxHandle= pInSet->m_MaxHandle;

    if (pOutSet && (pOutSet->m_MaxHandle > MaxHandle))
        MaxHandle= pOutSet->m_MaxHandle;

    if (pOOBSet && (pOOBSet->m_MaxHandle > MaxHandle))
        MaxHandle= pOOBSet->m_MaxHandle;

    /* load and init TCP/IP libraries */
    if( !hTCP32DLL )
        if( !ImplLoadTCPIP() )
            return 0;

    return bsdselect(MaxHandle + 1,
                     pInSet   ? &pInSet->m_Set  : 0,
                     pOutSet  ? &pOutSet->m_Set : 0,
                     pOOBSet  ? &pOOBSet->m_Set : 0,
                     pTimeout ? &tv : NULL);
}

/*****************************************************************************/




