/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: st_root.hxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 17:06:32 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef ARY_STORE_ST_ROOT_HXX
#define ARY_STORE_ST_ROOT_HXX

// USED SERVICES
    // BASE CLASSES
    // COMPONENTS
#include "st_unit.hxx"
#include "st_iterator.hxx"
    // PARAMETERS
#include <ary/ids.hxx>
#include <ary/x_ary.hxx>


namespace ary
{
namespace store
{



template <class UNIT>
class RootBlock
{
  public:
    typedef UNIT                unit_type;
    typedef const unit_type *   const_iterator;
    typedef unit_type *         iterator;

    // LIFECYCLE
                        RootBlock()             : pData(0),
                                                  pEnd(0) {}
    void                Init(
                            uintt               i_nBlockSize );
    void                ReleaseData()           { pData = pEnd = 0; }
    void                EraseAll()              { delete [] pData; ReleaseData(); }
                        ~RootBlock()            { delete [] pData; }

    // OPERATORS
    const unit_type &   operator[](
                            uintt               i_nIndex ) const
                                                { return pData[i_nIndex]; }
    unit_type &         operator[](
                            uintt               i_nIndex )
                                                { return pData[i_nIndex]; }
    // INQUIRY
    const_iterator      Begin() const           { return pData; }
    const_iterator      End() const             { return pEnd; }

    // ACCESS
    iterator            Begin()                 { return pData; }
    iterator            End()                   { return pEnd; }

  private:
    unit_type *         pData;
    unit_type *         pEnd;
};

template <class UNIT>
void
RootBlock<UNIT>::Init( uintt i_nBlockSize )
{
    pData =  new unit_type[i_nBlockSize];
    pEnd  =  pData + i_nBlockSize;
}



/** @descr Container for large amounts of data, which shall be
    accessed mainly by indices and most elements are added at the end
    or replaced. This is typical for data ordered by a serial number.
**/
template <class UNIT>
class Root
{
  public:
    typedef UNIT                                    unit_type;
    typedef typename UNIT::element_type             element_type;
    typedef TypedId<element_type>                   key_type;
    typedef RootBlock<UNIT>                         block_type;
    typedef block_type *                            block_list;
    typedef RootConstIterator<UNIT>                 const_iterator;
    typedef RootIterator<UNIT>                      iterator;

    // LIFECYCLE
                        Root(
                            uintt               i_nBLOCK_SIZE_LOG_2,
                            uintt               i_nReservedElements = 1,
                            uintt               i_nInitialNrOfBlocks = 2 );
                        ~Root();

    // OPERATORS
    const unit_type &   operator[](
                            key_type            i_nIndex ) const;
    unit_type &         operator[](
                            key_type            i_nIndex );

    // OPERATIONS
    /// Assigns id to pass_io_drNewElement.
    void                Add(
                            DYN element_type &  pass_io_drNewElement );
    /// Assigns id to pass_io_drNewElement.
    void                Set_Reserved(
                            key_type            i_nIndex,
                            DYN element_type &  pass_io_drNewElement );
    void                EraseAll();

    // INQUIRY
    uintt               Size() const;
                                                
    /// @return unit at index 1, because 0 is always empty.
    const_iterator      Begin() const;          

    // ACCESS
    /// @return unit at index 1, because 0 is always empty.
    iterator            Begin();

  private:
    // Locals
    void                Expand();
    block_type *        ResizeBlockList();
    const unit_type &   Get_Unit(                /// @precond  i_nIndex < Size().
                            uintt               i_nIndex ) const;
    unit_type &         Get_Unit(                /// @precond  i_nIndex < Size().
                            uintt               i_nIndex );
    // DATA
    block_list          dpBlocks;            // Array of block_type;
    block_type *        pBlocksEnd;

    iterator            pCurEnd;
    uintt               nCurSize;

        // The following three values depend on each other, but are pre-calculated for optimisation.
    uintt               nBlockSize;
    uintt               nLowerMask;
    uintt               nBLOCK_SIZE_LOG_2;
};



// IMPLEMENTATION

template <class UNIT>
const UNIT &
Root<UNIT>::Get_Unit( uintt i_nIndex ) const
{
    return dpBlocks[i_nIndex >> nBLOCK_SIZE_LOG_2][i_nIndex & nLowerMask];
}

template <class UNIT>
UNIT &
Root<UNIT>::Get_Unit(  uintt  i_nIndex )
{
    return dpBlocks[i_nIndex >> nBLOCK_SIZE_LOG_2][i_nIndex & nLowerMask];
}

template <class UNIT>
inline uintt
Root<UNIT>::Size() const
{
    return nCurSize;
}

template <class UNIT>
Root<UNIT>::Root( uintt i_nBLOCK_SIZE_LOG_2,
                                    uintt i_nReservedElements,
                                    uintt i_nInitialNrOfBlocks )
    :   dpBlocks( new block_type[i_nInitialNrOfBlocks] ),
        pBlocksEnd( dpBlocks + i_nInitialNrOfBlocks ),
        pCurEnd(),
        nCurSize(0),
        nBlockSize( uintt(1) << i_nBLOCK_SIZE_LOG_2 ),
        nLowerMask(nBlockSize - 1),
        nBLOCK_SIZE_LOG_2(i_nBLOCK_SIZE_LOG_2)
{
    precond(i_nInitialNrOfBlocks > 0);
    precond(nBlockSize > 1);
    precond(nBlockSize > i_nReservedElements);

    dpBlocks[0].Init(nBlockSize);
    pCurEnd.Set( dpBlocks, pBlocksEnd, (*dpBlocks).Begin() );
    for ( uintt i = 0; i < i_nReservedElements; ++i )
        Expand();     // There should be no element with id "0" and there may be some more reserved ones.
}

template <class UNIT>
Root<UNIT>::~Root()
{
    delete [] dpBlocks;
}

template <class UNIT>
const UNIT &
Root<UNIT>::operator[]( key_type i_nIndex ) const
{
    if (i_nIndex.Value() < nCurSize )
        return Get_Unit(i_nIndex.Value());
    return Get_Unit(0);
}

template <class UNIT>
UNIT &
Root<UNIT>::operator[]( key_type i_nIndex )
{
    if (i_nIndex.Value() < nCurSize )
        return Get_Unit(i_nIndex.Value());
    return Get_Unit(0);
}

template <class UNIT>
void
Root<UNIT>::Add( DYN element_type &  pass_io_drNewElement )
{
    pass_io_drNewElement.Set_Id( static_cast<Rid>(Size()) );
    (*pCurEnd).Set_Entity(pass_io_drNewElement);
    Expand();
}

template <class UNIT>
void
Root<UNIT>::Set_Reserved( key_type            i_nIndex,
                          DYN element_type &  pass_io_drNewElement )
{
    Rid nIndex = i_nIndex.Value();
    csv_assert( nIndex < Size() );

    pass_io_drNewElement.Set_Id(nIndex);
    Get_Unit(nIndex).Set_Entity(pass_io_drNewElement);
}

template <class UNIT>
void
Root<UNIT>::EraseAll()
{
    block_type * itEnd = pBlocksEnd;
    for ( block_type * it = dpBlocks; it != itEnd; ++it )
    {
        (*it).EraseAll();
    }   // end for

    dpBlocks[0].Init(nBlockSize);
}

template <class UNIT>
typename Root<UNIT>::const_iterator
Root<UNIT>::Begin() const
{
    const_iterator ret(dpBlocks, pBlocksEnd, (*dpBlocks).Begin());
    ++ret;
    return ret;
}

template <class UNIT>
typename Root<UNIT>::iterator
Root<UNIT>::Begin()
{
    iterator ret(dpBlocks, pBlocksEnd, (*dpBlocks).Begin());
    ++ret;
    return ret;
}                                  

template <class UNIT>
void
Root<UNIT>::Expand()
{
    ++pCurEnd;
    ++nCurSize;

    if ( nCurSize == (nCurSize & (~nLowerMask)) )
    {   // Init next block:
        block_type *
            pNextBlock = pCurEnd.CurBlock();
		if (pNextBlock == pBlocksEnd)
			pNextBlock = ResizeBlockList();
        pNextBlock->Init(nBlockSize);
        pCurEnd.Set( pNextBlock, pBlocksEnd, (*pNextBlock).Begin() );
    }
}

/*  Resizes the block list, but copies only the pointers to the block-internal memory,
    which can stay where it is.
*/
template <class UNIT>
typename Root<UNIT>::block_type *
Root<UNIT>::ResizeBlockList()
{
    uintt nOldSize = pBlocksEnd - dpBlocks;
    uintt nNewSize = nOldSize << 1;

    block_list
        pNew = new block_type[ nNewSize ];

    // This is presently possible, depends on the properties of class block_type.
    uintt nOldSizeInBytes = nOldSize * sizeof(block_type);
    memcpy( pNew, dpBlocks, nOldSizeInBytes );

    for ( block_type * it = dpBlocks; it != pBlocksEnd; ++it )
        it->ReleaseData();
    delete [] dpBlocks;

    dpBlocks = pNew;
    pBlocksEnd = dpBlocks + nNewSize;
    return dpBlocks + nOldSize; // The first new block
}


}   // namespace store
}   // namespace ary

#endif

