/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: shapeeventbroadcaster.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: obo $ $Date: 2005/10/11 08:35:15 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "canvas/debug.hxx"
#include "shapeeventbroadcaster.hxx"
#include "com/sun/star/awt/MouseButton.hpp"
#include "com/sun/star/awt/SystemPointer.hpp"
#include "com/sun/star/presentation/XShapeEventListener.hpp"
#include "com/sun/star/presentation/XSlideShowListener.hpp"
#include "com/sun/star/awt/MouseButton.hpp"
#include "boost/bind.hpp"

namespace css = com::sun::star;
using namespace css;

namespace presentation {
namespace internal {

// Disposable interface
void ShapeEventBroadcaster::dispose()
{
    disable(); // for safety... removes listener registration
    osl::MutexGuard const guard( m_mutex );
    m_shapeToCursor.clear();
    m_shapeToListeners.clear();
    m_hyperlinkShapes.clear();
    m_this.reset();
}

bool ShapeEventBroadcaster::handleMousePressed( awt::MouseEvent const& )
{
    // not used here
    return false; // did not handle the event
}

rtl::OUString ShapeEventBroadcaster::checkForHyperlink(
    basegfx::B2DPoint const& hitPos ) const
{
    // find matching region (scan reversely, to coarsely match
    // paint order): set is ordered by priority
    ShapeSet::const_reverse_iterator iPos( m_hyperlinkShapes.rbegin() );
    ShapeSet::const_reverse_iterator const iEnd( m_hyperlinkShapes.rend() );
    for ( ; iPos != iEnd; ++iPos ) {
        ShapeSharedPtr const& pShape = *iPos;
        OSL_ASSERT( pShape->hasHyperlinks() );
        basegfx::B2DPoint const shapeUpperLeft(
            pShape->getPosSize().getMinimum() );
        std::vector<Shape::HyperLinkRegion> const linkRegions(
            pShape->getHyperlinkRegions() );
        for ( std::size_t i = linkRegions.size(); i--; ) {
            basegfx::B2DRectangle const& relRegion = linkRegions[i].first;
            basegfx::B2DRectangle const absRegion(
                shapeUpperLeft + relRegion.getMinimum(),
                shapeUpperLeft + relRegion.getMaximum() );
            if (absRegion.isInside( hitPos ))
                return linkRegions[i].second;
        }
    }
    return rtl::OUString();
}

bool ShapeEventBroadcaster::handleMouseReleased( awt::MouseEvent const& e )
{
    if (e.Buttons != awt::MouseButton::LEFT)
        return false;
    
    osl::ClearableMutexGuard guard( m_mutex );
    
    basegfx::B2DPoint const aPosition( e.X, e.Y );

    // first check for hyperlinks, because these have
    // highest prio:
    rtl::OUString const hyperlink( checkForHyperlink(aPosition) );
    if (hyperlink.getLength() > 0) {
        guard.clear();
        m_rMultiplexer.notifyHyperlinkClicked(hyperlink);
        return true; // event consumed
    }
    
    // find matching shape (scan reversely, to coarsely match
    // paint order)
    ShapeToListenersMap::reverse_iterator aCurrBroadcaster(
        m_shapeToListeners.rbegin() );
    ShapeToListenersMap::reverse_iterator const aEndBroadcasters(
        m_shapeToListeners.rend() );
    while( aCurrBroadcaster != aEndBroadcasters )
    {
        // TODO(F2): Get proper geometry polygon from the
        // shape, to avoid having areas outside the shape
        // react on the mouse
        if( aCurrBroadcaster->first->getPosSize().isInside( aPosition ) &&
            aCurrBroadcaster->first->isVisible() )
        {
            // shape hit, and shape is visible. Raise
            // event.
            
            boost::shared_ptr<cppu::OInterfaceContainerHelper> const pCont(
                aCurrBroadcaster->second );
            uno::Reference<drawing::XShape> const xShape(
                aCurrBroadcaster->first->getXShape() );
            
            // release object mutex, we're calling out!
            guard.clear();
            
            // DON'T do anything with /this/ after this point!
            pCont->forEach<css::presentation::XShapeEventListener>(
                boost::bind(
                    &css::presentation::XShapeEventListener::click,
                    _1, boost::cref(xShape), boost::cref(e) ) );
            
            return true; // handled this event
        }
        
        ++aCurrBroadcaster;
    }
    
    return false; // did not handle this event
}

bool ShapeEventBroadcaster::handleMouseEntered( const awt::MouseEvent& )
{
    // not used here
    return false; // did not handle the event
}

bool ShapeEventBroadcaster::handleMouseExited( const awt::MouseEvent& )
{
    // not used here
    return false; // did not handle the event
}

bool ShapeEventBroadcaster::handleMouseDragged( const awt::MouseEvent& )
{
    // not used here
    return false; // did not handle the event
}

bool ShapeEventBroadcaster::handleMouseMoved( const awt::MouseEvent& e )
{
    osl::MutexGuard const guard( m_mutex );
    
    // find hit shape in map
    const ::basegfx::B2DPoint aPosition( e.X, e.Y );
    
    if (checkForHyperlink(aPosition).getLength() > 0) {
        m_rMultiplexer.setVolatileMouseCursor( awt::SystemPointer::REFHAND );
        return false; // event not consumed
    }
    
    // find matching shape (scan reversely, to coarsely match
    // paint order)
    ShapeToCursorMap::reverse_iterator aCurrCursor(
        m_shapeToCursor.rbegin() );
    ShapeToCursorMap::reverse_iterator const aEndCursors(
        m_shapeToCursor.rend() );
    while( aCurrCursor != aEndCursors )
    {
        // TODO(F2): Get proper geometry polygon from the
        // shape, to avoid having areas outside the shape
        // react on the mouse
        if( aCurrCursor->first->getPosSize().isInside( aPosition ) &&
            aCurrCursor->first->isVisible() )
        {
            // shape found, and it's visible. set
            // requested cursor to shape's
            m_rMultiplexer.setVolatileMouseCursor( aCurrCursor->second );
            break;
        }
        
        ++aCurrCursor;
    }
    
    return false; // we don't /eat/ this event. Lower prio
    // handler should see it, too.
}

ShapeEventBroadcaster::ShapeEventBroadcaster( EventMultiplexer & rMultiplexer )
    : m_mutex(),
      m_this(),
      m_rMultiplexer(rMultiplexer),
      m_shapeToListeners(),
      m_shapeToCursor(),
      m_hyperlinkShapes(),
      m_bEnabled(false)
{
}

boost::shared_ptr<ShapeEventBroadcaster> ShapeEventBroadcaster::create(
    EventMultiplexer & rMultiplexer )
{
    boost::shared_ptr<ShapeEventBroadcaster> const pRet(
        new ShapeEventBroadcaster(rMultiplexer) );
    pRet->m_this = pRet;
    return pRet;
}

ShapeEventBroadcaster::~ShapeEventBroadcaster()
{
}

void ShapeEventBroadcaster::enable()
{
    if (! m_bEnabled) {
        m_bEnabled = true;
        // register this handler on EventMultiplexer
        // prio (overrides engine handlers)
        m_rMultiplexer.addMouseMoveHandler( m_this, 2.0 );
        m_rMultiplexer.addClickHandler( m_this, 2.0 );
    }
}

void ShapeEventBroadcaster::disable()
{
    if (m_bEnabled) {
        m_bEnabled = false;
        // register this handler on EventMultiplexer
        // prio (overrides engine handlers)
        m_rMultiplexer.removeMouseMoveHandler( m_this );
        m_rMultiplexer.removeClickHandler( m_this );
    }
}

void ShapeEventBroadcaster::addShapeEventListener(
    const uno::Reference<css::presentation::XShapeEventListener >&  xListener, 
    const ShapeSharedPtr&                                   rShape )
{
    ENSURE_AND_THROW(
        rShape.get(),
        "ShapeEventBroadcaster::addShapeEventListener(): Invalid shape" );

    osl::MutexGuard const guard( m_mutex );
    
    ShapeToListenersMap::iterator aIter;
    if( (aIter = m_shapeToListeners.find( rShape )) ==
        m_shapeToListeners.end() )
    {
        // no entry for this shape -> create a ShapeListener
        aIter = m_shapeToListeners.insert( 
            ShapeToListenersMap::value_type(
                rShape, 
                boost::shared_ptr<cppu::OInterfaceContainerHelper>( 
                    new cppu::OInterfaceContainerHelper(m_mutex)) ) ).first;    
        // aIter now contains the newly added entry
    }
    
    // add new listener to broadcaster
    if( aIter->second.get() )
        aIter->second->addInterface( xListener );
}

void ShapeEventBroadcaster::removeShapeEventListener(
    const uno::Reference<css::presentation::XShapeEventListener >&  xListener, 
    const ShapeSharedPtr&                                   rShape )
{
    ENSURE_AND_THROW(
        rShape.get(),
        "ShapeEventBroadcaster::removeShapeEventListener(): Invalid shape" );
    
    osl::MutexGuard const guard( m_mutex );
    
    ShapeToListenersMap::iterator aIter;
    if( (aIter = m_shapeToListeners.find( rShape )) !=
        m_shapeToListeners.end() )
    {
        // entry for this shape found -> remove listener from
        // helper object
        ENSURE_AND_THROW(
            aIter->second.get(),
            "ShapeClickHandler::removeShapeEventListener(): "
            "listener map contains NULL broadcast helper" );
        
        aIter->second->removeInterface( xListener );
    }
}

void ShapeEventBroadcaster::setShapeCursor(
    const ShapeSharedPtr&   rShape, 
    sal_Int16               nPointerShape )
{
    ENSURE_AND_THROW(
        rShape.get(),
        "ShapeEventBroadcaster::setShapeCursor(): Invalid shape" );

    osl::MutexGuard const guard( m_mutex );
    
    ShapeToCursorMap::iterator aIter;
    if( (aIter = m_shapeToCursor.find( rShape )) == m_shapeToCursor.end() ) {
        
        // no entry for this shape -> add a new one
        // ========================================
        
        if( nPointerShape != awt::SystemPointer::ARROW ) {
            // add new entry, unless shape shall display
            // normal pointer arrow -> no need to handle that
            // case
            
            // no entry -> create one
            m_shapeToCursor.insert( 
                ShapeToCursorMap::value_type( rShape, nPointerShape ) );
        }
    }
    else if( nPointerShape == awt::SystemPointer::ARROW )
    {
        // shape shall display normal cursor -> can disable
        // the cursor and clear the entry
        m_shapeToCursor.erase( rShape );
    }
    else
    {
        // existing entry found, update with new cursor ID
        aIter->second = nPointerShape;
    }
}

void ShapeEventBroadcaster::addHyperlinkShape( ShapeSharedPtr const& pShape )
{
    osl::MutexGuard const guard( m_mutex );
    m_hyperlinkShapes.insert( pShape );
}

void ShapeEventBroadcaster::removeHyperlinkShape( ShapeSharedPtr const& pShape )
{
    osl::MutexGuard const guard( m_mutex );
    m_hyperlinkShapes.erase( pShape );
}

} // namespace internal
} // namespace presentation
