/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: groupset.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/08 15:02:33 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#define _GROUPSET_CXX

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
#ifndef _SV_IMAGE_HXX
#include <vcl/image.hxx>
#endif
#ifndef _SV_DECOVIEW_HXX
#include <vcl/decoview.hxx>
#endif
#ifndef _SV_SVAPP_HXX
#include <vcl/svapp.hxx>
#endif
#ifndef _VCL_POINTR_HXX
#include <vcl/pointr.hxx>
#endif
#ifndef _SV_TIMER_HXX
#include <vcl/timer.hxx>
#endif
#ifndef _SV_EDIT_HXX
#include <vcl/edit.hxx>
#endif
#ifndef _SV_HELP_HXX
#include <vcl/help.hxx>
#endif
#ifndef _SV_BUTTON_HXX
#include <vcl/button.hxx>
#endif
#ifndef _TOOLS_COLOR_HXX
#include <tools/color.hxx>
#endif

#pragma hdrstop

#include "groupset.hxx"

// =======================================================================

// alle Scrollbuttons rechts oben, links unten
#define GROUP_SCROLLBUTTONS

#define GROUPSET_ACC_RETURN 			1
#define GROUPSET_ACC_ESCAPE 			2
#define CURGROUP_FONTWEIGHT 			WEIGHT_BOLD

// -----------------------------------------------------------------------

static void DeleteButton( PushButton** ppButton )
{
	if( *ppButton )
	{
		(*ppButton)->EndTracking();
		DELETEZ(*ppButton);
	}
}

// =======================================================================

class ImplGroupEdit : public Edit
{
	Link			aCallBackHdl;
	Accelerator 	aAccReturn;
	Accelerator 	aAccEscape;
	Timer			aTimer;
	BOOL			bCanceled;
	BOOL			bAlreadyInCallback;
	BOOL			bGrabFocus;
	void			CallCallBackHdl_Impl();
					DECL_LINK( Timeout_Impl, Timer * );
					DECL_LINK( ReturnHdl_Impl, Accelerator * );
					DECL_LINK( EscapeHdl_Impl, Accelerator * );

protected:
	virtual void	KeyInput( const KeyEvent& rKEvt );
	virtual long	PreNotify( NotifyEvent& rNEvt );

public:

					ImplGroupEdit(
						Window* pParent,
						const Point& rPos,
						const Size& rSize,
						const XubString& rData,
						const Link& rNotifyEditEnd );

					~ImplGroupEdit();
	BOOL			EditingCanceled() const { return bCanceled; }
	void			StopEditing( BOOL bCancel = FALSE );
	BOOL			IsGrabFocus() const { return bGrabFocus; }
};

// -----------------------------------------------------------------------

ImplGroupEdit::ImplGroupEdit( Window* pParent, const Point& rPos,
	const Size& rSize, const XubString& rData, const Link& rNotifyEditEnd ) :
	Edit( pParent, WB_CENTER  ),
	aCallBackHdl( rNotifyEditEnd ),
	bCanceled( FALSE ),
	bAlreadyInCallback( FALSE ),
	bGrabFocus( FALSE )
{
	const StyleSettings& rStyleSettings = GetSettings().GetStyleSettings();
	Font aFont( pParent->GetPointFont() );
	aFont.SetTransparent( FALSE );
	SetControlFont( aFont );
	SetControlForeground( aFont.GetColor() );
	SetControlBackground( rStyleSettings.GetCheckedColor() );
	SetTextColor( pParent->GetTextColor() );
	SetPosPixel( rPos );
	SetSizePixel( rSize );
	SetText( rData );
	SaveValue();
	aAccReturn.InsertItem( GROUPSET_ACC_RETURN, KeyCode(KEY_RETURN) );
	aAccEscape.InsertItem( GROUPSET_ACC_ESCAPE, KeyCode(KEY_ESCAPE) );
	aAccReturn.SetActivateHdl( LINK( this, ImplGroupEdit, ReturnHdl_Impl) );
	aAccEscape.SetActivateHdl( LINK( this, ImplGroupEdit, EscapeHdl_Impl) );
	GetpApp()->InsertAccel( &aAccReturn );
	GetpApp()->InsertAccel( &aAccEscape );
	Show();
	GrabFocus();
}

// -----------------------------------------------------------------------

ImplGroupEdit::~ImplGroupEdit()
{
	if( !bAlreadyInCallback )
	{
		GetpApp()->RemoveAccel( &aAccReturn );
		GetpApp()->RemoveAccel( &aAccEscape );
	}
}

// -----------------------------------------------------------------------

void ImplGroupEdit::CallCallBackHdl_Impl()
{
	aTimer.Stop();
	if ( !bAlreadyInCallback )
	{
		bAlreadyInCallback = TRUE;
		GetpApp()->RemoveAccel( &aAccReturn );
		GetpApp()->RemoveAccel( &aAccEscape );
		Hide();
		aCallBackHdl.Call( this );
	}
}

// -----------------------------------------------------------------------

IMPL_LINK( ImplGroupEdit, Timeout_Impl, Timer*, EMPTYARG )
{
	CallCallBackHdl_Impl();
	return 0;
}

// -----------------------------------------------------------------------

IMPL_LINK( ImplGroupEdit, ReturnHdl_Impl, Accelerator*, EMPTYARG  )
{
	bCanceled = FALSE;
	bGrabFocus = TRUE;
	CallCallBackHdl_Impl();
	return 1;
}

// -----------------------------------------------------------------------

IMPL_LINK( ImplGroupEdit, EscapeHdl_Impl, Accelerator*, EMPTYARG  )
{
	bCanceled = TRUE;
	bGrabFocus = TRUE;
	CallCallBackHdl_Impl();
	return 1;
}

// -----------------------------------------------------------------------

void ImplGroupEdit::KeyInput( const KeyEvent& rKEvt )
{
	KeyCode aCode = rKEvt.GetKeyCode();
	USHORT nCode = aCode.GetCode();

	switch ( nCode )
	{
		case KEY_ESCAPE:
			bCanceled = TRUE;
			bGrabFocus = TRUE;
			CallCallBackHdl_Impl();
			break;

		case KEY_RETURN:
			bCanceled = FALSE;
			bGrabFocus = TRUE;
			CallCallBackHdl_Impl();
			break;

		default:
			Edit::KeyInput( rKEvt );
	}
}

// -----------------------------------------------------------------------

long ImplGroupEdit::PreNotify( NotifyEvent& rNEvt )
{
	if( rNEvt.GetType() == EVENT_LOSEFOCUS )
	{
		if ( !bAlreadyInCallback &&
			((!Application::GetFocusWindow()) || !IsChild(Application::GetFocusWindow())))
		{
			bCanceled = FALSE;
			aTimer.SetTimeout(10);
			aTimer.SetTimeoutHdl(LINK(this,ImplGroupEdit,Timeout_Impl));
			aTimer.Start();
		}
	}
	return 0;
}

// -----------------------------------------------------------------------

void ImplGroupEdit::StopEditing( BOOL bCancel )
{
	if ( !bAlreadyInCallback )
	{
		bCanceled = bCancel;
		CallCallBackHdl_Impl();
	}
}

// =======================================================================

struct ImplGroup
{
	Rectangle	maRect;
	XubString	maText;
	XubString	maHelpText;
	Image		maImage;
	ULONG		mnHelpId;
	void*		mpGroupData;
	USHORT		mnId;

				ImplGroup( USHORT nId ) { mnHelpId = 0; mpGroupData = 0; mnId = nId; }
};

DECLARE_LIST( ImplGroups, ImplGroup* );

// -----------------------------------------------------------------------

class ImplGroupSet
{
public:
	ImplGroups		maGroups;
	Rectangle		maContentTotArea;
	Rectangle		maContentVisArea;
	Size			maContentLineSize;
	Size			maContentPageSize;
	Point			maContentWinPos;
	Size			maContentWinSize;
	Timer			maDropTimer;
	ImplGroupEdit*	mpEdit;
	WinBits 		mnStyle;
	long			mnButtonHeightPixel;
	USHORT			mnCurHighlightedPos; // unter der Maus stehende Gruppe
	USHORT			mnCurPressedGroupId; // MausButtonDown auf Gruppe
	BOOL			mbGroupPressed; // TRUE: Maus beruehrt mnCurPressedGroupId
	USHORT			mnDropTargetPos;
	USHORT			mnEditingPos;
	USHORT			mnMoveToPos; // Verschieben von Gruppen per D&D
	long			nBorderWidthPixel;
	long			nBorderHeightPixel;
	BOOL			mbMoveToPosEmphasized;
	BOOL			mbFormatted;
	BOOL			mbTracking;

					ImplGroupSet();
					~ImplGroupSet();
	ImplGroup*		GetGroup( USHORT nId ) const;
	ImplGroup*		GetGroup( const Point& rPos ) const;
	USHORT			GetGroupPos( const Point& rPos ) const;
	XubString		GetGroupHelpText( ImplGroup* ) const;
	void			CutBorder( Rectangle& rRect ) const;
};

// -----------------------------------------------------------------------

ImplGroupSet::ImplGroupSet()
{
	mbFormatted = TRUE;
	mbTracking = FALSE;
	mpEdit = 0;
	mnCurHighlightedPos = GROUPSET_GROUP_NOTFOUND;
	mnCurPressedGroupId = GROUPSET_GROUP_NOTFOUND;
	mbGroupPressed = FALSE;
	mnDropTargetPos = GROUPSET_GROUP_NOTFOUND;
	mnEditingPos = GROUPSET_GROUP_NOTFOUND;
	mnMoveToPos = GROUPSET_GROUP_NOTFOUND;
	mbMoveToPosEmphasized = FALSE;
	nBorderWidthPixel = 2;
	nBorderHeightPixel = 2;
}

// -----------------------------------------------------------------------

ImplGroupSet::~ImplGroupSet()
{
	delete mpEdit;
	ULONG nCount = maGroups.Count();
	for( ULONG nCur = 0; nCur < nCount; nCur++ )
	{
		ImplGroup* pGroup = maGroups.GetObject( nCur );
		delete pGroup;
	}
}

// -----------------------------------------------------------------------

ImplGroup* ImplGroupSet::GetGroup( USHORT nId ) const
{
	ULONG nCount = maGroups.Count();
	for( ULONG nCur = 0; nCur < nCount; nCur++ )
	{
		ImplGroup* pGroup = maGroups.GetObject( nCur );
		if( pGroup->mnId == nId )
			return pGroup;
	}
	return 0;
}

// -----------------------------------------------------------------------

ImplGroup* ImplGroupSet::GetGroup( const Point& rPos ) const
{
	if( !mbFormatted )
		return 0;
	ULONG nCount = maGroups.Count();
	// in umgekehrter Paint-Reihenfolge wg. Z-Order!
	for( ULONG nCur = nCount; nCur > 0; nCur-- )
	{
		ImplGroup* pGroup = maGroups.GetObject( nCur - 1 );
		if( pGroup->maRect.IsInside( rPos ) )
			return pGroup;
	}
	return 0;
}

// -----------------------------------------------------------------------

USHORT ImplGroupSet::GetGroupPos( const Point& rPos ) const
{
	if( !mbFormatted )
		return GROUPSET_GROUP_NOTFOUND;
	ULONG nCount = maGroups.Count();
	// in umgekehrter Paint-Reihenfolge wg. Z-Order!
	for( ULONG nCur = nCount; nCur > 0; nCur-- )
	{
		ImplGroup* pGroup = maGroups.GetObject( nCur - 1 );
		if( pGroup->maRect.IsInside( rPos ) )
			return (USHORT)(nCur-1);
	}
	return GROUPSET_GROUP_NOTFOUND;
}

// -----------------------------------------------------------------------

XubString ImplGroupSet::GetGroupHelpText( ImplGroup* pGroup ) const
{
	if( pGroup )
	{
		if ( !pGroup->maHelpText.Len() && pGroup->mnHelpId )
		{
			Help* pHelp = Application::GetHelp();
			if ( pHelp )
				pGroup->maHelpText = pHelp->GetHelpText( pGroup->mnHelpId, NULL );
		}
		return pGroup->maHelpText;
	}
	return XubString();
}

// -----------------------------------------------------------------------

void ImplGroupSet::CutBorder( Rectangle& rRect ) const
{
	rRect.Left() += nBorderWidthPixel;
	rRect.Right() -= nBorderWidthPixel;
	rRect.Top() += nBorderHeightPixel;
	rRect.Bottom() -= nBorderHeightPixel;
}

// ============================================================================

GroupSet::GroupSet( Window* pParent, WinBits nWinStyle ) :
	Control( pParent, nWinStyle | WB_CLIPCHILDREN ),
	mpContentWindow(0),
	mnCurGroupId(GROUPSET_GROUP_NOTFOUND)
{
	mpUp = 0; mpDown = 0; mpLeft = 0; mpRight = 0;
	mpPageUp = 0; mpPageDown = 0; mpPageLeft = 0; mpPageRight = 0;
	mpImpl = new ImplGroupSet;
	mpImpl->mnStyle = nWinStyle;
	ImplInitSettings();
	EnableDrop();
	mpImpl->maDropTimer.SetTimeout( 750 );
	mpImpl->maDropTimer.SetTimeoutHdl( LINK(this, GroupSet, ImplDropTimeoutHdl));
}

// -----------------------------------------------------------------------

GroupSet::GroupSet( Window* pParent, const ResId& rResId ) :
	Control( pParent, rResId ),
	mpContentWindow(0),
	mnCurGroupId(GROUPSET_GROUP_NOTFOUND)
{
	mpUp = 0; mpDown = 0; mpLeft = 0; mpRight = 0;
	mpPageUp = 0; mpPageDown = 0; mpPageLeft = 0; mpPageRight = 0;
	mpImpl = new ImplGroupSet;
	mpImpl->mnStyle = GetStyle();
	ImplInitSettings();
	EnableDrop();
	mpImpl->maDropTimer.SetTimeout( 1500 );
	mpImpl->maDropTimer.SetTimeoutHdl( LINK(this, GroupSet, ImplDropTimeoutHdl));
}

// -----------------------------------------------------------------------

GroupSet::~GroupSet()
{
	ImplReleaseScrollButtons();
	delete mpImpl;
}

// -----------------------------------------------------------------------

void GroupSet::Select()
{
	maSelectHdl.Call( this );
}

// -----------------------------------------------------------------------

void GroupSet::ContentWindowScrolled()
{
	maContentScrolledHdl.Call( this );
}

// -----------------------------------------------------------------------

void GroupSet::Resize()
{
	if( mpImpl->mbFormatted )
	{
		mpImpl->mbFormatted = FALSE;
		ImplFormat( FALSE, TRUE );
	}
	Control::Resize();
}

// -----------------------------------------------------------------------

void GroupSet::Paint( const Rectangle& rRect )
{
	if( !mpImpl->mbFormatted )
		ImplFormat( TRUE );

	USHORT nCount = (USHORT)mpImpl->maGroups.Count();
	for( USHORT nCur = 0; nCur < nCount; nCur++ )
	{
		ImplGroup* pGroup = mpImpl->maGroups.GetObject( (ULONG)nCur );
		BOOL bHighlight = mpImpl->mnCurHighlightedPos == nCur ? TRUE : FALSE;
		BOOL bPressed = ((mpImpl->mnCurPressedGroupId == pGroup->mnId) &&
			(mpImpl->mbGroupPressed)) ? TRUE : FALSE;

		ImplDrawGroup( nCur, bHighlight, bPressed );
	}
}

// -----------------------------------------------------------------------

void GroupSet::ImplDrawGroup( USHORT nPos, BOOL bHigh, BOOL bDrag,
							  BOOL bClip, const Rectangle* pRect )
{
	ImplGroup* pGroup = mpImpl->maGroups.GetObject( (ULONG)nPos );
	if( pGroup )
	{
		Region aOldClipRegion;
		BOOL bRestoreClipRegion = FALSE;

		DecorationView aDecoView( this );
		Rectangle aRect( pGroup->maRect );
		if( pRect )
			aRect = *pRect;

		ULONG nCount = mpImpl->maGroups.Count();
		if( bClip && (ULONG)(nPos + 1) < nCount )
		{
			Region aClipRegion( aRect );
			// alle nachfolgenden Rechtecke koennen das aktuelle
			// Rechteck [teilweise] ueberlappen
			for( ULONG nCur = nPos+1; nCur < nCount; nCur++ )
			{
				ImplGroup* pNextGroup = mpImpl->maGroups.GetObject( nCur );
				if( pNextGroup->maRect.IsOver( aRect ) )
				{
					aClipRegion.Exclude( pNextGroup->maRect );
					bRestoreClipRegion = TRUE;
				}
			}
			if( bRestoreClipRegion )
			{
				aOldClipRegion = GetClipRegion();
				SetClipRegion( aClipRegion );
			}
		}

		USHORT nStyle = 0;
		if( bHigh )
			nStyle =  BUTTON_DRAW_HIGHLIGHT;
		if( bDrag )
			nStyle = BUTTON_DRAW_PRESSED;
		nStyle |= BUTTON_DRAW_FLAT;

		const BOOL bIsCurrentGroup = (pGroup->mnId == mnCurGroupId) ? TRUE : FALSE;
		if( bIsCurrentGroup )
			nStyle |= BUTTON_DRAW_NOBOTTOMSHADOWBORDER;
		else if( nPos && (GetGroupId( nPos - 1) == mnCurGroupId) )
			nStyle |= BUTTON_DRAW_NOTOPLIGHTBORDER;

		aDecoView.DrawButton( aRect, nStyle );
		mpImpl->CutBorder( aRect );
		Font aOldFont;
		if( bIsCurrentGroup )
		{
			aOldFont = GetFont();
			Font aFont( aOldFont );
			aFont.SetWeight( CURGROUP_FONTWEIGHT );
			SetFont( aFont );
		}

		DrawText( aRect, pGroup->maText,
				TEXT_DRAW_ENDELLIPSIS | TEXT_DRAW_CENTER | TEXT_DRAW_VCENTER );

		if( bIsCurrentGroup )
			SetFont( aOldFont );

		if( bRestoreClipRegion )
			SetClipRegion( aOldClipRegion );
	}
}

// -----------------------------------------------------------------------

void GroupSet::ImplInvalidateGroup( ImplGroup& rGroup )
{
	if( IsUpdateMode() && IsVisible() )
	{
		if( !mpImpl->mbFormatted )
			ImplFormat();
		Invalidate( rGroup.maRect );
	}
}

// -----------------------------------------------------------------------

void GroupSet::InsertGroup( USHORT nGroupId, const XubString& rStr, USHORT nPos )
{
	EndRenamingGroup( TRUE );
	ImplGroup* pGroup = new ImplGroup( nGroupId );
	pGroup->maText = rStr;
	mpImpl->maGroups.Insert( pGroup, (ULONG)nPos );
	mpImpl->mbFormatted = FALSE;
	ImplFormat( FALSE, TRUE );
}

// -----------------------------------------------------------------------

void GroupSet::InsertGroup( USHORT nGroupId, const XubString& rStr,
							const Image& rImage, USHORT nPos )
{
	EndRenamingGroup( TRUE );
	ImplGroup* pGroup = new ImplGroup( nGroupId );
	pGroup->maText = rStr;
	pGroup->maImage = rImage;
	mpImpl->maGroups.Insert( pGroup, (ULONG)nPos );
	mpImpl->mbFormatted = FALSE;
	ImplFormat( FALSE, TRUE );
}

// -----------------------------------------------------------------------

void GroupSet::RemoveGroup( USHORT nGroupId, BOOL bAutoSelect )
{
	EndRenamingGroup( TRUE );
	USHORT nPos = GetGroupPos( nGroupId );
	if( nPos != GROUPSET_GROUP_NOTFOUND )
	{
		BOOL bIsCurGroup = (BOOL)(nGroupId == mnCurGroupId );
		ImplGroup* pGroup = mpImpl->maGroups.GetObject( (ULONG)nPos );
		delete pGroup;
		mpImpl->maGroups.Remove( (ULONG)nPos );
		mpImpl->mbFormatted = FALSE;
		if( bIsCurGroup )
		{
			mnCurGroupId = GROUPSET_GROUP_NOTFOUND;
			if( bAutoSelect )
			{
				USHORT nCount = GetGroupCount();
				if( nCount )
				{
					if( nPos < nCount )
						SetCurGroupId( GetGroupId( nPos ) );
					else
						SetCurGroupId( GetGroupId( nPos - 1 ) );
				}
			}
			// Select-Hdl auch rufen, wenn die letzte Gruppe geloescht wurde!
			Select();
		}
		if( !GetGroupCount() )
			Invalidate(INVALIDATE_NOCHILDREN);
		ImplFormat( FALSE, TRUE );

	}
}

// -----------------------------------------------------------------------

void GroupSet::Clear()
{
	EndRenamingGroup( TRUE );
	if( GetGroupCount() )
	{
		delete mpImpl;
		mpImpl = new ImplGroupSet;
		mnCurGroupId = GROUPSET_GROUP_NOTFOUND;
		ImplInitSettings();
		mpImpl->mbFormatted = FALSE;
		ImplFormat();
		if( IsUpdateMode() )
			Invalidate();
	}
}

// -----------------------------------------------------------------------

USHORT GroupSet::GetGroupCount() const
{
	return (USHORT)mpImpl->maGroups.Count();
}

// -----------------------------------------------------------------------

USHORT GroupSet::GetGroupPos( USHORT nGroupId ) const
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	return pGroup ? (USHORT)mpImpl->maGroups.GetPos( pGroup ) : GROUPSET_GROUP_NOTFOUND;
}

// -----------------------------------------------------------------------

USHORT GroupSet::GetGroupId( USHORT nPos ) const
{
	ImplGroup* pGroup = mpImpl->maGroups.GetObject( nPos );
	return pGroup ? pGroup->mnId : GROUPSET_GROUP_NOTFOUND;
}

// -----------------------------------------------------------------------

USHORT GroupSet::GetGroupId( const Point& rPos ) const
{
	ImplGroup* pGroup = mpImpl->GetGroup( rPos );
	return pGroup ? pGroup->mnId : GROUPSET_GROUP_NOTFOUND;
}

// -----------------------------------------------------------------------

USHORT GroupSet::GetGroupPos( const Point& rPos ) const
{
	return mpImpl->GetGroupPos( rPos );
}

// -----------------------------------------------------------------------

Rectangle GroupSet::GetGroupRect( USHORT nGroupId ) const
{
	if( !mpImpl->mbFormatted )
		((GroupSet*)this)->ImplFormat();
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	return pGroup ? pGroup->maRect : Rectangle();
}

// -----------------------------------------------------------------------

void GroupSet::SetCurGroupId( USHORT nGroupId )
{
	EndRenamingGroup( TRUE );
	if( mnCurGroupId != nGroupId )
	{
		if( mpImpl->mbFormatted && mnCurGroupId != GROUPSET_GROUP_NOTFOUND )
		{
			// wg. entfallendem BUTTON_DRAW_NOBOTTOMSHADOWBORDER invalidieren
			Invalidate( GetGroupRect( mnCurGroupId ), INVALIDATE_NOCHILDREN);
			// wg. entfallendem BUTTON_DRAW_NOTOPLIGHTBORDER invalidieren
			Invalidate( GetGroupRect( GetGroupId(GetGroupPos(mnCurGroupId)+1)),
				INVALIDATE_NOCHILDREN );
		}
		mnCurGroupId = nGroupId;
		const USHORT nCurGroupPos = GetGroupPos( nGroupId );
		// die aktuelle Gruppe wird nicht hervorgehoben, wenn sich die
		// Maus ueber ihr befindet
		if( nCurGroupPos == mpImpl->mnCurHighlightedPos )
			mpImpl->mnCurHighlightedPos = GROUPSET_GROUP_NOTFOUND;
		if( mpImpl->mbFormatted )
		{
			mpImpl->mbFormatted = FALSE;
			ImplFormat( FALSE, TRUE );
			Invalidate( GetGroupRect( mnCurGroupId ), INVALIDATE_NOCHILDREN);
			// die aktuelle Gruppe wird mit BUTTON_DRAW_NOBOTTOMSHADOWBORDER und ihr
			// Nachfolger mit BUTTON_DRAW_NOTOPLIGHTBORDER gezeichnet. Deshalb muessen
			// diese beiden Gruppen explizit invalidiert werden, da dies durch das
			// Neuformatieren nicht zwangslaeufig erfolgt.
			const USHORT nNextId = GetGroupId( nCurGroupPos + 1 );
			if( nNextId != GROUPSET_GROUP_NOTFOUND )
				Invalidate( GetGroupRect( nNextId ), INVALIDATE_NOCHILDREN);
		}
	}
}

// -----------------------------------------------------------------------

void GroupSet::SetGroupText( USHORT nGroupId, const XubString& rStr )
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	if( pGroup && pGroup->maText != rStr )
	{
		pGroup->maText = rStr;
		ImplInvalidateGroup( *pGroup );
	}
}

// -----------------------------------------------------------------------

void GroupSet::SetGroupImage( USHORT nGroupId, const Image& rImage )
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	if( pGroup )
	{
		pGroup->maImage = rImage;
		ImplInvalidateGroup( *pGroup );
	}
}

// -----------------------------------------------------------------------

XubString GroupSet::GetGroupText( USHORT nGroupId ) const
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	return pGroup ? pGroup->maText : XubString();
}

// -----------------------------------------------------------------------

Image GroupSet::GetGroupImage( USHORT nGroupId ) const
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	return pGroup ? pGroup->maImage : Image();
}

// -----------------------------------------------------------------------

void GroupSet::SetGroupData( USHORT nGroupId, void* pData )
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	if( pGroup )
		pGroup->mpGroupData = pData;
}

// -----------------------------------------------------------------------

void* GroupSet::GetGroupData( USHORT nGroupId ) const
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	return pGroup ? pGroup->mpGroupData : 0;
}

// -----------------------------------------------------------------------

void GroupSet::SetGroupHelpId( USHORT nGroupId, ULONG nHelpId )
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	if( pGroup )
		pGroup->mnHelpId = nHelpId;
}

// -----------------------------------------------------------------------

void GroupSet::SetGroupHelpText( USHORT nGroupId, const XubString& rStr )
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	if( pGroup )
		pGroup->maHelpText = rStr;
}

// -----------------------------------------------------------------------

void GroupSet::SetContentWindow( Window* pWindow )
{
	if( pWindow == mpContentWindow )
		return;

	mpImpl->maContentTotArea.SetEmpty();
	mpImpl->maContentVisArea.SetEmpty();
	ImplReleaseScrollButtons();
	EndRenamingGroup( TRUE );
	mpContentWindow = pWindow;
	if( pWindow && mpImpl->mbFormatted )
	{
		SetContentWindowPosSizePixel(
			mpImpl->maContentWinPos, mpImpl->maContentWinSize );
	}
	ImplUpdateScrollButtons();
}

// -----------------------------------------------------------------------

void GroupSet::RequestHelp( const HelpEvent& rHEvt )
{
	ImplGroup* pGroup = mpImpl->GetGroup( ScreenToOutputPixel( rHEvt.GetMousePosPixel() ) );
	if( pGroup )
	{
		if( rHEvt.GetMode() & (HELPMODE_QUICK | HELPMODE_BALLOON) )
		{
			Rectangle aGroupRect( pGroup->maRect );
			mpImpl->CutBorder( aGroupRect );
			Point aPt( OutputToScreenPixel( aGroupRect.TopLeft() ) );
			aGroupRect.SetPos( aPt );

			XubString aStr( mpImpl->GetGroupHelpText( pGroup ) );
			if ( !aStr.Len() || !(rHEvt.GetMode() & HELPMODE_BALLOON) )
			{
				// Wir zeigen die Quick-Hilfe nur an, wenn Text nicht
				// vollstaendig sichtbar, ansonsten zeigen wir den Hilfetext
				// an, wenn das Item keinen Text besitzt

				Font aOldFont;
				if( pGroup->mnId == mnCurGroupId )
				{
					aOldFont = GetFont();
					Font aFont( aOldFont );
					aFont.SetWeight( CURGROUP_FONTWEIGHT );
					SetFont( aFont );
				}
				long nTextWidth = GetTextWidth( pGroup->maText );
				if ( pGroup->mnId == mnCurGroupId )
					SetFont( aOldFont );
				if ( nTextWidth > aGroupRect.GetWidth() )
				{
					aGroupRect.Right() = aGroupRect.Left() + nTextWidth + 5;
					aStr = pGroup->maText;
				}
				else if ( pGroup->maText.Len() )
					aStr.Erase();
			}

			if ( aStr.Len() )
			{
				if ( rHEvt.GetMode() & HELPMODE_BALLOON )
					Help::ShowBalloon( mpContentWindow, aGroupRect.Center(), aGroupRect, aStr );
				else
					Help::ShowQuickHelp( mpContentWindow, aGroupRect, aStr, QUICKHELP_LEFT | QUICKHELP_VCENTER );
				return;
			}
		}
		else if ( rHEvt.GetMode() & HELPMODE_EXTENDED )
		{
			ULONG nHelpId = pGroup->mnHelpId;
			if ( nHelpId )
			{
				// Wenn eine Hilfe existiert, dann ausloesen
				Help* pHelp = Application::GetHelp();
				if ( pHelp )
					pHelp->Start( nHelpId, NULL );
				return;
			}
		}
	}
	Control::RequestHelp( rHEvt );
}

// -----------------------------------------------------------------------

void GroupSet::ImplFormat( BOOL bForce, BOOL bInvalidateChangedRects )
{
	if( mpImpl->mbFormatted )
		return;
	if( bForce || (IsUpdateMode() && IsVisible()) )
	{
		mpImpl->mbFormatted = TRUE;
		Size aOutputSize( GetOutputSizePixel() );
		Size aButtonSize( aOutputSize.Width(), mpImpl->mnButtonHeightPixel );
		const ULONG nCount = mpImpl->maGroups.Count();
		if( nCount )
		{
			ImplGroup* pTopGroup = 0, *pBottomGroup = 0;
			Point aTopLeft;
			ULONG nCur;

			// bis zum aktuellen Eintrag untereinander anordnen
			for( nCur = 0; nCur < nCount; nCur++ )
			{
				ImplGroup* pCurGroup = mpImpl->maGroups.GetObject( nCur );
				Rectangle aNewRect( aTopLeft, aButtonSize );
				if( bInvalidateChangedRects && pCurGroup->maRect != aNewRect )
				{
					Invalidate( pCurGroup->maRect );
					Invalidate( aNewRect );
				}
				pCurGroup->maRect = aNewRect;
				pTopGroup = pCurGroup;
				if( pCurGroup->mnId == mnCurGroupId )
					break;
				aTopLeft.Y() += aButtonSize.Height();
			}

			// vom letzten bis zum Nachfolger des aktuellen Eintrags
			// uebereinander anordnen
			if( nCur + 1 < nCount )
			{
				ULONG nLast = nCur + 1;
				aTopLeft.Y() = aOutputSize.Height() - aButtonSize.Height();
				pBottomGroup = mpImpl->maGroups.GetObject( nLast );
				for( nCur = nCount - 1; nCur >= nLast; nCur-- )
				{
					ImplGroup* pCurGroup = mpImpl->maGroups.GetObject( nCur );
					Rectangle aNewRect( aTopLeft, aButtonSize );
					if( bInvalidateChangedRects && pCurGroup->maRect != aNewRect )
					{
						Invalidate( pCurGroup->maRect );
						Invalidate( aNewRect );
					}
					pCurGroup->maRect = aNewRect;
					aTopLeft.Y() -= aButtonSize.Height();
				}
			}

			// Position und Groesse des Content-Windows berechnen
			Point aPos( pTopGroup->maRect.BottomLeft() );
			aPos.Y() += 1;
			mpImpl->maContentWinPos = aPos;
			long nHeight;
			if( pBottomGroup )
			{
				nHeight =
					pBottomGroup->maRect.Top() - pTopGroup->maRect.Bottom();
				nHeight--;
			}
			else
			{
				nHeight = aOutputSize.Height() - pTopGroup->maRect.Bottom();
				nHeight--;
			}
			mpImpl->maContentWinSize.Width() = aOutputSize.Width();
			mpImpl->maContentWinSize.Height() = nHeight;
		}
		else
		{
			mpImpl->maContentWinPos = Point();
			mpImpl->maContentWinSize = GetOutputSizePixel();
		}

		// Groesse und Position setzen
		if( mpContentWindow )
		{
			SetContentWindowPosSizePixel(
				mpImpl->maContentWinPos, mpImpl->maContentWinSize );
		}
	}
}

// -----------------------------------------------------------------------

void GroupSet::StateChanged( StateChangedType nStateChange )
{
	Control::StateChanged( nStateChange );
}

// -----------------------------------------------------------------------

void GroupSet::DataChanged( const DataChangedEvent& rDCEvt )
{
	if( (rDCEvt.GetType()==DATACHANGED_SETTINGS) && (rDCEvt.GetFlags() & SETTINGS_STYLE) )
	{
		mpImpl->mbFormatted = FALSE;
		ImplInitSettings();
		ImplUpdateScrollButtons();
		Invalidate();
	}
	Control::DataChanged( rDCEvt );
}

// -----------------------------------------------------------------------

void GroupSet::MouseMove( const MouseEvent& rMEvt )
{
	USHORT nPos = mpImpl->GetGroupPos( rMEvt.GetPosPixel() );
	ImplGroup* pGroup = (nPos != GROUPSET_GROUP_NOTFOUND) ?
		mpImpl->maGroups.GetObject( (ULONG)nPos ) : 0;

	PointerStyle eStyle = POINTER_ARROW;
	if( pGroup )
	{
		eStyle = POINTER_REFHAND;
		BOOL bGroupChanged = (BOOL)(mpImpl->mnCurHighlightedPos != nPos);
		if( bGroupChanged )
		{
			if( mpImpl->mnCurHighlightedPos != mpImpl->mnEditingPos )
				ImplDrawGroup( mpImpl->mnCurHighlightedPos, FALSE, FALSE, TRUE );
			// die aktive Gruppe nicht hervorheben
			if( nPos != GetGroupPos( mnCurGroupId ) && nPos != mpImpl->mnEditingPos )
			{
				mpImpl->mnCurHighlightedPos = nPos;
				ImplDrawGroup( nPos, TRUE, FALSE, TRUE	);
			}
			else
				mpImpl->mnCurHighlightedPos = GROUPSET_GROUP_NOTFOUND;
		}
	}
	else
	{
		if( !ImplInDrag() &&
			mpImpl->mnCurHighlightedPos != GROUPSET_GROUP_NOTFOUND &&
			mpImpl->mnCurHighlightedPos != mpImpl->mnEditingPos )
		{
			ImplDrawGroup( mpImpl->mnCurHighlightedPos, FALSE, FALSE, TRUE );
			mpImpl->mnCurHighlightedPos = GROUPSET_GROUP_NOTFOUND;
		}
	}
	Pointer aPtr( eStyle );
	SetPointer( aPtr );
}

// -----------------------------------------------------------------------

void GroupSet::MouseButtonDown( const MouseEvent& rMEvt )
{
	EndRenamingGroup( TRUE );
	if ( rMEvt.IsLeft() )
	{
		ImplStartDrag( rMEvt.GetPosPixel(), FALSE );
	}
}

// -----------------------------------------------------------------------

void GroupSet::MouseButtonUp( const MouseEvent& rMEvt )
{
}

// -----------------------------------------------------------------------

void GroupSet::ImplInitSettings()
{
	const StyleSettings& rStyleSettings = GetSettings().GetStyleSettings();
	Font aFont = rStyleSettings.GetToolFont();
	if ( IsControlFont() )
		aFont.Merge( GetControlFont() );
	Font aNonBoldFont( aFont );
	aFont.SetWeight( CURGROUP_FONTWEIGHT );
	SetZoomedPointFont( aFont );
	mpImpl->mnButtonHeightPixel = GetTextHeight();
	USHORT nStyle = FRAME_DRAW_NODRAW | BUTTON_DRAW_HIGHLIGHT | BUTTON_DRAW_FLAT;
	DecorationView	aDecoView( this );
	Rectangle aRect( 0, 0, 10, 10 );
	Rectangle aCalcRect( aDecoView.DrawFrame( aRect, nStyle ) );
	// Platz fuer den Border draufrechnen
	mpImpl->mnButtonHeightPixel += aCalcRect.Top();
	mpImpl->nBorderHeightPixel = aCalcRect.Top();
	mpImpl->nBorderWidthPixel = aCalcRect.Left();
	mpImpl->mnButtonHeightPixel += (aRect.Bottom()-aCalcRect.Bottom());

	SetZoomedPointFont( aNonBoldFont );
	if ( IsControlForeground() )
		SetTextColor( GetControlForeground() );
	else
		SetTextColor( rStyleSettings.GetButtonTextColor() );

	SetBackground( rStyleSettings.GetCheckedColor() );
}

// -----------------------------------------------------------------------

void GroupSet::Tracking( const TrackingEvent& rTEvt )
{
	Point aMousePos = rTEvt.GetMouseEvent().GetPosPixel();

	if ( rTEvt.IsTrackingEnded() )
		ImplEndDrag( rTEvt.IsTrackingCanceled() );
	else
		ImplDrag( aMousePos );

}

// -----------------------------------------------------------------------

void GroupSet::Command( const CommandEvent& rCEvt )
{
	if(rCEvt.IsMouseEvent() &&
		(rCEvt.GetCommand()==COMMAND_STARTDRAG) && !ImplInDrag() )
	{
		ImplStartDrag( rCEvt.GetMousePosPixel(), TRUE );
		return;
	}
	Window::Command( rCEvt );
}

// -----------------------------------------------------------------------

BOOL GroupSet::ImplInDrag() const
{
	return mpImpl->mbTracking;
}

// -----------------------------------------------------------------------

void GroupSet::ImplStartDrag( const Point& rPosPixel, BOOL bCommand )
{
	USHORT nPos = mpImpl->GetGroupPos( rPosPixel );
	if( ((!(mpImpl->mnStyle & WB_GROUPS_MOVEABLE)) && (nPos==GetGroupPos(mnCurGroupId))) ||
		 nPos == GROUPSET_GROUP_NOTFOUND)
		return;
	USHORT nGroupId = GetGroupId( nPos );
	mpImpl->mbTracking = TRUE;
	mpImpl->mnCurPressedGroupId = nGroupId;
	mpImpl->mbGroupPressed = TRUE;
	ImplDrawGroup( nPos, FALSE, TRUE, TRUE );
	StartTracking();
}

// -----------------------------------------------------------------------

void GroupSet::ImplEndDrag( BOOL bCancel )
{
	SetPointer( Pointer( POINTER_REFHAND ));
	mpImpl->mbTracking = FALSE;
	USHORT nId = mpImpl->mnCurPressedGroupId;
	mpImpl->mnCurPressedGroupId = GROUPSET_GROUP_NOTFOUND;
	if( nId != GROUPSET_GROUP_NOTFOUND )
		ImplDrawGroup( GetGroupPos(nId), FALSE, FALSE, TRUE );

	if( mpImpl->mnMoveToPos != GROUPSET_GROUP_NOTFOUND )
	{
		USHORT nNewPos = mpImpl->mnMoveToPos;
		ImplSetMoveToPos( GROUPSET_GROUP_NOTFOUND );
		if( !bCancel )
			MoveGroup( nId, nNewPos );
	}
	else
	{
		BOOL bHit = mpImpl->mbGroupPressed;
		if( bHit )
		{
			USHORT nPos = GetGroupPos( nId );
			ImplDrawGroup( nPos, FALSE, FALSE, TRUE );
			if( !bCancel )
			{
				if( mnCurGroupId != nId )
				{
					SetCurGroupId( nId );
					Select();
				}
				else
				{
					// Bei Klick auf die aktive Gruppe soll ihre Nachfolderin
					// aktiviert werden.
					USHORT nNextId = GetGroupId( nPos + 1);
					if( nNextId != GROUPSET_GROUP_NOTFOUND )
					{
						SetCurGroupId( nNextId );
						Select();
					}
				}
			}
		}
	}
}

// -----------------------------------------------------------------------

void GroupSet::ImplDrag( const Point& rPosPixel )
{
	USHORT nPos = mpImpl->GetGroupPos( rPosPixel );
	USHORT nId = GetGroupId( nPos );
	BOOL bDown = (BOOL)(mpImpl->mnCurPressedGroupId == nId );

	USHORT nNewPos;
	if( mpImpl->mnStyle & WB_GROUPS_MOVEABLE )
	{
		nNewPos = GetMovePos( rPosPixel, mpImpl->mnCurPressedGroupId );
		ImplSetMoveToPos( nNewPos );
	}
	else
		nNewPos = GROUPSET_GROUP_NOTFOUND;

	BOOL bRepaint = (bDown != mpImpl->mbGroupPressed) ? TRUE : FALSE;
	mpImpl->mbGroupPressed = bDown;
	if( nNewPos != GROUPSET_GROUP_NOTFOUND )
	{
		bDown = TRUE;
		if( bDown != mpImpl->mbGroupPressed )
			bRepaint = TRUE;
	}
	if( bRepaint )
		ImplDrawGroup( GetGroupPos(mpImpl->mnCurPressedGroupId), FALSE, bDown, TRUE );

	PointerStyle eStyle = POINTER_REFHAND;
	if( !Rectangle( Point(), GetOutputSizePixel()).IsInside(rPosPixel) )
		eStyle = POINTER_NOTALLOWED;
	else if( Rectangle( mpImpl->maContentWinPos, mpImpl->maContentWinSize
			).IsInside( rPosPixel))
		eStyle = POINTER_ARROW;
	else if( nNewPos != GROUPSET_GROUP_NOTFOUND )
		eStyle = POINTER_MOVEDATA;
	SetPointer( Pointer( eStyle ));
}

// -----------------------------------------------------------------------

IMPL_LINK( GroupSet, ImplDropTimeoutHdl, Timer*, EMPTYARG )
{
	mpImpl->maDropTimer.Stop();
	if( mpImpl->mnDropTargetPos != GROUPSET_GROUP_NOTFOUND )
	{
		//ImplDrawGroup( mpImpl->mnDropTargetPos, FALSE, FALSE, TRUE );
		SetCurGroupId( GetGroupId( mpImpl->mnDropTargetPos ) );
		mpImpl->mnDropTargetPos = GROUPSET_GROUP_NOTFOUND;
		Select();
	}
	return 0;
}

// -----------------------------------------------------------------------

BOOL GroupSet::QueryDrop( DropEvent& rDEvt )
{
	if( ImplInDrag() )
		return FALSE;

	if( rDEvt.IsLeaveWindow() )
	{
		mpImpl->maDropTimer.Stop();
		if( mpImpl->mnDropTargetPos != GROUPSET_GROUP_NOTFOUND )
		{
			ImplDrawGroup( mpImpl->mnDropTargetPos, FALSE, FALSE, TRUE );
			mpImpl->mnDropTargetPos = GROUPSET_GROUP_NOTFOUND;
		}
		return FALSE;
	}

	USHORT nPos = GetGroupPos( rDEvt.GetPosPixel() );
	if( mnCurGroupId == GetGroupId(nPos)  )
		nPos = GROUPSET_GROUP_NOTFOUND;

	if( nPos != mpImpl->mnDropTargetPos )
	{
		if( mpImpl->mnDropTargetPos != GROUPSET_GROUP_NOTFOUND )
			ImplDrawGroup( mpImpl->mnDropTargetPos, FALSE, FALSE, TRUE );
		if( nPos != GROUPSET_GROUP_NOTFOUND )
			ImplDrawGroup( nPos, TRUE, FALSE, TRUE );
	}
	mpImpl->mnDropTargetPos = nPos;
	if( nPos == GROUPSET_GROUP_NOTFOUND )
	{
		mpImpl->maDropTimer.Stop();
		return FALSE;
	}
	else if( !mpImpl->maDropTimer.IsActive() )
		mpImpl->maDropTimer.Start();
	//rDEvt.SetAction( DROP_MOVE );
	return FALSE;
}

// -----------------------------------------------------------------------

BOOL GroupSet::Drop( const DropEvent& rDEvt )
{
	return FALSE;
}

// -----------------------------------------------------------------------

IMPL_LINK( GroupSet, ImplEditedHdl, void*, EMPTYARG )
{
	USHORT nEditedPos = mpImpl->mnEditingPos;
	USHORT nEditedId = GetGroupId( nEditedPos );

	mpImpl->mnEditingPos = GROUPSET_GROUP_NOTFOUND;
	if( nEditedPos == GROUPSET_GROUP_NOTFOUND )
	{
		mpImpl->mpEdit->Hide();
		if( mpImpl->mpEdit->IsGrabFocus() )
			GrabFocus();
		return 0;
	}

	XubString aGroupText( GetGroupText( nEditedId ) );
	XubString aText;
	if ( !mpImpl->mpEdit->EditingCanceled() )
		aText = mpImpl->mpEdit->GetText();
	else
		aText = aGroupText;

	mpImpl->mpEdit->Hide();
	if( mpImpl->mpEdit->IsGrabFocus() )
		GrabFocus();

	// die Aenderung verfaellt, wenn im Callback der Text geaendert wird
	if( GroupRenamed(nEditedId,aText) && aGroupText==GetGroupText(nEditedId))
		SetGroupText( nEditedId, aText );
	return 0;
}

// -----------------------------------------------------------------------

BOOL GroupSet::IsRenamingGroup() const
{
	return mpImpl->mnEditingPos == GROUPSET_GROUP_NOTFOUND ?
		   FALSE : TRUE;
}

// -----------------------------------------------------------------------

void GroupSet::EndRenamingGroup( BOOL bCancel )
{
	if( IsRenamingGroup() )
		mpImpl->mpEdit->StopEditing( bCancel );
}

// -----------------------------------------------------------------------

void GroupSet::RenameGroup( USHORT nGroupId, const String* pText )
{
	EndRenamingGroup( TRUE );
	USHORT nEditedPos = GetGroupPos( nGroupId );
	if( nEditedPos == GROUPSET_GROUP_NOTFOUND )
		return;

	DELETEZ(mpImpl->mpEdit);
	// MakeFieldVisible( (long)nRow, nColumnId, FALSE );
	Update();
	mpImpl->mnEditingPos = nEditedPos;
	Rectangle aRect( GetGroupRect( nGroupId ) );
	Point aPos( aRect.TopLeft() );
	if( aPos.X() < 0 || aPos.Y() < 0 )
		aRect.SetPos( Point() );
	mpImpl->CutBorder( aRect );
	mpImpl->mpEdit = new ImplGroupEdit(
		this,
		aRect.TopLeft(),
		aRect.GetSize(),
		pText ? *pText : GetGroupText( nGroupId ),
		LINK( this, GroupSet, ImplEditedHdl ) );
}

// -----------------------------------------------------------------------

BOOL GroupSet::GroupRenamed( USHORT nGroupId, const XubString& rStr )
{
	return TRUE;
}

// ============================================================================

// -----------------------------------------------------------------------

long GroupSet::PreNotify( NotifyEvent& rNEvt )
{
	if( rNEvt.GetType() == EVENT_MOUSEBUTTONDOWN )
	{
		if( mpImpl->mnEditingPos != GROUPSET_GROUP_NOTFOUND &&
			rNEvt.GetWindow() != mpImpl->mpEdit )
		{
			EndRenamingGroup( TRUE );
		}
	}
	return Control::PreNotify( rNEvt );
}

// -----------------------------------------------------------------------

XubString GroupSet::GetGroupHelpText( USHORT nGroupId ) const
{
	return mpImpl->GetGroupHelpText( mpImpl->GetGroup( nGroupId ) );
}

// -----------------------------------------------------------------------

ULONG GroupSet::GetGroupHelpId( USHORT nGroupId ) const
{
	ImplGroup* pGroup = mpImpl->GetGroup( nGroupId );
	return	pGroup ? pGroup->mnHelpId : 0;
}

// -----------------------------------------------------------------------

void GroupSet::SetContentWindowArea( const Rectangle& rRect )
{
	mpImpl->maContentTotArea = rRect;
	ImplUpdateScrollButtons();
}

// -----------------------------------------------------------------------

void GroupSet::SetContentWindowVisArea( const Rectangle& rRect )
{
	mpImpl->maContentVisArea = rRect;
	ImplUpdateScrollButtons();
}

// -----------------------------------------------------------------------

void GroupSet::SetContentWindowLineSize( const Size& rSize )
{
	mpImpl->maContentLineSize = rSize;
}

// -----------------------------------------------------------------------

void GroupSet::SetContentWindowPageSize( const Size& rSize )
{
	mpImpl->maContentPageSize = rSize;
}

// -----------------------------------------------------------------------

Rectangle GroupSet::GetContentWindowVisArea() const
{
	return mpImpl->maContentVisArea;
}

// -----------------------------------------------------------------------

Rectangle GroupSet::GetContentWindowArea() const
{
	return mpImpl->maContentTotArea;
}

// -----------------------------------------------------------------------

Size GroupSet::GetContentWindowLineSize() const
{
	return mpImpl->maContentLineSize;
}

// -----------------------------------------------------------------------

Size GroupSet::GetContentWindowPageSize() const
{
	return mpImpl->maContentPageSize;
}

// -----------------------------------------------------------------------

PushButton* GroupSet::ImplCreateButton( SymbolType eSymbol, const Link& rHdl )
{
	PushButton* pButton = new PushButton( mpContentWindow,
		WB_REPEAT | WB_NOTABSTOP| WB_NOPOINTERFOCUS |
		WB_RECTSTYLE | WB_NOLIGHTBORDER );
	pButton->SetSymbol( eSymbol );
	pButton->SetClickHdl( rHdl );
	return pButton;
}

// -----------------------------------------------------------------------

void GroupSet::ImplCreateButtons( int bUp, int bDown, int bLeft, int bRight )
{
	BOOL bDisableUp = FALSE, bDisableDown = FALSE;
	BOOL bDisableLeft = FALSE, bDisableRight = FALSE;

	if( bUp && !bDown )
	{
		bDown = TRUE;
		bDisableDown = TRUE;
	}
	if( !bUp && bDown )
	{
		bUp = TRUE;
		bDisableUp = TRUE;
	}
	if( bLeft && !bRight )
	{
		bRight = TRUE;
		bDisableRight = TRUE;
	}
	if( !bLeft && bRight )
	{
		bLeft = TRUE;
		bDisableLeft = TRUE;
	}

	Link aClickHdl( LINK(this, GroupSet, ImplScrollBtnClickHdl));
	if( bUp )
	{
		if( !mpUp )
			mpUp = ImplCreateButton( SYMBOL_SPIN_UP, aClickHdl );
		if( !mpPageUp )
			mpPageUp = ImplCreateButton( SYMBOL_PAGEUP, aClickHdl );
		mpUp->Enable( !bDisableUp );
		mpPageUp->Enable( !bDisableUp );
	}
	else
	{
		DeleteButton( &mpUp );
		DeleteButton( &mpPageUp );
	}

	if( bDown )
	{
		if( !mpDown )
			mpDown = ImplCreateButton( SYMBOL_SPIN_DOWN, aClickHdl );
		if( !mpPageDown )
			mpPageDown = ImplCreateButton( SYMBOL_PAGEDOWN, aClickHdl );
		mpDown->Enable( !bDisableDown );
		mpPageDown->Enable( !bDisableDown );
	}
	else
	{
		DeleteButton( &mpDown );
		DeleteButton( &mpPageDown );
	}

	if( bLeft )
	{
		if( !mpLeft )
			mpLeft = ImplCreateButton( SYMBOL_SPIN_LEFT, aClickHdl );
		if( !mpPageLeft )
			mpPageLeft = ImplCreateButton( SYMBOL_PAGEUP, aClickHdl );
		mpLeft->Enable( !bDisableLeft );
		mpPageLeft->Enable( !bDisableLeft );
	}
	else
	{
		DeleteButton( &mpLeft );
		DeleteButton( &mpPageLeft );
	}

	if( bRight )
	{
		if( !mpRight )
			mpRight = ImplCreateButton( SYMBOL_SPIN_RIGHT, aClickHdl );
		if( !mpPageRight )
			mpPageRight = ImplCreateButton( SYMBOL_PAGEDOWN, aClickHdl );
		mpRight->Enable( !bDisableRight );
		mpPageRight->Enable( !bDisableRight );
	}
	else
	{
		DeleteButton( &mpRight );
		DeleteButton( &mpPageRight );
	}
}

// -----------------------------------------------------------------------

void GroupSet::ImplPositionButtons()
{
	if( !mpContentWindow )
		return;

	const StyleSettings& rSettings = GetSettings().GetStyleSettings();
	long nScrBarSize = rSettings.GetScrollBarSize();
	long nBorderDistance = nScrBarSize / 4;
	Size aButtonSizePixel( nScrBarSize, nScrBarSize );
	const Size& rOutSize = mpImpl->maContentWinSize;

	Point aPos;

#ifndef GROUP_SCROLLBUTTONS
	if( mpUp )
	{
		aPos.Y() = nBorderDistance;
		aPos.X() = rOutSize.Width() - nScrBarSize - nBorderDistance;
		// PageUp/Down/Left/Right liegen aussen
		if( mpPageUp )
		{
			mpPageUp->SetPosSizePixel( aPos, aButtonSizePixel );
			mpPageUp->Show();
			aPos.Y() += aButtonSizePixel.Height();
		}
		mpUp->SetPosSizePixel( aPos, aButtonSizePixel );
		mpUp->Show();
	}
	if( mpDown )
	{
		aPos.Y() = rOutSize.Height() - nScrBarSize - nBorderDistance;
		if( mpRight )
			aPos.Y() -= nScrBarSize;
		aPos.X() = rOutSize.Width() - nScrBarSize - nBorderDistance;
		if( mpPageDown )
		{
			mpPageDown->SetPosSizePixel( aPos, aButtonSizePixel );
			mpPageDown->Show();
			aPos.Y() -= aButtonSizePixel.Height();
		}
		mpDown->SetPosSizePixel( aPos, aButtonSizePixel );
		mpDown->Show();
	}
	if( mpLeft )
	{
		aPos.Y() = rOutSize.Height() - nScrBarSize - nBorderDistance;
		aPos.X() = nBorderDistance;
		if( mpPageLeft )
		{
			mpPageLeft->SetPosSizePixel( aPos, aButtonSizePixel );
			mpPageLeft->Show();
			aPos.X() += aButtonSizePixel.Width();
		}
		mpLeft->SetPosSizePixel( aPos, aButtonSizePixel );
		mpLeft->Show();
	}
	if( mpRight )
	{
		aPos.Y() = rOutSize.Height() - nScrBarSize - nBorderDistance;
		aPos.X() = rOutSize.Width() - nScrBarSize - nBorderDistance;
		if( mpDown )
			aPos.X() -= nScrBarSize;
		if( mpPageRight )
		{
			mpPageRight->SetSizePixel( aButtonSizePixel );
			mpPageRight->Show();
			aPos.X() -= aButtonSizePixel.Width();
		}
		mpRight->SetSizePixel( aButtonSizePixel );
		mpRight->Show();
	}
#else
	aPos.Y() = nBorderDistance;
	aPos.X() = rOutSize.Width() - nScrBarSize - nBorderDistance;
	if( mpPageUp )
	{
		mpPageUp->SetPosSizePixel( aPos, aButtonSizePixel );
		mpPageUp->Show();
		aPos.Y() += aButtonSizePixel.Height();
	}
	if( mpUp )
	{
		mpUp->SetPosSizePixel( aPos, aButtonSizePixel );
		mpUp->Show();
		aPos.Y() += aButtonSizePixel.Height();
	}
	if( mpDown )
	{
		mpDown->SetPosSizePixel( aPos, aButtonSizePixel );
		mpDown->Show();
		aPos.Y() += aButtonSizePixel.Height();
	}
	if( mpPageDown )
	{
		mpPageDown->SetPosSizePixel( aPos, aButtonSizePixel );
		mpPageDown->Show();
	}

	aPos.Y() = rOutSize.Height() - nScrBarSize - nBorderDistance;
	aPos.X() = nBorderDistance;
	if( mpPageLeft )
	{
		mpPageLeft->SetPosSizePixel( aPos, aButtonSizePixel );
		mpPageLeft->Show();
		aPos.X() += aButtonSizePixel.Width();
	}
	if( mpLeft )
	{
		mpLeft->SetPosSizePixel( aPos, aButtonSizePixel );
		mpLeft->Show();
		aPos.X() += aButtonSizePixel.Width();
	}
	if( mpRight )
	{
		mpRight->SetPosSizePixel( aPos, aButtonSizePixel );
		mpRight->Show();
		aPos.X() += aButtonSizePixel.Width();
	}
	if( mpPageRight )
	{
		mpPageRight->SetPosSizePixel( aPos,aButtonSizePixel );
		mpPageRight->Show();
	}
#endif
}

// -----------------------------------------------------------------------

void GroupSet::ImplUpdateScrollButtons()
{
	if( !mpContentWindow )
		return;

	Rectangle rTot( mpImpl->maContentTotArea );
	Rectangle rVis( mpImpl->maContentVisArea );
	rVis = rVis.Intersection( rTot );

	BOOL bTotEmpty = rTot.IsEmpty();
	BOOL bVisEmpty = rVis.IsEmpty();
	if( bTotEmpty || bVisEmpty )
	{
		ImplReleaseScrollButtons();
		return;
	}

	WinBits nStyle = GetStyle();

	int bUp = (nStyle & WB_VSCROLL) && (rVis.Top() != rTot.Top());
	int bDown = (nStyle & WB_VSCROLL) && (rVis.Bottom() != rTot.Bottom());
	int bLeft = (nStyle & WB_HSCROLL) && (rVis.Left() != rTot.Left());
	int bRight = (nStyle & WB_HSCROLL) && (rVis.Right() != rTot.Right());
	ImplCreateButtons( bUp, bDown, bLeft, bRight );
	ImplPositionButtons();
}

// -----------------------------------------------------------------------

void GroupSet::ImplReleaseScrollButtons()
{
	DeleteButton( &mpUp );
	DeleteButton( &mpPageUp );
	DeleteButton( &mpDown );
	DeleteButton( &mpPageDown );
	DeleteButton( &mpLeft );
	DeleteButton( &mpPageLeft );
	DeleteButton( &mpRight );
	DeleteButton( &mpPageRight );
}

// -----------------------------------------------------------------------

IMPL_LINK( GroupSet, ImplScrollBtnClickHdl, PushButton*, pButton )
{
	long nHorzMove = 0;
	long nVertMove = 0;
	if( pButton == mpUp )
		nVertMove = -1 * mpImpl->maContentLineSize.Height();
	else if( pButton == mpDown )
		nVertMove = mpImpl->maContentLineSize.Height();
	else if( pButton == mpRight )
		nHorzMove = -1 * mpImpl->maContentLineSize.Width();
	else if( pButton == mpLeft )
		nHorzMove = mpImpl->maContentLineSize.Width();
	else if( pButton == mpPageUp )
		nVertMove = -1 * mpImpl->maContentPageSize.Height();
	else if( pButton == mpPageDown )
		nVertMove = mpImpl->maContentPageSize.Height();
	else if( pButton == mpPageRight )
		nHorzMove = -1 * mpImpl->maContentPageSize.Width();
	else if( pButton == mpPageLeft )
		nHorzMove = mpImpl->maContentPageSize.Width();

	if( nHorzMove || nVertMove )
	{
		mpImpl->maContentVisArea.Move( nHorzMove, nVertMove );
		mpImpl->maContentVisArea.Intersection( mpImpl->maContentTotArea );
		ContentWindowScrolled();
		ImplUpdateScrollButtons();
	}
	return 0;
}

// -----------------------------------------------------------------------

void GroupSet::SetContentWindowPosSizePixel( const Point& rPos, const Size& rSize )
{
	mpContentWindow->SetPosSizePixel(
			mpImpl->maContentWinPos, mpImpl->maContentWinSize );
}

// -----------------------------------------------------------------------

const Point& GroupSet::GetContentWindowPos() const
{
	return mpImpl->maContentWinPos;
}

// -----------------------------------------------------------------------

const Size& GroupSet::GetContentWindowSize() const
{
	return mpImpl->maContentWinSize;
}

// -----------------------------------------------------------------------

void GroupSet::MoveGroup( USHORT nGroupId, USHORT nPos )
{
	SetGroupPos( nGroupId, nPos );
}

// -----------------------------------------------------------------------

BOOL GroupSet::SetGroupPos( USHORT nGroupId, USHORT nPos )
{
	USHORT nOldPos = GetGroupPos( nGroupId );
	if( nOldPos != GROUPSET_GROUP_NOTFOUND )
	{
		const USHORT nTargetId = GetGroupId( nPos );
		if( nGroupId == nTargetId )
			return FALSE;
		ImplGroup* pGroup = (ImplGroup*)mpImpl->maGroups.GetObject( nOldPos );
		mpImpl->maGroups.Remove( (ULONG)nOldPos );
		const USHORT nNewPos = (nTargetId != GROUPSET_GROUP_NOTFOUND) ?
				GetGroupPos( nTargetId ) : (USHORT)mpImpl->maGroups.Count();
		mpImpl->maGroups.Insert( pGroup, (ULONG)nNewPos );
		BOOL bChanged = (nNewPos != nOldPos) ? TRUE : FALSE;
		if( bChanged )
		{
			mpImpl->mbFormatted = FALSE;
			Invalidate();
		}
		return bChanged;
	}
	return FALSE;
}

// -----------------------------------------------------------------------

USHORT GroupSet::GetMovePos( const Point& rPos, USHORT nCurGroupId ) const
{
	if( !mpImpl->mbFormatted )
		return GROUPSET_GROUP_NOTFOUND;

	const USHORT nCount = (USHORT)mpImpl->maGroups.Count();
	if( nCount <= 1 )
		return GROUPSET_GROUP_NOTFOUND;

	USHORT nPos = GROUPSET_GROUP_NOTFOUND;
	const USHORT nCurGroupPos = GetGroupPos( mnCurGroupId );

	for( USHORT nCur = 0; nCur < nCount; nCur++ )
	{
		const ImplGroup* pGroup = (const ImplGroup*)mpImpl->maGroups.GetObject( nCur );
		if( pGroup->maRect.IsInside( rPos ))
		{
			if( rPos.Y() > pGroup->maRect.Center().Y() )
				nPos = nCur + 1;
			else
				nPos = nCur;
			break;
		}
	}
	if(nCurGroupId != GROUPSET_GROUP_NOTFOUND && nPos != GROUPSET_GROUP_NOTFOUND)
	{
		USHORT nCurPos = GetGroupPos( nCurGroupId );
		if( nCurPos == nPos || ((nCurPos+1) == nPos ))
			nPos = GROUPSET_GROUP_NOTFOUND;
	}
	return nPos;
}

// -----------------------------------------------------------------------

void GroupSet::ImplDrawMoveToEmphasis( USHORT nPos, BOOL bShow )
{
	USHORT nCount = (USHORT)mpImpl->maGroups.Count();
	if( !nCount || nPos == GROUPSET_GROUP_NOTFOUND )
		return;
	BOOL bBottom = FALSE;
	BOOL bSplitted = FALSE;
	if( nPos >= nCount )
	{
		bBottom = TRUE;
		nPos--;
	}
	else
	{
		if( GetGroupPos(mnCurGroupId) + 1 == nPos )
		{
			nPos--;
			bBottom = TRUE;
			bSplitted = TRUE;
		}
	}

	const ImplGroup* pGroup = (const ImplGroup*)mpImpl->maGroups.GetObject( nPos );
	Rectangle aRect( pGroup->maRect );
	if( bBottom )
		aRect.Top() = aRect.Bottom();
	else
		aRect.Bottom() = aRect.Top();
#if 0 // ein dicker roter Strich
	aRect.Bottom() += 2;
	aRect.Top() -= 2;
	if( !bShow )
	{
		Invalidate( aRect );
		Update();
	}
	else
	{
		Update();
		Color aOld( GetFillColor());
		SetFillColor( Color( COL_LIGHTRED ));
		DrawRect( aRect );
		SetFillColor( aOld );
	}
#else // kleine Pfeilchen
	aRect.Bottom() += 2;
	aRect.Top() -= 2;
	long n3 = 1 + (mpImpl->mnButtonHeightPixel / 3);
	aRect.Bottom() += n3;
	aRect.Top() -= n3;
	aRect.Right() -= n3;
	aRect.Left() += n3;
	Size aSize3( n3,n3);
	long nYOffs = 0;
	if( bSplitted )
		nYOffs = mpImpl->maContentWinSize.Height() + 2;

	if( !bShow )
	{
		if( bSplitted )
			aRect.Bottom() += nYOffs;
		Invalidate( aRect, INVALIDATE_NOCHILDREN );
		Update();
	}
	else
	{
		Update();
		DecorationView aDecoView( this );
		Color aColor( GetTextColor() );
		Point aPos( aRect.TopLeft() );
		Rectangle aCur( aPos, aSize3 );
		aDecoView.DrawSymbol( aCur, SYMBOL_SPIN_DOWN, aColor );
		aPos.X() = aRect.Right() - n3;
		aCur.SetPos( aPos );
		aDecoView.DrawSymbol( aCur, SYMBOL_SPIN_DOWN, aColor  );

		aPos.Y() = aRect.Bottom() - n3 + nYOffs;
		aCur.SetPos( aPos );
		aDecoView.DrawSymbol( aCur, SYMBOL_SPIN_UP, aColor	);
		aPos.X() = aRect.Left();
		aCur.SetPos( aPos );
		aDecoView.DrawSymbol( aCur, SYMBOL_SPIN_UP, aColor	);
	}
#endif

}

// -----------------------------------------------------------------------

// Einfuegung zwischen Eintrag nPos-1 und Eintrag nPos
void GroupSet::ImplSetMoveToPos( USHORT nPos )
{
	if( nPos == mpImpl->mnMoveToPos && mpImpl->mbMoveToPosEmphasized )
		return;

	if( mpImpl->mbMoveToPosEmphasized )
	{
		ImplDrawMoveToEmphasis( mpImpl->mnMoveToPos, FALSE );
		mpImpl->mbMoveToPosEmphasized = FALSE;
	}
	mpImpl->mnMoveToPos = nPos;
	if( nPos != GROUPSET_GROUP_NOTFOUND )
	{
		ImplDrawMoveToEmphasis( nPos, TRUE );
		mpImpl->mbMoveToPosEmphasized = TRUE;
	}
}

