/*
 * This file is part of the QPackageKit project
 * Copyright (C) 2008 Adrien Bustany <madcat@mymadcat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef CLIENT_H
#define CLIENT_H

#include <QtCore>

namespace PackageKit {

class ClientPrivate;
class Package;
class Transaction;

/**
 * \class Client client.h Client
 * \author Adrien Bustany <madcat@mymadcat.com>
 *
 * \brief Base class used to interact with the PackageKit daemon
 *
 * This class holds all the functions enabling the user to interact with the PackageKit daemon.
 * The user should always use this class to initiate transactions.
 *
 * All the function returning a pointer to a Transaction work in an asynchronous way. The returned
 * object can be used to alter the operation's execution, or monitor it's state. The Transaction
 * object will be automatically deleted after it emits the finished() signal.
 *
 * \note This class is a singleton, its constructor is private. Call Client::instance() to get
 * an instance of the Client object
 */
class Client : public QObject
{

	Q_OBJECT
	Q_ENUMS(Action)
	Q_ENUMS(Filter)
	Q_ENUMS(Group)
	Q_ENUMS(NetworkState)
	Q_ENUMS(SignatureType)
	Q_ENUMS(ProvidesType)
	Q_ENUMS(ErrorType)
	Q_ENUMS(MessageType)
	Q_ENUMS(RestartType)
	Q_ENUMS(UpdateState)
	Q_ENUMS(DistroUpgradeType)

public:
	/**
	 * \brief Returns an instance of the Client
	 *
	 * The Client class is a singleton, you can call this method several times,
	 * a single Client object will exist.
	 */
	static Client* instance();

	/**
	 * Destructor
	 */
	~Client();

	// Daemon functions

	/**
	 * Lists all the available actions
	 * \sa getActions
	 */
	typedef enum {
		ActionCancel		 = 0x00000001,
		ActionGetDepends	 = 0x00000002,
		ActionGetDetails	 = 0x00000004,
		ActionGetFiles		 = 0x00000008,
		ActionGetPackages	 = 0x00000010,
		ActionGetRepoList	 = 0x00000020,
		ActionGetRequires	 = 0x00000040,
		ActionGetUpdateDetail	 = 0x00000080,
		ActionGetUpdates	 = 0x00000100,
		ActionInstallFiles	 = 0x00000200,
		ActionInstallPackages	 = 0x00000400,
		ActionInstallSignature	 = 0x00000800,
		ActionRefreshCache	 = 0x00001000,
		ActionRemovePackages	 = 0x00002000,
		ActionRepoEnable	 = 0x00004000,
		ActionRepoSetData	 = 0x00008000,
		ActionResolve		 = 0x00010000,
		ActionRollback		 = 0x00020000,
		ActionSearchDetails	 = 0x00040000,
		ActionSearchFile	 = 0x00080000,
		ActionSearchGroup	 = 0x00100000,
		ActionSearchName	 = 0x00200000,
		ActionUpdatePackages	 = 0x00400000,
		ActionUpdateSystem	 = 0x00800000,
		ActionWhatProvides	 = 0x01000000,
		ActionAcceptEula	 = 0x02000000,
		ActionDownloadPackages	 = 0x04000000,
		ActionGetDistroUpgrades	 = 0x08000000,
		ActionGetCategories	 = 0x10000000,
		ActionGetOldTransactions = 0x20000000,
		UnknownAction		 = 0x40000000
	} Action;
	Q_DECLARE_FLAGS(Actions, Action);

	/**
	 * Returns all the actions supported by the current backend
	 */
	Actions getActions();

	/**
	 * Holds a backend's detail
	 * \li \c name is the name of the backend
	 * \li \c author is the name of the person who wrote the backend
	 */
	typedef struct {
		QString name;
		QString author;
	} BackendDetail;

	/**
	 * Gets the current backend's details
	 * \return a BackendDetail struct holding the backend's details. You have to free this structure.
	 */
	BackendDetail getBackendDetail();

	/**
	 * Describes the different filters
	 */
	typedef enum {
		NoFilter		 = 0x0000001,
		FilterInstalled		 = 0x0000002,
		FilterNotInstalled	 = 0x0000004,
		FilterDevelopment	 = 0x0000008,
		FilterNotDevelopment	 = 0x0000010,
		FilterGui		 = 0x0000020,
		FilterNotGui		 = 0x0000040,
		FilterFree		 = 0x0000080,
		FilterNotFree		 = 0x0000100,
		FilterVisible		 = 0x0000200,
		FilterNotVisible	 = 0x0000400,
		FilterSupported		 = 0x0000800,
		FilterNotSupported	 = 0x0001000,
		FilterBasename		 = 0x0002000,
		FilterNotBasename	 = 0x0004000,
		FilterNewest		 = 0x0008000,
		FilterNotNewest		 = 0x0010000,
		FilterArch		 = 0x0020000,
		FilterNotArch		 = 0x0040000,
		FilterSource		 = 0x0080000,
		FilterNotSource		 = 0x0100000,
		FilterCollections	 = 0x0200000,
		FilterNotCollections	 = 0x0400000,
		FilterApplication	 = 0x0800000,
		FilterNotApplication	 = 0x1000000,
		UnknownFilter		 = 0x2000000
	} Filter;
	Q_DECLARE_FLAGS(Filters, Filter);

	/**
	 * Returns the filters supported by the current backend
	 */
	Filters getFilters();

	/**
	 * Describes the different groups
	 */
	typedef enum {
		GroupAccessibility,
		GroupAccessories,
		GroupAdminTools,
		GroupCommunication,
		GroupDesktopGnome,
		GroupDesktopKde,
		GroupDesktopOther,
		GroupDesktopXfce,
		GroupEducation,
		GroupFonts,
		GroupGames,
		GroupGraphics,
		GroupInternet,
		GroupLegacy,
		GroupLocalization,
		GroupMaps,
		GroupMultimedia,
		GroupNetwork,
		GroupOffice,
		GroupOther,
		GroupPowerManagement,
		GroupProgramming,
		GroupPublishing,
		GroupRepos,
		GroupSecurity,
		GroupServers,
		GroupSystem,
		GroupVirtualization,
		GroupScience,
		GroupDocumentation,
		GroupElectronics,
		GroupCollections,
		GroupVendor,
		GroupNewest,
		UnknownGroup
	} Group;
	typedef QSet<Group> Groups;

	/**
	 * Returns the groups supported by the current backend
	 */
	Groups getGroups();

	/**
	 * Returns a list containing the MIME types supported by the current backend
	 */
	QStringList getMimeTypes();

	/**
	 * Describes the current network state
	 */
	typedef enum {
		NetworkOffline,
		NetworkOnline,
		NetworkWired,
		NetworkWifi,
		NetworkMobile,
		UnknownNetworkState
	} NetworkState;

	/**
	 * Returns the current network state
	 */
	NetworkState getNetworkState();

	/**
	 * Returns the time (in seconds) since the specified \p action
	 */
	uint getTimeSinceAction(Action action);

	/**
	 * Returns the list of current transactions
	 */
	QList<Transaction*> getTransactions();

	/**
	 * Sets a global locale for all the transactions to be created
	 */
	void setLocale(const QString& locale);

	/**
	 * Sets a proxy to be used for all the network operations
	 */
	void setProxy(const QString& http_proxy, const QString& ftp_proxy);

	/**
	 * \brief Tells the daemon that the system state has changed, to make it reload its cache
	 *
	 * \p reason can be resume or posttrans
	 */
	void stateHasChanged(const QString& reason);

	/**
	 * Asks PackageKit to quit, for example to let a native package manager operate
	 */
	void suggestDaemonQuit();

	// Other enums
	/**
	 * Describes a signature type
	 */
	typedef enum {
		SignatureGpg,
		UnknownSignatureType
	} SignatureType;

	/**
	 * Describes a package signature
	 * \li \c package is a pointer to the signed package
	 * \li \c repoId is the id of the software repository containing the package
	 * \li \c keyUrl, \c keyId, \c keyFingerprint and \c keyTimestamp describe the key
	 * \li \c type is the signature type
	 */
	typedef struct {
		Package* package;
		QString repoId;
		QString keyUrl;
		QString keyUserid;
		QString keyId;
		QString keyFingerprint;
		QString keyTimestamp;
		SignatureType type;
	} SignatureInfo;

	/**
	 * Enum used to describe a "provides" request
	 * \sa whatProvides
	 */
	typedef enum {
		ProvidesAny,
		ProvidesModalias,
		ProvidesCodec,
		ProvidesMimetype,
		ProvidesFont,
		ProvidesHardwareDriver,
		UnknownProvidesType
	} ProvidesType;

	/**
	 * Lists the different types of error
	 */
	typedef enum {
		ErrorOom,
		ErrorNoNetwork,
		ErrorNotSupported,
		ErrorInternalError,
		ErrorGpgFailure,
		ErrorPackageIdInvalid,
		ErrorPackageNotInstalled,
		ErrorPackageNotFound,
		ErrorPackageAlreadyInstalled,
		ErrorPackageDownloadFailed,
		ErrorGroupNotFound,
		ErrorGroupListInvalid,
		ErrorDepResolutionFailed,
		ErrorFilterInvalid,
		ErrorCreateThreadFailed,
		ErrorTransactionError,
		ErrorTransactionCancelled,
		ErrorNoCache,
		ErrorRepoNotFound,
		ErrorCannotRemoveSystemPackage,
		ErrorProcessKill,
		ErrorFailedInitialization,
		ErrorFailedFinalise,
		ErrorFailedConfigParsing,
		ErrorCannotCancel,
		ErrorCannotGetLock,
		ErrorNoPackagesToUpdate,
		ErrorCannotWriteRepoConfig,
		ErrorLocalInstallFailed,
		ErrorBadGpgSignature,
		ErrorMissingGpgSignature,
		ErrorCannotInstallSourcePackage,
		ErrorRepoConfigurationError,
		ErrorNoLicenseAgreement,
		ErrorFileConflicts,
		ErrorPackageConflicts,
		ErrorRepoNotAvailable,
		ErrorInvalidPackageFile,
		ErrorPackageInstallBlocked,
		ErrorPackageCorrupt,
		ErrorAllPackagesAlreadyInstalled,
		ErrorFileNotFound,
		ErrorNoMoreMirrorsToTry,
		ErrorNoDistroUpgradeData,
		ErrorIncompatibleArchitecture,
		ErrorNoSpaceOnDevice,
		ErrorMediaChangeRequired,
		UnknownErrorType
	} ErrorType;

	/**
	 * Describes a message's type
	 */
	typedef enum {
		MessageBrokenMirror,
		MessageConnectionRefused,
		MessageParameterInvalid,
		MessagePriorityInvalid,
		MessageBackendError,
		MessageDaemonError,
		MessageCacheBeingRebuilt,
		MessageUntrustedPackage,
		MessageNewerPackageExists,
		MessageCouldNotFindPackage,
		MessageConfigFilesChanged,
		MessagePackageAlreadyInstalled,
		UnknownMessageType
	} MessageType;

	/**
	 * Describes an EULA
	 * \li \c id is the EULA identifier
	 * \li \c package is the package for which an EULA is required
	 * \li \c vendorName is the vendor name
	 * \li \c licenseAgreement is the EULA text
	 */
	typedef struct {
		QString id;
		Package* package;
		QString vendorName;
		QString licenseAgreement;
	} EulaInfo;

	/**
	 * Describes a restart type
	 */
	typedef enum {
		RestartNone,
		RestartApplication,
		RestartSession,
		RestartSystem,
		UnknownRestartType
	} RestartType;

	/**
	 * Describes an update's state
	 */
	typedef enum {
		UpdateStable,
		UpdateUnstable,
		UpdateTesting,
		UnknownUpdateState
	} UpdateState;

	/**
	 * Describes an distro upgrade state
	 */
	typedef enum {
		DistroUpgradeStable,
		DistroUpgradeUnstable,
		UnknownDistroUpgrade
	} DistroUpgradeType;

	/**
	 * Describes an error at the daemon level (for example, PackageKit crashes or is unreachable)
	 */
	typedef enum {
		DaemonUnreachable,
		UnkownDaemonError
	} DaemonError;
	/**
	 * Returns the last daemon error that was caught
	 */
	DaemonError getLastError();

	/**
	 * Describes a software update
	 * \li \c package is the package which triggered the update
	 * \li \c updates are the packages to be updated
	 * \li \c obsoletes lists the packages which will be obsoleted by this update
	 * \li \c vendorUrl, \c bugzillaUrl and \c cveUrl are links to webpages describing the update
	 * \li \c restart indicates if a restart will be required after this update
	 * \li \c updateText describes the update
	 * \li \c changelog holds the changelog
	 * \li \c state is the category of the update, eg. stable or testing
	 * \li \c issued and \c updated indicate the dates at which the update was issued and updated
	 */
	typedef struct {
		Package* package;
		QList<Package*> updates;
		QList<Package*> obsoletes;
		QString vendorUrl;
		QString bugzillaUrl;
		QString cveUrl;
		RestartType restart;
		QString updateText;
		QString changelog;
		UpdateState state;
		QDateTime issued;
		QDateTime updated;
	} UpdateInfo;

	// Transaction functions

	/**
	 * \brief Accepts an EULA
	 *
	 * The EULA is identified by the EulaInfo structure \p info
	 *
	 * \note You need to restart the transaction which triggered the EULA manually
	 *
	 * \sa Transaction::eulaRequired
	 */
	Transaction* acceptEula(EulaInfo info);

	/**
	 * Download the given \p packages to a temp dir
	 */
	Transaction* downloadPackages(const QList<Package*>& packages);

	/**
	 * This is a convenience function
	 */
	Transaction* downloadPackage(Package* package);

	/**
	 * Returns the collection categories
	 *
	 * \sa Transaction::category
	 */
	Transaction* getCategories();

	/**
	 * \brief Gets the list of dependencies for the given \p packages
	 *
	 * You can use the \p filters to limit the results to certain packages. The
	 * \p recursive flag indicates if the package manager should also fetch the
	 * dependencies's dependencies.
	 *
	 * \sa Transaction::package
	 *
	 */
	Transaction* getDepends(const QList<Package*>& packages, Filters filters = NoFilter, bool recursive = true);
	Transaction* getDepends(Package* package, Filters filters = NoFilter, bool recursive = true);

	/**
	 * Gets more details about the given \p packages
	 *
	 * \sa Transaction::details
	 */
	Transaction* getDetails(const QList<Package*>& packages);
	Transaction* getDetails(Package* package);

	/**
	 * Gets the files contained in the given \p packages
	 *
	 * \sa Transaction::files
	 */
	Transaction* getFiles(const QList<Package*>& packages);
	Transaction* getFiles(Package* packages);

	/**
	 * \brief Gets the last \p number finished transactions
	 *
	 * \note You must delete these transactions yourself
	 */
	Transaction* getOldTransactions(uint number);

	/**
	 * Gets all the packages matching the given \p filters
	 *
	 * \sa Transaction::package
	 */
	Transaction* getPackages(Filters filters = NoFilter);

	/**
	 * Gets the list of software repositories matching the given \p filters
	 */
	Transaction* getRepoList(Filters filter = NoFilter);

	/**
	 * \brief Searches for the packages requiring the given \p packages
	 *
	 * The search can be limited using the \p filters parameter. The recursive flag is used to tell
	 * if the package manager should also search for the package requiring the resulting packages.
	 */
	Transaction* getRequires(const QList<Package*>& packages, Filters filters = NoFilter, bool recursive = true);
	Transaction* getRequires(Package* package, Filters filters = NoFilter, bool recursive = true);

	/**
	 * Retrieves more details about the update for the given \p packages
	 */
	Transaction* getUpdateDetail(const QList<Package*>& packages);
	Transaction* getUpdateDetail(Package* package);

	/**
	 * \p Gets the available updates
	 *
	 * The \p filters parameters can be used to restrict the updates returned
	 */
	Transaction* getUpdates(Filters filters = NoFilter);

	/**
	 * Retrieves the available distribution upgrades
	 */
	Transaction* getDistroUpgrades();

	/**
	 * \brief Installs the local packages \p files
	 *
	 * \p trusted indicate if the packages are signed by a trusted authority
	 */
	Transaction* installFiles(const QStringList& files, bool trusted);
	Transaction* installFile(const QString& file, bool trusted);

	/**
	 * Install the given \p packages
	 */
	Transaction* installPackages(const QList<Package*>& packages);
	Transaction* installPackage(Package* p);

	/**
	 * \brief Installs a signature
	 *
	 * \p type, \p key_id and \p p generally come from the Transaction::repoSignatureRequired
	 */
	Transaction* installSignature(SignatureType type, const QString& key_id, Package* p);

	/**
	 * Refreshes the package manager's cache
	 */
	Transaction* refreshCache(bool force);

	/**
	 * \brief Removes the given \p packages
	 *
	 * \p allow_deps if the package manager has the right to remove other packages which depend on the
	 * pacakges to be removed. \p autoremove tells the package manager to remove all the package which
	 * won't be needed anymore after the packages are uninstalled.
	 */
	Transaction* removePackages(const QList<Package*>& packages, bool allow_deps = false, bool autoremove = false);
	Transaction* removePackage(Package* p, bool allow_deps = false, bool autoremove = false);

	/**
	 * Activates or disables a repository
	 */
	Transaction* repoEnable(const QString& repo_id, bool enable);

	/**
	 * Sets a repository's parameter
	 */
	Transaction* repoSetData(const QString& repo_id, const QString& parameter, const QString& value);

	/**
	 * \brief Tries to create a Package object from the package's name
	 *
	 * The \p filters can be used to restrict the search
	 */
	Transaction* resolve(const QStringList& packageNames, Filters filters = NoFilter);
	Transaction* resolve(const QString& packageName, Filters filters = NoFilter);

	/**
	 * Rolls back the given \p transactions
	 */
	Transaction* rollback(Transaction* oldtrans);

	/**
	 * \brief Search in the packages files
	 *
	 * \p filters can be used to restrict the returned packages
	 */
	Transaction* searchFile(const QString& search, Filters filters = NoFilter);

	/**
	 * \brief Search in the packages details
	 *
	 * \p filters can be used to restrict the returned packages
	 */
	Transaction* searchDetails(const QString& search, Filters filters = NoFilter);

	/**
	 * \brief Lists all the packages in the given \p group
	 *
	 * \p filters can be used to restrict the returned packages
	 */
	Transaction* searchGroup(Client::Group group, Filters filters = NoFilter);

	/**
	 * \brief Search in the packages names
	 *
	 * \p filters can be used to restrict the returned packages
	 */
	Transaction* searchName(const QString& search, Filters filters = NoFilter);

	/**
	 * \brief Tries to find a package name from a desktop file
	 *
	 * This function looks into /var/lib/PackageKit/desktop-files.db and searches for the associated package name.
	 *
	 * \p path the path to the desktop file (as shipped by the package)
	 * \return The associated package, or NULL if there's no result
	 */
	Package* searchFromDesktopFile(const QString& path);

	/**
	 * Update the given \p packages
	 */
	Transaction* updatePackages(const QList<Package*>& packages);
	Transaction* updatePackage(Package* package);

	/**
	 * Updates the whole system
	 */
	Transaction* updateSystem();

	/**
	 * Searchs for a package providing a file/a mimetype
	 */
	Transaction* whatProvides(ProvidesType type, const QString& search, Filters filters = NoFilter);

Q_SIGNALS:
	/**
	 * \brief Emitted when PolicyKit doesn't grant the necessary permissions to the user
	 *
	 * \p action is the PolicyKit name of the action
	 */
	void authError(const QString& action);

	/**
	 * Emitted when the PackageKit daemon is not reachable anymore
	 */
	void daemonError(PackageKit::Client::DaemonError e);

	/**
	 * Emitted when the daemon's locked state changes
	 */
	void locked(bool locked);

	/**
	 * Emitted when the network state changes
	 */
	void networkStateChanged(PackageKit::Client::NetworkState state);

	/**
	 * Emitted when the list of repositories changes
	 */
	void repoListChanged();

	/**
	 * Emmitted when a restart is scheduled
	 */
	void restartScheduled();

	/**
	 * \brief Emitted when the current transactions list changes.
	 *
	 * \note This is mostly useful for monitoring the daemon's state.
	 */
	void transactionListChanged(const QList<PackageKit::Transaction*>&);

	/**
	 * Emitted when new updates are available
	 */
	void updatesChanged();

private:
	Client(QObject* parent = 0);
	static Client* m_instance;
	friend class ClientPrivate;
	ClientPrivate* d;
};
Q_DECLARE_OPERATORS_FOR_FLAGS(Client::Actions)
Q_DECLARE_OPERATORS_FOR_FLAGS(Client::Filters)

} // End namespace PackageKit

#endif
