#include "policy.h"

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <mntent.h>
#include <stdlib.h>
#include <errno.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sysfs/libsysfs.h>

/*************************************************************************
 *
 * Helper functions
 *
 *************************************************************************/

/**
 * Check whether a fstab-type file (fstab, /etc/mtab or /proc/mounts)
 * contains a mount point. Exits the program if file could not be opened.
 * @param fname file of the fstab-type file to check
 * @param mntpt mount point directory to scan for
 * @return 1: fname has mount point, 0 otherwise
 */
int
fstab_has_mntpt( const char* fname, const char* mntpt )
{
    FILE* f;
    struct mntent *entry;

    if( !( f = fopen( fname, "r" ) ) ) {
	perror( "could not open fstab-type file" );
	exit( -50 );
    }

    while( ( entry = getmntent( f ) ) != NULL ) {
	if( !strcmp( entry->mnt_dir, mntpt ) ) {
	    endmntent( f );
	    return 1;
	}
    }

    endmntent( f );
    return 0;
}

/**
 * Check whether a bus occurs anywhere in the ancestry of a device.
 * @param dev sysfs device
 * @param buses NULL-terminated array of bus names to scan for
 * @return 0 if not found, 1 if found
 */
int
find_bus_ancestry( struct sysfs_device* dev, char** buses ) {
    char **i;

    if( !dev || !buses )
	return 0;

    for( i = buses; *i; ++i ) {
	if( !strcmp( dev->bus, *i ) )
	    return 1;
    }

    return find_bus_ancestry( sysfs_get_device_parent( dev ), buses );
}

/**
 * Return major device number of given device. Exit if device does not exist.
 */
unsigned char
get_major( const char* dev )
{
    struct stat devstat;
    
    if( stat( dev, &devstat ) ) {
	perror( "could not stat device node" );
	exit( -1 );
    }

    return devstat.st_rdev >> 8;
}

/**
 * Read the first number (separated by colon) from given file and return it;
 * allowed range is 0 to 255.
 * @return -1 on error, otherwise the read number
 */
int
read_number( const char* file )
{
    FILE* f;
    char buf[100];
    char* colon;
    char* endptr;
    int bufsize;
    long int number;
    
    /* read a chunk from the file that is big enough to hold any number */
    f = fopen( file, "r" );
    if( !f )
	return -1;

    bufsize = fread( buf, 1, sizeof(buf)-1, f );
    fclose( f );
    buf[bufsize] = 0;

    /* find ':' and cut string before it */
    if( ( colon = strchr( buf, ':' ) ) == NULL )
	return -1;
    *colon = 0;

    /* convert to a number */
    errno = 0;
    number = strtol( buf, &endptr, 10 );

    if( errno || *endptr || number < 0 || number > 255 )
	return -1;

    return number;
}

/**
 * Find sysfs node that matches the major device number of the given device.
 */
struct sysfs_device*
find_sysfs_device( const char* dev ) {
    unsigned char major = get_major( dev );
    char mntpath[PATH_MAX];
    char blockdirname[255];
    char devdirname[512]; // < 255 chars blockdir + max. 255 chars subdir
    char devfilename[520]; // < 255 chars blockdir + max. 255 chars subdir + '/dev'
    char linkfilename[1024];
    DIR* bldir;
    struct dirent *devdir;
    struct sysfs_device *sysdev = NULL;
    int number;

    /* get /sys/block/ */
    if( sysfs_get_mnt_path( mntpath, sizeof(mntpath) ) ) {
	fputs( "Error: could not get sysfs directory\n", stderr );
	exit( -1 );
    }
    snprintf( blockdirname, sizeof( blockdirname ), "%s/block/", mntpath );

    bldir = opendir( blockdirname );
    if( !bldir ) {
	perror( "Error: could not open <sysfs dir>/block/ " );
	exit( -1 );
    }

    /* open each subdirectory and see whether major device matches */
    while( ( devdir = readdir( bldir ) ) != NULL ) {
	/* construct /sys/block/<device> */
        snprintf( devdirname, sizeof( devdirname ), "%s%s", blockdirname, devdir->d_name );

	/* construct /sys/block/<device>/dev */
        snprintf( devfilename, sizeof( devfilename ), "%s%s", devdirname, "/dev" );

	number = read_number( devfilename );
	if( number >= 0 && major == (unsigned char) number ) {
            snprintf( devfilename, sizeof( devfilename ), "%s/device", devdirname );

	    /* read out the link */
	    if( !sysfs_get_link( devfilename, linkfilename, 1024 ) )
		sysdev = sysfs_open_device_path( linkfilename );
	    break;
	}
    }

    closedir( bldir );

    return sysdev;
}

/*************************************************************************
 *
 * Policy functions
 *
 *************************************************************************/

int
device_valid( const char* device )
{
    struct stat st;

    if( stat( device, &st ) ) {
	fprintf( stderr, "Error: device %s does not exist\n", device );
	return 0;
    }

    if( !S_ISBLK( st.st_mode ) ) {
	fprintf( stderr, "Error: %s is not a block device\n", device );
	return 0;
    }

    return 1;
}

int
fstab_has_device( const char* fname, const char* device, char* mntpt, int *uid )
{
    FILE* f;
    struct mntent *entry;
    char* uidopt;

    if( !( f = fopen( fname, "r" ) ) ) {
	perror( "could not open fstab-type file" );
	exit( -50 );
    }

    while( ( entry = getmntent( f ) ) != NULL ) {
	if( !strcmp( entry->mnt_fsname, device ) ) {
		endmntent( f );
		if( mntpt ) {
		    snprintf( mntpt, MEDIA_STRING_SIZE-1, "%s", entry->mnt_dir );
		}
		if( uid ) {
		    uidopt = hasmntopt( entry, "uid" );
		    if( uidopt )
			uidopt = strchr( uidopt, '=' );
		    if( uidopt ) {
			++uidopt; /* skip the '=' */
			/* FIXME: this probably needs more checking */
			*uid = atoi( uidopt );
		    } else
			*uid = -1;
		}
		return 1;
	}
    }

    /* just for safety */
    if( mntpt )
	*mntpt = 0; 

    endmntent( f );
    return 0;
}

int
device_mounted( const char* device, int expect, char* mntpt )
{
    char mp[MEDIA_STRING_SIZE];
    int uid;
    int mounted = fstab_has_device( "/etc/mtab", device, mp, &uid ) ||
		  fstab_has_device( "/proc/mounts", device, mp, &uid );
    if( mounted && !expect )
	fprintf( stderr, "Error: device %s is already mounted to %s\n", device, mp );
    else if( !mounted && expect )
	fprintf( stderr, "Error: device %s is not mounted\n", device );
    if( mounted && expect && uid > 0 && (uid_t) uid != getuid() ) {
	fprintf( stderr, "Error: device %s was not mounted by you\n", device );
	return 0;
    }

    if( mntpt ) {
        snprintf( mntpt, MEDIA_STRING_SIZE-1, "%s", mp);
    }

    return mounted;
}

int
device_removable( const char* device )
{
    struct sysfs_device *dev;
    static char* hotplug_buses[] = { "usb", "ieee1394", NULL };
    int removable;

    dev = find_sysfs_device( device );
    if( !dev )
	return 0; 

    removable = find_bus_ancestry( dev, hotplug_buses );
    sysfs_close_device( dev );

    if( !removable )
	fprintf( stderr, "Error: device %s is not removable\n", device );
    
    return removable;
}

int
mntpt_valid( const char* mntpt ) 
{
    struct stat st;
    int result;
    DIR* dir;
    struct dirent* dirent;

    /* we need root for both creating and checking the mount point dir */
    if( setreuid( -1, 0 ) ) {
	perror( "Internal error: mntpt_valid: could not change effective uid" );
	return 0;
    }

    if( stat( mntpt, &st ) ) {
	/* does not exist, create */
	result = mkdir( mntpt, 0750 );
	setreuid( -1, getuid() );
	if( result ) {
	    perror( "Error: could not create mount point" );
	    return 0;
	}
    } else {
	/* exists, check that it is an empty directory */
	if( !S_ISDIR( st.st_mode ) ) {
	    fprintf( stderr, "Error: mount point %s is not a directory\n", mntpt );
	    return 0;
	}

	dir = opendir( mntpt );
	setreuid( -1, getuid() );
	if( !dir ) {
	    perror( "Error: could not open mount point directory" );
	    return 0;
	}

        while ( ( dirent = readdir( dir ) ) ) {
	    if( strcmp( dirent->d_name, "." ) && strcmp( dirent->d_name, ".." ) ) {
		closedir( dir );
		fprintf( stderr, "Error: existing mount point directory %s is not empty\n", mntpt );
		return 0;
	    }
	}

	closedir( dir );
    }

    /* ensure moint point has tight permissions */
    if( setreuid( -1, 0 ) ) {
	perror( "Internal error: mntpt_valid: could not change effective uid" );
	return 0;
    }
    result = chown( mntpt, 0, 0 );
    setreuid( -1, getuid() );
    if( result ) {
	perror( "Error: could not set owner and group of mount point" );
	return 0;
    }
    return 1;
}

int
mntpt_mounted( const char* mntpt, int expect )
{
    int mounted = fstab_has_mntpt( "/etc/mtab", mntpt ) ||
	          fstab_has_mntpt( "/proc/mounts", mntpt );

    if( mounted && !expect )
	fprintf( stderr, "Error: directory %s already contains a mounted file system\n", mntpt );
    else if( !mounted && expect )
	fprintf( stderr, "Error: directory %s does not contain a mounted file system\n", mntpt );

    return mounted;
}

