/*
 * access to SPU privileged registers
 */
#include <linux/module.h>

#include <asm/io.h>
#include <asm/spu.h>
#include <asm/prom.h>

#include "interrupt.h"


/*  platform private data  */

struct spu_priv_data {
	u32 isrc;
};


/*  spu sysdev nodes  */

static ssize_t spu_show_isrc(struct sys_device *sysdev, char *buf)
{
	struct spu *spu = container_of(sysdev, struct spu, sysdev);
	return sprintf(buf, "%d", spu->priv_data->isrc);

}
static SYSDEV_ATTR(isrc, 0400, spu_show_isrc, NULL);

#if 0 /* FIXME: nid patch is not applied */
static ssize_t spu_show_numa_id(struct sys_device *sysdev, char *buf)
{
	struct spu *spu = container_of(sysdev, struct spu, sysdev);
	return sprintf(buf, "%d", spu->priv_data->nid);

}
static SYSDEV_ATTR(numa_id, 0400, spu_show_numa_id, NULL);
#endif

void spu_create_sysdev_files(struct spu *spu)
{
	sysdev_create_file(&spu->sysdev, &attr_isrc);
#if 0 /* FIXME: nid patch is not applied */
	sysdev_create_file(&spu->sysdev, &attr_numa_id);
#endif
}

void spu_destroy_sysdev_files(struct spu *spu)
{
	sysdev_remove_file(&spu->sysdev, &attr_isrc);
#if 0 /* FIXME: nid patch is not applied */
	sysdev_remove_file(&spu->sysdev, &attr_numa_id);
#endif
}


/*  spu setup funcs  */

static int __init find_spu_node_id(struct device_node *spe)
{
	unsigned int *id;
	struct device_node *cpu;

	cpu = spe->parent->parent;
	id = (unsigned int *)get_property(cpu, "node-id", NULL);

	return id ? *id : 0;
}

static void __init get_spe_property(struct device_node *n, const char *name,
	struct spu_phys* phys)
{
	struct address_prop {
		unsigned long address;
		unsigned int len;
	} __attribute__((packed)) *prop;

	void *p;
	int proplen;

	p = get_property(n, name, &proplen);
	if (proplen != sizeof (struct address_prop)) {
		phys->addr = phys->size = 0;
		return;
	}
	prop = p;

	phys->addr = prop->address;
	phys->size = prop->len;
}

int __init enum_and_create_spu(void)
{
	struct device_node *node;
	int ret;

	ret = -ENODEV;
	for (node = of_find_node_by_type(NULL, "spe");
			node; node = of_find_node_by_type(node, "spe")) {
		ret = create_spu(node);
		if (ret)
			break;
	}
	/* in some old firmware versions, the spe is called 'spc', so we
	   look for that as well */
	for (node = of_find_node_by_type(NULL, "spc");
			node; node = of_find_node_by_type(node, "spc")) {
		ret = create_spu(node);
		if (ret)
			break;
	}
	return ret;
}
EXPORT_SYMBOL_GPL(enum_and_create_spu);

int __init spu_setup(struct spu *spu, void *spu_setup_data)
{
	struct device_node *spe = spu_setup_data;
	char *prop;

	spu->priv_data = kmalloc(sizeof(struct spu_priv_data), GFP_KERNEL);
	if (!spu->priv_data)
		return -ENOMEM;

	spu->node = find_spu_node_id(spe);

	prop = get_property(spe, "isrc", NULL);
	if (!prop)
		goto out_free;
	spu->priv_data->isrc = *(unsigned int *)prop;

	spu->name = get_property(spe, "name", NULL);
	if (!spu->name)
		goto out_free;

	get_spe_property(spe, "local-store", &spu->local_store_phys);
	if (!spu->local_store_phys.addr)
		goto out_free;

	get_spe_property(spe, "problem", &spu->problem_phys);
	if (!spu->problem_phys.addr)
		goto out_free;

	get_spe_property(spe, "priv1", &spu->priv1_phys);

	get_spe_property(spe, "priv2", &spu->priv2_phys);
	if (!spu->priv2_phys.addr)
		goto out_free;

	return 0;

out_free:
	printk(KERN_WARNING "%s: no spu device found\n", __func__);
	pr_debug("%s: error (%p)\n", __func__, spu);
	kfree(spu->priv_data);
	spu->priv_data = NULL;
	return -ENODEV;
}
EXPORT_SYMBOL_GPL(spu_setup);

void spu_free_priv_data(struct spu *spu)
{
	if (spu->priv_data) {
		kfree(spu->priv_data);
		spu->priv_data = NULL;
	}
}
EXPORT_SYMBOL_GPL(spu_free_priv_data);

int spu_request_irqs(struct spu *spu,
	irqreturn_t (*cls0)(int, void *, struct pt_regs *),
	irqreturn_t (*cls1)(int, void *, struct pt_regs *),
	irqreturn_t (*cls2)(int, void *, struct pt_regs *))
{
	int ret;
	int irq_base;

	BUG_ON(spu->priv_data);

	if(!spu->priv_data) {
		spu->priv_data = kmalloc(sizeof(struct spu_priv_data),
			GFP_KERNEL);

		if(!spu->priv_data) {
			ret = -ENOMEM;
			goto out;
		}
	}

	irq_base = IIC_NODE_STRIDE * spu->node + IIC_SPE_OFFSET;

	snprintf(spu->irq_c0, sizeof (spu->irq_c0), "spe%02d.0", spu->number);
	ret = request_irq(irq_base + spu->priv_data->isrc,
		 cls0, SA_INTERRUPT, spu->irq_c0, spu);
	if (ret)
		goto out;

	snprintf(spu->irq_c1, sizeof (spu->irq_c1), "spe%02d.1", spu->number);
	ret = request_irq(irq_base + IIC_CLASS_STRIDE + spu->priv_data->isrc,
		 cls1, SA_INTERRUPT, spu->irq_c1, spu);
	if (ret)
		goto out1;

	snprintf(spu->irq_c2, sizeof (spu->irq_c2), "spe%02d.2", spu->number);
	ret = request_irq(irq_base + 2*IIC_CLASS_STRIDE + spu->priv_data->isrc,
		 cls2, SA_INTERRUPT, spu->irq_c2, spu);
	if (ret)
		goto out2;
	goto out;

out2:
	free_irq(irq_base + IIC_CLASS_STRIDE + spu->priv_data->isrc, spu);
out1:
	free_irq(irq_base + spu->priv_data->isrc, spu);
out:
	return ret;
}
EXPORT_SYMBOL_GPL(spu_request_irqs);

void spu_free_irqs(struct spu *spu)
{
	int irq_base;

	irq_base = IIC_NODE_STRIDE * spu->node + IIC_SPE_OFFSET;

	free_irq(irq_base + spu->priv_data->isrc, spu);
	free_irq(irq_base + IIC_CLASS_STRIDE + spu->priv_data->isrc, spu);
	free_irq(irq_base + 2*IIC_CLASS_STRIDE + spu->priv_data->isrc, spu);

	kfree(spu->priv_data);
	spu->priv_data = NULL;
}
EXPORT_SYMBOL_GPL(spu_free_irqs);


/*  spu priv1 access funcs.  */

void spu_int_mask_and(struct spu *spu, int class, u64 mask)
{
	u64 old_mask;

	old_mask = in_be64(&spu->priv1->int_mask_RW[class]);
	out_be64(&spu->priv1->int_mask_RW[class], old_mask & mask);
}
EXPORT_SYMBOL_GPL(spu_int_mask_and);

void spu_int_mask_or(struct spu *spu, int class, u64 mask)
{
	u64 old_mask;

	old_mask = in_be64(&spu->priv1->int_mask_RW[class]);
	out_be64(&spu->priv1->int_mask_RW[class], old_mask | mask);
}
EXPORT_SYMBOL_GPL(spu_int_mask_or);

void spu_int_mask_set(struct spu *spu, int class, u64 mask)
{
	out_be64(&spu->priv1->int_mask_RW[class], mask);
}
EXPORT_SYMBOL_GPL(spu_int_mask_set);

u64 spu_int_mask_get(struct spu *spu, int class)
{
	return in_be64(&spu->priv1->int_mask_RW[class]);
}
EXPORT_SYMBOL_GPL(spu_int_mask_get);

void spu_int_stat_clear(struct spu *spu, int class, u64 stat)
{
	out_be64(&spu->priv1->int_stat_RW[class], stat);
}
EXPORT_SYMBOL_GPL(spu_int_stat_clear);

u64 spu_int_stat_get(struct spu *spu, int class)
{
	return in_be64(&spu->priv1->int_stat_RW[class]);
}
EXPORT_SYMBOL_GPL(spu_int_stat_get);

void spu_cpu_affinity_set(struct spu *spu, int cpu)
{
	u64 target = iic_get_target_id(cpu);
	u64 route = target << 48 | target << 32 | target << 16;

	out_be64(&spu->priv1->int_route_RW, route);
}
EXPORT_SYMBOL_GPL(spu_cpu_affinity_set);

u64 spu_mfc_dar_get(struct spu *spu)
{
	return in_be64(&spu->priv1->mfc_dar_RW);
}
EXPORT_SYMBOL_GPL(spu_mfc_dar_get);

u64 spu_mfc_dsisr_get(struct spu *spu)
{
	return in_be64(&spu->priv1->mfc_dsisr_RW);
}
EXPORT_SYMBOL_GPL(spu_mfc_dsisr_get);

void spu_mfc_dsisr_set(struct spu *spu, u64 dsisr)
{
	out_be64(&spu->priv1->mfc_dsisr_RW, dsisr);
}
EXPORT_SYMBOL_GPL(spu_mfc_dsisr_set);

void spu_mfc_sdr_setup(struct spu *spu)
{
	out_be64(&spu->priv1->mfc_sdr_RW, mfspr(SPRN_SDR1));
}
EXPORT_SYMBOL_GPL(spu_mfc_sdr_setup);

void spu_mfc_sr1_set(struct spu *spu, u64 sr1)
{
	out_be64(&spu->priv1->mfc_sr1_RW, sr1);
}
EXPORT_SYMBOL_GPL(spu_mfc_sr1_set);

u64 spu_mfc_sr1_get(struct spu *spu)
{
	return in_be64(&spu->priv1->mfc_sr1_RW);
}
EXPORT_SYMBOL_GPL(spu_mfc_sr1_get);

void spu_mfc_tclass_id_set(struct spu *spu, u64 tclass_id)
{
	out_be64(&spu->priv1->mfc_tclass_id_RW, tclass_id);
}
EXPORT_SYMBOL_GPL(spu_mfc_tclass_id_set);

u64 spu_mfc_tclass_id_get(struct spu *spu)
{
	return in_be64(&spu->priv1->mfc_tclass_id_RW);
}
EXPORT_SYMBOL_GPL(spu_mfc_tclass_id_get);

void spu_tlb_invalidate(struct spu *spu)
{
	out_be64(&spu->priv1->tlb_invalidate_entry_W, 0ul);
}
EXPORT_SYMBOL_GPL(spu_tlb_invalidate);

void spu_resource_allocation_groupID_set(struct spu *spu, u64 id)
{
	out_be64(&spu->priv1->resource_allocation_groupID_RW, id);
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_groupID_set);

u64 spu_resource_allocation_groupID_get(struct spu *spu)
{
	return in_be64(&spu->priv1->resource_allocation_groupID_RW);
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_groupID_get);

void spu_resource_allocation_enable_set(struct spu *spu, u64 enable)
{
	out_be64(&spu->priv1->resource_allocation_enable_RW, enable);
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_enable_set);

u64 spu_resource_allocation_enable_get(struct spu *spu)
{
	return in_be64(&spu->priv1->resource_allocation_enable_RW);
}
EXPORT_SYMBOL_GPL(spu_resource_allocation_enable_get);
