/*
 * irq.c
 *
 * PS3PF specific IRQ routines.
 *
 * Author: Frank Rowand frowand@mvista.com or source@mvista.com
 *
 * 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/kernel.h>
#include <linux/threads.h>
#include <linux/pci.h>
#include <asm/lv1call.h>
#include <linux/config.h>

#include "spu_priv.h"

static spinlock_t ps3pf_io_irq_lock = SPIN_LOCK_UNLOCKED;
static unsigned char ps3pf_io_irq_connected[LV1_PLUG_ID_LIMIT];
unsigned long ps3pf_io_irq_outlet_id[LV1_RESERVED_PLUG_ID_LIMIT];

static spinlock_t lv1_irq_plug_lock = SPIN_LOCK_UNLOCKED;

/* ps3pf_plug_allocated[] could have been implemented as a bitmap, but the */
/* extra code complexity isn't worth it.                                 */
static unsigned char ps3pf_plug_allocated[LV1_PLUG_ID_LIMIT];
unsigned long ps3pf_max_plug_id = 0;

int
ps3pf_reserve_irq_plug(unsigned long plug_id)
{

	if (LV1_RESERVED_PLUG_ID_LIMIT <= plug_id)
		return -EINVAL;

	spin_lock(&lv1_irq_plug_lock);

	if (ps3pf_plug_allocated[plug_id] != 0) {
		spin_unlock(&lv1_irq_plug_lock);
		return -EBUSY;
	}

	ps3pf_plug_allocated[plug_id] = 1;

	if (plug_id > ps3pf_max_plug_id)
		ps3pf_max_plug_id = plug_id;

	spin_unlock(&lv1_irq_plug_lock);

	return 0;
}

unsigned long
ps3pf_allocate_irq_plug(void)
{
	unsigned long		new_plug_id;
	unsigned long		tmp_plug_id;

	new_plug_id = 0;
	spin_lock(&lv1_irq_plug_lock);

	for (tmp_plug_id = LV1_RESERVED_PLUG_ID_LIMIT;
	     tmp_plug_id < LV1_PLUG_ID_LIMIT;
	     tmp_plug_id++) {
		if (ps3pf_plug_allocated[tmp_plug_id] == 0) {
			ps3pf_plug_allocated[tmp_plug_id] = 1;
			new_plug_id = tmp_plug_id;
			break;
		}
	}

	if (new_plug_id > ps3pf_max_plug_id)
		ps3pf_max_plug_id = new_plug_id;

	spin_unlock(&lv1_irq_plug_lock);

	return new_plug_id;
}

static void
ps3pf_free_irq_plug(unsigned long plug_id)
{
	int			k;
	unsigned long		new_ps3pf_max_plug_id;

	spin_lock(&lv1_irq_plug_lock);

	ps3pf_plug_allocated[plug_id] = 0;

	new_ps3pf_max_plug_id = 0;
	for (k = 0; k < ps3pf_max_plug_id; k++) {
		if (ps3pf_plug_allocated[k])
			new_ps3pf_max_plug_id = k;
	}
	ps3pf_max_plug_id = new_ps3pf_max_plug_id;

	spin_unlock(&lv1_irq_plug_lock);
}

int
__ps3pf_free_irq(
	unsigned long outlet_id,
	unsigned int irq_no,
	unsigned int irq_cpu,
	void *dev_id)
{
	unsigned long pu_id;
	int ret;

	lv1_get_logical_pu_id(&pu_id);
	ret = lv1_disconnect_irq_plug(pu_id, irq_cpu, irq_no);

	if (ret)
		ret = -EPERM;

	free_irq(irq_no, dev_id);

	ps3pf_free_irq_plug(irq_no);

	return ret;
}
EXPORT_SYMBOL(__ps3pf_free_irq);

/*
 * You must not call this function with disabled interrupts or from a
 * interrupt handler or from a bottom half handler.
 * (This restriction is the smp_call_funtion_on_cpu() restriction.)
 */
int
ps3pf_free_irq(
	unsigned long outlet_id,
	unsigned int irq_no,
	unsigned int irq_cpu,
	void *dev_id)
{
	int			ret;
	unsigned long		status;

	ret = __ps3pf_free_irq(outlet_id, irq_no, irq_cpu, dev_id);

	status = lv1_destruct_event_receive_port(outlet_id);
	if (status) {
		ret = -EPERM;
		printk(KERN_ERR "%s(): lv1_destruct_event_receive_port() "
		       "failed: %ld\n", __FUNCTION__, status);
	}

	return ret;
}
EXPORT_SYMBOL(ps3pf_free_irq);

int __ps3pf_connect_irq(

	unsigned long	outlet_id,

	unsigned int	*irq_no,
	unsigned int	*irq_cpu,

	/* arguments to request_irq() */
	irqreturn_t     (*handler)(int, void *, struct pt_regs *),
	unsigned long   flags,
	const char      *irq_name,
	void            *dev_id
	)
{
	int				ret;
	long				status;

	*irq_no = ps3pf_allocate_irq_plug();
	if (unlikely(*irq_no == 0)) {
		printk(KERN_ERR "%s(): ps3pf_allocate_irq_plug() failed\n",
		      __FUNCTION__);
		return -ENOMEM;
	}

	ret = request_irq(*irq_no, handler,
			  flags, irq_name,
			  dev_id);
	if (ret) {
		printk(KERN_ERR "%s(): request_irq() failed: %d\n",
			__FUNCTION__, ret);
		return ret;
	}

	preempt_disable();

	*irq_cpu = smp_processor_id();
	status = lv1_construct_and_connect_irq_plug(*irq_no, outlet_id);

	preempt_enable();

	if (unlikely(status)) {
		printk(KERN_ERR "%s(): lv1_construct_and_connect_irq_plug() "
		       "failed: %ld\n", __FUNCTION__, status);
		return -EPERM;
	}

	return 0;
}
EXPORT_SYMBOL(__ps3pf_connect_irq);

/* outlet_id and irq_no are returned to the caller.  All other parameters
 * are input parameters.
 */
int ps3pf_connect_irq(

	unsigned long	*outlet_id,
	unsigned int	*irq_no,
	unsigned int	*irq_cpu,

	/* arguments to request_irq() */
	irqreturn_t     (*handler)(int, void *, struct pt_regs *),
	unsigned long   flags,
	const char      *irq_name,
	void            *dev_id
	)
{
	long				status;

	status = lv1_construct_event_receive_port(outlet_id);
	if (unlikely(status)) {
		printk(KERN_ERR "%s(): lv1_construct_event_receive_port() "
		       "failed: %ld\n", __FUNCTION__, status);
		return -EPERM;
	}

	return __ps3pf_connect_irq(*outlet_id,
				 irq_no, irq_cpu,
				 handler, flags, irq_name, dev_id);
}
EXPORT_SYMBOL(ps3pf_connect_irq);

int
ps3pf_connect_io_irq(unsigned long intr_id, unsigned int *irq_no)
{
	int err;
	unsigned long status, outlet_id;
	unsigned long flags;

	if (LV1_PLUG_ID_LIMIT <= intr_id)
		return -EINVAL;

	spin_lock_irqsave(&ps3pf_io_irq_lock, flags);
	if (ps3pf_io_irq_connected[intr_id]) {
		*irq_no = intr_id;
		spin_unlock_irqrestore(&ps3pf_io_irq_lock, flags);

		return 0;
	}

	err = ps3pf_reserve_irq_plug(intr_id);
	if (err != 0) {
		spin_unlock_irqrestore(&ps3pf_io_irq_lock, flags);
		printk(KERN_ERR "%s(): ps3pf_reserve_io_irq_plug(%ld) failed\n",
		      __FUNCTION__, intr_id);
		return err;
	}

	status = lv1_construct_io_irq_outlet(intr_id, &outlet_id);
	if (unlikely(status)) {
		spin_unlock_irqrestore(&ps3pf_io_irq_lock, flags);
		printk(KERN_ERR "%s(): lv1_construct_io_irq_outlet() "
		       "failed: %ld\n", __FUNCTION__, status);
		return -EPERM;
	}
	ps3pf_io_irq_outlet_id[intr_id] = outlet_id;

	if (intr_id == 0) {
		/*
		 * XXX, fix me
		 * We don't handle this case for now.
		 */
		panic("%s(): intr_id=0", __FUNCTION__);
	}
	*irq_no = intr_id;

	status = lv1_construct_and_connect_irq_plug(*irq_no, outlet_id);
	if (unlikely(status)) {
		spin_unlock_irqrestore(&ps3pf_io_irq_lock, flags);
		/*

		XXX, fix me

		status = lv1_destruct_io_irq_outlet(outlet_id);
		*/
		printk(KERN_ERR "%s(): lv1_construct_and_connect_irq_plug() "
		       "failed: %ld\n", __FUNCTION__, status);
		return -EPERM;
	}

	ps3pf_io_irq_connected[intr_id] = 1;
	spin_unlock_irqrestore(&ps3pf_io_irq_lock, flags);

	return 0;
}

EXPORT_SYMBOL(ps3pf_connect_io_irq);

/*
 * XXX, this remains just for backward compatibility
 */
int
ps3pf_get_io_irq(unsigned long intr_id, int *irq_no)
{
	return ps3pf_connect_io_irq(intr_id, irq_no);
}

EXPORT_SYMBOL(ps3pf_get_io_irq);

int ps3pf_connect_spu_irq(
	unsigned long	lspu_id,
	unsigned int	class,

	unsigned long	*outlet_id,
	unsigned int	*irq_no,
	unsigned int	*irq_cpu,

	/* arguments to request_irq() */
	irqreturn_t	(*handler)(int, void *, struct pt_regs *),
	unsigned long	flags,
	const char	*irq_name,
	void		*dev_id
	)
{
#if defined(CONFIG_SPU_FS) || defined(CONFIG_SPU_FS_MODULE)
	long				status;

	status = lv1_get_spu_irq_outlet(lspu_id,
					class,
					outlet_id);
	if (unlikely(status)) {
		printk(KERN_ERR "%s(): construct / get spu irq outlet :"
		       "failed: %ld\n", __FUNCTION__, status);
		return -EPERM;
	}

	return __ps3pf_connect_irq(*outlet_id,
				 irq_no, irq_cpu,
				 handler, flags, irq_name, dev_id);
#else
	BUG();
	return 0;
#endif /* CONFIG_SPU_FS || CONFIG_SPU_FS_MODULE */
}
EXPORT_SYMBOL(ps3pf_connect_spu_irq);

int ps3pf_free_spu_irq(
	unsigned long	outlet_id,
	unsigned int	irq_no,
	unsigned int	irq_cpu,
	void		*dev_id
	)
{
#if defined(CONFIG_SPU_FS) || defined(CONFIG_SPU_FS_MODULE)
	int ret;

	ret = __ps3pf_free_irq(outlet_id, irq_no, irq_cpu, dev_id);

	/* outlet will be destructed with logical spu */

	return ret;
#else
	BUG();
	return 0;
#endif /* CONFIG_SPU_FS || CONFIG_SPU_FS_MODULE */
}
EXPORT_SYMBOL(ps3pf_free_spu_irq);
