/* Copyright (C) 2003, 2004 Peter J. Verveer
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      
 */

#include "ni_support.h"
#include "ni_morphology.h"
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <limits.h>
#include <float.h>

#define NI_ERODE_POINT(_pi, _out, _offsets, _filter_size, _type, _mv,     \
                       _border_value, _bv, _true, _false, _ivo, _changed) \
{                                                                         \
  int _ii, _oo = *_offsets;                                               \
  int _in = *(_type*)_pi ? 1 : 0;                                         \
  if (_mv) {                                                              \
    _out = 1;                                                             \
    for(_ii = 0; _ii < _filter_size; _ii++) {                             \
      _oo = _offsets[_ii];                                                \
      if (_oo == _bv) {                                                   \
        if (!_border_value) {                                             \
          _out = 0;                                                       \
          break;                                                          \
        }                                                                 \
      } else {                                                            \
        int _nn = *(_type*)(_pi + _oo) ? _true : _false;                  \
        if (!_nn) {                                                       \
          _out = 0;                                                       \
          break;                                                          \
        }                                                                 \
      }                                                                   \
    }                                                                     \
    _out = _ivo ? 1 - _out : _out;                                        \
    if (_in != _out)                                                      \
      _changed = 1;                                                       \
  } else {                                                                \
    _out = _in;                                                           \
  }                                                                       \
}                                                                         

int NI_BinaryErosion(PyArrayObject* input, PyArrayObject* strct, 
                     PyArrayObject* mask, PyArrayObject** output,  
                     PyObject* output_in, int bdr_value, int *shifts, 
                     int invert_input, int invert_output,  int* changed)
{
  int struct_size = 0, *offsets = NULL, size, *oo, jj, border_flag_value;
  int ssize, mtype = tAny, true, false, msk_value, irank, itype;
  int idims[NI_MAXDIM], sdims[NI_MAXDIM];
  NI_Iterator ii, io, mi;
  NI_FilterIterator fi;
  Bool *ps, out = 0;
  char *pi, *po, *pm = NULL;

  assert(input != NULL);
  assert(output != NULL);
  assert(strct != NULL);

  /* complex type not supported: */
  itype = NI_GetArrayType(input);
  if (itype == tComplex32 || itype == tComplex64) {
    PyErr_SetString(PyExc_RuntimeError, "complex arrays not supported");
    goto exit;
  }

  irank = NI_GetArrayRank(input);
  /* input and structure must have equal rank: */
  if (irank != NI_GetArrayRank(strct)) {
    PyErr_SetString(PyExc_RuntimeError, 
                    "structure and input arrays must have equal rank");
    goto exit;
  }

  /* structuring element must be of bool type: */
  if (NI_GetArrayType(strct) != tBool) {
    PyErr_SetString(PyExc_RuntimeError, "structure type must be boolean");
    goto exit;
  }

  /* the structure array must be contigous: */
  if (!PyArray_ISCONTIGUOUS(strct)) {
    PyErr_SetString(PyExc_RuntimeError,
                    "structure array must be contiguous");
    goto exit;
  }
  ps = (Bool*)NI_GetArrayData(strct);
  ssize = NI_Elements(strct);
  for(jj = 0; jj < ssize; jj++) 
    if (ps[jj]) ++struct_size;

  /* allocate output */
  NI_GetArrayDimensions(input, idims);
  if (!NI_OutputArray(tBool, irank, idims, output_in, output))
    goto exit;

  /* structure must be not be emtpy: */
  if (struct_size < 1) {
    PyErr_SetString(PyExc_RuntimeError, "structure size must be > 0");
    goto exit;
  }

  if (mask) {
    /* input and mask must have equal size: */
    if (!NI_ShapeEqual(input, mask)) {
      PyErr_SetString(PyExc_RuntimeError, 
                      "input and mask sizes must be equal");
      return 0;
    }
    /* iterator, data pointer and type of mask array: */
    if (!NI_InitPointIterator(mask, &mi))
      return 0;
    pm = NI_GetArrayData(mask);
    mtype = NI_GetArrayType(mask);
  }

  /* calculate the filter offsets: */
  if (!NI_InitFilterOffsetsFromArray(input, strct, shifts,
                                     NI_EXTEND_CONSTANT, &offsets,
                                     &border_flag_value))
    goto exit;

  /* initialize input element iterator: */
  if (!NI_InitPointIterator(input, &ii))
    goto exit;

  /* initialize output element iterator: */
  if (!NI_InitPointIterator(*output, &io))
    goto exit;

  /* initialize filter iterator: */
  NI_GetArrayDimensions(strct, sdims);
  if (!NI_InitFilterIterator(irank, sdims, struct_size, idims, shifts, &fi))
    goto exit;
    
  /* get data pointers an size: */
  pi = NI_GetArrayData(input);
  po = NI_GetArrayData(*output);
  size = NI_Elements(input);

  if (invert_input) {
    bdr_value = bdr_value ? 0 : 1;
    true = 0;
    false = 1;
  } else {
    bdr_value = bdr_value ? 1 : 0;
    true = 1;
    false = 0;
  }

  /* iterator over the elements: */
  oo = offsets;
  *changed = 0;
  msk_value = 1;
  for(jj = 0; jj < size; jj++) {
    if (mask) {                                                        
      switch(mtype) {                                                
      case tBool:                        
        msk_value = *(Bool*)pm ? 1 : 0;                
        break;                        
      case tUInt8:                
        msk_value = *(UInt8*)pm ? 1 : 0;        
        break;                        
      case tUInt16:                
        msk_value = *(UInt16*)pm ? 1 : 0;        
        break;                        
      case tUInt32:                
        msk_value = *(UInt32*)pm ? 1 : 0;        
        break;                        
#if HAS_UINT64
      case tUInt64:                
        msk_value = *(UInt64*)pm ? 1 : 0;        
        break;                        
#endif
      case tInt8:                        
        msk_value = *(Int8*)pm ? 1 : 0;                
        break;                        
      case tInt16:                
        msk_value = *(Int16*)pm ? 1 : 0;        
        break;                        
      case tInt32:                
        msk_value = *(Int32*)pm ? 1 : 0;        
        break;
      case tInt64:                
        msk_value = *(Int64*)pm ? 1 : 0;        
        break;
      case tFloat32:                
        msk_value = *(Float32*)pm ? 1 : 0;
        break;
      case tFloat64:                
        msk_value = *(Float64*)pm ? 1 : 0;        
        break;                        
      default:                        
        PyErr_SetString(PyExc_RuntimeError, "data type not supported");        
        return 0; 
      }        
    }
    switch (itype) {
    case tBool:
      NI_ERODE_POINT(pi, out, oo, struct_size, Bool, msk_value, bdr_value, 
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
    case tUInt8:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt8, msk_value, bdr_value,
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
    case tUInt16:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt16, msk_value, bdr_value,
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
    case tUInt32:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt32, msk_value, bdr_value,
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
#if HAS_UINT64
    case tUInt64:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt64, msk_value, bdr_value,
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
#endif
    case tInt8:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int8, msk_value, bdr_value, 
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
    case tInt16:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int16, msk_value, bdr_value,
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
    case tInt32:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int32, msk_value, bdr_value, 
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
    case tInt64:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int64, msk_value, bdr_value, 
                     border_flag_value, true, false, invert_output,
                     *changed);
      break;
    case tFloat32:
      NI_ERODE_POINT(pi, out, oo, struct_size, Float32, msk_value, 
                     bdr_value, border_flag_value, true, false,
                    invert_output, *changed);
      break;
    case tFloat64:
      NI_ERODE_POINT(pi, out, oo, struct_size, Float64, msk_value,
                     bdr_value, border_flag_value, true, false,
                     invert_output, *changed);
      break;
    default:
      PyErr_SetString(PyExc_RuntimeError, "data type not supported");
      goto exit;
    }
    *(Bool*)po = out;
    if (mask) {
      NI_FILTER_NEXT3(fi, ii, io, mi, oo, pi, po, pm);
    } else {
      NI_FILTER_NEXT2(fi, ii, io, oo, pi, po);
    }
  }

 exit:
  if (offsets) 
    free(offsets);
  return PyErr_Occurred() ? 0 : 1;
}


#define NI_DISTANCE_EUCLIDIAN  1
#define NI_DISTANCE_CITY_BLOCK 2
#define NI_DISTANCE_CHESSBOARD 3

typedef struct {
  int *coordinates;
  int index;
  void *next;
} NI_BorderElement;

int NI_DistanceTransformBruteForce(PyArrayObject* input, int metric,
                                   double *sampling, int return_distances,
                                   int return_features,
                                   PyArrayObject** distances, 
                                   PyArrayObject** features,
                                   PyObject *dt_object, PyObject *ft_object)
{
  int irank, itype, idims[NI_MAXDIM], size, otype, jj, kk, min_index = 0;
  NI_BorderElement *border_elements = NULL, *temp;
  NI_Iterator ii, di, fi;
  char *pi, *pd = NULL, *pf = NULL;

  assert(input != NULL);
  assert(return_distances || return_features);

  /* get input rank, type and shape */
  irank = NI_GetArrayRank(input);
  itype = NI_GetArrayType(input);
  NI_GetArrayDimensions(input, idims);

  if (itype != tInt8) {
    PyErr_SetString(PyExc_RuntimeError,  "input type not correct");
    goto exit;
  }

  size = NI_Elements(input);
  pi = NI_GetArrayData(input);

  if (!NI_InitPointIterator(input, &ii))
    goto exit;
  
  for(jj = 0; jj < size; jj++) {
    if (*(Int8*)pi < 0) {
      temp = (NI_BorderElement*)malloc(sizeof(NI_BorderElement));
      if (!temp) {
        PyErr_NoMemory();
        goto exit;
      }
      temp->next = border_elements;
      border_elements = temp;
      temp->index = jj;
      temp->coordinates = (int*)malloc(irank * sizeof(int));
      for(kk = 0; kk < irank; kk++)
        temp->coordinates[kk] = ii.coordinates[kk];
    }
    NI_ITERATOR_NEXT(ii, pi);
  }

  NI_ITERATOR_RESET(ii);
  pi = NI_GetArrayData(input);

  /* allocate outputs */
  if (return_distances) {
    if (metric == NI_DISTANCE_EUCLIDIAN)
      otype = tFloat64;
    else
      otype = tUInt32;
    if (!NI_OutputArray(otype, irank, idims, dt_object, distances))
      goto exit;
    pd = NI_GetArrayData(*distances);

    if (!NI_InitPointIterator(*distances, &di))
      goto exit;
  }

  if (return_features) {
    if (!NI_OutputArray(tInt32, irank, idims, ft_object, features))
      goto exit;
    pf = NI_GetArrayData(*features);
    if (!NI_InitPointIterator(*features, &fi))
      goto exit;
  }

  switch(metric) {
  case NI_DISTANCE_EUCLIDIAN:
    for(jj = 0; jj < size; jj++) {
      if (*(Int8*)pi > 0) {
        double distance = DBL_MAX;
        temp = border_elements;
        while(temp) {
          double d = 0.0, t;
          for(kk = 0; kk < irank; kk++) {
            t = ii.coordinates[kk] - temp->coordinates[kk];
            if (sampling)
              t *= sampling[kk];
            d += t * t;
          }
          if (d < distance) {
            distance = d;
            if (return_features)
              min_index = temp->index;
          }
          temp = temp->next;
        }
        if (return_distances) 
          *(Float64*)pd = sqrt(distance);
        if (return_features)
          *(Int32*)pf = min_index;
      } else {
        if (return_distances) 
          *(Float64*)pd = 0.0;
        if (return_features)
          *(Int32*)pf = jj;
      }
      if (return_features && return_distances) {
        NI_ITERATOR_NEXT3(ii, di, fi, pi, pd, pf);
      } else if (return_distances) {
        NI_ITERATOR_NEXT2(ii, di, pi, pd);
      } else {
         NI_ITERATOR_NEXT2(ii, fi, pi, pf);
      }
    }
    break;
  case NI_DISTANCE_CITY_BLOCK:
  case NI_DISTANCE_CHESSBOARD:
    for(jj = 0; jj < size; jj++) {
      if (*(Int8*)pi > 0) {
        unsigned int distance = UINT_MAX;
        temp = border_elements;
        while(temp) {
          unsigned int d = 0;
          int t;
          for(kk = 0; kk < irank; kk++) {
            t = ii.coordinates[kk] - temp->coordinates[kk];
            if (t < 0)
              t = -t;
            if (metric == NI_DISTANCE_CITY_BLOCK) {
              d += t;
            } else {
              if ((unsigned int)t > d)
                d = t;
            }
          }
          if (d < distance) {
            distance = d;
            if (return_features)
              min_index = temp->index;
          }
          temp = temp->next;
        }
        if (return_distances) 
          *(UInt32*)pd = distance;
        if (return_features)
          *(Int32*)pf = min_index;
      } else {
        if (return_distances) 
          *(UInt32*)pd = 0;
        if (return_features)
          *(Int32*)pf = jj;
      }
      if (return_features && return_distances) {
        NI_ITERATOR_NEXT3(ii, di, fi, pi, pd, pf);
      } else if (return_distances) {
        NI_ITERATOR_NEXT2(ii, di, pi, pd);
      } else {
         NI_ITERATOR_NEXT2(ii, fi, pi, pf);
      }
    }
    break;
  default:
    PyErr_SetString(PyExc_RuntimeError,  "distance metric not supported");
    goto exit;
  }

 exit:
  while (border_elements) {
    temp = border_elements;
    border_elements = border_elements->next;
    if (temp->coordinates)
      free(temp->coordinates);
    free(temp);
  }
  return PyErr_Occurred() ? 0 : 1;
}


int NI_DistanceTransformOnePass(PyArrayObject *strct,
                                PyArrayObject* distances, 
                                PyArrayObject *features)
{
  int kk, jj, ssize, size, filter_size, shifts[NI_MAXDIM], *offsets = NULL;
  int mask_value, *oo, arank, itype, adims[NI_MAXDIM], sdims[NI_MAXDIM];
  int *foffsets = NULL, *foo = NULL;
  PyArrayObject *tmp = NULL;
  Bool *ps, *pf = NULL;
  char *pd;
  NI_FilterIterator si, ti;
  NI_Iterator di, fi;

  assert(distances != NULL);
  assert(strct != NULL);

  /* type must be tInt32: */
  itype = NI_GetArrayType(distances);
  if (itype != tInt32) {
    PyErr_SetString(PyExc_RuntimeError, "array type must be tInt32");
    goto exit;
  }
  /* structuring element must be of bool type: */
  if (NI_GetArrayType(strct) != tBool) {
    PyErr_SetString(PyExc_RuntimeError, "structure type must be Bool");
    goto exit;
  }

  arank = NI_GetArrayRank(distances);
  /* array and structure must have equal rank: */
  if (NI_GetArrayRank(strct) != arank) {
    PyErr_SetString(PyExc_RuntimeError, 
                    "structure rank must be equal to array rank");
    goto exit;
  }

  ssize = 1;
  NI_GetArrayDimensions(strct, sdims);
  for(kk = 0; kk < arank; kk++) {
    ssize *= sdims[kk];
    if (sdims[kk] != 3) {
    PyErr_SetString(PyExc_RuntimeError, "structure dimensions must "
                    "equal to 3");
      goto exit;
    }
  }

  /* we only use the first half of the structure data, so we make a 
     temporary structure for use with the filter functions: */
  tmp = NI_NewArray(tBool, arank, sdims);
  if (!tmp) {
    PyErr_NoMemory();
    goto exit;
  }
  NI_CopyArray(tmp, strct);
  ps = (Bool*)NI_GetArrayData(tmp);
  filter_size = 0;
  for(kk = 0; kk < ssize / 2; kk++)
    if (ps[kk])
      ++filter_size;
  for(kk = ssize / 2; kk < ssize; kk++)
    ps[kk] = 0;
  
  /* get data and size */
  NI_GetArrayDimensions(distances, adims);
  pd = NI_GetArrayData(distances);
  size = NI_Elements(distances);
  if (!NI_InitPointIterator(distances, &di))
    goto exit;

  /* we don't use shifts: */
  for(kk = 0; kk < arank; kk++)
    shifts[kk] = 0;
  /* calculate the filter offsets: */
  if (!NI_InitFilterOffsetsFromArray(distances, tmp, shifts, 
                                     NI_EXTEND_CONSTANT, 
                                     &offsets, &mask_value))
    goto exit;
  /* initialize filter iterator: */
  if (!NI_InitFilterIterator(arank, sdims, filter_size, adims, shifts, &si))
    goto exit;
  
  if (features) {
    int dummy;
    /* initialize point iterator: */
    pf = NI_GetArrayData(features);
    if (!NI_InitPointIterator(features, &fi))
      goto exit;
    /* calculate the filter offsets: */
    if (!NI_InitFilterOffsetsFromArray(features, tmp, shifts, 
                                       NI_EXTEND_CONSTANT, 
                                       &foffsets, &dummy))
      goto exit;
    /* initialize filter iterator: */
    if (!NI_InitFilterIterator(arank, sdims, filter_size, adims, shifts,
        &ti))
      goto exit;
  }

  /* iterator over the elements: */
  oo = offsets;
  if (features)
    foo = foffsets;
  for(jj = 0; jj < size; jj++) {
    Int32 value = *(Int32*)pd;
    if (value != 0) {
      Int32 min = value;
      int min_offset = 0;
      /* iterate over structuring element: */
      for(kk = 0; kk < filter_size; kk++) {
        int offset = oo[kk];
        Int32 tt = -1;
        if (offset < mask_value)
          tt = *(Int32*)(pd + offset);
        if (tt >= 0) {
          if ((min < 0) || (tt + 1 < min)) {
            min = tt + 1;
            if (features)
              min_offset = foo[kk];
          }
        }
      }
      *(Int32*)pd = min;
      if (features)
        *(Int32*)pf = *(Int32*)(pf + min_offset);
    }
    if (features) {
      NI_FILTER_NEXT(ti, fi, foo, pf);
    }
    NI_FILTER_NEXT(si, di, oo, pd);
  }
  
 exit:
  if (offsets) free(offsets);
  if (foffsets) free(foffsets);
  Py_XDECREF(tmp);
  return PyErr_Occurred() ? 0 : 1;
}

static void _VoronoiFT(char *pf, int len, int *coor, int rank, int d,
                       int stride, int cstride, int **f, int *g, 
                       double *sampling)
{
  int l = -1, ii, jj, maxl, idx1, idx2;
  for(ii = 0; ii < len; ii++)
    for(jj = 0; jj < rank; jj++)
      f[ii][jj] = *(Int32*)(pf + ii * stride + cstride * jj);
  for(ii = 0; ii < len; ii++) {
    if (*(Int32*)(pf + ii * stride) >= 0) {
      double fd = f[ii][d];
      double wR = 0.0;
      for(jj = 0; jj < rank; jj++) {
        if (jj != d) {
          double tw = f[ii][jj] - coor[jj];
          if (sampling)
            tw *= sampling[jj];
          wR += tw * tw;
        }
      }
      while(l >= 1) {
        double a, b, c, uR = 0.0, vR = 0.0, f1;
        idx1 = g[l];
        f1 = f[idx1][d];
        idx2 = g[l - 1];
        a = f1 - f[idx2][d];
        b = fd - f1;
        if (sampling) {
          a *= sampling[d];
          b *= sampling[d];
        }
        c = a + b;
        for(jj = 0; jj < rank; jj++) {
          if (jj != d) {
            double cc = coor[jj];
            double tu = f[idx2][jj] - cc;
            double tv = f[idx1][jj] - cc;
            if (sampling) {
              tu *= sampling[jj];
              tv *= sampling[jj];
            }
            uR += tu * tu;
            vR += tv * tv;
          }
        }
        if (c * vR - b * uR - a * wR - a * b * c <= 0.0)
          break;
        --l;
      }
      ++l;
      g[l] = ii;
    }
  }
  maxl = l;
  if (maxl >= 0) {
    l = 0;
    for (ii = 0; ii < len; ii++) {
      double delta1 = 0.0, t;
      for(jj = 0; jj < rank; jj++) {
        t = jj == d ? f[g[l]][jj] - ii : f[g[l]][jj] - coor[jj];
        if (sampling)
          t *= sampling[jj];
        delta1 += t * t;
      }
      while (l < maxl) {
        double delta2 = 0.0;
        for(jj = 0; jj < rank; jj++) {
          t = jj == d ? f[g[l + 1]][jj] - ii : f[g[l + 1]][jj] - coor[jj];
          if (sampling)
            t *= sampling[jj];
          delta2 += t * t;
        }
        if (delta1 <= delta2)
          break;
        delta1 = delta2;
        ++l;
      }
      idx1 = g[l];
      for(jj = 0; jj < rank; jj++)
        *(Int32*)(pf + ii * stride + jj * cstride) = f[idx1][jj];
    }
  }
}


/* Recursive feature transform */
static void _ComputeFT(char *pi, char *pf, int *ishape, int *istrides, 
                       int *fstrides, int rank, int d, int *coor, 
                       int **f, int *g, PyArrayObject *features,
                       double *sampling)
{
  int jj, kk;

  if (d == 0) {
    char *tf1 = pf;
    for(jj = 0; jj < ishape[0]; jj++) {
      if (*(Int8*)pi) {
        *(Int32*)tf1 = -1;
      } else {
        char *tf2 = tf1;
        *(Int32*)tf2 = jj;
        for(kk = 1; kk < rank; kk++) {
          tf2 += fstrides[0];
          *(Int32*)tf2 = coor[kk];
        }
      }
      pi += istrides[0];
      tf1 += fstrides[1];
    }
    _VoronoiFT(pf, ishape[0], coor, rank, 0, fstrides[1], fstrides[0], f, g,
               sampling);
  } else {
    UInt32 axes = 0;
    char *tf = pf;
    int size = 1;
    NI_Iterator ii;

    for(jj = 0; jj < ishape[d]; jj++) {
      coor[d] = jj;
      _ComputeFT(pi, tf, ishape, istrides, fstrides, rank, d - 1, coor, f,
                 g, features, sampling);
      pi += istrides[d];
      tf += fstrides[d + 1];
    }

    for(jj = 0; jj < d; jj++) {
      axes |= (UInt32)1 << (jj + 1);
      size *= ishape[jj];
    }
    NI_InitSubSpaceIterator(features, &ii, axes);
    tf = pf;
    for(jj = 0; jj < size; jj++) {
      for(kk = 0; kk < d; kk++)
        coor[kk] = ii.coordinates[kk];
      _VoronoiFT(tf, ishape[d], coor, rank, d, fstrides[d + 1], fstrides[0],
                 f, g, sampling);
      NI_ITERATOR_NEXT(ii, tf);
    }
    for(kk = 0; kk < d; kk++)
      coor[kk] = 0;
  }
}

/* Exact euclidean feature transform, as described in: C. R. Maurer,
   Jr., R. Qi, V. Raghavan, "A linear time algorithm for computing
   exact euclidean distance transforms of binary images in arbitrary
   dimensions. IEEE Trans. PAMI 25, 265-270, 2003. */
int NI_EuclideanFeatureTransform(PyArrayObject* input, double *sampling,
                                 PyArrayObject** features,
                                 PyObject *output_in)
{
  int irank, itype, ishape[NI_MAXDIM], odims[NI_MAXDIM], ii;
  int coor[NI_MAXDIM], istrides[NI_MAXDIM], fstrides[NI_MAXDIM], mx = 0;
  int *tmp = NULL, **f = NULL, *g = NULL;
  char *pi, *pf;

  assert(input != NULL);

  /* get input rank, type and shape */
  irank = NI_GetArrayRank(input);
  itype = NI_GetArrayType(input);
  NI_GetArrayDimensions(input, ishape);

  if (itype != tInt8) {
    PyErr_SetString(PyExc_RuntimeError,  "input type not correct");
    goto exit;
  }

  pi = NI_GetArrayData(input);

  /* output is an array of indices, with an extra axis for the indices */
  odims[0] = irank;
  for(ii = 0; ii < irank; ii++)
    odims[ii + 1] = ishape[ii];
  if (!NI_OutputArray(tInt32, irank + 1, odims, output_in, features))
    goto exit;
  pf = NI_GetArrayData(*features);

  /* We start with zero coordinates */
  for(ii = 0; ii < irank; ii++)
    coor[ii] = 0;

  NI_GetArrayStrides(input, istrides);
  NI_GetArrayStrides(*features, fstrides);

  for(ii = 0; ii < irank; ii++)
    if (ishape[ii] > mx)
      mx = ishape[ii];

  /* Some temporaries */
  f = (int**)malloc(mx * sizeof(int*));
  g = (int*)malloc(mx * sizeof(int));
  tmp = (int*)malloc(mx * irank * sizeof(int));
  if (!f || !g || !tmp) {
    PyErr_NoMemory();
    goto exit;
  }
  for(ii = 0; ii < mx; ii++) 
    f[ii] = tmp + ii * irank;
  
  /* First call of recursive feature transform */
  _ComputeFT(pi, pf, ishape, istrides, fstrides, irank, irank - 1, coor, f,
             g, *features, sampling);

 exit:
  if (f)
    free(f);
  if (g)
    free(g);
  if (tmp)
    free(tmp);
      
  return PyErr_Occurred() ? 0 : 1;
}
