/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qzion.h"

#include <QtGui>
#include <math.h>
#include "scene.h"


Scene::Scene(QZionAbstractCanvas *canvas)
{
    this->canvas = canvas;
}


void Scene::add(QZionObject *obj)
{
    objects.append(obj);
}


void Scene::valueChanged(qreal value)
{
    int v = (int) (initV + (finalV - initV) * value);
    target1->setSize(v, target1->size().height());
    target2->setSize(v, target2->size().height());
}


void Scene::startAnim(QZionObject *obj1, QZionObject *obj2)
{
    if (timeline.state() == QTimeLine::Running)
        return;

    target1 = obj1;
    target2 = obj2;

    timeline.setDuration(4000);
    timeline.setUpdateInterval(1);
    timeline.setLoopCount(0);

    connect(&timeline, SIGNAL(valueChanged(qreal)),
            this, SLOT(valueChanged(qreal)));

    initV = target1->size().width();
    finalV = (initV >= 600) ? 50 : 600;

    timeline.start();
}


int main(int argc, char **argv)
{
    QApplication app(argc, argv);
    QZionCanvas canvas;
    Scene scene(&canvas);

    canvas.setSize(QSize(600, 425));
    canvas.show();

    // loading image from image file
    QZionObject *background = new QZionImage(&canvas, "../images/bg.jpg");
    background->setPos(0, 0);
    background->show();
    scene.add(background);

    // Ellipsizing

    QZionRectangle *labelRect = new QZionRectangle(&canvas);
    labelRect->setColor(QColor(200, 200, 200));
    labelRect->setPos(0, 0);
    labelRect->setSize(QSize(600, 50));
    labelRect->setOpacity(128);
    labelRect->show();
    scene.add(labelRect);

    QZionText *label = new QZionText(&canvas, QString("Ellipsizing"),
                                     QFont("SansSerif", 20, QFont::Bold));
    label->setColor(Qt::white);
    label->setStyle(QZionText::Shadow);
    label->setAlignment(QPointF(0.5, 0.5));
    label->setPos(labelRect->pos().x(), labelRect->pos().y());
    label->setSize(labelRect->size());
    label->show();
    scene.add(label);

    scene.startAnim(label, labelRect);

    // Left aligned

    QZionRectangle *label1Rect = new QZionRectangle(&canvas);
    label1Rect->setColor(QColor(200, 0, 200));
    label1Rect->setOpacity(128);
    label1Rect->setPos(0, labelRect->pos().y() + labelRect->size().height() + 5);
    label1Rect->setSize(QSize(300, 50));
    label1Rect->show();
    scene.add(label1Rect);

    QZionText *label1 = new QZionText(&canvas, QString("Left"),
                                     QFont("SansSerif", 20, QFont::Bold),
                                     QColor(0, 0, 0));
    label1->setAlignment(QPointF(0.0, 0.5));
    label1->setPos(label1Rect->pos().x(), label1Rect->pos().y());
    label1->setSize(label1Rect->size());
    label1->show();
    scene.add(label1);

    // Right aligned

    QZionRectangle *label2Rect = new QZionRectangle(&canvas);
    label2Rect->setColor(QColor(0, 200, 200));
    label2Rect->setOpacity(128);
    label2Rect->setPos(0, label1Rect->pos().y() + label1Rect->size().height() + 5);
    label2Rect->setSize(QSize(600, 50));
    label2Rect->show();
    scene.add(label2Rect);

    QZionText *label2 = new QZionText(&canvas, QString("Right"),
                                     QFont("SansSerif", 20, QFont::Bold),
                                     QColor(0, 0, 0));
    label2->setAlignment(QPointF(1.0, 0.5));
    label2->setPos(label2Rect->pos().x(), label2Rect->pos().y());
    label2->setSize(label2Rect->size());
    label2->show();
    scene.add(label2);

    // Centered

    QZionRectangle *label3Rect = new QZionRectangle(&canvas);
    label3Rect->setColor(QColor(200, 200, 0));
    label3Rect->setOpacity(128);
    label3Rect->setPos(0, label2Rect->pos().y() + label2Rect->size().height() + 5);
    label3Rect->setSize(QSize(600, 50));
    label3Rect->show();
    scene.add(label3Rect);

    QZionText *label3 = new QZionText(&canvas, QString("Centered"),
                                     QFont("SansSerif", 20, QFont::Bold),
                                     QColor(0, 0, 0));
    label3->setAlignment(QPointF(0.5, 0.5));
    label3->setColor(Qt::white);
    label3->setStyle(QZionText::Shadow);
    label3->setPos(label3Rect->pos().x(), label3Rect->pos().y());
    label3->setSize(label3Rect->size());
    label3->show();
    scene.add(label3);

    // Top

    QZionRectangle *label4Rect = new QZionRectangle(&canvas);
    label4Rect->setColor(QColor(200, 0, 0));
    label4Rect->setOpacity(128);
    label4Rect->setPos(0, label3Rect->pos().y() + label3Rect->size().height() + 5);
    label4Rect->setSize(QSize(600, 100));
    label4Rect->show();
    scene.add(label4Rect);

    QZionText *label4 = new QZionText(&canvas,
                                     QString("Top"),
                                     QFont("SansSerif", 20, QFont::Bold),
                                     QColor(0, 0, 0));
    label4->setAlignment(QPointF(0.5, 0.0));
    label4->setColor(Qt::white);
    label4->setStyle(QZionText::OutLine);
    label4->setPos(label4Rect->pos().x(), label4Rect->pos().y());
    label4->setSize(label4Rect->size());
    label4->show();
    scene.add(label4);

    // Bottom

    QZionRectangle *label5Rect = new QZionRectangle(&canvas);
    label5Rect->setColor(QColor(0, 0, 200));
    label5Rect->setOpacity(128);
    label5Rect->setPos(0, label4Rect->pos().y() + label4Rect->size().height() + 5);
    label5Rect->setSize(QSize(600, 100));
    label5Rect->show();
    scene.add(label5Rect);

    QZionText *label5 = new QZionText(&canvas,
                                     QString("Bottom"),
                                     QFont("SansSerif", 20, QFont::Bold),
                                     QColor(0, 0, 0));
    label5->setAlignment(QPointF(0.5, 1.0));
    label5->setPos(label5Rect->pos().x(), label5Rect->pos().y());
    label5->setSize(label5Rect->size());
    label5->show();
    scene.add(label5);

    app.exec();

    while (!scene.objects.isEmpty())
        delete scene.objects.takeFirst();

    return 1;
}
