# Copyright (C) 2013 Canonical, Ltd.
#
# This library is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License
# version 3.0 as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License version 3.0 for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library. If not, see
# <http://www.gnu.org/licenses/>.
"""Tests for the main plug-in module."""

from __future__ import print_function, unicode_literals

import os
import ubuntuone.ubuntuone as rbplugin

from dirspec.utils import user_home
from gi.repository import Gio, GObject
from twisted.internet.defer import inlineCallbacks
from ubuntuone.devtools.testcases.dbus import DBusTestCase


class FakeSettings(GObject.GObject):
    """Fake settings object."""
    __gsignals__ = {
        b'changed': (
            GObject.SIGNAL_RUN_LAST | GObject.SIGNAL_DETAILED,
            GObject.TYPE_NONE, (str,)),
    }
    values = {}

    def get_strv(self, key):
        """Get the value for key."""
        value = self.values.get(key, [])
        assert isinstance(value, list), 'Value must be type list.'
        return value

    def set_strv(self, key, value):
        """Set the value for key."""
        assert isinstance(value, list), 'Value must be type list.'
        self.values[key] = value

    @property
    def locations(self):
        """Return the currently set locations key value."""
        return self.get_strv('locations')

    def set_locations(self, value):
        """Set the locations key value."""
        self.set_strv('locations', value)


class FakeDB(GObject.GObject):
    """Fake DB object for tests."""
    entries = {}

    def entry_lookup_by_location(self, uri):
        """Lookup an entry for the URI."""
        return self.entries.get(uri, None)

    def add_uri(self, uri):
        """Add a URI to the entries list."""
        self.entries[uri] = uri


class FakeShell(GObject.GObject):
    """Fake shell class to store the db property object."""
    db = GObject.property(type=GObject.GObject)


class ExtensionTestCase(DBusTestCase):
    """Tests for the main extension class."""

    @inlineCallbacks
    def setUp(self):
        """Prepare the test."""
        yield super(ExtensionTestCase, self).setUp()
        self.settings = FakeSettings()
        self.patch(Gio, 'Settings', lambda _: self.settings)
        self.plugin = rbplugin.UbuntuOnePlugin()
        self.db = FakeDB()
        shell = FakeShell()
        shell.db = self.db
        self.plugin.object = shell
        self.lib_uri = Gio.File.new_for_path(rbplugin.U1LIBRARYPATH).get_uri()

    def test_add_location_exists(self):
        """Test that the library location doesn't get added twice."""
        expected = [self.lib_uri]
        self.settings.set_locations(expected)
        self.plugin._locations_changed()
        self.assertEqual(self.settings.locations, expected)

    def test_add_location_missing(self):
        """Test that adding the library location works when missing."""
        expected = [self.lib_uri]
        self.plugin._locations_changed()
        self.assertEqual(self.settings.locations, expected)

    def test_add_location_with_other(self):
        """Test that location gets added, when another library path exists."""
        music_uri = Gio.File.new_for_path(
            os.path.join(user_home, 'Music')).get_uri()
        self.settings.set_locations([music_uri])
        expected = [music_uri, self.lib_uri]
        self.plugin._locations_changed()
        self.assertEqual(self.settings.locations, expected)

    def test_del_location_bad_uri(self):
        """Test that removing the old badly formated URI works."""
        expected = [self.lib_uri]
        self.settings.set_locations(['file://%s' % rbplugin.U1LIBRARYPATH])
        self.plugin._locations_changed()
        self.assertEqual(self.settings.locations, expected)

    def test_del_location_non_uri(self):
        """Test that removing the old non-URI path value works."""
        expected = [self.lib_uri]
        self.settings.set_locations([rbplugin.U1LIBRARYPATH])
        self.plugin._locations_changed()
        self.assertEqual(self.settings.locations, expected)

    def test_download_finished(self):
        """Test that the download finished call adds a track to the library."""
        self.addCleanup(self.plugin.do_deactivate)
        self.plugin.do_activate()
        path = os.path.join(rbplugin.U1LIBRARYPATH,
                            'Artist', 'Album', 'Track.mp3')
        expected = Gio.File.new_for_path(path).get_uri()
        self.plugin.download_finished(path, None)
        self.assertEqual(self.db.entry_lookup_by_location(expected), expected)
