/*  Screem:  colourWizard.c
 *
 *  (C) Copyright 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <libgnomeui/gnome-stock-icons.h>
#include <libgnomeui/gnome-uidefs.h>

#include <libgnome/gnome-i18n.h>

#include <gtk/gtkcolorsel.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkstock.h>

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include "screem-plugin.h"

#include "screem-window.h"
#include "screem-editor.h"

#include "libegg/menu/egg-menu.h"

/* per wizard struct */
typedef struct {
	ScreemWindow *window;
	ScreemEditor *editor;

	GtkWidget *dialog;

	gchar *selected_text;
	gboolean applied;
} ColourWizard;


/* keep track of how many wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3

#define NAME "Color Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG NULL
;

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );

static GtkWidget* create_dialog( ColourWizard *colour_wizard );
void colour_wizard_display( EggAction *action, gpointer user_data );

static void clicked( GtkWidget *widget, gint button, GtkColorSelection *cs );


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	ColourWizard *colour_wizard;

	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"ColourWizard\" verb=\"ColourWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"ColourWizard\" verb=\"ColourWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "ColourWizard", "label",
	  	GTK_STOCK_SELECT_COLOR, NULL, "tip",
	  	G_CALLBACK( colour_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;
	
	gchar *label;
	gchar *tip;
	
	colour_wizard = g_new0( ColourWizard, 1 );
	colour_wizard->window = SCREEM_WINDOW( window );
	colour_wizard->editor = SCREEM_EDITOR( editor );
	
	label = g_strdup( _( "Color Wizard" ) );
	tip = g_strdup( _( "Select a Color to Insert" ) );

	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = colour_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( colour_wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( colour_wizard->window->merge ),
						ui, strlen( ui ), NULL );

	/* NOTE: make sure that the window being obtained isn't visible */
	colour_wizard->dialog = create_dialog( colour_wizard );

	g_object_set_data( G_OBJECT( colour_wizard->dialog ),
			   "colour_wizard", colour_wizard );

	g_free( label );
	g_free( tip );

	wizards = g_list_append( wizards, colour_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	ColourWizard *colour_wizard;

	for( list = wizards; list; list = list->next ) {
		colour_wizard = (ColourWizard*)list->data;
		if( colour_wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* link_wizard is the one to erase */
	wizards = g_list_remove( wizards, colour_wizard );
	g_free( colour_wizard );
}

/* End of required section */

void colour_wizard_display( EggAction *action, gpointer user_data )
{
	ColourWizard *colour_wizard;
	ScreemPage *page;

	colour_wizard = (ColourWizard*)user_data;
	page = screem_window_get_document( colour_wizard->window );
	
	if( ! page )
		return;
	
	gtk_widget_show_all( colour_wizard->dialog );
	gdk_window_raise( colour_wizard->dialog->window ); 
}

static void clicked( GtkWidget *widget, gint button, GtkColorSelection *cs )
{
	gint r, g, b;
	gchar *text;
	gint16 pos;

	gint start;
	gint len;

	ColourWizard *colour_wizard;

	GdkColor colour;

	colour_wizard = (ColourWizard*)g_object_get_data( G_OBJECT( widget ), 
							  "colour_wizard" );

	if( button == GTK_RESPONSE_OK ||
	    button == GTK_RESPONSE_APPLY ) {
		gtk_color_selection_get_current_color(GTK_COLOR_SELECTION(cs),
						      &colour);
		/* we want 8 bit values, not 16 */
		r = colour.red >> 8;
		g = colour.green >> 8;
		b = colour.blue >> 8;

		if( screem_editor_has_selection( colour_wizard->editor,
						 &start, &len ) ) {
			/* keep a copy of the selected text */
			if( !colour_wizard->selected_text )
				g_free( colour_wizard->selected_text );
			if(!colour_wizard->applied) {
				colour_wizard->selected_text = 
					screem_editor_get_text( colour_wizard->editor,
								start, len );
			}
			
			/* clear it */
			screem_editor_clear_selection( colour_wizard->editor );
		}
		
		text = g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
		pos = screem_editor_get_pos( colour_wizard->editor );
		screem_editor_insert( colour_wizard->editor, pos, text );
		g_free( text );

		colour_wizard->applied = TRUE;

		if( button == GTK_RESPONSE_APPLY ) {
			screem_editor_select_region( colour_wizard->editor, 
						     pos, 9 );
			return;
		}
	}
	else { /* cancel */
		if( screem_editor_has_selection( colour_wizard->editor, 
						 &start, &len ) )
			screem_editor_clear_selection( colour_wizard->editor );
		if(colour_wizard->applied )
			screem_editor_insert( colour_wizard->editor, start, 
					      colour_wizard->selected_text );
		colour_wizard->applied = FALSE;
	}

	if( colour_wizard->selected_text )
		g_free( colour_wizard->selected_text );
	colour_wizard->selected_text = NULL;

	gtk_widget_hide( widget );
}


static GtkWidget* create_dialog( ColourWizard *colour_wizard )
{
	GtkWidget *dialog;
	GtkWidget *colourSel;
	
	dialog = gtk_dialog_new_with_buttons(_("Insert a color"),
					     NULL, 0,
					     GTK_STOCK_CLOSE, 
					     GTK_RESPONSE_CLOSE,
					     GTK_STOCK_APPLY,
					     GTK_RESPONSE_APPLY,
					     GTK_STOCK_OK, GTK_RESPONSE_OK,
					     NULL );
      
	colourSel = gtk_color_selection_new();
	gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ), colourSel,
			    TRUE, TRUE, GNOME_PAD );

	g_signal_connect( G_OBJECT( dialog ), "response", 
			  G_CALLBACK( clicked ),
			  colourSel );

	gtk_window_set_wmclass(GTK_WINDOW( dialog ), "color_wizard", "Screem");

	g_signal_connect( G_OBJECT( dialog ), "delete_event",
			  G_CALLBACK( gtk_widget_hide ), NULL );

	g_object_set_data( G_OBJECT( dialog ), "colour_wizard",colour_wizard );

	return dialog;
}
