/*  Screem:  css-wizard.c
 *
 *  A Wizard for building up CSS selector patterns
 * 
 *  (C) Copyright 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-entry.h>

#include <gtk/gtk.h>

#include <glade/glade.h>

#include <string.h>

#include "screem-plugin.h"

#include "screem-window.h"
#include "screem-editor.h"

#include "libegg/menu/egg-menu.h"

/* per wizard struct */
typedef struct {
	ScreemWindow *window;
	ScreemEditor *editor;

} CSSWizard;


/* keep track of how many wizards we have */
static GList *wizards = NULL;


enum {
	ANYWHERE = 0,
	SOMEWHERE,
	CHILD_OF,
	AFTER
};

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3;

#define NAME "CSS Selector Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG NULL


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );

void css_selector_wizard_display( EggAction *action, gpointer user_data );


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	CSSWizard *wizard;

	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"CSSSelectorWizard\" verb=\"CSSSelectorWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"CSSSelectorWizard\" verb=\"CSSSelectorWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "CSSSelectorWizard", N_( "CSS Selector" ),
	  	GTK_STOCK_EXECUTE, NULL, N_( "A Wizard to help you construct selectors for applying css properties to " ),
	  	G_CALLBACK( css_selector_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;
	
	gchar *label;
	gchar *tip;
	
	wizard = g_new0( CSSWizard, 1 );
	wizard->window = SCREEM_WINDOW( window );
	wizard->editor = SCREEM_EDITOR( editor );
	
	label = g_strdup( entries[ 0 ].label );
	tip = g_strdup( entries[ 0 ].tooltip );

	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( wizard->window->merge ),
						ui, strlen( ui ), NULL );

	g_free( label );
	g_free( tip );

	wizards = g_list_append( wizards, wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	CSSWizard *wizard;

	for( list = wizards; list; list = list->next ) {
		wizard = (CSSWizard*)list->data;
		if( wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* wizard is the one to erase */
	wizards = g_list_remove( wizards, wizard );
	g_free( wizard );
}

/* End of required section */

static void create_tag_menu( CSSWizard *wizard, GladeXML *xml )
{
	GtkWidget *widget;
	GtkWidget *menu;
	GtkWidget *item;
	ScreemPage *page;
	ScreemDTD *dtd;
	const GSList *elements;
	const ScreemDTDElement *elem;
	const gchar *name;
	
	widget = glade_xml_get_widget( xml, "tag_menu" );
	menu = gtk_menu_new();

	page = screem_window_get_document( wizard->window );
	if( page ) {
		dtd = screem_page_get_dtd( page );
		for( elements = screem_dtd_get_elements( dtd );
				elements; elements = elements->next ) {

			elem = (const ScreemDTDElement*)elements->data;
			name = screem_dtd_element_get_name( elem );
			item = gtk_menu_item_new_with_label( name );
			g_object_set_data( G_OBJECT( item ),
					"name", (gpointer)name );
			gtk_widget_show( item );
			gtk_menu_append( GTK_MENU( menu ), item );
		}
		gtk_widget_show( menu );
		gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ),
				menu );
	}
}

static void create_action_menu( CSSWizard *wizard, GladeXML *xml )
{
	GtkWidget *widget;
	GtkWidget *menu;
	GtkWidget *item;
	gint i;
	
	const gchar *actions[] = {
		_( "activated" ), "active",
		_( "hovered" ), "hover",
		_( "focused" ), "focus",
		NULL
	};

	widget = glade_xml_get_widget( xml, "action_menu" );
	menu = gtk_menu_new();
	
	for( i = 0; actions[ i ]; i += 2 ) {

		item = gtk_menu_item_new_with_label( actions[ i ] );
		g_object_set_data( G_OBJECT( item ), "action",
				(gpointer)actions[ i + 1 ] );
		gtk_widget_show( item );
		gtk_menu_append( GTK_MENU( menu ), item );
	}
	gtk_widget_show( menu );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );
}

static void add_page( GtkWidget *widget, const gchar *label_txt )
{
	GladeXML *xml;
	GtkWidget *notebook;
	GtkWidget *label;
	CSSWizard *wizard;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "match_box" );
	notebook = g_object_get_data( G_OBJECT( widget ), "notebook" );
	wizard = g_object_get_data( G_OBJECT( widget ), "wizard" );
	
	xml = glade_xml_new( GLADE_PATH"/css-wizard.glade",
			"match_box", NULL );
	widget = glade_xml_get_widget( xml, "match_box" );
	g_object_set_data( G_OBJECT( widget ),
			"notebook", notebook );
	g_object_set_data( G_OBJECT( widget ), "wizard",
			wizard );

	label = gtk_label_new( label_txt );
	
	create_tag_menu( wizard, xml );
	create_action_menu( wizard, xml );

	gtk_widget_show( widget );
	gtk_widget_show( label );
	gtk_notebook_append_page( GTK_NOTEBOOK( notebook ), widget,
				  label );

	glade_xml_signal_autoconnect( xml );
}

static gchar *create_selector( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *notebook;
	CSSWizard *wizard;
	gint pages;
	gint i;
	gint item;
	GString *pattern;
	gchar *ret;
	
	pattern = g_string_new( " {\n\n}" );
	notebook = g_object_get_data( G_OBJECT( widget ), "notebook" );
	wizard = g_object_get_data( G_OBJECT( widget ), "wizard" );

	pages = gtk_notebook_get_n_pages( GTK_NOTEBOOK( notebook ) );
	for( i = 0; i < pages; ) {
		widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( notebook ), i );
		xml = glade_get_widget_tree( widget );

		widget = glade_xml_get_widget( xml, "action_menu" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* apply action */
			widget = gtk_option_menu_get_menu( GTK_OPTION_MENU( widget ) );
			widget = gtk_menu_get_active( GTK_MENU( widget ) );
			g_string_prepend( pattern,
					g_object_get_data( G_OBJECT( widget ), "action" ) );
			g_string_prepend( pattern, ":" );
		}
		widget = glade_xml_get_widget( xml, "hyperlink_box" );
		if( GTK_WIDGET_VISIBLE( widget ) ) {
			widget = glade_xml_get_widget( xml, 
					"hyperlink_menu" );
			if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
				item = gtk_option_menu_get_history( GTK_OPTION_MENU( widget ) );
				if( item == 0 ) {
					g_string_prepend( pattern,
							":link" );
				} else {
					g_string_prepend( pattern,
							":visited" );
				}
			}
		}
		widget = glade_xml_get_widget( xml, "id_entry" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* match id */
			widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
			g_string_prepend( pattern,
					gtk_entry_get_text( GTK_ENTRY( widget ) ) );
			g_string_prepend_c( pattern, '#' );
		}
		widget = glade_xml_get_widget( xml, "class_entry" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* match class */
			widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
			g_string_prepend( pattern,
					gtk_entry_get_text( GTK_ENTRY( widget ) ) );
			g_string_prepend_c( pattern, '.' );
		}
		widget = glade_xml_get_widget( xml, "tag_menu" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* match tag */
			widget = gtk_option_menu_get_menu( GTK_OPTION_MENU( widget ) );
			widget = gtk_menu_get_active( GTK_MENU( widget ) );
			g_string_prepend( pattern,
					g_object_get_data( G_OBJECT( widget ), "name" ) );
		}

		i ++;
		
		if( i != pages ) {
			widget = glade_xml_get_widget( xml, 
					"location_menu" );
			item = gtk_option_menu_get_history( GTK_OPTION_MENU( widget ) );
			switch( item ) {
				case SOMEWHERE:
					g_string_prepend_c( pattern, ' ' );
					break;
				case CHILD_OF:
					g_string_prepend( pattern, " > " );
					break;
				case AFTER:
					g_string_prepend( pattern, " + " );
					break;
				default:
					break;
			}
		}
	}
	ret = pattern->str;
	g_string_free( pattern, FALSE );

	return ret;
}

void css_selector_wizard_display( EggAction *action, gpointer user_data )
{
	CSSWizard *wizard;
	ScreemPage *page;
	GtkWidget *widget;
	GtkWidget *notebook;
	GladeXML *xml;
	gint button;
	gchar *pattern;
	
	wizard = (CSSWizard*)user_data;
	page = screem_window_get_document( wizard->window );
	
	if( page ) {
		xml = glade_xml_new( GLADE_PATH"/css-wizard.glade",
				"csspattern", NULL );
		widget = glade_xml_get_widget( xml, "match_box" );
		notebook = glade_xml_get_widget( xml, "notebook" );
		g_object_set_data( G_OBJECT( widget ),
				"notebook", notebook );
		g_object_set_data( G_OBJECT( widget ), "wizard",
				wizard );
		create_tag_menu( wizard, xml );
		create_action_menu( wizard, xml );
		
		widget = glade_xml_get_widget( xml, "csspattern" );
		gtk_widget_show( widget );
		glade_xml_signal_autoconnect( xml );

		button = gtk_dialog_run( GTK_DIALOG( widget ) );

		if( button == GTK_RESPONSE_APPLY ) {
			widget = glade_xml_get_widget( xml, 
					"match_box" );
			pattern = create_selector( widget );
			
			screem_editor_insert( wizard->editor, -1,
					pattern );
			g_free( pattern );

		}
		
		widget = glade_xml_get_widget( xml, "csspattern" );
		gtk_widget_destroy( widget );
		g_object_unref( G_OBJECT( xml ) );
	}
}

void css_selector_next_change( GtkOptionMenu *omenu )
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkWidget *notebook;
	gint page;
	gint pages;
	gint item;
	const gchar *labels[] = {
		_( "Rule" ),
		_( "Contained in" ),
		_( "Child of" ),
		_( "After a" )
	};
		
	xml = glade_get_widget_tree( GTK_WIDGET( omenu ) );
	widget = glade_xml_get_widget( xml, "match_box" );
	notebook = g_object_get_data( G_OBJECT( widget ), "notebook" );
	page = gtk_notebook_get_current_page( GTK_NOTEBOOK( notebook ) );

	item = gtk_option_menu_get_history( omenu );
	pages = gtk_notebook_get_n_pages( GTK_NOTEBOOK( notebook ) );

	switch( item ) {
		case ANYWHERE:
			/* remove all subsequent pages */
			while( pages > ( page + 1 ) ) {
				widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( notebook ), page + 1 );
				xml = glade_get_widget_tree( widget );
				gtk_notebook_remove_page( GTK_NOTEBOOK( notebook ), page + 1 );
				g_object_unref( xml );
				pages --;
			}
			break;
		case SOMEWHERE:
		case CHILD_OF:
		case AFTER:
			if( pages == ( page + 1 ) ) {
				add_page( notebook, labels[ item ] );
			} else {
				/* change label */
				widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( notebook ), page + 1 );
				gtk_notebook_set_tab_label_text( GTK_NOTEBOOK( notebook ),
						widget, labels[ item ] );
			}

			break;
	}
}

void css_selector_tag_change( GtkWidget *widget )
{
	GladeXML *xml;
	CSSWizard *wizard;
	const gchar *name;
	ScreemPage *page;
	ScreemDTD *dtd;
	const ScreemDTDElement *elem;
	const ScreemDTDAttribute *attr;
	const GSList *attrs;
	gboolean show;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "match_box" );
	wizard = g_object_get_data( G_OBJECT( widget ), "wizard" );

	widget = glade_xml_get_widget( xml, "tag_menu" );
	show = ( ! GTK_WIDGET_IS_SENSITIVE( widget ) );

	widget = gtk_option_menu_get_menu( GTK_OPTION_MENU( widget ) );
	widget = gtk_menu_get_active( GTK_MENU( widget ) );
	name = (const gchar*)g_object_get_data( G_OBJECT( widget ), 
			"name" );

	page = screem_window_get_document( wizard->window );
	if( page && ! show ) {
		dtd = screem_page_get_dtd( page );
		elem = screem_dtd_valid_element( dtd, name );
		attrs = screem_dtd_element_get_attrs( elem );
		while( attrs ) {
			attr = (const ScreemDTDAttribute*)attrs->data;
			name = screem_dtd_attribute_get_name( attr );
			if( ! g_strcasecmp( "href", name ) ) {
				show = TRUE;
				break;
			}
			attrs = attrs->next;
		}
	}
	widget = glade_xml_get_widget( xml, "hyperlink_box" );
	gtk_widget_set_sensitive( widget, show );
}

