#include <config.h>

#include <string.h>

#include <libgnomeui/gnome-stock-icons.h>
#include <libgnomeui/gnome-color-picker.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnome/gnome-i18n.h>

#include <gtk/gtktoolbar.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtkdialog.h>

#include <glade/glade.h>

#include "screem-plugin.h"
#include "screem-window.h"
#include "screem-editor.h"

#include "libegg/menu/egg-menu.h"

/* per wizard struct */
typedef struct {
	GladeXML *xml;
	ScreemWindow *window;
	ScreemEditor *editor;

	GtkWidget *dialog;
} TableWizard;

/* keep track of how many table wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3;

#define NAME "Table Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG "table"

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );

void table_wizard_clicked( GtkWidget *w, gint button );
void table_wizard_display( EggAction *action, gpointer user_data );
G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	TableWizard *table_wizard;
	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"TableWizard\" verb=\"TableWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"TableWizard\" verb=\"TableWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "TableWizard", "label",
	  	"Screem_Table", NULL, "tip",
	  	G_CALLBACK( table_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;
	
	gchar *label;
	gchar *tip;

	table_wizard = g_new0( TableWizard, 1 );
	table_wizard->xml = glade_xml_new( GLADE_PATH"/tablewizard.glade", 
					   "tablewizard", NULL );
	table_wizard->window = SCREEM_WINDOW( window );
	table_wizard->editor = SCREEM_EDITOR( editor );

	label = g_strdup( _( "Table Wizard" ) );
	tip = g_strdup( _( "Insert a Table" ) );
	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = table_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( table_wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( table_wizard->window->merge ),
						ui, strlen( ui ), NULL );
	g_free( label );
	g_free( tip );
	
	table_wizard->dialog = glade_xml_get_widget( table_wizard->xml, 
						    "tablewizard" );

	g_object_set_data( G_OBJECT( table_wizard->dialog ),
			   "table_wizard", table_wizard );

	glade_xml_signal_autoconnect( table_wizard->xml );

	wizards = g_list_append( wizards, table_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	TableWizard *table_wizard;

	for( list = wizards; list; list = list->next ) {
		table_wizard = (TableWizard*)list->data;
		if( table_wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* table_wizard is the one to erase */
	wizards = g_list_remove( wizards, table_wizard );
	g_object_unref( G_OBJECT( table_wizard->xml ) );
	g_free( table_wizard );
}

/* End of required section */

void table_wizard_clicked( GtkWidget *w, gint button )
{
	gint c;
	gint pos;
	GtkWidget *widget;
	GtkWidget *widget2;
	gint border = 0;
	GString *text;
	gint cellSpacing;
	gint cellPadding;
	gint width_value;
	guint8 r, g, b, a;
	gint rows;
	gint cols;
	gboolean caption;

	GladeXML *xml;
	TableWizard *wizard;
	GList *list;
	xml = glade_get_widget_tree( w );

	wizard = NULL;
	for( list = wizards; list; list = list->next ) {
		wizard = (TableWizard*)list->data;
		if( wizard->xml == xml )
			break;
		wizard = NULL;
	}
	g_assert( wizard );

    	if( button == GTK_RESPONSE_APPLY || button == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( 	wizard->xml, "caption" );
		caption = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
		widget = glade_xml_get_widget( 	wizard->xml, "border" );
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON( widget ))) {
			widget = glade_xml_get_widget( 	wizard->xml, "border_width" );
			border = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON ( widget ) );
		}

		text = g_string_new( "" );
		g_string_sprintf( text, "<table border=\"%d\"", border );

		widget = glade_xml_get_widget( wizard->xml, "left" );
		widget2 = glade_xml_get_widget( wizard->xml, "centre" );

		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(widget) ) )
			g_string_append( text, " align=\"left\"" );
		else if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget2) ) )
			g_string_append( text, " align=\"center\"" );
		else
			g_string_append( text, " align=\"right\"" );

		widget = glade_xml_get_widget( wizard->xml, "spacing" );
	        cellSpacing = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));
		widget = glade_xml_get_widget( wizard->xml, "padding" );
	        cellPadding = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));
		g_string_append_printf( text, 
				   " cellspacing=\"%d\" cellpadding=\"%d\"",
				   cellSpacing, cellPadding );
	
		widget = glade_xml_get_widget( wizard->xml, "rows" );
	        rows = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));
		widget = glade_xml_get_widget( wizard->xml, "cols" );
	        cols = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(widget));

		widget = glade_xml_get_widget( wizard->xml, "width" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			widget = glade_xml_get_widget( wizard->xml, "table_width" );
			width_value = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
			g_string_append_printf( text, " width=\"%d",
					   width_value );
					
			widget = glade_xml_get_widget( wizard->xml, "percent" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) )
				g_string_append_c( text, '%' );
			g_string_append_c( text, '"' );
		}

		widget = glade_xml_get_widget( wizard->xml, "use_colour" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			widget = glade_xml_get_widget( wizard->xml, "colour" );
			gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ), &r, &g, &b, &a);
			g_string_append_printf( text, 
					   " bgcolor=\"#%.2x%.2x%.2x\"",r,g,b);
		}

		widget = glade_xml_get_widget( wizard->xml, "use_image" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			widget = glade_xml_get_widget( wizard->xml, 
						       "image_entry" );
			widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
			g_string_append_printf( text, " background=\"%s\"",
					   gtk_entry_get_text( GTK_ENTRY( widget ) ) );
		}

		widget = glade_xml_get_widget( wizard->xml, "summary" );
		if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
			widget = glade_xml_get_widget( wizard->xml,
						       "summary_entry" );
			widget = gnome_entry_gtk_entry( GNOME_ENTRY(widget) );
			g_string_append_printf( text, " summary=\"%s\"",
					   gtk_entry_get_text( GTK_ENTRY( widget ) ) );
		}

		g_string_append_c( text, '>' );

		widget = glade_xml_get_widget( wizard->xml, "above" );
		if( caption &&
		    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ) {
			g_string_append( text,  "\n<caption> </caption>" );
			caption = FALSE;
		}

		for( ; rows; rows -- ) {
			g_string_append( text, "\n<tr>\n" );
			for( c = 0; c < cols; c ++ )
				g_string_append( text, "\n<td> </td>" );
			g_string_append( text, "\n</tr>" );
		}

		if( caption )
			g_string_append( text,  "\n<caption> </caption>" );

		g_string_append( text, "\n</table>" );

		pos = screem_editor_get_pos( wizard->editor );
		screem_editor_insert( wizard->editor, pos, text->str );
		screem_editor_set_pos( wizard->editor, pos + strlen( text->str ) );

		g_string_free( text, TRUE );
	}

	if( button == GTK_RESPONSE_OK || button == GTK_RESPONSE_CLOSE )
		gtk_widget_hide( wizard->dialog );
}

void table_wizard_display( EggAction *action, gpointer user_data )
{
	TableWizard *wizard;
	ScreemPage *page;

	wizard = (TableWizard*)user_data;

        page = screem_window_get_document( wizard->window );
        
	if( ! page )
		return;

	gtk_widget_show_all( wizard->dialog );
	gdk_window_raise( wizard->dialog->window );
}
