/*  Screem:  uploadWizard.c,
 *  This file handles the uploading of a site
 *
 *  This file Copyright (C) 1999 Lee Mallabone.
 *  Original 'sitecopy' by Joe Orton.
 *  Sitecopy functionality integrated into transfer.c by Lee Mallabone.
 *
 *  TODO:
 *  add fetch site functionality
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>

#include <errno.h>
#include <gmodule.h>
#include <gnome.h>
#include <unistd.h>
#include <stdio.h>
#include <pthread.h>

#include <semaphore.h>
#include <setjmp.h>
#include <signal.h>

#include <sys/stat.h>

#include <glade/glade.h>

/* add sitecopy functionality */
#include "common.h"
#include "sites.h"
#include "rcfile.h"
#include "frontend.h"
#include "ftp.h"

/* screem stuff */
#include "screem-plugin.h"
#include "screem-window.h"

#include "uploadWizard.h"

#include "screem-site.h"

#include "libegg/menu/egg-menu.h"

/* per wizard struct */
typedef struct {
	ScreemWindow *window;

	GladeXML *xml;
	GladeXML *exml;

	GtkWidget *dialog;

	struct site *site;

	volatile int in_critical_section;
	volatile int want_abort;
	sigjmp_buf abort_buf;

	sem_t *update_semaphore;
	pthread_t update_tid;
	gboolean prompting;
	gboolean closing;

	gint button;

	float upload_total;
	float uploaded_bytes;

	EggAction *action;
} UploadWizard;

/* keep track of how many link wizards we have */
static GList *wizards = NULL;


/* the wizard currently being used */
static UploadWizard *currentWiz = NULL;

/* can't allow 2 wizards to run at once */
static GMutex *wizard_lock;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3;

#define NAME "Upload Wizard (sitecopy)"
#define AUTHOR "Lee Mallabone, David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG "sitecopy_hack"

int my_abortable_transfer_wrapper(struct site *site,
                                  enum site_op operation );
void handle_abort( int sig );
void do_abort( UploadWizard *wizard );
void abort_site_update(GtkWidget * button, gpointer data);
int fe_gtk_question(char *question, GnomeReplyCallback yes_action);
void fe_login_clicked( GtkDialog *dialog, gint number );
void fe_login_close( GtkDialog *dialog );

static gboolean create_sitecopy_directory( void );
static void destroy_fnlists( struct site *site );
static void destroy_fnlist( struct fnlist *list );


/* the drivers */
extern const struct proto_driver ftp_driver;
extern const struct proto_driver dav_driver;
extern const struct proto_driver rsh_driver;
extern const struct proto_driver local_driver;


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );

static struct site* screem_site_to_sitecopy_site( ScreemSite *site, 
						  gboolean show_alert );


static void upload_wizard_error( const gchar *message );
static void upload_wizard_message( const gchar *message );
static void upload_site( EggAction *action, gpointer user_data );
static gboolean should_init( void );
static gboolean verify_site( struct site *site );

static void *update_thread( void *arg );

void upload_button_clicked( GtkWidget *button, gpointer user_data );
void error_button_clicked( GtkButton *button, gpointer user_data );
void close_button_clicked(GtkButton *button, gpointer user_data );

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	if(!wizard_lock) {
		wizard_lock = g_mutex_new();
	}

	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	UploadWizard *upload_wizard;
	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Tools\" verb=\"Tools\">\
<menuitem name=\"UploadWizard\" verb=\"UploadWizard\"/>\
</submenu>\
</menu>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "UploadWizard", "label",
	  	GTK_STOCK_JUMP_TO, NULL, "tip",
	  	G_CALLBACK( upload_site ), NULL },
	};
	EggActionGroupEntry *entry;

	gchar *label;
	gchar *tip;

	GtkSizeGroup *group;
	GtkWidget *widget;
	
	upload_wizard = g_new0( UploadWizard, 1 );
	upload_wizard->window = SCREEM_WINDOW( window );
	
	label = g_strdup( _( "Upload" ) );
	tip = g_strdup( _( "Upload the Current Site or Page" ) );
	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = upload_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( upload_wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( upload_wizard->window->merge ),
						ui, strlen( ui ), NULL );
	g_free( label );
	g_free( tip );

	upload_wizard->action = egg_action_group_get_action( EGG_ACTION_GROUP( upload_wizard->window->action_group ), "UploadWizard" );
	
	upload_wizard->xml = glade_xml_new( GLADE_PATH, "update_dialog",
					    NULL );
	upload_wizard->dialog = glade_xml_get_widget( upload_wizard->xml,
						      "update_dialog" );
	g_object_set_data( G_OBJECT( upload_wizard->dialog ),
			     "wizard", upload_wizard );

	group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );
	
	widget = glade_xml_get_widget( upload_wizard->xml, "host_label" );
	gtk_size_group_add_widget( group, widget );
	widget = glade_xml_get_widget( upload_wizard->xml, "status_label" );
	gtk_size_group_add_widget( group, widget );
	widget = glade_xml_get_widget( upload_wizard->xml, "current_label" );
	gtk_size_group_add_widget( group, widget );
	widget = glade_xml_get_widget( upload_wizard->xml, "total_label" );
	gtk_size_group_add_widget( group, widget );
	
	glade_xml_signal_autoconnect( upload_wizard->xml );

	wizards = g_list_append( wizards, upload_wizard );

	fe_initialize();

	upload_wizard->update_semaphore = g_new( sem_t, 1 );
	sem_init(upload_wizard->update_semaphore, 0, 0);

	upload_wizard->closing = FALSE;
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	UploadWizard *wizard;

	for( list = wizards; list; list = list->next ) {
		wizard = (UploadWizard*)list->data;
		if( wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* link_wizard is the one to erase */
	wizards = g_list_remove( wizards, wizard );
	g_free( wizard );
}


static struct site* screem_site_to_sitecopy_site( ScreemSite *ssite,
						  gboolean show_alert )
{
	struct site *sitecopy_site;
	struct site_host *host;
	const gchar *string;
	GList *list;
	struct fnlist *exclude;
	struct fnlist *ignores;
	struct fnlist *ascii;

	const gchar* home;

	gchar *temp;

	sitecopy_site = g_new0( struct site, 1 );

	host = &sitecopy_site->server;

	/* the site name */
	string = screem_site_get_name( ssite );
	sitecopy_site->name = g_strdup( string );

	string = screem_site_get_remote_url( ssite );
	host->hostname = g_strdup( string );
	host->username = g_strdup( screem_site_get_remote_user( ssite ) );
	host->password = g_strdup( screem_site_get_remote_pass( ssite ) );

	if( host->username && strlen( host->username ) == 0 )
		host->username = NULL;

	if( host->password && strlen( host->password ) == 0 )
		host->password = NULL;

	if( host->hostname ) {
		if( ( temp = strchr( host->hostname, ':' ) ) ) {
			host->port = atoi( temp + 1 );
			temp = g_strndup( host->hostname, 
					 temp - host->hostname );
			host->hostname = temp;
			/* FIXME: memory leak */
		}
	}

	/* the view url */
	string = screem_site_get_http_url( ssite );
	sitecopy_site->url = g_strdup( string );

	/* get transfer options */
	switch( screem_site_get_remote_method( ssite ) ) {
	case FTP:
		sitecopy_site->protocol = siteproto_ftp;
		sitecopy_site->proto_string = "ftp";
		sitecopy_site->driver = &ftp_driver;
		break;
	case WEBDAV:
		sitecopy_site->protocol = siteproto_dav;
		sitecopy_site->proto_string = "dav";
		sitecopy_site->driver = &dav_driver;
		break;
	case RSH:
		sitecopy_site->protocol = siteproto_rsh;
		sitecopy_site->proto_string = "rsh";
		sitecopy_site->driver = &rsh_driver;
		sitecopy_site->rsh_cmd = "rsh";
		sitecopy_site->rcp_cmd = "rcp";
		break;
	case SSH:
		sitecopy_site->protocol = siteproto_rsh;
                sitecopy_site->proto_string = "ssh";
                sitecopy_site->driver = &rsh_driver;
		sitecopy_site->rsh_cmd = "ssh";
		sitecopy_site->rcp_cmd = "scp";
                break;
	case LOCAL:
	default:
		sitecopy_site->protocol = siteproto_local;
		sitecopy_site->proto_string = "local";
		sitecopy_site->driver = &local_driver;
		break;
	}

	switch( screem_site_get_permissions( ssite ) ) {
	case PERMS_EXEC:
		sitecopy_site->perms = sitep_exec;
		break;
	case PERMS_IGNORE:
		sitecopy_site->perms = sitep_ignore;
		break;
	case PERMS_ALL:
		sitecopy_site->perms = sitep_all;
		break;
	}

	switch( screem_site_get_symlinks( ssite ) ) {
	case SYM_IGNORE:
		sitecopy_site->symlinks = sitesym_ignore;
		break;
	case SYM_FOLLOW:
		sitecopy_site->symlinks = sitesym_follow;
		break;
	case SYM_MAINTAIN:
		sitecopy_site->symlinks = sitesym_maintain;
		break;
	}

	/* the remote path */
	string = screem_site_get_remote_path( ssite );
	sitecopy_site->remote_root_user = g_strdup( string );

	sitecopy_site->remote_isrel = ( string[ 0 ] == '~' );
	if( string[ 0 ] != '~' && string[ 0 ] != G_DIR_SEPARATOR ) {
		/* must start with ~ or /  */
		if( show_alert )
			upload_wizard_error( _("Remote path must being with ~ or /") );
		g_free( sitecopy_site );
		return NULL;
	}

	if( screem_site_get_remote_method( ssite ) == LOCAL ) {
		host->hostname = g_strdup( string );
	}


	/* the local pathname */
	string = screem_site_get_pathname( ssite );

	/* Screem maybe using a URI as the pathname,
	   Sitecopy can't deal with this, so we need to strip the method/hostname off,
	   although if the site isn't a local method we can't really use sitecopy */
	{
		GnomeVFSURI *uri;
		gboolean islocal;
		gchar *temp;

		uri = gnome_vfs_uri_new( string );
		islocal = gnome_vfs_uri_is_local( uri );

		gnome_vfs_uri_unref( uri );
		
		if( ( ! islocal ) && strncmp( "file:", string,
					strlen( "file:" ) ) ) {
			if( show_alert )
				upload_wizard_error( _( "The upload wizard can only be used with local sites" ) );
			g_free( sitecopy_site );
			return NULL;
		}

		temp = gnome_vfs_get_local_path_from_uri( string );
		if( ! temp ) {
			string = screem_site_get_pathname( ssite );
		} else {
			/* memleak */
			string = temp;
		}
	}

	sitecopy_site->local_root_user = g_strdup( string );
	sitecopy_site->local_isrel = ( string[ 0 ] == '~' );
	if( string[ 0 ] != '~' && string[ 0 ] != G_DIR_SEPARATOR ) {
		/* should never occur, screem should not allow this
		   case to happen */
		if( show_alert )
			upload_wizard_error( _("Local path must being with ~ or /") );
		g_free( sitecopy_site );
		return NULL;
	}

	string = screem_site_get_pathname( ssite );
	
	/* misc settings */
	sitecopy_site->ftp_pasv_mode = screem_site_get_passive_ftp( ssite );
	sitecopy_site->nodelete = screem_site_get_no_delete( ssite );
	sitecopy_site->checkmoved = screem_site_get_check_moved( ssite );
	sitecopy_site->nooverwrite = screem_site_get_no_overwrite( ssite );

	/* setup excludes */
	exclude = g_new0( struct fnlist, 1 );
	exclude->haspath = FALSE;
	exclude->pattern = "*.screem";
	exclude->prev = g_new0( struct fnlist, 1 );
	exclude->prev->next = exclude;
	exclude = exclude->prev;
	exclude->haspath = FALSE;
	exclude->pattern = ".project.screem";
	for(list = screem_site_get_excludes( ssite ); list; list = list->next){
		struct fnlist *newexclude;

		newexclude = g_new0( struct fnlist, 1 );

		if( ! strncmp( "glob:", (gchar*)list->data, 
				strlen( "glob:" ) ) ) {
			newexclude->haspath = FALSE;
			newexclude->pattern = list->data + strlen( "glob:" );
		} else if( ! strncmp( "regex:", (gchar*)list->data, 
				strlen( "regex:" ) ) ) {
			newexclude->haspath = FALSE;
			newexclude->pattern = list->data + strlen( "regex: " );

			/* don't support this yet */
			g_free( newexclude );
			newexclude = NULL;
			g_warning( "regex: patterns not supported yet\n" );
		} else {
			newexclude->haspath = TRUE;
			newexclude->pattern = ((gchar*)list->data) + strlen( string );
		}

		if( newexclude ) {
			exclude->prev = newexclude;
			exclude->prev->next = exclude;
			exclude = exclude->prev;
		}
	}
	sitecopy_site->excludes = exclude;
	
	/* setup ignores */
	for( ignores = NULL, list = screem_site_get_ignores( ssite ); list;
	     list = list->next ) {
		struct fnlist *newignore;

		newignore = g_new0( struct fnlist, 1 );
		if( ! strncmp( "glob:", (gchar*)list->data, 
				strlen( "glob:" ) ) ) {
			newignore->haspath = FALSE;
			newignore->pattern = list->data + strlen( "glob:" );
		} else if( ! strncmp( "regex:", (gchar*)list->data, 
				strlen( "regex:" ) ) ) {
			newignore->haspath = FALSE;
			newignore->pattern = list->data + strlen( "regex: " );

			/* don't support this yet */
			g_free( newignore );
			newignore = NULL;
			g_warning( "regex: patterns not supported yet\n" );
		} else {
			newignore->haspath = TRUE;
			newignore->pattern = ((gchar*)list->data) + strlen( string );
		}

		if( newignore ) {
			if( ! ignores ) {
				ignores = newignore;
			} else {
				ignores->prev = newignore;
				ignores->prev->next = ignores;
				ignores = ignores->prev;
			}
		}
	}
	sitecopy_site->ignores = ignores;

	/* setup asciis */
	for( ascii = NULL, list = screem_site_get_asciis( ssite ); list;
	     list = list->next ) {
		struct fnlist *newascii;

		newascii = g_new0( struct fnlist, 1 );
		if( ! strncmp( "glob:", (gchar*)list->data, 
				strlen( "glob:" ) ) ) {
			newascii->haspath = FALSE;
			newascii->pattern = list->data + strlen( "glob:" );
		} else if( ! strncmp( "regex:", (gchar*)list->data, 
				strlen( "regex:" ) ) ) {
			newascii->haspath = FALSE;
			newascii->pattern = list->data + strlen( "regex: " );

			/* don't support this yet */
			g_free( newascii );
			newascii = NULL;
			g_warning( "regex: patterns not supported yet\n" );
		} else {
			newascii->haspath = TRUE;
			newascii->pattern = ((gchar*)list->data) + strlen( string );
		}

		if( newascii ) {
			if( ! ascii ) {
				ascii = newascii;
			} else {
				ascii->prev = newascii;
				ascii->prev->next = ascii;
				ascii = ascii->prev;
			}
		}
	}
	sitecopy_site->asciis = ascii;

	sitecopy_site->use_this = TRUE;

	/* FIXME: this should be an option */
	sitecopy_site->state_method = state_timesize;
	sitecopy_site->stored_state_method = state_timesize;

	/* set info file path */
	home = g_get_home_dir();
       	sitecopy_site->infofile = g_strconcat( home, G_DIR_SEPARATOR_S, 
					       ".sitecopy",
					       G_DIR_SEPARATOR_S, 
					       sitecopy_site->name, NULL );

	return sitecopy_site;
}

static void upload_page(ScreemPage *page)
{
	GladeXML *sxml;
	GtkWidget *dialog;
	GtkWidget *widget;
	gint button;

	const gchar *file;
	gchar *rfile;

	ftp_session *session;
	struct site_host *server;

	file = g_basename( screem_page_get_pathname( page ) );

	sxml = glade_xml_new( GLADE_PATH, "single_upload", NULL );
	dialog = glade_xml_get_widget( sxml, "single_upload" );
	
	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );
	
	if( button == 0 ) {
		server = g_new0( struct site_host, 1 );
	
		widget = glade_xml_get_widget( sxml, "uri" );
		server->hostname = g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
		server->port = 21;
		widget = glade_xml_get_widget( sxml, "user" );
		server->username = g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );
		widget = glade_xml_get_widget( sxml, "pass" );
		server->password = g_strdup( gtk_entry_get_text( GTK_ENTRY( widget ) ) );

		widget = glade_xml_get_widget( sxml, "path" );
		
		/* do upload, FIXME: handle reporting of errors +
		   give the user some feed back */
		session = ftp_init();
		ftp_set_server( session, server );
		
		rfile = g_strconcat( gtk_entry_get_text( GTK_ENTRY( widget ) ),
				     "/", file, NULL );
		ftp_put( session, file, rfile, TRUE );
		ftp_finish( session );
		g_free( rfile );
		g_free( server );
	}

	gtk_widget_destroy( dialog );
}

static void upload_site( EggAction *action, gpointer user_data )
{
	UploadWizard *wizard;
	struct stat s;
	ScreemSite *site;
	ScreemPage *page;
	GtkWidget *widget;
	
	wizard = (UploadWizard*)user_data;

	site = screem_window_get_current( wizard->window );

	if( screem_site_get_fake_flag( site ) ) {
		/* upload a single page, we can use the ftp code
		   in sitecopy */
		page = screem_window_get_document( wizard->window );

		upload_page( page );
		return;
	}

	/* create ~/.sitecopy if necessary */
	if( ! create_sitecopy_directory() ) {
		return;
	}

	g_mutex_lock(wizard_lock);
	
	currentWiz = wizard;

	if( ! ( wizard->site = screem_site_to_sitecopy_site( site,TRUE ) ) ) {
		g_mutex_unlock(wizard_lock);
		return;
	}

	/* verify that the site is setup properly */
	if( ! verify_site( wizard->site ) ) {
		g_free( wizard->site->infofile );
		g_free( wizard->site );
		g_mutex_unlock(wizard_lock);
		return;
	}

	widget = glade_xml_get_widget( wizard->xml, "host_text" );
	gtk_label_set_text( GTK_LABEL( widget ), wizard->site->server.hostname );
	widget = glade_xml_get_widget( wizard->xml, "status_text" );
	gtk_label_set_text( GTK_LABEL( widget ), _( "Idle" ) );
	widget = glade_xml_get_widget( wizard->xml, "main_progressbar" );
	gtk_progress_set_percentage( GTK_PROGRESS( widget ), 0.0 );
	widget = glade_xml_get_widget( wizard->xml, "job_progressbar" );
	gtk_progress_set_percentage( GTK_PROGRESS( widget ), 0.0 );
	widget = glade_xml_get_widget( wizard->xml, "keep_going_button" );
	gtk_widget_set_sensitive( widget, TRUE );
	
	/* info file exist? */
	if( stat( wizard->site->infofile, &s ) < 0 ) {
		/* Displays a dialog asking if the user wants to init or
		   catchup their website. */
		site_write_stored_state( wizard->site );

		if( should_init() ) {
			site_initialize( wizard->site );
		} else if ( (site_readfiles (wizard->site)) < 0 ) {
			upload_wizard_error(_("Could not retrieve information about your local files."));
			g_free( wizard->site->infofile );
			g_free( wizard->site );
			g_mutex_unlock(wizard_lock);
			return;
		} else {
			site_catchup( wizard->site );
			site_write_stored_state( wizard->site );
			/* If the site is up to date, we really don't need 
			   to upload it! */
			g_free( wizard->site->infofile );
			g_free( wizard->site );
			g_mutex_unlock(wizard_lock);
			return;
		}
	}

	wizard->site->files = NULL;
	wizard->site->files_tail = NULL;
	
	g_object_set( G_OBJECT( wizard->action ), "sensitive", FALSE, NULL );
	
	if( site_readfiles( wizard->site ) < 0 ) {
		/* couldn't get info on local site files */
		upload_wizard_error( _("Couldn't retrieve information about local files") );
		g_free( wizard->site->infofile );
		g_free( wizard->site );
		g_object_set( G_OBJECT( wizard->action ), "sensitive", TRUE, NULL );
		g_mutex_unlock(wizard_lock);
		return;
	} else if( ! wizard->site->remote_is_different ) {
		/* nothing needs uploading */
		upload_wizard_message( _( "The remote site is already uptodate" ) );
		g_free( wizard->site->infofile );
		g_free( wizard->site );
		g_object_set( G_OBJECT( wizard->action ), "sensitive", TRUE, NULL );
		g_mutex_unlock(wizard_lock);
		return;
	}

	/* reset transfer counter */
	wizard->upload_total = wizard->site->totalnew + 
		wizard->site->totalchanged;
	g_print( "upload_total: %f", wizard->upload_total );
	wizard->uploaded_bytes = 0.0;

	sem_init(wizard->update_semaphore, 0, 0);
	wizard->closing = FALSE;

	/* claim the wizard lock */

	pthread_create(&wizard->update_tid, NULL, 
		       update_thread, wizard);

       	/* all setup, display upload dialog */
	gtk_widget_show_all( wizard->dialog );
	gdk_window_raise( wizard->dialog->window );	

	widget = glade_xml_get_widget( wizard->xml, "upload_button" );
	gtk_widget_show( widget );
	widget = glade_xml_get_widget( wizard->xml, "close_button" );
	gtk_widget_show( widget );
	widget = glade_xml_get_widget( wizard->xml, "stop_button" );
	gtk_widget_hide( widget );

}

static gboolean verify_site( struct site *site )
{
	gint ret;
	gchar *message = NULL;
	
	switch( ( ret = rcfile_verify( site ) ) ) {
	case SITE_NONAME:
		message =  _("No site name given");
		break;
	case SITE_NOSERVER:
		message = _("No server name given.");
		break;
	case SITE_NOREMOTEDIR:
		message = _("No remote directory given.");
		break;
	case SITE_NOLOCALDIR:
		message = _("No local directory given.");
		break;
	case SITE_ACCESSLOCALDIR:
		message = _("The local dir couldn't be accessed.");
		break;
	case SITE_INVALIDPORT:
		message = _("There was a problem with the port number for this site.");
		break;
	case SITE_NOMAINTAIN:
		message = _("Sorry, the symbolic links option you chose is not supported\nby this transfer protocol. Please choose another option.");
		break;
	case SITE_NOREMOTEREL:
		if( site->remote_isrel )
			message = _("This protocol does not support relative remote directories.");
		break;
	case SITE_NOPERMS:
		message = _("The protocol you are attempting to use does\nnot currently support maintaining permissions.");
        break;

	case SITE_NOLOCALREL:
		message = _("The local dir is invalid.");
		break;
	case 0:
		/* Site integrity is okay. */
		break;
	default:
		message = _("There was an undetermined problem verifying the correctness of your site definition. Please report this to the maintainer.");
		g_print( "ERROR: %i\n", ret );
		break;
	}

	if( message ) {
		upload_wizard_error( message );
	}
	
	return ( ret == 0 );
}


static gboolean should_init()
{
	GtkWidget *state_question;
	gint button;
	
	state_question = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL, 
						 GTK_MESSAGE_QUESTION,
						 GTK_BUTTONS_YES_NO,
						 _("This appears to be the first time you have attempted to upload\nthis website.  Does the site already exist on the server?") );

	button = gtk_dialog_run( GTK_DIALOG( state_question ) );

	gtk_widget_destroy( state_question );

	return ( button == GTK_RESPONSE_NO );
}

static void *update_thread( void *arg ) 
{
	UploadWizard *wizard = (UploadWizard*)arg;
	int ret;
	gboolean tmp;
	GtkWidget *keep_going_button;
	GtkWidget *begin_button;
	GtkWidget *stop_button;
	GtkWidget *close_button;
	GtkWidget *update_dialog;
	GtkWidget *widget;
	const gchar *message;
	
	pthread_detach(wizard->update_tid);

	for( ;; ) {
		/* sleep */
		sem_wait( wizard->update_semaphore );

		if( wizard->closing )
			break;
		
		gdk_threads_enter();
		keep_going_button = 
			glade_xml_get_widget( wizard->xml, 
					      "keep_going_button" );
		begin_button = glade_xml_get_widget( wizard->xml,
						     "upload_button" );
		close_button = glade_xml_get_widget( wizard->xml,
						     "close_button" );
		stop_button = glade_xml_get_widget( wizard->xml,
						    "stop_button" );
		update_dialog = glade_xml_get_widget( wizard->xml,
						      "update_dialog" );

		tmp = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( keep_going_button ) );

		gtk_widget_set_sensitive(begin_button, FALSE);
		gtk_widget_set_sensitive(keep_going_button, FALSE);
		gdk_threads_leave();

		ret = my_abortable_transfer_wrapper(wizard->site, 
						    site_op_update);

		gdk_threads_enter();

		message = NULL;
		
		switch( ret ) {
		case SITE_OK:
			/* be sure to add some 'free' commands in here */
			site_write_stored_state( wizard->site );
			/*rescan_selected(1);*/
			break;
		case SITE_AUTH:
			message = _("There was a problem authenticating with the remote server.");
			break;
		case SITE_LOOKUP:
			message = _("Could not resolve the remote site's hostname.");
			break;
		case SITE_CONNECT:
			message = _("Could not establish a connection to the remote site.");
			break;
		case SITE_FAILED:
			message = _("Update Failed (Authentication problems)" );
			break;
		case SITE_ABORTED:
			message = _("Update was aborted." );
			break;
		case SITE_ERRORS:
			/* replace this with a custom dialog containing the errors, a la Xsitecopy */
			message = _("There were errors.");
			break;
		}
		if( message ) {
			upload_wizard_error( message );
		}
		gtk_widget_hide (stop_button);
		gtk_widget_show (close_button); 
		gtk_widget_set_sensitive(close_button, TRUE);
		gtk_widget_set_sensitive(begin_button, TRUE);
		gtk_widget_set_sensitive(keep_going_button, TRUE );
		widget = glade_xml_get_widget( wizard->xml, "status_text" );
		gtk_label_set_text( GTK_LABEL( widget ), _( "Idle" ) );
		gtk_window_set_modal(GTK_WINDOW(update_dialog), FALSE);
		gdk_threads_leave();
	}
	g_object_set( G_OBJECT( wizard->action ), "sensitive", TRUE, NULL );
	g_mutex_unlock( wizard_lock );

	return NULL;
}

/* GUI part */

static void upload_wizard_error( const gchar *message )
{
	gchar *temp;

	temp = g_strdup_printf( _( "Upload Wizard: %s" ), message );
	
	screem_window_show_error( SCREEM_WINDOW( currentWiz->window ),
				    temp );

	g_free( temp );	
}

static void upload_wizard_message( const gchar *message )
{
	gchar *temp;

	temp = g_strdup_printf( _( "Upload Wizard: %s" ), message );
	
	screem_window_show_message( SCREEM_WINDOW( currentWiz->window ),
				    temp );

	g_free( temp );
}

/* callbacks */
void upload_button_clicked( GtkWidget *button, gpointer user_data )
{
	/* We don't want to begin again. Also, there doesn't seem to be any way
	 * to explictly terminate a pthread, so we don't want the dialog to be
	 * closed while the upload would carry on in the background.
	 * We also don't want the user changing files while we're doing an 
	 * upload, so we make the upload dialog modal.
	 */
	GladeXML *xml;
	UploadWizard *wizard;
	
	GtkWidget *update_dialog;
	GtkWidget *close_button;
	GtkWidget *stop_button;
	GtkWidget *upload_button;

	xml = glade_get_widget_tree( button );
	update_dialog = glade_xml_get_widget( xml, "update_dialog" );

	wizard = (UploadWizard*)g_object_get_data( G_OBJECT(update_dialog),
						     "wizard" );

	upload_button = glade_xml_get_widget( xml, "upload_button" );
	close_button = glade_xml_get_widget( xml, "close_button" );
	stop_button = glade_xml_get_widget( xml, "stop_button" );

	gtk_widget_hide( close_button );
	gtk_widget_show( stop_button );
	gtk_widget_set_sensitive(upload_button, FALSE);
	gtk_window_set_modal( GTK_WINDOW( update_dialog ), TRUE );
	sem_post(wizard->update_semaphore);
}

void close_button_clicked(GtkButton *button, gpointer user_data )
{
	GladeXML *xml;
	GtkWidget *update_dialog;
	UploadWizard *wizard;

	xml = glade_get_widget_tree( GTK_WIDGET( button ) );
	update_dialog = glade_xml_get_widget( xml, "update_dialog" );
	wizard = (UploadWizard*)g_object_get_data( G_OBJECT(update_dialog),
						     "wizard" );

	wizard->closing = TRUE;
	sem_post( wizard->update_semaphore );
       
	/* the update thread will terminate itself as the for loop
	   is broken out of */
	
	/* Free the memory created in site_readfiles. */
	site_destroy( wizard->site );

	/* clean up */
	g_free( wizard->site );

	gtk_widget_hide( wizard->dialog );
}

/* Signal handler, ok how are we supposed to get the wizard struct here? */
void handle_abort( int sig ) 
{
	if( currentWiz->in_critical_section ) {
		/* Can't abort now, so remember we want to for later */
		currentWiz->want_abort = 1;
	} else {
		do_abort( currentWiz );
	}
}

void abort_site_update(GtkWidget * button, gpointer data)
{
	GladeXML *xml;
	GtkWidget *update_dialog;
	UploadWizard *wizard;
	GtkWidget *status_label;

	xml = glade_get_widget_tree( GTK_WIDGET( button ) );
	update_dialog = glade_xml_get_widget( xml, "update_dialog" );
	wizard = (UploadWizard*)g_object_get_data( G_OBJECT(update_dialog),
						   "wizard" );

	status_label = glade_xml_get_widget( xml, "status_text" );

	gtk_label_set(GTK_LABEL(status_label), "Aborting...");
	
	/* Free the memory created in site_readfiles. 
	 *  hmmm, should we be doing this at this point? - David
	*/
	site_destroy( wizard->site );

	pthread_kill(wizard->update_tid, SIGUSR1);
}


int my_abortable_transfer_wrapper(struct site *site,
                                  enum site_op operation ) 
{
	int ret;

	/* lets find site in our list of wizards */
	GList *list;
	UploadWizard *wizard = NULL;

	for( list = wizards; list; list = list->next ) {
		wizard = (UploadWizard*)list->data;

		if( site == wizard->site )
			break;
		wizard = NULL;
	}
		
	/* we must have a wizard which owns the site
	   or something strange is going on */
	g_assert( wizard != NULL );

	signal( SIGUSR1, handle_abort );
	if( !sigsetjmp( wizard->abort_buf, 1 ) ) {
		/* Normal execution */
		switch (operation) {
			case site_op_update:
				ret = site_update(site);
				break;
			case site_op_fetch:
				ret = site_fetch(site);
				break;
			case site_op_resync:
				ret = site_synch(site);
				break;
			default:
				ret = 0;
				break;
			}
	} else {
		/* The update was aborted */
		ret = SITE_ABORTED;
	}
	signal( SIGUSR1, SIG_IGN );

	return ret;
}

/* Actually abort the update */
void do_abort( UploadWizard *wizard ) {
	wizard->want_abort = 0;
	siglongjmp( wizard->abort_buf, 1 );
}



/********************************************************************/
/**** Implementations of all the functions defined in frontend.h ****/
/**** These functions should be considered 'private' and never   ****/
/**** be called directly.                                        ****/
/********************************************************************/

int fe_gtk_question(char *question, GnomeReplyCallback yes_action)
{
    //GtkWidget *question_window;

    gdk_threads_enter();
    //    question_window = gnome_app_question(GNOME_APP(app),
    //                                   (const gchar *) question,
    //                                       yes_action, NULL);
//gtk_widget_show(question_window);
    gdk_threads_leave();
g_print( "QUESTION: %s\n", question );
    return 1;
}

/* Enter critical section */
void fe_disable_abort(struct site *site)
{
	/* lets find site in our list of wizards */
	GList *list;
	UploadWizard *wizard = NULL;

	for( list = wizards; list; list = list->next ) {
		wizard = (UploadWizard*)list->data;

		if( site == wizard->site )
			break;
		wizard = NULL;
	}
		
	/* we must have a wizard which owns the site
	   or something strange is going on */
	g_assert( wizard != NULL );

	wizard->in_critical_section = 1;
}

/* Leave critical section */
void fe_enable_abort(struct site *site)
{
	/* lets find site in our list of wizards */
	GList *list;
	UploadWizard *wizard = NULL;

	for( list = wizards; list; list = list->next ) {
		wizard = (UploadWizard*)list->data;

		if( site == wizard->site )
			break;
		wizard = NULL;
	}
		
	/* we must have a wizard which owns the site
	   or something strange is going on */
	g_assert( wizard != NULL );

	wizard->in_critical_section = 0;
	/* Carry out the abort if we were aborted while in the
	 * critical section */
	if( wizard->want_abort )
		do_abort( wizard );
}

void fe_warning(const char *description, const char *subject,
                const char *error)
{
	gchar *temp;

	gdk_threads_enter();
	
	temp = g_strdup_printf( _( "Upload Wizard: %s" ), description );
	
	screem_window_show_message( SCREEM_WINDOW( currentWiz->window ),
				    temp );

	gdk_threads_leave();
	
	g_free( temp );
}

/** Informs the user when we're trying to (re)establish a connection
 *  or lookup a hostname. 
 */
void fe_connection( fe_status status, const char *info ) 
{
	GtkWidget *label;
	gchar *text = "";

	gdk_threads_enter();

	label = glade_xml_get_widget( currentWiz->xml, "status_text" );

	switch( status ) {
	case fe_namelookup:
		text = g_strconcat( _("Looking up hostname: "), info, " ...", NULL );
		break;
	case fe_connecting:
		text = g_strconcat( _("Attemptng to connect "), " ...", NULL );
		break;
	case fe_connected:
		text = g_strconcat( _("Connected "), NULL );
 		break;
	}
	gtk_label_set( GTK_LABEL( label ), text );

	g_free( text );

	gdk_threads_leave();
}


int fe_can_update( const struct site_file *file ) 
{
   return TRUE;
}

void fe_updating( const struct site_file *file ) 
{
	gchar *file_status;
	GtkWidget *status_label;
	
	gdk_threads_enter();

	status_label = glade_xml_get_widget( currentWiz->xml, "status_text" );

	file_status = g_strdup_printf(_("Commiting updates to %s..."),
				      currentWiz->site->server.hostname);
	gtk_label_set(GTK_LABEL(status_label), file_status);

	
	if (file->type == file_dir) {
		if (file->diff == file_new) {
			file_status = g_strdup_printf( _( "Creating directory %s" ),
							file_name( file ) );
		} else {
			/* can we move dirs yet? */
			file_status = g_strdup( _("Deleting directory" ) );
		}
	        gtk_label_set (GTK_LABEL (status_label), file_status);
		g_free( file_status );
	} else {
		switch (file->diff) {
		case file_changed:
		case file_new:
			file_status = g_strdup_printf( _( "Uploading %s" ),
							file_name( file ) );
			break;
		case file_deleted:
			file_status = g_strdup_printf( _( "Deleting %s" ),
							file_name( file ) );
			break;
		case file_moved:
			file_status = g_strdup_printf( _( "Moving %s" ),
							file_name( file ) );
			break;
		case file_unchanged:
			file_status = g_strdup_printf( _( "Unchanged File %s" ),
					file_name( file ) );
		}
		gtk_label_set (GTK_LABEL (status_label), file_status);
		g_free( file_status );
	}

	gdk_threads_leave();

}

/* Once a file has been updated, any errors with it are recorded in the 
 * error list, and the progress bar is reset. 
 */
void fe_updated( const struct site_file *file, const int success,
		 const char *error )
{
	GtkWidget *bar;
	gchar *temp;

	gdk_threads_enter();


	if( ! success ) {
		temp = g_strdup_printf( "%s: %s", file_name( file ), error );
		screem_window_show_message( SCREEM_WINDOW( currentWiz->window ),
					    temp );
		g_free( temp );
	}

	bar = glade_xml_get_widget( currentWiz->xml, "main_progressbar" );
	gtk_progress_set_percentage( GTK_PROGRESS( bar ), 0.0 );
	
	currentWiz->uploaded_bytes += (float) file->local.size;

	gdk_threads_leave();
}

void fe_transfer_progress( off_t progress, off_t total )
{
	float div1, div2;

	GtkWidget *mainbar;
	GtkWidget *jobbar;

	gdk_threads_enter();

	mainbar = glade_xml_get_widget( currentWiz->xml, "main_progressbar" );
	jobbar = glade_xml_get_widget( currentWiz->xml, "job_progressbar" );

	/* Calculate the amount to display the current file's progress */
	div1 = (float) progress;
	div2 = (float) total;

	if( div2 ) {
		gtk_progress_bar_update(GTK_PROGRESS_BAR(mainbar), 
					div1 / div2);
	}
      

	gtk_progress_bar_update(GTK_PROGRESS_BAR(jobbar),
				(currentWiz->uploaded_bytes + 
				 (float) progress) / 
				(float) currentWiz->upload_total);

	gdk_threads_leave();
}

/* These can be implemented if we ever want to use the respective 
 * functionality.
 */
void fe_synching( const struct site_file *file )
{

}

void fe_synched( const struct site_file *file, const int success,
		 const char *error )
{

}

void fe_setting_perms(const struct site_file *file)
{
}
void fe_set_perms(const struct site_file *file, int success,
                  const char *error)
{
}

void fe_checksumming( const char *filename )
{

}

void fe_checksummed(const char *file, int success, const char *err)
{
}

void fe_fetch_found(const struct site_file *file) 
{
}

int fe_login(fe_login_context ctx, const char *realm, const char *hostname,
             char *username, char *password) 
{
	const char *server = 
		(ctx==fe_login_server)?N_("server"):N_("proxy server");

	gchar *message;
	GtkWidget *widget;
	GladeXML *lxml;
	guint size;
	const gchar *value;

	gdk_threads_enter();

	lxml = glade_xml_new( GLADE_PATH, "auth_dialog", NULL );
	widget = glade_xml_get_widget( lxml, "message_label" );

	if (realm) {
		message = g_strdup_printf( _("Authentication required for %s on %s `%s':\n"), realm, server, hostname );
	} else {
		message = g_strdup_printf( _("Authentication required on %s `%s':\n"), server, hostname );
	}

	gtk_label_set( GTK_LABEL( widget ), message );
	g_free( message );

	if( *username ) {
		widget = glade_xml_get_widget( lxml, "username" );
		gtk_entry_set_text( GTK_ENTRY( widget ), username );
	}
	if( *password ) {
		widget = glade_xml_get_widget( lxml, "password" );
		gtk_entry_set_text( GTK_ENTRY( widget ), password );
	}

	widget = glade_xml_get_widget( lxml, "auth_dialog" );
	gtk_widget_show_all( widget );

	glade_xml_signal_autoconnect( lxml );

	/* sleep, we will be woken up by one of the dialogs callbacks */
	currentWiz->button = -1 ;
	gdk_threads_leave();

	sem_wait( currentWiz->update_semaphore );

	if( currentWiz->button != GTK_RESPONSE_OK ) {
		/* canceled */
		gtk_widget_destroy( widget );
		return -1;
	}
	gdk_threads_enter();
	widget = glade_xml_get_widget( lxml, "username" );
      
	
	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	size = strlen( value );
	if( size > FE_LBUFSIZ ) {
		size = FE_LBUFSIZ;
	}
	strncpy( username, value, size );

	widget = glade_xml_get_widget( lxml, "password" );

	value = gtk_entry_get_text( GTK_ENTRY( widget ) );
	size = strlen( value );
	if( size > FE_LBUFSIZ ) {
		size = FE_LBUFSIZ;
	}
	strncpy( password, value, size );


	widget = glade_xml_get_widget( lxml, "auth_dialog" );
	gtk_widget_destroy( widget );

	gdk_threads_leave();

	return 0;
}

void fe_login_clicked( GtkDialog *dialog, gint number )
{
	currentWiz->button = number;
	sem_post(currentWiz->update_semaphore);
}
void fe_login_close( GtkDialog *dialog )
{
	sem_post(currentWiz->update_semaphore);
}

static gboolean create_sitecopy_directory()
{
	const gchar *home;
	gchar *dir;
	struct stat s;

	home = g_get_home_dir();
	dir = g_strconcat( home, G_DIR_SEPARATOR_S, ".sitecopy", NULL );
	if( stat( dir, &s ) != 0 ) {
		if( errno != ENOENT ) {
			/* couldn't access it */
			upload_wizard_error(_("Couldn't access ~/.sitecopy"));
			g_free( dir );
			return FALSE;
		}
		/* it doesn't exist so lets make it */
		if( mkdir( dir, S_IRWXU ) != 0 ) {
			/* couldn't make it */
			upload_wizard_error(_("Couldn't create ~/.sitecopy"));
			g_free( dir );
			return FALSE;
		}
	}
	g_free( dir );
	return TRUE;
}

gboolean screem_site_get_sync_status( ScreemSite *ssite, GHashTable **table )
{
	struct site *site;
	struct site_file *files;
	struct file_state *file;
	gchar *filename;

	UploadWizard *wizard;

	g_return_val_if_fail( ssite != NULL, FALSE );
	g_return_val_if_fail( table != NULL, FALSE );

	*table = NULL;

	if( screem_site_get_fake_flag( ssite ) ) {
		/* fake sites have no sync status */
		return FALSE;
	}

	if( ! create_sitecopy_directory() ) {
		return FALSE;
	}

	site = screem_site_to_sitecopy_site( ssite, FALSE );

	if( ! site ) {
		return FALSE;
	}

	wizard = g_new0( UploadWizard, 1 );
	wizards = g_list_append( wizards, wizard );

	wizard->site = site;

	if( ! verify_site( site ) ) {
		destroy_fnlists( site );
		g_free( site->infofile );
		g_free( site );
		wizards = g_list_remove( wizards, wizard );
		g_free( wizard );
		return FALSE;
	}

	if( site_readfiles( site ) < 0 ) {
		/* couldn't get info on local site files, this probably
		   means the user hasn't attempted to upload yet */
		destroy_fnlists( site );
		g_free( site->infofile );
		g_free( site );
		wizards = g_list_remove( wizards, wizard );
		g_free( wizard );
		return FALSE;
	}

	*table = g_hash_table_new( g_str_hash, g_str_equal );

	/* ok we've got the info, and there are changes */
	files = site->files;
	while( files ) {
		file = &files->local;
		
		if( file->filename ) {
			gchar *tmp;
			filename = file_full_local( file, site );

			tmp = g_strconcat( "file://", filename, NULL );
			
			free( filename );

			/* add to hash table with the status */
			g_hash_table_insert( *table, tmp, 
					     GINT_TO_POINTER( files->diff ) );
		}

		files = files->next;
	}

	site_destroy( site );

	destroy_fnlists( site );
	g_free( site->infofile );
	g_free( site );

	wizards = g_list_remove( wizards, wizard );
	g_free( wizard );

	return TRUE;
}

/* destroy the fnlists in the sitecopy site object,
   we don't want to free that pattern though */
static void destroy_fnlists( struct site *site )
{
	struct fnlist *list;

	list = site->excludes;
	destroy_fnlist( list );
	list = site->ignores;
	destroy_fnlist( list );
	list = site->asciis;
	destroy_fnlist( list );
	
}

static void destroy_fnlist( struct fnlist *list )
{
	if( ! list )
		return;

	for( ;; ) {
		if( list->next ) {
			list = list->next;
			g_free( list->prev );
		} else {
			g_free( list );
			break;
		}
	}
}
