/*  Screem:  screem-page-druid.c,
 *  A page druid for guiding users through the process of creating a page
 *
 *  Copyright (C) 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

#include <gconf/gconf-client.h>

#include <glade/glade.h>

#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-color-picker.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <gtk/gtkcombo.h>
#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtkimage.h>

#include <string.h>

#include "screem-application.h"
#include "screem-window.h"
#include "screem-preview.h"
#include "screem-site.h"
#include "screem-page.h"
#include "screem-markup.h"
#include "screem-dtd.h"

#include "fileops.h"

void screem_page_druid_finish( GtkWidget *widget );
void screem_page_druid_cancel( GtkWidget *widget );

void screem_page_druid_switch_page( GtkNotebook *book,
				    GtkNotebookPage *page,
				    guint page_num );

gboolean screem_page_druid_init_details( GtkWidget *widget);
gboolean screem_page_druid_init_templates( GtkWidget *widget );
gboolean screem_page_druid_init_colours( GtkWidget *widget );
gboolean screem_page_druid_init_preview( GtkWidget *widget );

static void screem_page_druid_create_page( ScreemPage *page, GladeXML *xml,
					   ScreemWindow *window );


/* MUST have an entry in the array for each page in the druid,
   use NULL if there is to be no init function */
typedef gboolean (*InitPage)( GtkWidget *widget );
static const InitPage init_funcs[] = {
	NULL,
	screem_page_druid_init_details,
	screem_page_druid_init_templates,
	screem_page_druid_init_colours,
	screem_page_druid_init_preview
};

void screem_page_druid_begin( ScreemApplication *application,
			      ScreemWindow *window )
{
	ScreemSite *site;
	ScreemPage *page;
	GladeXML *xml;
	GtkWidget *druid;
	GtkWidget *book;
	GtkWidget *image;
	gchar *gladepath;

	site = screem_window_get_current( window );

	gladepath = gconf_client_get_string( application->client,
					     "/apps/screem/general/glade_path",
					     NULL );
	xml = glade_xml_new( gladepath, "pagedruid", NULL );

	druid = glade_xml_get_widget( xml, "pagedruid" );
	page = screem_page_new( G_OBJECT( application ) );

	g_object_set_data( G_OBJECT( druid ), "page", page );
	g_object_set_data( G_OBJECT( druid ), "application", application );
	g_object_set_data( G_OBJECT( druid ), "window", window );
	g_object_set_data( G_OBJECT( druid ), "site", site );

	book = glade_xml_get_widget( xml, "druid-notebook" );
	gtk_notebook_set_show_tabs( GTK_NOTEBOOK( book ), FALSE );

	image = glade_xml_get_widget( xml, "druid-logo" );
	gtk_image_set_from_file( GTK_IMAGE( image ),
				 DATADIR"/pixmaps/screem.png" );

	glade_xml_signal_autoconnect( xml );

	gtk_window_set_transient_for( GTK_WINDOW( druid ),
				      GTK_WINDOW( window ) );

	screem_page_druid_switch_page( GTK_NOTEBOOK( book ),
				       NULL, 0 );
}


void screem_page_druid_finish( GtkWidget *widget )
{
	ScreemSite *site;
	ScreemPage *page;
	ScreemWindow *window;
	GladeXML *xml;
	GtkWidget *book;

	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );
	widget = glade_xml_get_widget( xml, "pagedruid" );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );
	page = SCREEM_PAGE( g_object_get_data( G_OBJECT( widget ), "page" ) );
	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( widget ), 
						   "window" ) );

	screem_page_set_changed( page, TRUE );
	if( screem_page_save( page ) ) {
		if( ! screem_site_add_page( site, page ) ) {
			/* already got a page in the site with the
			   same pathname, FIXME: what now? */
		}
		
		if( site == screem_window_get_current( window ) ) {
			screem_window_set_document( window, page );
			
		}
	} else {
		GtkWidget *msg;

		msg = gtk_message_dialog_new( GTK_WINDOW( window ),
					GTK_DIALOG_MODAL |
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_ERROR,
					GTK_BUTTONS_OK,
					_( "Failed to create %s" ),
					screem_page_get_pathname( page ) );
		gtk_dialog_run( GTK_DIALOG( msg ) );
		gtk_widget_destroy( msg );
		g_object_unref( page );
	}

	book = glade_xml_get_widget( xml, "druid-notebook" );
	g_signal_handlers_disconnect_by_func( G_OBJECT( book ),
					      screem_page_druid_switch_page,
					      NULL );

	gtk_widget_destroy( widget );
}


void screem_page_druid_cancel( GtkWidget *widget )
{
	ScreemPage *page;
	GladeXML *xml;
	GtkWidget *book;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "pagedruid" );

	page = SCREEM_PAGE( g_object_get_data( G_OBJECT( widget ), "page" ) );
	g_object_unref( page );

	book = glade_xml_get_widget( xml, "druid-notebook" );
	g_signal_handlers_disconnect_by_func( G_OBJECT( book ),
					      screem_page_druid_switch_page,
					      NULL );

	gtk_widget_destroy( widget );
}

void screem_page_druid_switch_page( GtkNotebook *book,
				    GtkNotebookPage *page,
				    guint page_num )
{
	GladeXML *xml;

	GtkWidget *back;
	GtkWidget *forward;
	GtkWidget *apply;
	GtkWidget *current;

	xml = glade_get_widget_tree( GTK_WIDGET( book ) );

	back = glade_xml_get_widget( xml, "back" );
	forward = glade_xml_get_widget( xml, "forward" );
	apply = glade_xml_get_widget( xml, "apply" );

	if( page_num == 0 ) {
		gtk_widget_set_sensitive( apply, FALSE );
		gtk_widget_set_sensitive( back, FALSE );
		gtk_widget_set_sensitive( forward, TRUE );
	} else if( page_num == g_list_length( book->children ) - 1 ) {
		gtk_widget_set_sensitive( apply, TRUE );
		gtk_widget_set_sensitive( back, TRUE );
		gtk_widget_set_sensitive( forward, FALSE );
	} else {
		gtk_widget_set_sensitive( apply, FALSE );
		gtk_widget_set_sensitive( back, TRUE );
		gtk_widget_set_sensitive( forward, TRUE );
	}

	/* call init function for page if needed */
	current = gtk_notebook_get_nth_page( book, page_num );
	if( ! g_object_get_data( G_OBJECT( current ), "inited" ) ) {
		if( init_funcs[ page_num ] && 
		    init_funcs[ page_num ]( current ) ) {
			g_object_set_data( G_OBJECT( current ), "inited",
					   GINT_TO_POINTER( 1 ) );
		}
	}
}

gboolean screem_page_druid_init_details( GtkWidget *widget )
{
	/* add all loaded doctypes to "doctype" combo */
	ScreemApplication *application;
	ScreemDTDDB *db;
	GladeXML *xml;
	GtkWidget *druid;
	gchar *default_dtd;
	gchar *default_charset;

	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );

	if( xml ) {
		ScreemSite *site;
		const gchar *sitepath;
		GnomeVFSURI *uri;
		gchar *spath;
		
		druid = glade_xml_get_widget( xml, "pagedruid" );

		site = SCREEM_SITE( g_object_get_data( G_OBJECT( druid ),
						       "site" ) );
		sitepath = screem_site_get_pathname( site );

		uri = NULL;
		if( sitepath ) {
			uri = gnome_vfs_uri_new( sitepath );
		}
		if( ! uri ) {
			spath = NULL;
		} else if( gnome_vfs_uri_is_local( uri ) ) {
			/* strip method */
			spath = gnome_vfs_get_local_path_from_uri( sitepath );
			gnome_vfs_uri_unref( uri );
		} else {
			spath = gnome_vfs_uri_to_string( uri, 0 );
			gnome_vfs_uri_unref( uri );
		}
		widget = glade_xml_get_widget( xml, "page_path" );
		gnome_file_entry_set_default_path( GNOME_FILE_ENTRY( widget ),
						   spath );
		widget = glade_xml_get_widget( xml, "page_stylesheet" );
		gnome_file_entry_set_default_path( GNOME_FILE_ENTRY( widget ),
						   spath );
		g_free( spath );
					   
		widget = glade_xml_get_widget( xml, "doctype" );
		
		application = g_object_get_data( G_OBJECT( druid ), 
						 "application" );
		db = screem_application_get_dtd_db( application );
		screem_dtd_db_fill_combo( db, widget );

		default_dtd = gconf_client_get_string( application->client,
						       "/apps/screem/editor/default_dtd",
						       NULL );
		widget = GTK_COMBO( widget )->entry;
		gtk_entry_set_text( GTK_ENTRY( widget ), default_dtd );
		
		g_free( default_dtd );
		
		widget = glade_xml_get_widget( xml, "page_author" );
		widget = gnome_entry_gtk_entry( GNOME_ENTRY(widget) );
		gtk_entry_set_text( GTK_ENTRY( widget ), 
				    g_get_real_name() );
				
		widget = glade_xml_get_widget( xml, "page_charset" );
		widget = GTK_COMBO( widget )->entry;
		default_charset = gconf_client_get_string( application->client,
							   "/apps/screem/editor/default_charset",
							   NULL );
		gtk_entry_set_text( GTK_ENTRY( widget ), default_charset );
		g_free( default_charset );
	}

	return TRUE;
}


gboolean screem_page_druid_init_templates( GtkWidget *widget )
{
	GtkListStore *store;
	GtkTreeView *view;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GladeXML *xml;
	ScreemSite *site;
	const gchar *template;
	gchar *spath;
	
	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );
	if( xml ) {
		widget = glade_xml_get_widget( xml, "pagedruid" );

		site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ),
						       "site" ) );
		
		view = GTK_TREE_VIEW( glade_xml_get_widget( xml,
							    "templatelist" ) );

		store = gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_STRING,
					    NULL );

		gtk_tree_view_set_model( view, GTK_TREE_MODEL( store ) );
		
		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, "Name" );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( view, col );
		gtk_tree_view_column_set_attributes( col, renderer, 
						     "text", 0, NULL );
		
		renderer = gtk_cell_renderer_text_new();
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( col, "Path" );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		gtk_tree_view_append_column( view, col );
		gtk_tree_view_column_set_attributes( col, renderer, 
						     "text", 1, NULL );

		widget = glade_xml_get_widget( xml, 
					"custom_template" );
		spath = screem_get_local_site_path( site );
		if( spath ) {
			gnome_file_entry_set_default_path( GNOME_FILE_ENTRY( widget ), spath );
			g_free( spath );
		}
		
		template = screem_site_get_template_path( site );
		if( template ) {
			widget = glade_xml_get_widget( xml, 
					"custom_template" );

			gnome_file_entry_set_filename( GNOME_FILE_ENTRY( widget ), template );
			widget = glade_xml_get_widget( xml,
					"use_custom" );
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ), TRUE );
		}
	}

	return TRUE;
}


gboolean screem_page_druid_init_colours( GtkWidget *widget )
{
	GladeXML *xml;
	gboolean template;

	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );

	if( xml ) {
		widget = glade_xml_get_widget( xml, "notemplate" );
		template = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

		widget = glade_xml_get_widget( xml, "override" );

		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
					      template );
		gtk_widget_set_sensitive( widget, ! template );
	}

	return FALSE;
}


gboolean screem_page_druid_init_preview( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *druid;
	ScreemPreview *preview;
	ScreemPage *page;

	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );

	if( xml ) {
		GList *children;
		ScreemWindow *window;
		druid = glade_xml_get_widget( xml, "pagedruid" );

		window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( druid ),
							   "window" ) );

		widget = glade_xml_get_widget( xml, "page_preview" );
		children = gtk_container_get_children( GTK_CONTAINER( widget ) );
		children = children->next;

		if( ! children ) {
			preview = screem_preview_new();
			gtk_box_pack_start( GTK_BOX( widget ),
					    GTK_WIDGET( preview ),
					    TRUE, TRUE, 0 );
		} else {
			preview = SCREEM_PREVIEW( children->data );
		}
		gtk_widget_show_all( GTK_WIDGET( preview ) );

		
		/* now take the page template, colours, stylesheet etc
		   and construct a page */
		widget = glade_xml_get_widget( xml, "pagedruid" );
		page = SCREEM_PAGE( g_object_get_data( G_OBJECT( widget ),
						       "page" ) );

		g_return_val_if_fail( page != NULL, FALSE );
		screem_page_druid_create_page( page, xml,
					       window );

		screem_page_set_mime_type( page, "text/html" );

		g_object_set( G_OBJECT( preview ), 
			      "page", page,			      
			      "online", FALSE,
			      NULL );
		screem_view_display( SCREEM_VIEW( preview ) );
	}

	return FALSE;
}

static void screem_page_druid_create_page( ScreemPage *page, GladeXML *xml,
					   ScreemWindow *window )
{
	GtkWidget *widget;

	gchar *pathname;
	gchar *uri;
	
	gchar *stylesheet;
	const gchar *doctype;
	
	const gchar *title;
	const gchar *author;
	const gchar *charset;
	const gchar *keywords;

	gchar *template;

	gboolean override;
	gboolean frameset;

	GString *newtext;
	gchar *text;
	gint pos;
	gint start;
	gint end;
	gchar *name;
	gchar *tag;

	gboolean xhtml;

	gboolean setauthor;
	gboolean setgenerator;
	gboolean setkeywords;
	gboolean setcontent;
	gboolean setscript;
	gboolean setstyle;

	ScreemApplication *app;
	ScreemDTDDB *db;
	ScreemSite *site;
	const gchar *sitepath;
	gchar *spath;
	
	app = window->application;
	db = screem_application_get_dtd_db( app );

	site = screem_window_get_current( window );
	
	widget = glade_xml_get_widget( xml, "page_path" );
	pathname = gnome_file_entry_get_full_path( GNOME_FILE_ENTRY( widget ), FALSE );


	widget = glade_xml_get_widget( xml, "page_stylesheet" );
	stylesheet = gnome_file_entry_get_full_path( GNOME_FILE_ENTRY( widget ), FALSE );
	if( stylesheet && ! screem_site_get_fake_flag( site ) ) {
		spath = relative_path( stylesheet,
				screem_site_get_pathname( site ) );
		g_free( stylesheet );
		stylesheet = spath;
		spath = NULL;
	}
	
	widget = glade_xml_get_widget( xml, "doctype" );
	widget = GTK_COMBO( widget )->entry;
	doctype = gtk_entry_get_text( GTK_ENTRY( widget ) );

	/* bit of a hack */
	xhtml = ( strstr( doctype, " XHTML " ) != NULL );
	if( ! xhtml ) {
		xhtml = ! strcmp( ".xml", 
				g_extension_pointer( pathname ) );
		xhtml |= ! strcmp( ".xhtml",
				g_extension_pointer( pathname ) );
	}

	widget = glade_xml_get_widget( xml, "page_title" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	title = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_author" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	author = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_charset" );
	widget = GTK_COMBO( widget )->entry;
	charset = gtk_entry_get_text( GTK_ENTRY( widget ) );

	widget = glade_xml_get_widget( xml, "page_keywords" );
	widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	keywords = gtk_entry_get_text( GTK_ENTRY( widget ) );

	template = NULL;
	widget = glade_xml_get_widget( xml, "use_custom" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "custom_template" );

		template = gnome_file_entry_get_full_path( GNOME_FILE_ENTRY( widget ), TRUE );
	 	
	}
	widget = glade_xml_get_widget( xml, "use_standard" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		widget = glade_xml_get_widget( xml, "druid" );
		template = g_object_get_data( G_OBJECT( widget ),
					      "template" );
		template = g_strdup( template );
	}

	
	widget = glade_xml_get_widget( xml, "override" );
	override = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	widget = glade_xml_get_widget( xml, "frameset" );
	frameset = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	/* if we are using a template load it */
	if( template ) {
		screem_page_set_pathname( page, template );
		screem_page_load( page );
		text = screem_page_get_data( page );
	} else {
		/* create a basic page */

		text = screem_markup_basic_html( db, doctype,
				frameset, xhtml );
		screem_page_set_data( page, text );
	}

	newtext = g_string_new( NULL );
	pos = start = end = 0;

	setauthor = setgenerator = setkeywords = setcontent = 
		setscript = setstyle = FALSE;

	while( ( tag = screem_markup_next_tag( text, pos, &start, &end, &name ) ) ) {
		GSList *attrs;
		GSList *list;
		gboolean processed;

		g_assert( name );
		
		attrs = screem_markup_build_attributes_list( tag, NULL );
		attrs = g_slist_reverse( attrs );

		if( start > pos ) {
			g_string_append_len( newtext, text + pos, start - pos);
		}

		pos = end;
		
		processed = FALSE;

		if( ! g_strcasecmp( "/head", name ) ) {
			/* so we can append unset meta elements, and
			   a link to a stylesheet if specified */
			if( ! setauthor ) {
				g_string_append( newtext,
						 "\t<meta name=\"author\" content=\"" );
				g_string_append( newtext, author );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setauthor = TRUE;
			}
			if( ! setgenerator ) {
				g_string_append( newtext,
						 "\t<meta name=\"generator\" content=\"" );
				g_string_append( newtext, PACKAGE_STRING );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setgenerator = TRUE;
			}
			if( ! setkeywords ) {
				g_string_append( newtext,
						 "\t<meta name=\"keywords\" content=\"" );
				g_string_append( newtext, keywords );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setkeywords = TRUE;
			}
			if( ! setcontent && ! template ) {
				g_string_append( newtext,
						 "\t<meta http-equiv=\"content-type\" content=\"text/html; charset=" );
				g_string_append( newtext, charset );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
				setcontent = TRUE;
			}
			if( ! setscript ) {
				g_string_append( newtext,
						 "\t<meta http-equiv=\"Content-Script-Type\" content=\"text/javascript\" " );
				if( ! xhtml ) {
					g_string_append( newtext, ">\n\t" );
				} else {
					g_string_append( newtext, "/>\n\t" );
				}
				setscript = TRUE;
			}
			if( ! setstyle ) {
				g_string_append( newtext,
						 "\t<meta http-equiv=\"Content-Style-Type\" content=\"text/css\" " );
				if( ! xhtml ) {
					g_string_append( newtext, ">\n\t" );
				} else {
					g_string_append( newtext, "/>\n\t" );
				}
				setstyle = TRUE;
			}
			if( stylesheet && *stylesheet != '\0' ) {
				g_string_append( newtext,
						 "\t<link rel=\"stylesheet\" type=\"text/css\" href=\"" );
				g_string_append( newtext, stylesheet );
				if( ! xhtml ) {
					g_string_append( newtext, "\" >\n\t" );
				} else {
					g_string_append( newtext, "\" />\n\t" );
				}
			}
		} else if( ! g_strcasecmp( "title", name ) && *title != '\0' ) {
			processed = TRUE;
			g_string_append_len( newtext, text + start, end - start + 1 );
			g_string_append( newtext, title );
			/* shift to start of closing tag for <title> */
			while( text[ pos ] != '<' && text[ pos ] != '\0' ) {
				pos ++;
			}
		} else if( ! g_strcasecmp( "meta", name ) ) {
			/* we are interested in:
			   name="author"
			   name="generator"
			   name="keywords"

			   and http-equiv="content-type" partly
			*/
			GSList *nme;
			GSList *content;
			nme = content = NULL;
			for( list = attrs; list; list = list->next ) {
				GSList *attr;

				attr = list;
				list = list->next;
				if( ! g_strcasecmp( "name", attr->data ) &&
				    ( ! g_strcasecmp( "author", list->data ) ||
				      ! g_strcasecmp( "generator", list->data ) ||
				      ! g_strcasecmp( "keywords", list->data ) ) ) {
					nme = list;
				} else if( ! g_strcasecmp( "http-equiv", attr->data ) &&
					   ( ! g_strcasecmp( "content-type",
							     list->data ) ||
					     ! g_strcasecmp( "Content-Style-Type",
							     list->data ) ||
					     ! g_strcasecmp( "Content-Script-Type",
							     list->data ) ) ) {
					nme = list;
				}
				if( ! g_strcasecmp( "content", attr->data ) ) {
					content = list;
				}
			}
			if( nme ) {
				const gchar *type;
				gchar *value;

				value = NULL;
				type = (const gchar*)nme->data;

				processed = TRUE;
				g_string_append_c( newtext, '<' );
				g_string_append( newtext, name );
				if( ! g_strcasecmp( "author", type ) ) {
					value = g_strdup( author );
				} else if( ! g_strcasecmp( "generator", 
							   type ) ) {
					value = g_strdup( PACKAGE_STRING );
				} else if( ! g_strcasecmp( "keywords",
							   type ) ) {
					value = g_strdup( keywords );
				} else if( content ) {
					value = g_strdup( content->data );
				}
					
				if( content ) {
					content->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( type ) );
				}
					   
				g_string_append_c( newtext, '<' );
				g_string_append( newtext, name );

				for( list = attrs; list; list = list->next ) {
					g_string_append_c( newtext, ' ' );
					if( ! g_strcasecmp( "author", list->data ) ) {
						setauthor = TRUE;
					} else if( ! g_strcasecmp( "generator",
								   list->data ) ) {
						setgenerator = TRUE;
					} else if( ! g_strcasecmp( "keywords", 
								   list->data ) ) {
						setkeywords = TRUE;
					} else if( ! g_strcasecmp( "content-type",
								   list->data ) ) {
						setcontent = TRUE;
					} else if( ! g_strcasecmp( "Content-Style-Type",
								   list->data ) ) {
						setstyle = TRUE;
					} else if( ! g_strcasecmp( "Content-Script-Type",
								   list->data ) ) {
						setscript = TRUE;
					}

					g_string_append( newtext, list->data );
					g_string_append( newtext, "=\"" );
					if( ! list->next->data ) {
						g_string_append( newtext, list->data );
						list = list->next;
					} else {
						list = list->next;
						g_string_append( newtext, list->data );
					}
					g_string_append_c( newtext, '"' );
				}

			} else {
				processed = FALSE;
			}
		} else if( ! g_strcasecmp( "body", name ) && override ) {
			GSList *bgcolor;
			GSList *text;
			GSList *link;
			GSList *vlink;
			GSList *alink;
			GSList *background;
			guint8 r, g, b, a;
			gchar *value;

			processed = TRUE;

			g_string_append_c( newtext, '<' );
			g_string_append( newtext, name );

			/* find attributes in attrs */
			bgcolor = text = link = vlink = alink = background = NULL;
			for( list = attrs; list; list = list->next ) {
				if( ! g_strcasecmp( "bgcolor", list->data ) ) {
					bgcolor = list->next;
				} else if( ! g_strcasecmp( "text", list->data ) ) {
					text = list->next;
				} else if( ! g_strcasecmp( "link", list->data ) ) {
					link = list->next;
				} else if( ! g_strcasecmp( "vlink", list->data ) ) {
					vlink = list->next;
				} else if( ! g_strcasecmp( "alink", list->data ) ) {
					alink = list->next;
				} else if( ! g_strcasecmp( "background", list->data ) ) {
					background = list->next;
				}
				list = list->next;
			}

			widget = glade_xml_get_widget( xml, "bgcolor" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "bgcolorpicker" );
				gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
							   &r, &g, &b, &a );
				value = g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
				if( bgcolor ) {
					g_free( bgcolor->data );
					bgcolor->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "bgcolor" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "text" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "textpicker" );
				gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
							   &r, &g, &b, &a );
				value = g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
				if( text ) {
					g_free( text->data );
					text->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "text" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "link" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "linkpicker" );
				gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
							   &r, &g, &b, &a );
				value = g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
				if( link ) {
					g_free( link->data );
					link->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "link" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "vlink" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "vlinkpicker" );
				gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
							   &r, &g, &b, &a );
				value = g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
				if( vlink ) {
					g_free( vlink->data );
					vlink->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "vlink" ) );
				}
			}
			widget = glade_xml_get_widget( xml, "alink" );
			if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
				widget = glade_xml_get_widget( xml, "alinkpicker" );
				gnome_color_picker_get_i8( GNOME_COLOR_PICKER( widget ),
							   &r, &g, &b, &a );
				value = g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
				if( alink ) {
					g_free( alink->data );
					alink->data = value;
				} else {
					attrs = g_slist_prepend( attrs, value );
					attrs = g_slist_prepend( attrs, 
								 g_strdup( "alink" ) );
				}
			}
			for( list = attrs; list; list = list->next ) {
				g_string_append_c( newtext, ' ' );
				g_string_append( newtext, list->data );
				g_string_append( newtext, "=\"" );
				if( ! list->next->data ) {
					g_string_append( newtext, list->data );
					list = list->next;
				} else {
					list = list->next;
					g_string_append( newtext, list->data );
				}
				g_string_append_c( newtext, '"' );
			}
		} 

		if( ! processed ) {
			g_string_append_len( newtext, text + start, end - start );
		}

		g_slist_foreach( attrs, (GFunc)g_free, NULL );
		g_slist_free( attrs );
		g_free( tag );
	}
	g_string_append( newtext, text + end );

	screem_page_set_data( page, newtext->str );
	
	g_string_free( newtext, TRUE );
	g_free( text );
	
	if( screem_site_get_fake_flag( site ) ) {
		if( g_path_is_absolute( pathname ) ) {
			spath = NULL;
		} else {
			spath = g_get_current_dir();
		}
	} else {
		sitepath = screem_site_get_pathname( site );
		spath = g_strdup( sitepath );
	}
	if( spath ) {	
		uri = relative_to_full( pathname, spath );	
		g_free( spath );
	} else {
		uri = g_strdup( pathname );
	}

	screem_page_set_pathname( page, uri );

	g_free( uri );
	g_free( pathname );
	g_free( stylesheet );
	g_free( template );
}
