/*  Screem:  screem-page.c
 *
 *  The ScreemPage object
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <gconf/gconf-client.h>

#include <glib/gconvert.h>
#include <glib/gunicode.h>

#include <gtk/gtktreemodel.h>
#include <gtk/gtkliststore.h>

#include <gtksourceview/gtksourcebuffer.h>

#include "screem-page.h"
#include "screem-page-model.h"

#include "fileops.h"

/* yuk */
#include "pageUI.h"

/* yuk */
#include "screem-markup.h"

#include "screem-search.h"

/* yuk */
#include "support.h"

#include "screemmarshal.h"

enum {
	SAVED,
	/* these 2 are fired from the ScreemPageModel */
	BUILDING,
	BUILT,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_APP,
	PROP_PATHNAME,
	PROP_CHANGED,
	PROP_MIMETYPE,
	PROP_OPEN,
	PROP_CHARSET,
	PROP_DTD
};

static guint screem_page_signals[ LAST_SIGNAL ] = { 0 };


static void screem_page_class_init( ScreemPageClass *klass );
static void screem_page_init( ScreemPage *page );
static void screem_page_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec );
static void screem_page_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec );
static void screem_page_finalize( GObject *object );

static gboolean bookmark_store_find( GtkTreeModel *model, GtkTreePath *path,
				     GtkTreeIter *iter, gpointer data );

static void changed( GtkTextBuffer *buffer, ScreemPage *page );

static void screem_page_monitor_cb( GnomeVFSMonitorHandle *handle,
				    const gchar *monitor_uri,
				    const gchar *info_uri,
				    GnomeVFSMonitorEventType type,
				    gpointer data );

struct ScreemPagePrivate {
	ScreemPageModel *model;
	
	gchar *pathname;     /* pathname for the page */
	gchar *mime_type;
	gboolean changed;

	gboolean loaded;
	gboolean opened;

	GConfClient *client;
	gboolean highlight;  /* do we think highlighting is on? */
	gint highlight_notify;

	GHashTable *bookmarks;
	GtkListStore *bookmarkstore;

	GnomeVFSMonitorHandle *monitor;
	gboolean modified;

};

void screem_page_undo( ScreemPage *page )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	if( gtk_source_buffer_can_undo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_undo( GTK_SOURCE_BUFFER( page ) );
	}
}

void screem_page_redo( ScreemPage *page )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	if( gtk_source_buffer_can_redo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_redo( GTK_SOURCE_BUFFER( page ) );
	}
}

GSList *screem_page_get_mime_types( ScreemPage *page )
{
	GSList *ret;
	const GSList *langs;
	GtkSourceLanguagesManager *lm;

	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	ret = NULL;

	lm = screem_application_load_syntax_tables();
	langs = gtk_source_languages_manager_get_available_languages( lm );
	while( langs ) {
		GtkSourceLanguage *lang;
		GSList *types;

		lang = GTK_SOURCE_LANGUAGE( langs->data );
		types = gtk_source_language_get_mime_types( lang );
		ret = g_slist_concat( ret, types );

		langs = langs->next;
	}

	return ret;
}

ScreemPage *screem_page_new( GObject *application )
{
	ScreemPage *page;
	GType type;

	type = screem_page_get_type();

	page = SCREEM_PAGE( g_object_new( type, "tag-table", NULL, "app", application, NULL ) );

	return page;
}

gboolean screem_page_is_loaded( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return ( page->private->loaded || 
		 page->private->pathname == NULL );
}

gboolean screem_page_is_open( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return page->private->opened;
}

void screem_page_set_open( ScreemPage *page, gboolean opened )
{
	g_object_set( G_OBJECT( page ), "open", opened, NULL );
}

static void screem_page_set_open_real( ScreemPage *page, gboolean opened )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	page->private->opened = opened;
}


void screem_page_set_pathname( ScreemPage *page, const gchar *path )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	g_object_set( G_OBJECT( page ), "pathname", path, NULL );
}

static void screem_page_set_pathname_real( ScreemPage *page, const gchar *path )
{
	GtkSourceLanguage *lang;
	GtkSourceLanguagesManager *lm;
	GtkTextTagTable *table;
	GtkTextTag *tag;

	lm = screem_application_load_syntax_tables();
	
	if( page->private->pathname && path != page->private->pathname ) {
		g_free( page->private->pathname );
	}

	if( page->private->mime_type ) {
		g_free( page->private->mime_type );
	}
	page->private->mime_type = NULL;
	
	if( path ) {
		page->private->mime_type = 
			screem_get_mime_type( path );
	}
	if( ! page->private->mime_type ) {
		page->private->mime_type = g_strdup( "text/plain" );
	}

	if( path != page->private->pathname ) {
		if( path ) {
			page->private->pathname = g_strdup( path );
		} else {
			page->private->pathname = NULL;
		}
	}

	lang = gtk_source_languages_manager_get_language_from_mime_type( lm, page->private->mime_type );
	if( lang ) {
		gtk_source_buffer_set_language( GTK_SOURCE_BUFFER( page ), lang );
		table = gtk_text_buffer_get_tag_table( GTK_TEXT_BUFFER( page ) );
		tag = gtk_text_tag_new( SCREEM_INVALID_MARKUP_TAG );
		gtk_text_tag_table_add( table, tag );
		g_object_set( G_OBJECT( tag ),
			"background", "#ffffffff0000",
			"foreground", "#000000000000",
			NULL );
	} 
	if( page->private->monitor ) {
		gnome_vfs_monitor_cancel( page->private->monitor );
		page->private->monitor = NULL;
	}
	if( page->private->pathname ) {
		gnome_vfs_monitor_add( &page->private->monitor,
				       page->private->pathname,
				       GNOME_VFS_MONITOR_FILE,
				       screem_page_monitor_cb, page );
	}
}

const gchar *screem_page_get_pathname( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	return page->private->pathname;
}

void screem_page_set_mime_type( ScreemPage *page, const gchar *mime_type )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	g_object_set( G_OBJECT( page ), "mime-type", mime_type, NULL );
}

static void screem_page_set_mime_type_real( ScreemPage *page, const gchar *mime_type )
{
	GtkSourceLanguage *lang;
	GtkSourceLanguagesManager *lm;
	GtkTextTagTable *table;
	GtkTextTag *tag;

	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	lm = screem_application_load_syntax_tables();
	
	if( ! mime_type ) {
		mime_type = "text/plain";
	}

	if( page->private->mime_type ) {
		g_free( page->private->mime_type );
	}
	page->private->mime_type = g_strdup( mime_type );

	lang = gtk_source_languages_manager_get_language_from_mime_type( lm, page->private->mime_type );
	gtk_source_buffer_set_language( GTK_SOURCE_BUFFER( page ), lang );
	table = gtk_text_buffer_get_tag_table( GTK_TEXT_BUFFER( page ) );
	tag = gtk_text_tag_new( SCREEM_INVALID_MARKUP_TAG );
	gtk_text_tag_table_add( table, tag );
	g_object_set( G_OBJECT( tag ),
		"background", "#ffffffff0000",
		"foreground", "#000000000000",
		NULL );

	screem_page_model_force_check( page->private->model );
}

const gchar *screem_page_get_mime_type( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	return page->private->mime_type;
}

gchar *screem_page_get_charset( ScreemPage *page )
{
	gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	ret = NULL;

	g_object_get( G_OBJECT( page->private->model ),
			"charset", &ret, NULL );
	
	return ret;
}

void screem_page_set_data( ScreemPage *page, const gchar *data )
{
	gchar *cdata;
	
	g_return_if_fail( SCREEM_IS_PAGE( page ) );


	gtk_source_buffer_begin_not_undoable_action( GTK_SOURCE_BUFFER( page ) );
	
	/* ensure the text is UTF8, otherwise GtkTextBuffer can't
	   hold it */

	cdata = screem_support_charset_convert( data );
	
	if( cdata ) {
		gtk_text_buffer_set_text( GTK_TEXT_BUFFER( page ),
					  cdata,
					  strlen( cdata ) );
		g_free( cdata );
	}
	screem_page_model_force_check( page->private->model );
		
	page->private->loaded = TRUE;
	gtk_source_buffer_end_not_undoable_action( GTK_SOURCE_BUFFER( page ) );

}

gchar *screem_page_get_data( ScreemPage *page )
{
	GtkTextIter it;
	GtkTextIter eit;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	gtk_text_buffer_get_start_iter( GTK_TEXT_BUFFER( page ),
					&it );
	gtk_text_buffer_get_end_iter( GTK_TEXT_BUFFER( page ),
					&eit );

	return gtk_text_buffer_get_text( GTK_TEXT_BUFFER( page ), 
					&it, &eit, TRUE );
}

void screem_page_set_changed( ScreemPage *page, gboolean flag )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	g_object_set( G_OBJECT( page ), "changed", flag, NULL );
}

static void screem_page_set_changed_real( ScreemPage *page, gboolean flag )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	page->private->changed = flag;
}

gboolean screem_page_get_changed( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return page->private->changed;
}

gboolean screem_page_load( ScreemPage *page )
{
	const gchar *path;

	GString *data;

        g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	if( screem_page_is_loaded( page ) ) {
		return TRUE;
	}

	path = screem_page_get_pathname( page );

	g_return_val_if_fail( path != NULL, FALSE );
	
	data = load_file( path );

	if( data ) {
		screem_page_set_data( page, data->str );
		g_string_free( data, TRUE );

		screem_page_set_changed( page, FALSE );

		page->private->loaded = TRUE;
	}

	page->private->modified = FALSE;

	return ( data != NULL );
}

gboolean screem_page_revert( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	page->private->loaded = FALSE;

	return screem_page_load( page );
}

gboolean screem_page_save( ScreemPage *page )
{
	gchar *data;
	gboolean retval;

	gchar *charset;
	gchar *temp;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	if( ( ! screem_page_get_changed( page ) ) ||
	    ( ! screem_page_save_check( page ) ) ) {
		return TRUE;
	}

	data = screem_page_get_data( page );

	charset = screem_page_get_charset( page );
	
	if( ! charset ) {
		GConfClient *client;

		client = gconf_client_get_default();

		charset = gconf_client_get_string( client,
						   "/apps/screem/editor/default_charset",
						   NULL );
		g_object_unref( G_OBJECT( client ) );
	}

	temp = screem_support_charset_convert_to( data, charset );
	if( temp ) {
		g_free( data );
		data = temp;
	}
	g_free( charset );

	/* stop monitoring while we write to the file ourselves */
	if( page->private->monitor ) {
		gnome_vfs_monitor_cancel( page->private->monitor );
	}
	page->private->monitor = NULL;

	retval = save_file( screem_page_get_pathname( page ),
			    data, 
			    GNOME_VFS_PERM_USER_READ | 
			    GNOME_VFS_PERM_USER_WRITE |
			    GNOME_VFS_PERM_GROUP_READ |
			    GNOME_VFS_PERM_GROUP_WRITE |
			    GNOME_VFS_PERM_OTHER_READ );

	g_free( data );

	if( retval ) {
		g_signal_emit( G_OBJECT( page ),
				       screem_page_signals[ SAVED ], 0 );
	}
	
	screem_page_set_changed( page, FALSE );
	page->private->modified = FALSE;

      	if( ! screem_page_get_mime_type( page ) ) {
		screem_page_set_pathname( page, 
					  screem_page_get_pathname( page ) );
	} else {
		gnome_vfs_monitor_add( &page->private->monitor,
				       page->private->pathname,
				       GNOME_VFS_MONITOR_FILE,
				       screem_page_monitor_cb, page );
	}
	return retval;
}

gboolean screem_page_is_file_page( const gchar *path )
{
	gchar *mime_type;
	gboolean ret;
	
	g_return_val_if_fail( path != NULL, FALSE );
	

	mime_type = screem_get_mime_type( path );

	ret =  screem_page_is_mime_type_page( mime_type );
	g_free( mime_type );

	return ret;
}

gboolean screem_page_is_mime_type_page( const gchar *mime_type )
{
	gboolean is_page;

	is_page = FALSE;
	
	if(  mime_type ) {
		is_page = ! strncmp( mime_type, "text/", strlen( "text/" ) );
		is_page |= ! strcmp( mime_type, "application/x-php" );
		is_page |= ! strcmp( mime_type, "application/x-asp" );
		is_page |= ! strcmp( mime_type, "application/x-cgi" );
	}
	
	return is_page;
}


gboolean screem_page_get_modified( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return page->private->modified;
}

GtkListStore *screem_page_get_bookmark_model( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	return page->private->bookmarkstore;
}

void screem_page_add_bookmark( ScreemPage *page, gint line, const gchar *name )
{
	ScreemPagePrivate *private;
	ScreemPageBookMark *bookmark;
	GHashTable *table;
	GtkTreeIter it;
	GtkTextIter tit;

	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	private = page->private;

	table = private->bookmarks;

	gtk_text_buffer_get_iter_at_line( GTK_TEXT_BUFFER( page ),
					  &tit, line - 1 );
		
	bookmark = g_hash_table_lookup( table, name );
	if( bookmark ) {
		/* already got a bookmark with this name, just update
		   the line */
		g_free( bookmark->line_number );
		bookmark->line_number = g_strdup_printf( "%d", line );

		gtk_tree_model_foreach( GTK_TREE_MODEL(private->bookmarkstore),
					bookmark_store_find, bookmark );	
		it = *(GtkTreeIter*)bookmark->iter;
		gtk_source_buffer_move_marker( GTK_SOURCE_BUFFER( page ),
						bookmark->marker,
						&tit );
	} else {

		bookmark = g_new0( ScreemPageBookMark, 1 );
		bookmark->line_number = g_strdup_printf( "%d", line );
		bookmark->name = g_string_new( name );

		gtk_list_store_append( private->bookmarkstore, &it );

		g_hash_table_insert( table, bookmark->name->str, bookmark );
		
		bookmark->marker = 
			gtk_source_buffer_create_marker( GTK_SOURCE_BUFFER( page ), 
							 NULL,
							 BOOKMARK_MARKER, 
							 &tit );
	}

	gtk_list_store_set( private->bookmarkstore, &it,
			    0, bookmark->name->str,
			    1, bookmark->line_number,
			    2, bookmark, -1 );
	
}

void screem_page_remove_bookmark( ScreemPage *page, const gchar *name )
{
	ScreemPagePrivate *private;
	ScreemPageBookMark *bookmark;
	GHashTable *table;

	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	private = page->private;

	table = private->bookmarks;

	bookmark = g_hash_table_lookup( table, name );
	if( bookmark ) {
		/* found the bookmark we are after */
		GtkTreeIter it;

		g_hash_table_remove( table, name );

		gtk_source_buffer_delete_marker( GTK_SOURCE_BUFFER( page ),
						bookmark->marker );
		
		gtk_tree_model_foreach( GTK_TREE_MODEL(private->bookmarkstore),
					bookmark_store_find, bookmark );
		it = *(GtkTreeIter*)bookmark->iter;
		gtk_list_store_remove( private->bookmarkstore,
				       &it );

		g_string_free( bookmark->name, TRUE );
		g_free( bookmark );
	}
}

gint screem_page_get_bookmark( ScreemPage *page, const gchar *name )
{
	ScreemPagePrivate *private;
	ScreemPageBookMark *bookmark;
	GHashTable *table;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), -1 );

	private = page->private;

	table = private->bookmarks;

	bookmark = g_hash_table_lookup( table, name );
	if( bookmark ) {
		return atoi( bookmark->line_number );
	} else {
		return -1;
	}
}

static gboolean screem_page_clear_bookmarks( gpointer key, gpointer value, 
					     gpointer data )
{
	screem_page_remove_bookmark( SCREEM_PAGE( data ), key );

	return FALSE;
}

static gboolean bookmark_store_find( GtkTreeModel *model, GtkTreePath *path,
				     GtkTreeIter *iter, gpointer data )
{
	ScreemPageBookMark *bookmark;
	ScreemPageBookMark *bookmark2;
     	GValue value = {0,};

	bookmark = (ScreemPageBookMark*)data;
	
	gtk_tree_model_get_value( GTK_TREE_MODEL( model ),
				  iter, 2, &value );
	bookmark2 = g_value_get_pointer( &value );
	
	bookmark->iter = NULL;
	if( !strcmp( bookmark->name->str, bookmark2->name->str ) )
		bookmark->iter = iter;

	return ( bookmark->iter != NULL );
}

static void changed( GtkTextBuffer *buffer, ScreemPage *page )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	screem_page_set_changed( page, TRUE );
}

static void screem_page_highlight_notify( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data )
{
	ScreemPage *page;

	page = SCREEM_PAGE( data );

	if( entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state = gconf_value_get_bool( entry->value );

		page->private->highlight = state;
		gtk_source_buffer_set_highlight( GTK_SOURCE_BUFFER( page ),
						 state );
	}
}

static void screem_page_monitor_cb( GnomeVFSMonitorHandle *handle,
				    const gchar *monitor_uri,
				    const gchar *info_uri,
				    GnomeVFSMonitorEventType type,
				    gpointer data )
{
	ScreemPage *page;

	page = SCREEM_PAGE( data );

	switch( type ) {
	case GNOME_VFS_MONITOR_EVENT_CHANGED:
		page->private->modified = TRUE;
		break;
	case GNOME_VFS_MONITOR_EVENT_DELETED:
		/* FIXME: hmm, should alert the user really */
		break;
	case GNOME_VFS_MONITOR_EVENT_STARTEXECUTING:
		break;
	case GNOME_VFS_MONITOR_EVENT_STOPEXECUTING:
		break;
	case GNOME_VFS_MONITOR_EVENT_CREATED:
		page->private->modified = FALSE;
		break;
	case GNOME_VFS_MONITOR_EVENT_METADATA_CHANGED:
		break;
	default:
		break;
	}
}

gboolean screem_page_is_feature( ScreemPage *page,
				const gchar *feature )
{
	GtkSourceLanguagesManager *lm;
	GHashTable *table;
	const gchar *mime_type;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	mime_type = screem_page_get_mime_type( page );

	ret = FALSE;
	lm = screem_application_load_syntax_tables();
	if( lm ) {
		table = g_object_get_data( G_OBJECT( lm ), "features" );
		table = g_hash_table_lookup( table, feature );
		if( table && mime_type ) {
			ret = ( g_hash_table_lookup( table, mime_type ) != NULL );
		}
	}
	
	return ret;
}

gboolean screem_page_is_dynamic( ScreemPage *page )
{
	return screem_page_is_feature( page, "dynamic" );
}

gboolean screem_page_is_markup( ScreemPage *page )
{
	return screem_page_is_feature( page, "markup" );
}

gboolean screem_page_is_template( ScreemPage *page, gchar **tag,
				  gchar **path, GSList **blocks )
{
	gchar *text;
	gint len;
	gchar *name;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );
	g_return_val_if_fail( tag != NULL, FALSE );
	g_return_val_if_fail( path != NULL, FALSE );

	if( ! screem_page_is_markup( page ) ) {
		return FALSE;
	}
	
	text = screem_page_get_data( page );
	name = find_text( text, "<!-- #BeginTemplate[ \t]+\"[^\"]*\"[ \t]*-->",
			  NULL, &len );
	ret = ( name != NULL );
	if( ret ) {
		const gchar *pathname;
		gchar *end;
		*tag= g_strndup( name, len );
		name += strlen( "<!-- #BeginTemplate \"" );
		end = strchr( name, '"' );
		*path = g_strndup( name, end - name );
				
		pathname = screem_page_get_pathname( page );
		if( ! gnome_vfs_uris_match( pathname, *path ) ) {
			/* page is not the master template, it just uses it */
			g_free( *path );
			g_free( *tag );
			ret = FALSE;
		} else if( blocks ){
			/* build list of blocks */
			gchar *template_end;
			name = strstr( name, "-->" );
			g_assert( name );
			name += strlen( "-->" );
		
			template_end = strstr( name, "<!-- #EndTemplate -->" );
			
			do {
				gchar *ed_name = NULL;

				end = strstr( name, "<!-- #BeginEditable" );
				if( end ) {
					/* get name of editable section */
					
					ed_name = find_text( end, "\"[^\"]*\"",
								NULL, &len );
					end = strstr( end, "-->" );
					if( ed_name > end ) {
						ed_name = g_strdup( "" );
					} else {
						ed_name = g_strndup( ed_name,
									len );
					}
				}
				if( ! end ) {
					if( template_end ) {
						*blocks = g_slist_prepend( *blocks, g_strndup( name, template_end - name ) );		
					} else {
						*blocks = g_slist_prepend( *blocks, g_strdup( name ) );
					}
				} else {
					end += strlen( "-->" );
					*blocks = g_slist_prepend( *blocks, g_strndup( name, end - name ) );
					*blocks = g_slist_prepend( *blocks, ed_name );
					end = strstr( end, "<!-- #EndEditable -->" );
				}
				name = end;
			} while( name );
			*blocks = g_slist_reverse( *blocks );
		}
	}
	
	g_free( text );
	
	return ret;
}

GtkTreeModel *screem_page_get_model( ScreemPage *page )
{
	return GTK_TREE_MODEL( page->private->model );
}

void screem_page_build_model( ScreemPage *page )
{
	screem_page_model_build_model( page->private->model );
}

void screem_page_ensure_model_built( ScreemPage *page )
{
	screem_page_model_ensure_built( page->private->model );
}

gboolean screem_page_select_context( ScreemPage *page,
				  guint pos,
				  guint *start, guint *end,
				  gboolean select_text )
{
	return screem_page_model_select_context( page->private->model,
			pos, start, end, select_text );
}

gchar *screem_page_query_context( ScreemPage *page,
		guint pos, gboolean query_text,
		gboolean build, 
		guint *depth, guint *start, guint *end )
{
	return screem_page_model_query_context( page->private->model,
			pos, query_text, build, depth, start, end );
}

ScreemDTD *screem_page_get_dtd( ScreemPage *page )
{
	ScreemDTD *ret;

	ret = NULL;
	g_object_get( G_OBJECT( page->private->model ),
			"dtd", &ret, NULL );

	g_object_unref( ret );
	
	return ret;
}

void screem_page_model_emit_building( ScreemPage *page )
{
	g_signal_emit( G_OBJECT( page ),
		       screem_page_signals[ BUILDING ], 0 );
}

void screem_page_model_emit_built( ScreemPage *page )
{
	g_signal_emit( G_OBJECT( page ),
		       screem_page_signals[ BUILT ], 0 );
}

/* G Object stuff */
#define PARENT_TYPE GTK_TYPE_SOURCE_BUFFER

static gpointer parent_class;

static void screem_page_class_init( ScreemPageClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_page_finalize;
	object_class->get_property = screem_page_get_prop;
	object_class->set_property = screem_page_set_prop;

	g_object_class_install_property(object_class,
					PROP_APP,
					g_param_spec_object("app",
							    "Application",
							    "The Application",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE |
							    G_PARAM_CONSTRUCT)
					);
	g_object_class_install_property(object_class,
					PROP_PATHNAME,
					g_param_spec_string("pathname",
							    "Page pathname",
							    "The pathname of the page",
							    "",
							    G_PARAM_READWRITE)
					);
	g_object_class_install_property(object_class,
					PROP_CHANGED,
					g_param_spec_boolean("changed",
							     "changed",
							     "If the page has changed",
							     FALSE,
							     G_PARAM_READWRITE)
					);
	g_object_class_install_property(object_class,
					PROP_MIMETYPE,
					g_param_spec_string("mime-type",
							    "Page mime type",
							    "The mime-type of the page",
							    "",
							    G_PARAM_READWRITE)
					);
	g_object_class_install_property(object_class,
					PROP_OPEN,
					g_param_spec_boolean("open",
							     "open",
							     "If the page is open",
							     FALSE,
							     G_PARAM_READWRITE)
					);
	g_object_class_install_property(object_class,
					PROP_CHARSET,
					g_param_spec_string("charset",
							    "Page character set",
							    "The character set of the page",
							    "",
							    G_PARAM_READABLE)
					);
	g_object_class_install_property(object_class,
					PROP_DTD,
					g_param_spec_object("dtd",
							    "DTD",
							    "The ScreemDTD for the page",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE)
					);
	
	screem_page_signals[ SAVED ] =
		g_signal_new( "saved",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemPageClass, 
					       saved),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	screem_page_signals[ BUILDING ] =
		g_signal_new( "building",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemPageClass, 
					       building ),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	screem_page_signals[ BUILT ] =
		g_signal_new( "built",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemPageClass, 
					       built),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
}

static void screem_page_init( ScreemPage *page )
{
	page->private = g_new0( ScreemPagePrivate, 1 );

	page->private->client = gconf_client_get_default();
	page->private->highlight = gconf_client_get_bool(page->private->client, "/apps/screem/editor/highlight",
							 NULL );
	page->private->highlight_notify = 
		gconf_client_notify_add( page->private->client,
					 "/apps/screem/editor/highlight",
					 screem_page_highlight_notify,
					 page, NULL, NULL );
	
	page->private->bookmarks = g_hash_table_new( g_str_hash, g_str_equal );

	/* NAME, LINE, BOOKMARK STRUCT */
	page->private->bookmarkstore = gtk_list_store_new( 3, G_TYPE_STRING,
							   G_TYPE_STRING,
							   G_TYPE_POINTER,
							   NULL );	
	page->private->mime_type = g_strdup( "text/plain" );
	
	g_signal_connect( G_OBJECT( page ),
			  "changed", G_CALLBACK( changed ), page );

}

static void screem_page_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec )
{
	ScreemPage *page;
	GObject *obj;

	page = SCREEM_PAGE( object );

	switch( prop_id ) {
		case PROP_APP:
			obj = g_value_get_object( value );
			page->private->model = screem_page_model_new( page, obj );
			gtk_source_buffer_set_highlight( GTK_SOURCE_BUFFER( page ), page->private->highlight );
			break;
		case PROP_PATHNAME:
			screem_page_set_pathname_real( page, g_value_get_string( value ) );
			break;
		case PROP_CHANGED:
			screem_page_set_changed_real( page, g_value_get_boolean( value ) );
			break;
		case PROP_MIMETYPE:
			screem_page_set_mime_type_real( page, g_value_get_string( value ) );
			break;
		case PROP_OPEN:
			screem_page_set_open_real( page, g_value_get_boolean( value ) );
			break;
		case PROP_DTD:
			break;
	}
}

static void screem_page_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec )
{
	ScreemPage *page;
	GObject *obj;
	gchar *str;
	const gchar *cstr;
	
	page = SCREEM_PAGE( object );

	switch( prop_id ) {
		case PROP_APP:
			g_object_get( G_OBJECT( page->private->model ),
					"application", &obj, NULL );
			g_value_set_object( value, obj );
			g_object_unref( obj );
			break;
		case PROP_PATHNAME:
			cstr = screem_page_get_pathname( page );
			if( cstr ) {
				g_value_set_string( value, cstr );
			}
			break;
		case PROP_CHANGED:
			g_value_set_boolean( value, screem_page_get_changed( page ) );
			break;
		case PROP_MIMETYPE:
			cstr = screem_page_get_mime_type( page );
			if( cstr ) {
				g_value_set_string( value, cstr );
			}
			break;
		case PROP_OPEN:
			g_value_set_boolean( value, screem_page_is_open( page ) );
			break;
		case PROP_CHARSET:
			g_object_get( G_OBJECT( page->private->model ),
					"charset", &str, NULL );
			if( str ) {
				g_value_set_string( value, str );
				g_free( str );
			}
			break;
		case PROP_DTD:
			g_object_get( G_OBJECT( page->private->model ),
					"dtd", &obj, NULL );
			g_value_set_object( value, obj );
			g_object_unref( obj );
			break;
		default:
			break;
	}
}


static void screem_page_finalize( GObject *object )
{
	ScreemPage *page;

	page = SCREEM_PAGE( object );

	gconf_client_notify_remove( page->private->client, 
				    page->private->highlight_notify );

	g_hash_table_foreach( page->private->bookmarks,
			      (GHFunc)screem_page_clear_bookmarks, page );
	g_hash_table_destroy( page->private->bookmarks );
	g_object_unref( page->private->bookmarkstore );

	g_object_unref( page->private->client );

	g_object_unref( page->private->model );

	g_free( page->private->pathname );
	g_free( page->private->mime_type );

	if( page->private->monitor ) {
		gnome_vfs_monitor_cancel( page->private->monitor );
	}

	g_free( page->private );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType screem_page_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemPageClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_page_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemPage ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_page_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemPage",
					       &info, 0 );
	}

	return type;
}
