/*  Screem:  screem-site-view.c,
 * 
 *  file tree model handling for a ScreemSite
 *
 *  Copyright (C) 2001, 2002  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>

#include <time.h>

#include <pwd.h>
#include <grp.h>
#include <sys/types.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include <glade/glade.h>

#include <gconf/gconf-client.h>

#include <gtk/gtkmain.h>

#include <gtk/gtktreeview.h>
#include <gtk/gtktreeselection.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrendererpixbuf.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkimage.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtktooltips.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <libgnome/gnome-i18n.h>


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include "screem-site.h"
#include "screem-site-view.h"

#include "screem-application.h"

#include "fileops.h"

typedef struct {
	gchar *uri;
	gchar *mime_type;
	gint number;
	gint total_size;
} Stats;

enum {
	TARGET_URI_LIST
};

static const GtkTargetEntry drag_types[] = {
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static const int num_drag_types = sizeof(drag_types) / sizeof(drag_types[ 0 ]);


static const GtkTargetEntry drop_types[] = {
	{ "text/uri-list", 0, TARGET_URI_LIST }
};
static const int num_drop_types = sizeof(drop_types) / sizeof(drop_types[ 0 ]);

static GnomeUIInfo dnd_menu[] = {
        { GNOME_APP_UI_ITEM, N_("_Move here"), N_("Move file"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        { GNOME_APP_UI_ITEM, N_("_Copy here"), N_("Copy file"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_SEPARATOR,
        { GNOME_APP_UI_ITEM, N_("Cancel drag"), N_("Cancel drag"),
          0, NULL, NULL,
          GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BLANK, 0,
          GDK_CONTROL_MASK, NULL },
        GNOMEUIINFO_END
};


static void screem_site_view_row_deleted( GtkTreeModel *model, 
					  GtkTreePath *path,
					  GtkTreeIter *it, gpointer data );
static gboolean screem_site_view_motion( GtkWidget *widget, 
					 GdkDragContext *context, 
					 gint x, gint y, guint time, 
					 gpointer data );
static void screem_site_view_set_dnd_data( GtkWidget *widget, 
					   GdkDragContext *context,
					   GtkSelectionData *data,
					   guint info, guint time,
					   gpointer udata );
static void screem_site_view_dnd_delete( GtkWidget *widget,
					 GdkDragContext *context,
					 gpointer data );
static void screem_site_view_drop( GtkWidget *widget, GdkDragContext *context,
				   gint x, gint y, 
				   GtkSelectionData *selectionData,
				   guint info, guint time, 
				   gpointer data );

static gboolean screem_site_view_drop_uris( ScreemSite *site,
					    gchar **uris,
					    GtkTreeIter *it,
					    const gchar *target_uri,
					    gboolean moving );

static gboolean screem_site_view_find_iter( ScreemSiteViewNodeInfo *info,
					    const gchar *uri,
					    GtkTreeIter *parent,
					    GtkTreeIter *it );


static void screem_site_view_fm_expanded( GtkTreeView *view,
					  GtkTreeIter *iter,
					  GtkTreePath *path,
					  gpointer data );
static void screem_site_view_fm_activated( GtkTreeView *view,
					   GtkTreePath *path,
					   GtkTreeViewColumn *column,
					   gpointer data );

static GList *screem_site_view_calc_stats( GtkTreeModel *model, 
					   GtkTreeIter *it, GList *types );


gint screem_site_view_compare_func( GtkTreeModel *model, 
				    GtkTreeIter *a, GtkTreeIter *b,
				    ScreemSite *site )
{
        ScreemSiteViewNodeInfo *ainfo;
        ScreemSiteViewNodeInfo *binfo;
	gchar *aname;
	gchar *bname;
	const gchar *spath;
	
	int ret = 0;

	spath = screem_site_get_pathname( site );
	
	gtk_tree_model_get( model, a, 
			FILE_BROWSER_DATA_COL, &ainfo,
			FILE_BROWSER_NAME_COL, &aname,
			-1 );
	gtk_tree_model_get( model, b, 
			FILE_BROWSER_DATA_COL, &binfo,
			FILE_BROWSER_NAME_COL, &bname,
			-1 );

	/* special root case */
	if( ainfo && ! strcmp( spath, ainfo->fullname ) ) {
		ret = -1;
	} else if( binfo && ! strcmp( spath, binfo->fullname ) ) {
		ret = 1;
	/* folders come before files */
	} else if( ainfo && binfo && 
		   ainfo->type == SCREEM_SITE_VIEW_FOLDER &&
		   binfo->type != SCREEM_SITE_VIEW_FOLDER ) {
		ret = -1;
	} else if( ainfo && binfo && 
		 binfo->type == SCREEM_SITE_VIEW_FOLDER &&
		 ainfo->type != SCREEM_SITE_VIEW_FOLDER ) {
		ret = 1;
	/* handle unset names */
	} else if( aname && ! bname ) {
		ret = 1;
	} else if( bname && ! aname ) {
		ret = -1;
	/* default name comparison */
	} else {
		ret = strcmp( aname, bname );
	}

	g_free( aname );
	g_free( bname );

	return ret;
}


ScreemSiteViewNodeInfo* 
screem_site_view_node_info_new( ScreemSiteViewNodeType type, ScreemSite *site,
				gpointer data, struct stat *s )
{
	ScreemSiteViewNodeInfo *info;

	info = g_new0( ScreemSiteViewNodeInfo, 1 );

	info->type = type;
	if( data ) {
		info->fullname = g_strdup( data );
	}
	info->site = site;

	return info;
}

void screem_site_view_node_info_destroy( ScreemSiteViewNodeInfo *info )
{
	if( info ) {
		g_free( info->fullname );
		g_free( info );
	}
}

void screem_site_view_set_val_at( GtkTreeStore *store, GtkTreeIter *iter,
				  ScreemSiteViewNodeInfo *info)
{
	ScreemSite *site;
	gchar flags[ 3 ] = { '.', '.', '.' };
	gchar *flagstxt;
	
	site = info->site;

	if( screem_site_is_excluded( site, info->fullname ) ) {
		flags[ 0 ] = 'E';
	}
	if( screem_site_is_ignored( site, info->fullname ) ) {
		flags[ 2 ] = 'I';
	}
	if( screem_site_is_ascii( site, info->fullname ) ) {
		flags[ 1 ] = 'A';
	}

	flagstxt = g_strdup_printf( "%c%c%c", 
				    flags[ 0 ], flags[ 1 ], flags[ 2 ] );

	/* name is what we want to display */
	gtk_tree_store_set( store, iter, 
			    FILE_BROWSER_USER_COL, flagstxt, 
			    FILE_BROWSER_DATA_COL, info, 
			    FILE_BROWSER_URI_COL, info->fullname,
			    -1 );
	g_free( flagstxt );
}

void screem_site_view_destroy( ScreemSite *site )
{
	ScreemSitePrivate *private;
	ScreemSiteView *view;

	private = site->private;
	view = private->view;

	g_free( view->images );
	g_free( view->styles );
	g_free( view->scripts );
		
	g_object_unref( view->browser );

	g_free( view );
}

void screem_site_init_tree_view( GtkWidget *tree )
{
	GtkCellRenderer *prenderer;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *selection;
	GtkTooltips *tip;
	
	gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( tree ), TRUE );
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( tree ),
					   TRUE );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree ) );
	gtk_tree_selection_set_mode( GTK_TREE_SELECTION( selection ),
				     GTK_SELECTION_SINGLE );
	
	/* column 1 */
	prenderer = gtk_cell_renderer_pixbuf_new();
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_pack_start( column, prenderer, FALSE );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_column_set_resizable( column, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( tree ), column );
	
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 
					     FILE_BROWSER_NAME_COL,
					     NULL );
	gtk_tree_view_column_set_attributes( column, prenderer,
					     "pixbuf",
					     FILE_BROWSER_ICON_COL,
					     NULL );
	
	gtk_tree_view_column_set_sort_column_id( column,
						 FILE_BROWSER_NAME_COL );

	tip = gtk_tooltips_new();
	gtk_tooltips_set_tip( tip, column->button,
				_( "Site name" ),
				"" );
	
	/* column 2 */
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, "" );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( tree ), column );
	
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 
					     FILE_BROWSER_USER_COL,
					     NULL );
	
	tip = gtk_tooltips_new();
	gtk_tooltips_set_tip( tip, column->button,
				_( "Upload Flags" ),
				"" );
	
	gtk_tree_view_columns_autosize( GTK_TREE_VIEW( tree ) );
	
	gtk_drag_source_set(tree, GDK_BUTTON1_MASK | GDK_BUTTON2_MASK,
			    drag_types, num_drag_types,
			    GDK_ACTION_COPY |GDK_ACTION_MOVE | GDK_ACTION_ASK);
	gtk_drag_dest_set( tree,
			   GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP |
			   GTK_DEST_DEFAULT_MOTION,
			   drop_types, num_drop_types,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_ASK);
	
	g_signal_connect( G_OBJECT( tree ), "drag_motion",
			  G_CALLBACK( screem_site_view_motion ), NULL );
	g_signal_connect( G_OBJECT( tree ), "drag_data_get",
			  G_CALLBACK( screem_site_view_set_dnd_data ), NULL );
	g_signal_connect( G_OBJECT( tree ), "drag_data_delete",
			  G_CALLBACK( screem_site_view_dnd_delete ), NULL );
	g_signal_connect( G_OBJECT( tree ), "drag_data_received",
			  G_CALLBACK( screem_site_view_drop ), NULL );

	g_signal_connect( G_OBJECT( tree ), "row_activated",
			  G_CALLBACK( screem_site_view_fm_activated ),
			  NULL );
	g_signal_connect( G_OBJECT( tree ), "row_expanded",
			  G_CALLBACK( screem_site_view_fm_expanded ),
			  NULL );
}

GtkTreeModel *screem_site_get_model( ScreemSite *site )
{
	g_return_val_if_fail( SCREEM_IS_SITE( site ), NULL );

	return screem_file_browser_get_model( site->private->view->browser );

}

gboolean screem_site_view_create_dir( ScreemSite *site, GtkWidget *widget,
				      const gchar *dir )
{
	gboolean ret = FALSE;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	GValue value = {0,};
	ScreemSiteViewNodeInfo *node_info;

	gchar *newdir = NULL;

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	gtk_tree_selection_get_selected( selection, &model, &it );

	gtk_tree_model_get_value( model, &it, FILE_BROWSER_DATA_COL, &value );
	node_info = g_value_get_pointer( &value );

	if( ! node_info ) {
		newdir = g_strconcat( screem_site_get_pathname( site ),
				      "/", dir, NULL );
	} else if( node_info->type != SCREEM_SITE_VIEW_FOLDER ) {
		GtkTreeIter pit;
		GValue pvalue = { 0 };
		gtk_tree_model_iter_parent( model, &pit, &it );
		gtk_tree_model_get_value( model, &pit, FILE_BROWSER_DATA_COL,
					  &pvalue );
		node_info = g_value_get_pointer( &pvalue );
		if( ! node_info ) {
			return FALSE;
		}
		memcpy( &it, &pit, sizeof( GtkTreeIter ) );
	}

	if( node_info ) {
		newdir = g_strconcat( node_info->fullname, "/", dir, NULL );
	}

	ret = mkdir_recursive( newdir,
			       GNOME_VFS_PERM_USER_ALL |
			       GNOME_VFS_PERM_GROUP_ALL |
			       GNOME_VFS_PERM_OTHER_READ |
			       GNOME_VFS_PERM_OTHER_EXEC );

	g_free( newdir );

	return ret;
}

void screem_site_view_show_properties( ScreemSite *site,
					const gchar *uri,
				       GtkWindow *window )
{
	GladeXML *xml;
	GConfClient *client;
	GtkWidget *dialog;
	GtkWidget *widget;
	gchar *gladepath;

	GnomeVFSFileInfoOptions options;
	GnomeVFSFileInfo *vfsinfo;
	GtkWidget *label;

	GdkPixbuf *pixbuf;
	GtkWidget *image;

	struct passwd *userinfo;
	struct group *fgroup;
	struct group *group;
	GtkWidget *menu;
	gint grouppos;
	gint i;
	gchar *title;

	gchar *name;

	ScreemFileBrowser *browser;
	
	browser = site->private->view->browser;
	
	client = gconf_client_get_default();
	gladepath = gconf_client_get_string( client,
					     "/apps/screem/general/glade_path",
					     NULL );
	xml = glade_xml_new( gladepath, "file_properties", NULL );
	g_free( gladepath );

	g_object_unref( client );

	if( screem_site_get_fake_flag( site ) ) {
		widget = glade_xml_get_widget( xml, "exclude" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "ascii" );
		gtk_widget_set_sensitive( widget, FALSE );
		widget = glade_xml_get_widget( xml, "ignore" );
		gtk_widget_set_sensitive( widget, FALSE );
	}
	
	dialog = glade_xml_get_widget( xml, "file_properties" );

	vfsinfo = gnome_vfs_file_info_new();
	options = GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS;

	gnome_vfs_get_file_info( uri, vfsinfo, options );

	title = g_strconcat( vfsinfo->name, " - Properties", NULL );
	gtk_window_set_title( GTK_WINDOW( dialog ), title );
	g_free( title );

	label = glade_xml_get_widget( xml, "basename" );
	gtk_entry_set_text( GTK_ENTRY( label ), vfsinfo->name );

	label = glade_xml_get_widget( xml, "size" );
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE ) {
		gchar *txt;

		txt = gnome_vfs_format_file_size_for_display( vfsinfo->size );
		gtk_label_set_text( GTK_LABEL( label ), txt );
		g_free( txt );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}

	label = glade_xml_get_widget( xml, "location" );
	{
		gchar *dirname;

		dirname = g_dirname( uri );	
		gtk_label_set_text( GTK_LABEL( label ), dirname );
		g_free( dirname );
	}

	label = glade_xml_get_widget( xml, "mime_type" );
	image = glade_xml_get_widget( xml, "file_image" );
	
	pixbuf = screem_file_browser_get_icon( browser, uri, 48, 48,
						TRUE, NULL );	
			
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MIME_TYPE ) {
		gtk_label_set_text( GTK_LABEL( label ), vfsinfo->mime_type );
		label = glade_xml_get_widget( xml, "type" );
		gtk_label_set_text( GTK_LABEL( label ),
				    gnome_vfs_mime_get_description( vfsinfo->mime_type ) );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
		label = glade_xml_get_widget( xml, "type" );
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}

	gtk_image_set_from_pixbuf( GTK_IMAGE( image ), pixbuf );
	g_object_unref( pixbuf );
	
	label = glade_xml_get_widget( xml, "modified" );
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME ) {
		gchar *t = ctime( &vfsinfo->mtime );
		t[ strlen( t ) - 1 ] = '\0';
		gtk_label_set_text( GTK_LABEL( label ), t );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}


	label = glade_xml_get_widget( xml, "accessed" );
	if( vfsinfo->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME ) {
		gchar *t = ctime( &vfsinfo->atime );
		t[ strlen( t ) - 1 ] = '\0';
		gtk_label_set_text( GTK_LABEL( label ), t );
	} else {
		gtk_label_set_text( GTK_LABEL( label ), _( "Unknown" ) );
	}

	userinfo = getpwuid( vfsinfo->uid );
	if( userinfo ) {
		gchar *t;

		label = glade_xml_get_widget( xml, "user" );
		t = g_strconcat( userinfo->pw_name, " - ",
				 userinfo->pw_gecos, NULL );
		gtk_label_set_text( GTK_LABEL( label ), t );
		g_free( t );
	}
	
	userinfo = getpwuid( geteuid() );

	fgroup = getgrgid( vfsinfo->gid );
	grouppos = -1;
	i = 0;
	menu = gtk_menu_new();
	while( ( group = getgrent() ) ) {
		label = gtk_menu_item_new_with_label( group->gr_name );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), label );
		if( ! strcmp( group->gr_name, fgroup->gr_name ) ) {
			gtk_widget_show( label );
			grouppos = i;
		} else {
			gint ni;

			for( ni = 0; group->gr_mem[ ni ]; ++ ni ) {
				if( ! strcmp( group->gr_mem[ ni ],
					      userinfo->pw_name ) ) {
					gtk_widget_show( label );
					break;
				}
			}
		}
		g_object_set_data( G_OBJECT( label ), "gid",
				   GUINT_TO_POINTER( group->gr_gid ) );
		i ++;
	}
	endgrent();
	if( grouppos == -1 ) {
		label = gtk_menu_item_new_with_label( fgroup->gr_name );
		gtk_widget_show( label );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ), label );
		grouppos = i;
		g_object_set_data( G_OBJECT( label ), "gid",
				   GUINT_TO_POINTER( group->gr_gid ) );
	}

	label = glade_xml_get_widget( xml, "group" );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( label ),
				  menu );
	gtk_option_menu_set_history( GTK_OPTION_MENU( label ),
				     grouppos );
	
	label = glade_xml_get_widget( xml, "owner_read" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_USER_READ ) );	
	label = glade_xml_get_widget( xml, "owner_write" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_USER_WRITE ) );
	label = glade_xml_get_widget( xml, "owner_exec" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_USER_EXEC ) );
	label = glade_xml_get_widget( xml, "group_read" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_GROUP_READ ) );	
	label = glade_xml_get_widget( xml, "group_write" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_GROUP_WRITE ) );
	label = glade_xml_get_widget( xml, "group_exec" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_GROUP_EXEC ) );
	label = glade_xml_get_widget( xml, "all_read" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_OTHER_READ ) );	
	label = glade_xml_get_widget( xml, "all_write" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_OTHER_WRITE ) );
	label = glade_xml_get_widget( xml, "all_exec" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      ( vfsinfo->permissions & 
					GNOME_VFS_PERM_OTHER_EXEC ) );

	label = glade_xml_get_widget( xml, "exclude" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      screem_site_is_excluded(site,
							      uri));
	label = glade_xml_get_widget( xml, "ascii" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      screem_site_is_ascii(site,
							   uri));
	label = glade_xml_get_widget( xml, "ignore" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( label ),
				      screem_site_is_ignored(site,
							     uri));


	gnome_vfs_file_info_unref( vfsinfo );

	g_object_set_data( G_OBJECT( dialog ), "site", site );
	g_object_set_data( G_OBJECT( dialog ), "fullname",
			   g_strdup( uri ) );

	glade_xml_signal_autoconnect( xml );

	gtk_window_set_transient_for( GTK_WINDOW( dialog ), window );
	gtk_dialog_run( GTK_DIALOG( dialog ) );

	name = g_object_get_data( G_OBJECT( dialog ), "fullname" );
	if( name ) {
		g_free( name );
	}

	g_object_set_data( G_OBJECT( dialog ), "fullname", NULL );
			   
	gtk_widget_destroy( dialog );
}

void screem_site_view_properties_rename( GtkWidget *widget )
{
	GladeXML *xml;
	const gchar *name;
	ScreemSite *site;
	const gchar *fullname;
	gchar *base;
	name = gtk_entry_get_text( GTK_ENTRY( widget ) );

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );

	base = NULL;
	if( fullname ) {
		base = g_path_get_basename( fullname );
	}
	if( base && strcmp( name, base ) ) {
		/* file has been renamed */
		gchar *dirname;
		gchar *newpath;
		ScreemPage *page;
		gboolean done;

		page = screem_site_locate_page( site, fullname );
		
		dirname = g_dirname( fullname );
		newpath = g_strconcat( dirname, G_DIR_SEPARATOR_S,
				       name, NULL );

		if( page ) {
			screem_page_set_pathname( page, newpath );
		}
		if( screem_uri_is_dir( fullname ) ) {
	       		done = copy_dir( fullname, newpath, FALSE );
		} else {
			done = copy_file( fullname, newpath );
		}
		if( done ) {
			screem_site_file_change( site,
						 fullname,
						 newpath );
			
			delete_dir( fullname );
			
			g_object_set_data( G_OBJECT( widget ), "fullname",
					   newpath );
			g_free( (gchar*)fullname );
		} else if( page ) {
			screem_page_set_pathname( page, fullname );
			g_free( newpath );
		}

		g_free( dirname );
	}
	g_free( base );
}

void screem_site_view_properties_chown( GtkWidget *widget )
{
	GladeXML *xml;
	GnomeVFSFileInfo *vfsinfo;
	guint gid;
	const gchar *fullname;

	xml = glade_get_widget_tree( widget );

	widget = GTK_OPTION_MENU( widget )->menu_item;
	gid = GPOINTER_TO_UINT( g_object_get_data( G_OBJECT( widget ),
						   "gid" ) );

	widget = glade_xml_get_widget( xml, "file_properties" );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );
	if( fullname ) {
		vfsinfo = gnome_vfs_file_info_new();
		
		gnome_vfs_get_file_info( fullname, vfsinfo,
					 GNOME_VFS_FILE_INFO_FOLLOW_LINKS );
		
		vfsinfo->gid = gid;
		
		gnome_vfs_set_file_info( fullname, vfsinfo,
					 GNOME_VFS_SET_FILE_INFO_OWNER );
		
		gnome_vfs_file_info_unref( vfsinfo );
	}
}

void screem_site_view_properties_chmod( GtkWidget *widget )
{
	GladeXML *xml;
	GnomeVFSFileInfo *vfsinfo;
	GnomeVFSFilePermissions perms;
	const gchar *fullname;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );

	perms = 0;
	widget = glade_xml_get_widget( xml, "owner_read" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_USER_READ;
	}
	widget = glade_xml_get_widget( xml, "owner_write" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_USER_WRITE;
	}
	widget = glade_xml_get_widget( xml, "owner_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_USER_EXEC;
	}
	widget = glade_xml_get_widget( xml, "group_read" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_GROUP_READ;
	}
	widget = glade_xml_get_widget( xml, "group_write" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_GROUP_WRITE;
	}
	widget = glade_xml_get_widget( xml, "group_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_GROUP_EXEC;
	}
	widget = glade_xml_get_widget( xml, "all_read" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_OTHER_READ;
	}
	widget = glade_xml_get_widget( xml, "all_write" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_OTHER_WRITE;
	}
	widget = glade_xml_get_widget( xml, "all_exec" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) ) ) {
		perms |= GNOME_VFS_PERM_OTHER_EXEC;
	}

	if( fullname ) {
		vfsinfo = gnome_vfs_file_info_new();
		
		vfsinfo->permissions = perms;
		
		gnome_vfs_set_file_info( fullname, vfsinfo,
					 GNOME_VFS_SET_FILE_INFO_PERMISSIONS );
		
		gnome_vfs_file_info_unref( vfsinfo );
	}
}

void screem_site_view_properties_exclude( GtkWidget *widget )
{
	GladeXML *xml;
	gboolean exclude;
	const gchar *fullname;
	ScreemSite *site;

	exclude = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );
	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ),
					       "site" ) );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );

	if( fullname ) {
		if( exclude ) {
			screem_site_add_exclude( site, fullname );
		} else {
			screem_site_remove_exclude( site, fullname );
		}
	}
}

void screem_site_view_properties_ascii( GtkWidget *widget )
{
	GladeXML *xml;
	gboolean ascii;
	const gchar *fullname;
	ScreemSite *site;

	ascii = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );
	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ),
					       "site" ) );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );

	if( fullname ) {
		if( ascii ) {
			screem_site_add_ascii( site, fullname );
		} else {
			screem_site_remove_ascii( site, fullname );
		}
	}
}

void screem_site_view_properties_ignore( GtkWidget *widget )
{
	GladeXML *xml;
	gboolean ignore;
	const gchar *fullname;
	ScreemSite *site;

	ignore = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "file_properties" );
	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ),
					       "site" ) );
	fullname = (const gchar*)g_object_get_data( G_OBJECT( widget ),
						    "fullname" );
	
	if( fullname ) {
		if( ignore ) {
			screem_site_add_ignore( site, fullname );
		} else {
			screem_site_remove_ignore( site, fullname );
		}
	}
}

static void screem_site_view_row_deleted( GtkTreeModel *model, 
					  GtkTreePath *path,
					  GtkTreeIter *it, gpointer data )
{
	gchar *string;
	
	string = gtk_tree_path_to_string( path );

	if( string && *string == '0' ) {
		GValue value = { 0 };
		GtkTreeIter iter;
		ScreemSiteViewNodeInfo *info;
		ScreemSiteView *view;

		gtk_tree_model_get_value( model, it, FILE_BROWSER_DATA_COL,
					  &value );

		info = g_value_get_pointer( &value );
		g_value_unset( &value );

		/* before destroying we need to remove it from
		   the quick access nodes, Images, Scripts, Stylesheets etc */
		view = info->site->private->view;
		if( view->images &&
		    screem_site_view_find_iter( info, NULL, view->images,
						&iter ) ) {
			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &iter );
		}
		if( view->styles &&
		    screem_site_view_find_iter( info, NULL, view->styles,
						&iter ) ) {
			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &iter );
		}
		if( view->scripts &&
		    screem_site_view_find_iter( info, NULL, view->scripts,
						&iter ) ) {
			gtk_tree_store_remove( GTK_TREE_STORE( model ),
					       &iter );
		}
		
		screem_site_view_node_info_destroy( info );
	}
	g_free( string );
}

static gboolean screem_site_view_motion( GtkWidget *widget, 
					 GdkDragContext *context, 
					 gint x, gint y, guint time, 
					 gpointer data )
{
	GdkDragAction action;
       	GdkModifierType modifiers;
	GtkTreePath *path;
	GtkTreeViewColumn *column;
	gint cx;
	gint cy;
	
        gdk_window_get_pointer (NULL, NULL, NULL, &modifiers);
	
	gtk_tree_view_get_path_at_pos( GTK_TREE_VIEW( widget ),
				       x, y, &path, &column,
				       &cx, &cy );
      
        if( context->suggested_action != GDK_ACTION_ASK ) {
                action = GDK_ACTION_COPY;
	} else {
                action = GDK_ACTION_ASK;
	}

	if( ( modifiers & GDK_MOD1_MASK ) != 0 ) {
                action = GDK_ACTION_ASK;
	}
        
        gdk_drag_status( context, action, time );

        return TRUE;
}

static void screem_site_view_set_dnd_data( GtkWidget *widget, 
					   GdkDragContext *context,
					   GtkSelectionData *data,
					   guint info, guint time,
					   gpointer udata )
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	GValue value = {0,};
	const gchar *uri;
	ScreemSite *site;

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	
	gtk_tree_selection_get_selected( selection, &model, &it );

	gtk_tree_model_get_value( model, &it, FILE_BROWSER_URI_COL, &value );
	uri = g_value_get_string( &value );

	if( uri ) {
		gtk_selection_data_set( data, data->target,
					8, uri,
					strlen( uri ) );
	} else {
		gtk_selection_data_set( data, data->target,
					8, NULL, 0 );
	}
}

static void screem_site_view_dnd_delete( GtkWidget *widget,
					 GdkDragContext *context,
					 gpointer data )
{
	ScreemSite *site;

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );

	g_signal_stop_emission_by_name( G_OBJECT( widget ), "drag_data_delete" );
}


static void screem_site_view_drop( GtkWidget *widget, GdkDragContext *context,
				   gint x, gint y, 
				   GtkSelectionData *selectionData,
				   guint info, guint time, 
				   gpointer data)
{
	gboolean moving;
	gchar **uris;
	const gchar *site_path;
	GtkTreeModel *model;
	GtkTreePath *path;
	GtkTreeIter it;
	GtkTreeSelection *selection;
	GValue value = {0};
	const gchar *target_uri;
	ScreemSite *site;

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site" ) );

	g_signal_stop_emission_by_name( G_OBJECT( widget ),
					"drag_data_received" );
	
	if( context->action == GDK_ACTION_ASK ) {
		GtkWidget *popup;
		gint item;
		popup = gnome_popup_menu_new( dnd_menu );
		item = gnome_popup_menu_do_popup_modal( popup, 0, 0, 0, 0, widget );
		switch( item ) {
		case 0:
			context->action = GDK_ACTION_MOVE;
			break;
		case 1:
			context->action = GDK_ACTION_COPY;
			break;
		default:
			gtk_drag_finish( context, FALSE, FALSE, time );
			return;
			break;
		}
	}

	moving = ( context->action == GDK_ACTION_MOVE );

	site_path = screem_site_get_pathname( site );

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );

	gtk_tree_view_get_dest_row_at_pos( GTK_TREE_VIEW( widget ),
					   x, y, &path, NULL );
	if( ! path ) {
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}
		
	if( ! gtk_tree_model_get_iter( model, &it, path ) ) {
		gtk_tree_path_free( path );
		return;
	}

	gtk_tree_path_free( path );
	
	gtk_tree_model_get_value( model, &it, FILE_BROWSER_URI_COL, &value );
	target_uri = g_value_get_string( &value );

	if( ! target_uri ) {
		gtk_drag_finish( context, FALSE, FALSE, time );
		return;
	}

	/* what type of drop do we have? */
	switch( info ) {
	case TARGET_URI_LIST:
		uris = g_strsplit( selectionData->data, "\r\n", 0 );

		/* drops onto iters that aren't directories require
		   us to get the parent node, which should be a directory */
		if( ! screem_uri_is_dir( target_uri ) ) {
			GtkTreeIter pit;

			g_value_unset( &value );
			gtk_tree_model_iter_parent( model, &pit, &it );
			gtk_tree_model_get_value( model, &pit, 
						  FILE_BROWSER_URI_COL,
						  &value );
			target_uri = g_value_get_string( &value );
			if( ! target_uri ) {
				gtk_drag_finish( context, FALSE, FALSE, time );
				return;
			}
			memcpy( &it, &pit, sizeof( GtkTreeIter ) );
		}
		/* select the item we are dropping onto */
		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(widget));
		gtk_tree_selection_select_iter( GTK_TREE_SELECTION(selection),
						&it );
		screem_site_view_drop_uris( site, uris, &it, target_uri, 
					    moving );
		
		g_strfreev( uris );
		break;
	}
	
	g_value_unset( &value );

	gtk_drag_finish( context, TRUE, FALSE, time );
}

static gboolean screem_site_view_drop_uris( ScreemSite *site,
					    gchar **uris,
					    GtkTreeIter *it,
					    const gchar *target_uri,
					    gboolean moving )
{
	gint i;

	/* processed all uris, they have been
	   dropped onto target_uri */
	for( i = 0; uris[ i ] && *uris[ i ] != '\0'; ++ i ) {
		GnomeVFSURI *source;
		GnomeVFSURI *target;
		GnomeVFSURI *realtarget;
		gchar *basename;
		GnomeVFSFileInfo *vfsinfo;
		GnomeVFSFileInfoOptions options;
		gboolean res;
		gchar *tgt;
		/* make sure we aren't dropping onto the
		   same uri, first we need to append the
		   basename of source onto target */
		source = gnome_vfs_uri_new( uris[ i ] );
		target = gnome_vfs_uri_new( target_uri );
		basename = gnome_vfs_uri_extract_short_name( source );
		realtarget = gnome_vfs_uri_append_file_name(target,
							    basename );
		gnome_vfs_uri_unref( target );
		target = realtarget;
		
		if( ! gnome_vfs_uri_equal( source, target ) ) {
			tgt = gnome_vfs_uri_to_string(target,
						      GNOME_VFS_URI_HIDE_NONE);
			
			vfsinfo = gnome_vfs_file_info_new();
			options = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
			gnome_vfs_get_file_info( uris[ i ], vfsinfo, 
						 options );
			if( vfsinfo->type == GNOME_VFS_FILE_TYPE_DIRECTORY ) {
				res = copy_dir( uris[ i ], tgt, TRUE );
			} else {
				res = copy_file( uris[ i ], tgt );
			}
			gnome_vfs_file_info_unref( vfsinfo );
			
			if( res ) {
				/* we need to copy the buffer if
				   we are copying a file which is
				   already loaded */
				ScreemPage *page;
				ScreemPage *npage;
				page = screem_site_locate_page( site, 
								uris[ i ] );
				npage = screem_site_locate_page( site, tgt );
				if( page && npage ) {
					gchar *data;
					data = screem_page_get_data( page );
					if( screem_page_is_loaded(page))
						screem_page_set_data( npage,
								      data );
					g_free( data );
				}
				if( moving ) {
					screem_site_file_change( site, 
								 uris[i],tgt );
					/* will call delete_file()
					   if needed */
					delete_dir( uris[ i ] );
				}
			}
			g_free( tgt );
			g_free( basename );
			gnome_vfs_uri_unref( target );
			gnome_vfs_uri_unref( source );
		}
	}
		
	return TRUE;
}	


static gboolean screem_site_view_find_iter( ScreemSiteViewNodeInfo *info,
					    const gchar *uri,
					    GtkTreeIter *parent,
					    GtkTreeIter *it )
{
	GtkTreeModel *model;
	ScreemSiteViewNodeInfo *ninfo;
	gboolean got;
	gboolean found = FALSE;

	if( ! uri ) {
		uri = info->fullname;
	}

	model = screem_file_browser_get_model( SCREEM_FILE_BROWSER( info->view->browser ) );
	model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );

	if( ! parent ) {
		got = gtk_tree_model_get_iter_first( model, it );
	} else {
		got = gtk_tree_model_iter_children( model, it, parent );
	}

	while( got && ! found ) {
		GValue value = { 0 };

		gtk_tree_model_get_value( model, it, FILE_BROWSER_DATA_COL,
					  &value );
		ninfo = g_value_get_pointer( &value );
		g_value_unset( &value );

		if( ninfo && ! strcmp( uri, ninfo->fullname ) ) {
			/* found it */
			found = TRUE;
		} else {
			GtkTreeIter cit;
			if( gtk_tree_model_iter_has_child( model, it ) ) {
				if( screem_site_view_find_iter( info, uri,
								it, &cit ) ) {
					/* found it */
					*it = cit;
					found = TRUE;
				}
			}
		}
		/* if not found check siblings */
		if( ! found ) {
			got = gtk_tree_model_iter_next( model, it );
		}
	}

	return found;
}

static void screem_site_view_fm_expanded( GtkTreeView *view,
					  GtkTreeIter *iter,
					  GtkTreePath *path,
					  gpointer data )
{
	ScreemSite *site;

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( view ), "site" ) );

	if(  screem_site_get_fake_flag( site ) &&
	     ! g_object_get_data( G_OBJECT( view ), "block" ) ) {
		ScreemFileBrowser *browser; 
		
		browser = site->private->view->browser;

                screem_file_browser_scan_iter( browser, iter, -1 );

                g_object_set_data( G_OBJECT( view ), "block",
                                   GUINT_TO_POINTER( 1 ) );
                gtk_tree_view_expand_row( view, path, FALSE );
                g_object_set_data( G_OBJECT( view ), "block", NULL );
        }
}

static void screem_site_view_fm_activated( GtkTreeView *view,
					   GtkTreePath *path,
					   GtkTreeViewColumn *column,
					   gpointer data )
{
	ScreemFileBrowser *browser;
	GtkTreeModel *model;
	GtkTreeIter it;
	ScreemSite *site;

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( view ), "site" ) );

	browser = site->private->view->browser;

	model = gtk_tree_view_get_model( view );

        if( gtk_tree_model_get_iter( model, &it, path ) ) {
		GList *list;
		GList *ret;
		
		list = g_list_append( NULL, &it );
		ret = screem_file_browser_get_pathnames( browser,
							 list,
							 FALSE );
		g_list_free( list );
		if( ret ) {
			const gchar *current = ret->data;
			g_list_free( ret );
			g_object_set_data( G_OBJECT( view ),
					   "file_activated",
					   (gpointer)current );
		}
	}
}

static GList *screem_site_view_calc_stats( GtkTreeModel *model, 
					   GtkTreeIter *it, GList *types )
{
	GnomeVFSFileInfo *info;
	ScreemSiteViewNodeInfo *ninfo;
	GValue value = { 0 };
	GtkTreeIter iter;

	if( ! it ) {
		/* get root */
		if( gtk_tree_model_get_iter_first( model, &iter ) &&
		    gtk_tree_model_iter_has_child( model, &iter ) ) {
			it = gtk_tree_iter_copy( &iter );
			gtk_tree_model_iter_children( model, &iter, it );
			gtk_tree_iter_free( it );
			
			it = &iter;
		}
	}
	if( ! it ) {
		return types;
	}

	gtk_tree_model_get_value( model, it, FILE_BROWSER_DATA_COL, &value );
	ninfo = g_value_get_pointer( &value );
	g_value_unset( &value );
	
	info = gnome_vfs_file_info_new();

	gnome_vfs_get_file_info( ninfo->fullname, 
				 info, GNOME_VFS_FILE_INFO_GET_MIME_TYPE );

	if( info->type == GNOME_VFS_FILE_TYPE_DIRECTORY &&
	    gtk_tree_model_iter_has_child( model, it ) ) {
		GtkTreeIter child;

		if( gtk_tree_model_iter_children( model, &child, it ) ) {
			types = screem_site_view_calc_stats( model, &child,
							     types );
		}
	} else {
		GList *list;
		const gchar *mime_type;

		mime_type = gnome_vfs_file_info_get_mime_type( info );

		if( ! mime_type ) {
			mime_type = _( "unknown" );
		}

		for( list = types; list; list = list->next ) {
			Stats *stats;

			stats = (Stats*)list->data;
			if( ! strcmp( stats->mime_type, mime_type ) ) {
				stats->number ++;
				stats->total_size += info->size;
				break;
			}
		}
		if( ! list ) {
			Stats *stats;
			
			stats = g_new0( Stats, 1 );
			stats->uri = g_strdup( ninfo->fullname );
			stats->number = 1;
			stats->total_size = info->size;
			stats->mime_type = g_strdup( mime_type );
			
			types = g_list_prepend( types, stats );
		}
	}

	gnome_vfs_file_info_unref( info );

	if( gtk_tree_model_iter_next( model, it ) ) {
		types = screem_site_view_calc_stats( model, it, types );
	}

	return types;
}

GtkTreeModel *screem_site_get_statistics( ScreemSite *site )
{
	GtkTreeModel *store;
	GtkTreeModel *model;
	GList *list;
	GList *tmp;
	gint total;
	ScreemFileBrowser *browser;

	browser = site->private->view->browser;
	
	store = screem_file_browser_get_model( browser );
	store = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( store ) );

	list = screem_site_view_calc_stats( store, NULL, NULL );

	model = GTK_TREE_MODEL( gtk_list_store_new( 5, 
						    G_TYPE_STRING, 
						    GDK_TYPE_PIXBUF,
						    G_TYPE_INT,
						    G_TYPE_INT,
						    G_TYPE_INT,
						    NULL ) );

	for( tmp = list, total = 0; tmp; tmp = tmp->next ) {
		Stats *stats;
		GtkTreeIter it;
		GdkPixbuf *pixbuf;
		
		stats = (Stats*)tmp->data;

		total += stats->total_size;

		gtk_list_store_append( GTK_LIST_STORE( model ), &it );
		gtk_list_store_set( GTK_LIST_STORE( model ), &it,
				    0, stats->mime_type,
				    2, stats->number,
				    3, stats->total_size,
				    4, stats->total_size / stats->number,
				    -1 );

		pixbuf = screem_file_browser_get_icon( browser, stats->uri, 
							16, 16, FALSE,
							NULL );
		
		if( pixbuf ) {
			gtk_list_store_set( GTK_LIST_STORE( model ), &it, 1,
					    pixbuf, -1 );
			g_object_unref( pixbuf );
		}

		g_free( stats->mime_type );
		g_free( stats );
	}
	g_object_set_data( G_OBJECT( model ), "total",
			   GINT_TO_POINTER( total ) );
	g_list_free( list );

	return model;
}

static gboolean screem_site_view_update_node( GtkTreeModel *model, GtkTreePath *path,
					GtkTreeIter *iter, ScreemSite *site )
{
	ScreemSiteViewNodeInfo *info;

	info = NULL;
	gtk_tree_model_get( model, iter, FILE_BROWSER_DATA_COL, &info, -1 );
	
	if( info ) {
		gchar flags[ 3 ];
		strcpy( flags, "..." );

		if( screem_site_is_excluded( site, info->fullname ) ) {
			flags[ 0 ] = 'E';
		}
		if( screem_site_is_ignored( site, info->fullname ) ) {
			flags[ 2 ] = 'I';
		}
		if( screem_site_is_ascii( site, info->fullname ) ) {
			flags[ 1 ] = 'A';
		}

		gtk_tree_store_set( GTK_TREE_STORE( model ),
				    iter, FILE_BROWSER_USER_COL,
				    flags, -1 );
	}

	return FALSE;
}

void screem_site_view_update( ScreemSite *site )
{
	GtkTreeModel *store;
	
	store = screem_file_browser_get_model( site->private->view->browser );
	store = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( store ) );

	gtk_tree_model_foreach( store, 
				(GtkTreeModelForeachFunc)screem_site_view_update_node,
				site );
}

void screem_site_view_removed( ScreemFileBrowser *browser, 
		const gchar *uri, GtkTreeIter *it, gpointer data )
{
	GtkTreePath *path;
	GtkTreeModel *model;
	
	model = screem_file_browser_get_model( browser );
	model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );

	path = gtk_tree_model_get_path( model, it );
	if( path ) {	
		screem_site_view_row_deleted( model, path, it, NULL );
		gtk_tree_path_free( path );
	}

	/* FIXME: need to offer cvs delete */
}

void screem_site_view_added( ScreemFileBrowser *browser, const gchar *uri,
			     const gchar *mime_type,
			     GtkTreeIter *it,
			     gpointer data )
{
	ScreemSite *site;
	ScreemApplication *app;
	ScreemSiteView *private;
	ScreemSiteViewNodeInfo *info;
	GtkTreeModel *model;
	gboolean is_dir;

	site = SCREEM_SITE( data );

	if( screem_site_get_fake_flag( site ) || ! it ) {
		return;
	}

	g_object_get( G_OBJECT( site ), "app", &app, NULL );
	
	private = site->private->view;

	model = screem_file_browser_get_model( browser );
	model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );
	
	is_dir = screem_uri_is_dir( uri );

	/* mime type hack */
	if( ! mime_type ) {
		mime_type = "text/plain";
	}
	
	if( ! is_dir ) {
		info = screem_site_view_node_info_new( SCREEM_SITE_VIEW_FILE,
						       site,
						       uri,
						       NULL );
		info->view = private;

		if( screem_page_is_file_page( uri ) &&
			! screem_site_locate_page( site, uri ) ) {
			ScreemPage *page;
			
			page = screem_page_new( G_OBJECT( app ) );
			screem_page_set_pathname( page, uri );
			if( ! screem_site_add_page( site, page ) ) {
				g_object_unref( page );
				page = screem_site_locate_page( site, uri );
				g_assert( page );
			}
		}

		if( ! strncmp( "image/", mime_type, strlen( "image/" ) ) ) {
			/* add to images branch as well */
			GtkTreeIter iiter;
			gchar *base;
			GdkPixbuf *pixbuf;

			if( ! private->images ) {
				private->images = g_new0( GtkTreeIter, 1 );
				gtk_tree_store_append( GTK_TREE_STORE( model ),
						       private->images,
						       NULL );
				/* pass root as we know its a directory,
				   and we just want the directory icon */
				gtk_tree_store_set( GTK_TREE_STORE( model ),
						    private->images, 
						    FILE_BROWSER_NAME_COL,
						    _( "Images" ), 
						    FILE_BROWSER_URI_COL, NULL,
						    -1 );
				pixbuf = screem_file_browser_get_icon( browser,
									"/",
									16, 16,
									FALSE,
									NULL );
				if( pixbuf ) {
					gtk_tree_store_set(GTK_TREE_STORE(model),
							   private->images,
							   FILE_BROWSER_ICON_COL,
							   pixbuf,
							   -1 );
					g_object_unref( pixbuf );
				}
			}
			gtk_tree_store_append( GTK_TREE_STORE( model ),
					       &iiter, private->images );
			base = g_path_get_basename( uri );
			gtk_tree_model_get( model, it,
					    FILE_BROWSER_ICON_COL,
					    &pixbuf, -1 );

			gtk_tree_store_set( GTK_TREE_STORE( model ),
					    &iiter,
					    FILE_BROWSER_NAME_COL, base, 
					    FILE_BROWSER_ICON_COL, pixbuf,
					    FILE_BROWSER_URI_COL, uri,
					    -1 );
			g_free( base );
			screem_site_view_set_val_at( GTK_TREE_STORE( model),
						     &iiter, info );
		} else if( ! strcmp( "text/css", mime_type ) ) {
			/* add to images branch as well */
			GtkTreeIter iiter;
			gchar *base;
			GdkPixbuf *pixbuf;

			if( ! private->styles ) {
				private->styles = g_new0( GtkTreeIter, 1 );
				gtk_tree_store_append( GTK_TREE_STORE( model ),
						       private->styles,
						       NULL );
				/* pass root as we know its a directory,
				   and we just want the directory icon */
				gtk_tree_store_set( GTK_TREE_STORE( model ),
						    private->styles, 
						    FILE_BROWSER_NAME_COL,
						    _( "Stylesheets" ),
						    FILE_BROWSER_URI_COL, NULL,
						    -1 );
				
				pixbuf = screem_file_browser_get_icon( browser,
									"/",
									16, 16,
									FALSE,
									NULL );
				if( pixbuf ) {
					gtk_tree_store_set(GTK_TREE_STORE(model),
							   private->styles,
							   FILE_BROWSER_ICON_COL,
							   pixbuf,
							   -1 );
					g_object_unref( pixbuf );
				}
			}
			gtk_tree_store_append( GTK_TREE_STORE( model ),
					       &iiter, private->styles );
			base = g_path_get_basename( uri );
			gtk_tree_model_get( model, it,
					    FILE_BROWSER_ICON_COL,
					    &pixbuf, -1 );

			gtk_tree_store_set( GTK_TREE_STORE( model ),
					    &iiter,
					    FILE_BROWSER_NAME_COL, base, 
					    FILE_BROWSER_ICON_COL, pixbuf,
					    FILE_BROWSER_URI_COL, uri,
					    -1 );
			g_free( base );
			screem_site_view_set_val_at( GTK_TREE_STORE( model),
						     &iiter, info );
		} else if( ! strcmp( "text/javascript", mime_type ) ||
			   ! strcmp( "text/x-javascript", mime_type ) ) {
			/* add to images branch as well */
			GtkTreeIter iiter;
			gchar *base;
			GdkPixbuf *pixbuf;

			if( ! private->scripts ) {
				private->scripts = g_new0( GtkTreeIter, 1 );
				gtk_tree_store_append( GTK_TREE_STORE( model ),
						       private->scripts,
						       NULL );
				/* pass root as we know its a directory,
				   and we just want the directory icon */
				gtk_tree_store_set( GTK_TREE_STORE( model ),
						    private->scripts, 
						    FILE_BROWSER_NAME_COL,
						    _( "Scripts" ), 
						    FILE_BROWSER_URI_COL, NULL,
						    -1 );
				
				pixbuf = screem_file_browser_get_icon( browser,
									"/",
									16, 16,
									FALSE,
									NULL );
				if( pixbuf ) {
					gtk_tree_store_set(GTK_TREE_STORE(model),
							   private->scripts,
							   FILE_BROWSER_ICON_COL,
							   pixbuf,
							   -1 );
					g_object_unref( pixbuf );
				}
			}
			gtk_tree_store_append( GTK_TREE_STORE( model ),
					       &iiter, private->scripts );
			base = g_path_get_basename( uri );
			gtk_tree_model_get( model, it,
					    FILE_BROWSER_ICON_COL,
					    &pixbuf, -1 );

			gtk_tree_store_set( GTK_TREE_STORE( model ),
					    &iiter,
					    FILE_BROWSER_NAME_COL, base, 
					    FILE_BROWSER_ICON_COL, pixbuf,
					    FILE_BROWSER_URI_COL, uri,
					    -1 );
			g_free( base );
			screem_site_view_set_val_at( GTK_TREE_STORE( model),
						     &iiter, info );
		}  
	} else {
		info = screem_site_view_node_info_new( SCREEM_SITE_VIEW_FOLDER,
						       site,
						       uri,
						       NULL );
		info->view = private;
	}
	if( it ) {
		screem_site_view_set_val_at( GTK_TREE_STORE( model ),
					     it, info );
	}
	g_object_unref( app );
}

void screem_site_view_icon_change( ScreemFileBrowser *browser, const gchar *uri,
				     const gchar *mime_type,
				     GtkTreeIter *it,
				     gpointer data )
{
	ScreemSite *site;
	ScreemSiteView *private;
	GtkTreeModel *model;
	gchar *url;
	GdkPixbuf *pixbuf;
	
	site = SCREEM_SITE( data );
	private = site->private->view;

	model = screem_file_browser_get_model( browser );
	model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );

	gtk_tree_model_get( model, it,
			FILE_BROWSER_URI_COL, &url,
			-1 );
	
	if( ! url ) {
		pixbuf = screem_file_browser_get_icon( browser, "/",
							16, 16, FALSE,
							NULL );
		gtk_tree_store_set( GTK_TREE_STORE(model),
				    it,
				    FILE_BROWSER_ICON_COL,
				    pixbuf,
				    -1 );
		if( pixbuf ) {
			g_object_unref( pixbuf );
		}
	}

	
	g_free( url );
}
