/*  Screem:  screem-site.c
 *
 *  The ScreemSite object
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <unistd.h>
#include <dirent.h>

#include <glib/gfileutils.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-exec.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libxml/tree.h>
#include <libxml/parser.h>

#include "fileops.h"

#include "screem-site.h"
#include "screem-site-view.h"

#include "screem-todo.h"

#include "screem-markup.h" /* for the fix link code,
			      should it be moved else where? */

#include "support.h"

const gchar *permission_strings[] = {
	"p_exec",
	"p_ignore",
	"p_all"
};
const gchar *symlink_strings[] = {
	"s_follow",
	"s_ignore",
	"s_maintain"
};
const gchar *upload_strings[] = {
	"local",
	"ftp",
	"webdav",
	"rsh",
	"ssh"
};

enum {
	PROP_0,
	PROP_APP
};

static void screem_site_class_init( ScreemSiteClass *klass );
static void screem_site_init( ScreemSite *site );
static void screem_site_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec );
static void screem_site_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec );
static void screem_site_finalize( GObject *object );

static void screem_site_set_pages( ScreemSite *site, GList *list );

static void screem_site_view_initialise( ScreemSite *site );

static void screem_site_create_from_template( ScreemSite *site, 
					      const gchar *template,
					      const gchar *base,
					      xmlNodePtr node );

static void screem_site_check_template( ScreemPage *page,
					ScreemSite *site );
static gboolean screem_site_manage_present( ScreemSite *site,
					    const gchar *path,
					    GtkListStore *store );
static void screem_site_manage_add( ScreemSite *site, const gchar *pattern,
				    GtkListStore *store );
static void screem_site_manage_remove( ScreemSite *site, const gchar *pattern,
				    	GtkListStore *store );
static GList *screem_site_manage_get( ScreemSite *site,
					GtkListStore *store );

ScreemSite *screem_site_new( GObject *application )
{
	ScreemSite *site;
	GType type;

	type = screem_site_get_type();

	site = SCREEM_SITE( g_object_new( type, "app", application, NULL ) );

	return site;
}

void screem_site_purge( ScreemSite *site )
{
	const GList *list;
	ScreemPage *page;

	list = screem_site_get_pages( site );
	while( list ) {
		page = SCREEM_PAGE( list->data );
		screem_site_remove_page( site, page );
		g_object_unref( G_OBJECT( page ) );
		list = screem_site_get_pages( site );
	}
	screem_todo_clear( SCREEM_TODO( screem_site_get_todo( site ) ) ); 
}

void screem_site_set_name( ScreemSite *site, const gchar *name )
{

	/* does the site already have a name set? */
	if( site->private->name ) {
		g_free( site->private->name );
	}
	
	site->private->name = g_strdup( name );
}

const gchar* screem_site_get_name( ScreemSite *site )
{
	g_return_val_if_fail( site != NULL, NULL );

	return site->private->name;
}

void screem_site_set_pathname( ScreemSite *site, const gchar *pathname )
{
	GnomeVFSURI *uri;
	gchar *path;

	/* does the site already have a pathname set? */
	if( site->private->pathname ) {
		g_free( site->private->pathname );
	}

	if( pathname[ strlen( pathname ) - 1 ] != G_DIR_SEPARATOR ) {
		path = g_strconcat( pathname, G_DIR_SEPARATOR_S, NULL );
	} else {
		path = g_strdup( pathname );
	}

	uri = gnome_vfs_uri_new( path );
	g_free( path );
	site->private->pathname = gnome_vfs_uri_to_string( uri,
							   GNOME_VFS_URI_HIDE_NONE );

	gnome_vfs_uri_unref( uri );
}

const gchar* screem_site_get_pathname( ScreemSite *site )
{
	return site->private->pathname;
}

void screem_site_set_preview_static( ScreemSite *site, gboolean flag )
{
	g_return_if_fail( SCREEM_IS_SITE( site ) );

	site->private->preview_static = flag;
}

gboolean screem_site_get_preview_static( ScreemSite *site )
{
	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );

	return site->private->preview_static;
}

void screem_site_set_preview_dynamic( ScreemSite *site, gboolean flag )
{
	g_return_if_fail( SCREEM_IS_SITE( site ) );

	site->private->preview_dynamic = flag;
}

gboolean screem_site_get_preview_dynamic( ScreemSite *site )
{
	g_return_val_if_fail( SCREEM_IS_SITE( site ), FALSE );

	return site->private->preview_dynamic;
}


void screem_site_set_remote_url( ScreemSite *site, const gchar *remote_url )
{
	/* does the site already have a remote url set? */
	if( site->private->remote_url ) {
		g_free( site->private->remote_url );
	}

	site->private->remote_url = g_strdup( remote_url );
}

const gchar* screem_site_get_remote_url( ScreemSite *site )
{
	return site->private->remote_url;
}

void screem_site_set_remote_method( ScreemSite *site, UploadMethods remote_method )
{
	site->private->remote_method = remote_method;
}

UploadMethods screem_site_get_remote_method( ScreemSite *site )
{
	return site->private->remote_method;
}

void screem_site_set_remote_path( ScreemSite *site, const gchar *remote_path )
{
	/* does the site already have a remote path set? */
	if( site->private->remote_path ) {
		g_free( site->private->remote_path );
	}

	site->private->remote_path = g_strdup( remote_path );
}

const gchar* screem_site_get_remote_path( ScreemSite *site )
{
	return site->private->remote_path;
}

void screem_site_set_remote_user( ScreemSite *site, const gchar *remote_user )
{
	/* does the site already have a remote usernmae set? */
	if( site->private->remote_user ) {
		g_free( site->private->remote_user );
	}

	site->private->remote_user = g_strdup( remote_user );
}

const gchar* screem_site_get_remote_user( ScreemSite *site )
{
	return site->private->remote_user;
}

void screem_site_set_remote_pass( ScreemSite *site, const gchar *remote_pass )
{
	/* does the site already have a remote password set? */
	if( site->private->remote_pass ) {
		g_free( site->private->remote_pass );
	}

	site->private->remote_pass = g_strdup( remote_pass );
}

const gchar* screem_site_get_remote_pass( ScreemSite *site )
{
	return site->private->remote_pass;
}


void screem_site_set_cvs_root( ScreemSite *site, const gchar *cvs_root )
{
	/* does the site already have a cvs root set? */
	if( site->private->cvs_root ) {
		g_free( site->private->cvs_root );
	}

	if( cvs_root && strlen( cvs_root ) ) {
		site->private->cvs_root = g_strdup( cvs_root );
	} else {
		site->private->cvs_root = NULL;
	}
}

const gchar* screem_site_get_cvs_root( ScreemSite *site )
{
	return site->private->cvs_root;
}

void screem_site_set_auto_update_ask( ScreemSite *site, gboolean value )
{
	site->private->auto_update_ask = value;
}

gboolean screem_site_get_auto_update_ask( ScreemSite *site ) 
{
	return site->private->auto_update_ask;
}

void screem_site_set_auto_update( ScreemSite *site, gboolean value )
{
	site->private->auto_update = value;
}

gboolean screem_site_get_auto_update( ScreemSite *site ) 
{
	return site->private->auto_update;
}

void screem_site_set_use_ctags( ScreemSite *site, gboolean flag )
{
	site->private->ctags = flag;
}

gboolean screem_site_get_use_ctags( const ScreemSite *site )
{
	return site->private->ctags;
}

void screem_site_update_ctags_file( ScreemSite *site )
{
	const gchar *command = "ctags -R";
	const gchar *pathname;
	gchar *path;
	
	g_return_if_fail( SCREEM_IS_SITE( site ) );
	g_return_if_fail( ! screem_site_get_fake_flag( site ) );

	pathname = screem_site_get_pathname( site );

	if( screem_site_get_use_ctags( site ) ) {
		path = gnome_vfs_get_local_path_from_uri( pathname );
		if( path ) {
			gnome_execute_shell( path, command ); 
			g_free( path );
		}
	}
}

void screem_site_set_site_template( ScreemSite *site, const gchar *path )
{
	if( site->private->site_template_path ) {
		g_free( site->private->site_template_path );
	}

	if( path && strlen( path ) ) {
		site->private->site_template_path = g_strdup( path );
	} else {
		site->private->site_template_path = NULL;
	}
}

const gchar *screem_site_get_site_template( ScreemSite *site )
{
	return site->private->site_template_path;
}

void screem_site_set_template_path( ScreemSite *site, const gchar *template_path )
{
	/* does the site already have a template path set? */
	if( site->private->template_path ) {
		g_free( site->private->template_path );
	}

	if( template_path && strlen( template_path ) ) {
		site->private->template_path = g_strdup( template_path );
	} else {
		site->private->template_path = NULL;
	}
}

const gchar* screem_site_get_template_path( ScreemSite *site )
{
	return site->private->template_path;
}

void screem_site_set_http_url( ScreemSite *site, const gchar *http_url )
{
	/* does the site already have a remote usernmae set? */
	if( site->private->http_url ) {
		g_free( site->private->http_url );
	}

	site->private->http_url = g_strdup( http_url );
}

const gchar* screem_site_get_http_url( ScreemSite *site )
{
	return site->private->http_url;
}

gboolean screem_site_get_fake_flag( ScreemSite *site )
{
	return site->private->fake_site;
}

void screem_site_set_passive_ftp( ScreemSite *site, gboolean flag )
{
	site->private->passive_ftp = flag;
}

gboolean screem_site_get_passive_ftp( ScreemSite *site )
{
	return site->private->passive_ftp;
}

void screem_site_set_no_delete( ScreemSite *site, gboolean flag )
{
	site->private->no_delete = flag;
}

gboolean screem_site_get_no_delete( ScreemSite *site )
{
	return site->private->no_delete;
}

void screem_site_set_check_moved( ScreemSite *site, gboolean flag )
{
	site->private->check_moved = flag;
}

gboolean screem_site_get_check_moved( ScreemSite *site )
{
	return site->private->check_moved;
}

void screem_site_set_no_overwrite( ScreemSite *site, gboolean flag )
{
	site->private->no_overwrite = flag;
}

gboolean screem_site_get_no_overwrite( ScreemSite *site )
{
	return site->private->no_overwrite;
}

void screem_site_set_permissions( ScreemSite *site, ScreemSitePerms permissions )
{
	site->private->permissions = permissions;
}

ScreemSitePerms screem_site_get_permissions( ScreemSite *site )
{
	return site->private->permissions;
}

void screem_site_set_symlinks( ScreemSite *site, ScreemSiteSymlinks symlinks )
{
	site->private->symlinks = symlinks;
}

ScreemSiteSymlinks screem_site_get_symlinks( ScreemSite *site )
{
	return site->private->symlinks;
}


void screem_site_set_is_import( ScreemSite *site, gboolean val )
{
	site->private->is_import = val;
}

gboolean screem_site_get_is_import( ScreemSite *site )
{
	return site->private->is_import;
}

gboolean screem_site_is_excluded( ScreemSite *site, const gchar *path )
{
	gboolean ret;

	ret = ! screem_site_get_fake_flag( site );
	
	if( ret ) {
		ret = screem_site_manage_present( site, path,
					   site->private->excludes );
	}

	return ret;
}

void screem_site_add_exclude( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_add( site, pattern,
			   	 site->private->excludes );
}
		
void screem_site_remove_exclude( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_remove( site, pattern,
				   site->private->excludes );
}

gboolean screem_site_is_ascii( ScreemSite *site, const gchar *path )
{
	gboolean ret;

	ret = ! screem_site_get_fake_flag( site );
	
	if( ret ) {
		ret = screem_site_manage_present( site, path,
					   site->private->asciis );
	}

	return ret;
}

void screem_site_add_ascii( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_add( site, pattern,
				site->private->asciis );
}
void screem_site_remove_ascii( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_remove( site, pattern,
				   site->private->asciis );
}

gboolean screem_site_is_ignored( ScreemSite *site, const gchar *path )
{
	gboolean ret;

	ret = ! screem_site_get_fake_flag( site );
	
	if( ret ) {
		ret = screem_site_manage_present( site, path,
					   site->private->ignores );
	}

	return ret;
}

void screem_site_add_ignore( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_add( site, pattern,
				site->private->ignores );
}
void screem_site_remove_ignore( ScreemSite *site, const gchar *pattern )
{
	screem_site_manage_remove( site, pattern,
				   site->private->ignores );
}

void screem_site_add_auto_open( ScreemSite *site, const gchar *file )
{
	if( file && *file != '\0' ) {
		site->private->auto_open = 
			g_list_append( site->private->auto_open, g_strdup( file ) );
	}
}

GList *screem_site_get_auto_open( ScreemSite *site )
{
	return site->private->auto_open;
}

void screem_site_save( ScreemSite *site )
{
	g_return_if_fail( site );

	if( ! screem_site_get_fake_flag( site ) ) {
		/* first the project file */
		screem_site_write_project_file( site );
	}
}

gboolean screem_site_create( ScreemSite *site )
{
	const gchar *pathname;
	const gchar *template;

	g_return_val_if_fail( site != NULL, FALSE );

	site->private->fake_site = FALSE;

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	/* try and make the path, will just switch to it
	   if it already exists */
	if( ! mkdir_recursive( pathname,
			       GNOME_VFS_PERM_USER_ALL |
			       GNOME_VFS_PERM_GROUP_ALL |
			       GNOME_VFS_PERM_OTHER_READ |
			       GNOME_VFS_PERM_OTHER_EXEC ) ) {
		return FALSE;
	}

	screem_file_browser_set_mode( site->private->view->browser,
				      FILE_BROWSE_RECURSE | FILE_BROWSE_ROOT );
	g_signal_connect( G_OBJECT( site->private->view->browser ),
			  "added", 
			  G_CALLBACK( screem_site_view_added ),
			  site );
	g_signal_connect( G_OBJECT( site->private->view->browser ),
			  "removed", 
			  G_CALLBACK( screem_site_view_removed ),
			  site );
	g_signal_connect( G_OBJECT( site->private->view->browser ),
			  "icon_change", 
			  G_CALLBACK( screem_site_view_icon_change ),
			  site );
	screem_file_browser_set_sort_func( site->private->view->browser,
					   screem_site_view_compare_func,
					   site );
    
	/* write the initial project file */
	screem_site_write_project_file( site );

	/* process template */
	template = screem_site_get_site_template( site );
	if( template ) {
		screem_site_create_from_template( site, template,
						  pathname, NULL );
	}

	screem_file_browser_scan_directory( site->private->view->browser,
					    pathname, -1 );

	return TRUE;
}

gboolean screem_site_load( ScreemSite *site )
{
	gchar *project_file;
	const gchar *pathname;
	gchar *mime_type;

	GList *project_files = NULL;
	GList *file;
    
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfoOptions options;
	GnomeVFSFileInfo *info;

	g_return_val_if_fail( site != NULL, FALSE );

	site->private->fake_site = FALSE;

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	options = GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS;

	if( gnome_vfs_directory_open( &handle, pathname, options ) != 
	    GNOME_VFS_OK ) {
		return FALSE;
	}
		
	info = gnome_vfs_file_info_new();

	/* build a list of *.screem files */
	while( gnome_vfs_directory_read_next( handle,info ) == GNOME_VFS_OK ){
		gchar *name;
	
		name = g_build_filename( pathname, info->name, NULL );
		mime_type = screem_get_mime_type( name );
		
		if( mime_type &&
		! strcmp( "application/x-screem", mime_type ) ) {
			/* add it to the list */
		
			project_files = g_list_append( project_files,
						g_strdup( info->name ) );
		}
		g_free( name );
		g_free( mime_type );
	}
	gnome_vfs_directory_close( handle );
	
	gnome_vfs_file_info_unref( info );

	/* now check the list.  The priority we give is the following:
	   .project.screem
	   project.screem

	   Perhaps the project file could be named  .<sitename>.screem
	*/
	file = g_list_find_custom( project_files, (gpointer)".project.screem",
				   (GCompareFunc)strcmp );
	if( ! file ) {
		file = g_list_find_custom( project_files, 
					   (gpointer)"project.screem",
					   (GCompareFunc)strcmp );
	}

	project_file = NULL;
	if( file ) {
		project_file = g_strconcat( pathname, file->data, NULL );
	}
	g_list_foreach( project_files, (GFunc)g_free, NULL );

	if( ! screem_site_parse_project_file( site, project_file ) ) {
		screem_site_set_is_import( site, TRUE );
	}
	if( project_file ) {
		g_free( project_file );
	}
	screem_file_browser_set_mode( site->private->view->browser,
				      FILE_BROWSE_RECURSE | FILE_BROWSE_ROOT );
	g_signal_connect( G_OBJECT( site->private->view->browser ),
			  "added", 
			  G_CALLBACK( screem_site_view_added ),
			  site );
	g_signal_connect( G_OBJECT( site->private->view->browser ),
			  "removed", 
			  G_CALLBACK( screem_site_view_removed ),
			  site );
	g_signal_connect( G_OBJECT( site->private->view->browser ),
			  "icon_change", 
			  G_CALLBACK( screem_site_view_icon_change ),
			  site );
	screem_file_browser_set_sort_func( site->private->view->browser,
					   screem_site_view_compare_func,
					   site );
	screem_file_browser_scan_directory( site->private->view->browser,
					    pathname, -1 );

	screem_site_update_ctags_file( site );
	
	return TRUE;
}

gboolean screem_site_write_project_file( ScreemSite *site )
{
	gchar *project_file;
	xmlDocPtr doc;
	xmlNsPtr ns;
	xmlNodePtr tree;
	xmlNodePtr subtree;
	gint retval;
	GList *list;
	GList *tmp;

	const gchar *pathname;
	const gchar *name;
	const gchar *remote_url;
	UploadMethods remote_method;
	const gchar *remote_user;
	const gchar *remote_pass;
	const gchar *remote_path;
	gboolean passive_ftp;
	gboolean no_delete;
	gboolean check_moved;
	gboolean no_overwrite;

	ScreemSitePerms permissions;
	ScreemSiteSymlinks symlinks;

	const gchar *http_url;
	const gchar *cvs_root;
	const gchar *template_path;

	ScreemSitePrivate *private;
	ScreemTodo *todo;

	g_return_val_if_fail( site != NULL, FALSE );

	/* if we are a fake site pretend we wrote the file with no problems */
	if( screem_site_get_fake_flag( site ) ) {
		return TRUE;
	}

	private = site->private;

	pathname = screem_site_get_pathname( site );

	g_return_val_if_fail( pathname != NULL, FALSE );

	name = screem_site_get_name( site );
	remote_url = screem_site_get_remote_url( site );
	remote_method = screem_site_get_remote_method( site );
	remote_user = screem_site_get_remote_user( site );
	remote_pass = screem_site_get_remote_pass( site );
	remote_path = screem_site_get_remote_path( site );
	passive_ftp = screem_site_get_passive_ftp( site );
	no_delete = screem_site_get_no_delete( site );
	check_moved = screem_site_get_check_moved( site );
	no_overwrite = screem_site_get_no_overwrite( site );
	permissions = screem_site_get_permissions( site );
	symlinks = screem_site_get_symlinks( site );
	http_url = screem_site_get_http_url( site );
	cvs_root = screem_site_get_cvs_root( site );
	template_path = screem_site_get_template_path( site );

	doc = xmlNewDoc( XML_DEFAULT_VERSION );
	xmlDocSetRootElement( doc, 
			      xmlNewDocNode( doc, NULL, "SCREEM_PROJECT", NULL ) );
	ns = xmlNewNs( xmlDocGetRootElement( doc ),
		       "http://www.screem.org/", "screem" );
	xmlNewChild( xmlDocGetRootElement( doc ), ns, "title", name );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "remote", remote_url );
	xmlSetProp( tree, "method", upload_strings[ remote_method ] );

	xmlSetProp( tree, "username", remote_user );
	xmlSetProp( tree, "password", remote_pass );
	xmlSetProp( tree, "path", remote_path );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns,
			    "preview", NULL );
	xmlSetProp( tree, "static_publish",
		    screem_site_get_preview_static( site ) ? "TRUE" : "FALSE" );
	xmlSetProp( tree, "dynamic_publish",
		    screem_site_get_preview_dynamic( site ) ? "TRUE" : "FALSE" );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "options", NULL);
	xmlSetProp( tree, "passive_ftp", passive_ftp?"TRUE":"FALSE" );
	xmlSetProp( tree, "no_delete", no_delete?"TRUE":"FALSE" );
	xmlSetProp( tree, "check_moved", check_moved?"TRUE":"FALSE" );
	xmlSetProp( tree, "no_overwrite", no_overwrite?"TRUE":"FALSE" );

	xmlSetProp( tree, "permissions",  permission_strings[ permissions ] );

	xmlSetProp( tree, "symlinks",  symlink_strings[ symlinks ] );

	xmlNewChild( xmlDocGetRootElement( doc ), ns, "http", http_url );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "ctags",
			NULL );
	if( screem_site_get_use_ctags( site ) ) {
		xmlSetProp( tree, "use", "true" );
	} else {
		xmlSetProp( tree, "use", "false" );
	}
	
	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "cvs", cvs_root );
	if( cvs_root ) {
		xmlSetProp( tree, "use", "true" );
	} else {
		xmlSetProp( tree, "use", "false" );
	}

	if( screem_site_get_auto_update( site ) ) {
		xmlSetProp( tree, "auto_update", "true" );
	} else {
		xmlSetProp( tree, "auto_update", "false" );
	}

	if( screem_site_get_auto_update_ask( site ) ) {
		xmlSetProp( tree, "ask", "true" );
	} else {
		xmlSetProp( tree, "ask", "false" );
	}
      	
	xmlNewChild( xmlDocGetRootElement( doc ), ns, "template", template_path );

	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "tasks", NULL );
	todo = SCREEM_TODO( screem_site_get_todo( site ) );
        for( list = screem_todo_get_tasks( todo ); list; list = list->next ) {
                ScreemTodoItem *item;
		item = (ScreemTodoItem*)list->data;
		subtree = xmlNewChild( tree, ns, "task", NULL );
		xmlSetProp( subtree, "name", item->task );
		xmlSetProp( subtree, "assigned", item->assigned );
		xmlSetProp( subtree, "priority", item->priority );
		xmlSetProp( subtree, "linkedTo", item->linked_to );
		xmlSetProp( subtree, "description", item->description);
		if( item->complete ) {
			xmlSetProp( subtree, "completed", "true" );
		} else {
			xmlSetProp( subtree, "completed", "false" );
		}
	}

	/* write exclude,ignore, and ascii lists */
	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "excludes", NULL );
	for( tmp = list = screem_site_manage_get( site, site->private->excludes );
		list; list = list->next ) {
		subtree = xmlNewChild(tree, ns, "exclude", (gchar*)list->data);
		g_free( list->data );
	}
	g_list_free( tmp );
	/* write ignore list */
	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "ignores", NULL );
	for( tmp = list = screem_site_manage_get( site, site->private->ignores );
		list; list = list->next ) {
		subtree = xmlNewChild(tree, ns, "ignore", (gchar*)list->data);
		g_free( list->data );
	}	
	g_list_free( tmp );
	/* write ascii list */
	tree = xmlNewChild( xmlDocGetRootElement( doc ), ns, "asciis", NULL );
	for( tmp = list = screem_site_manage_get( site, site->private->asciis );
		list; list = list->next ) {
		subtree = xmlNewChild(tree, ns, "ascii", (gchar*)list->data);
		g_free( list->data );
	}
	g_list_free( tmp );

	if( FALSE && name && strlen( name ) ) {
		project_file = g_strconcat( pathname, name, ".screem", NULL );
	} else {
		project_file = g_strconcat( pathname,".project.screem", NULL );
	}


	{
		xmlChar *mem;
		gint size;

		xmlDocDumpMemory( doc, &mem, &size );

		retval = save_file( project_file, mem,
				    GNOME_VFS_PERM_USER_READ | 
				    GNOME_VFS_PERM_USER_WRITE );
		xmlFree( mem );
		xmlFreeDoc( doc );
		g_free( project_file );
	}

	return (retval < 0);
}

ScreemPage* screem_site_locate_page( ScreemSite *site, const gchar *path )
{
	const GList *list;
	ScreemPage *page = NULL;
	const gchar *pathname;
	gchar *pname;
	gchar *pthname;
	GnomeVFSURI *uri;

	g_return_val_if_fail( site, NULL );
	g_return_val_if_fail( SCREEM_IS_SITE( site ), NULL );
	g_return_val_if_fail( path, NULL );

	/* we want a full URI as the path */
	uri = gnome_vfs_uri_new( path );
	if( uri ) {
		pname = gnome_vfs_uri_to_string( uri, 0 );
		gnome_vfs_uri_unref( uri );
	} else {
		pname = g_strdup( path );
	}

	for( list = screem_site_get_pages(site ); list && ( ! page );
	     list = list->next ) {
		page = SCREEM_PAGE( list->data );
		pathname = screem_page_get_pathname( page );
		if( pathname ) {
			uri = gnome_vfs_uri_new( pathname );
			pthname = gnome_vfs_uri_to_string( uri, 0 );
			gnome_vfs_uri_unref( uri );

			if( strcmp( pname, pthname ) ) { 
				page = NULL;
			}
			g_free( pthname );
		} else {
			page = NULL;
		}
	}

	g_free( pname );

	return page;
}

gboolean screem_site_add_page( ScreemSite *site, ScreemPage *page )
{
	const GList *list;
	GList *nlist;

	g_return_val_if_fail( site != NULL, FALSE );
	g_return_val_if_fail( page != NULL, FALSE );

	list = screem_site_get_pages( site );
	nlist = g_list_append( (GList*)list, page );
		
	/* hook up to the save signal */
	g_signal_connect( G_OBJECT( page ), "saved",
			G_CALLBACK( screem_site_check_template ),
			site );
		
	screem_site_set_pages( site, nlist );

	return TRUE;
}
ScreemPage* screem_site_remove_page_with_path( ScreemSite *site, const gchar *path )
{
	ScreemPage *page;
	
	g_return_val_if_fail( site != NULL, NULL );
	g_return_val_if_fail( path != NULL, NULL );

	page = screem_site_locate_page( site, path );

	return screem_site_remove_page( site, page );
}

ScreemPage* screem_site_remove_page( ScreemSite *site, ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_SITE( site ), NULL );
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );
	
	if( page ) {
		GList *list;

		list = (GList*)screem_site_get_pages( site );

		list = g_list_remove( list, page );
		screem_site_set_pages( site, list );

	}
	return page;
}

const GList* screem_site_get_pages( ScreemSite *site )
{
	return site->private->pages;
}

GObject *screem_site_get_todo( ScreemSite *site )
{
	return site->private->tasks;
}

void screem_site_file_change( ScreemSite *site, const gchar *src, 
			      const gchar *dest )
{
	/* if we are a fake site we do not want to do this */
	if( ! screem_site_get_fake_flag( site ) ) {
		/* go through all pages in the site and update any links */
		const GList *list;

		for( list = screem_site_get_pages( site ); list; 
		     list = list->next ) {
			ScreemPage *page;
			const gchar *pathname;
			gchar *oldpathname;
			gchar *temp;
				
			page = SCREEM_PAGE( list->data );
			
			pathname = screem_page_get_pathname( page );

			/* if pathname is in src then we
			   need to update the pathname to point
			   to the new location */
			oldpathname = g_strdup( pathname );
			if( ! strncmp( pathname, src, strlen( src ) ) ) {
				temp = g_strconcat( dest,
						    pathname + strlen( src ),
						    NULL );
				screem_page_set_pathname( page, temp );
				g_free( temp );
			}
			screem_page_load( page );
			
			pathname = screem_page_get_pathname( page );
			screem_markup_fix_links( site, page, pathname,
						 oldpathname,
						 src, dest );
			g_free( oldpathname );
		}
	}
}

void screem_site_set_documents( ScreemSite *site, const gchar *name, 
				GList *list )
{
	GList *prev;

	prev = g_hash_table_lookup( site->private->documents,
					name );
	if( prev ) {
		g_hash_table_remove( site->private->documents, name );
		g_list_free( prev );
	}
	
	g_hash_table_insert( site->private->documents, 
			     (gchar*)name,  /* FIXME: must ensure we remove
					       the name from the table
					       before the window
					       it refers to is destroyed */
			     list );
}

GList* screem_site_get_documents( ScreemSite *site, const gchar *name )
{
	GList *documents;

	documents = g_hash_table_lookup( site->private->documents, name );

	return documents;
}

GList* screem_site_get_excludes( ScreemSite *site )
{
	return screem_site_manage_get( site, site->private->excludes );
}

GList* screem_site_get_ignores( ScreemSite *site )
{
	return screem_site_manage_get( site, site->private->ignores );
}

GList* screem_site_get_asciis( ScreemSite *site )
{
	return screem_site_manage_get( site, site->private->asciis );
}

gchar *screem_get_local_site_path( ScreemSite *site )
{
	gchar *spath;
	GnomeVFSURI *uri;
	const gchar *pathname;
	
	spath = NULL;
	if( ! screem_site_get_fake_flag( site ) ) {
		pathname = screem_site_get_pathname( site );
		uri = gnome_vfs_uri_new( pathname );
		if( ! uri ) {
			spath = NULL;
		} else if( gnome_vfs_uri_is_local( uri ) ) {
			/* strip method */
			spath = gnome_vfs_get_local_path_from_uri( pathname );
			gnome_vfs_uri_unref( uri );
		} else {
			spath = gnome_vfs_uri_to_string( uri, 0 );
			gnome_vfs_uri_unref( uri );
		}
	}

	return spath;
}

/* static stuff */
static void screem_site_set_pages( ScreemSite *site, GList *list )
{
	site->private->pages = list;
}

static void screem_site_view_initialise( ScreemSite *site )
{
	ScreemSiteView *view;

	view = site->private->view = g_new0( ScreemSiteView, 1 );

	view->browser = screem_file_browser_new();
	screem_file_browser_set_mode( site->private->view->browser,
				      FILE_BROWSE_NORMAL );
	
	screem_file_browser_scan_directory( view->browser, "/", -1 );
}

static void screem_site_create_from_template( ScreemSite *site, 
					      const gchar *template,
					      const gchar *base,
					      xmlNodePtr node )
{
	xmlDocPtr doc;

	if( ! node ) {
		doc = xmlParseFile( template );
		if( doc ) {
			node = xmlDocGetRootElement( doc )->xmlChildrenNode;
		}
	}
	
	while( node ) {
		if( ! strcmp( "directory", node->name ) ) {
			gchar *name;
			gchar *dir;
			
			name = xmlGetProp( node, "name" );
			dir = g_strconcat( base, G_DIR_SEPARATOR_S,
					   name, NULL );
			
			if( name ) {
				mkdir_recursive( dir,
						 GNOME_VFS_PERM_USER_ALL |
						 GNOME_VFS_PERM_GROUP_ALL |
						 GNOME_VFS_PERM_OTHER_READ |
						 GNOME_VFS_PERM_OTHER_EXEC );
				g_free( name );
			}
			name = xmlGetProp( node, "template" );
			if( name ) {
				screem_site_create_from_template( site,
								  name,
								  dir,
								  NULL );
			}
			if( node->xmlChildrenNode ) {
				screem_site_create_from_template( site,
								  template,
								  dir,
								  node->xmlChildrenNode );
			}
			g_free( dir );
		} else if( ! strcmp( "file", node->name ) ) {
			gchar *name;
			
			name = xmlGetProp( node, "name" );
			if( name ) {
				gchar *template;
				gchar *dir;

				dir = g_strconcat( base, G_DIR_SEPARATOR_S,
						   name,
						   NULL );

				template = xmlGetProp( node, "template" );
				if( ! template ) {
					template = (gchar*)screem_site_get_template_path( site );
					if( template ) {
						template = g_strdup(template);
					}
				}
				if( template ) {
					/* copy template to dir */
					copy_file( template, dir );
					g_free( template );
				} else {
					/* create empty file */
					save_file( dir, "",
						   GNOME_VFS_PERM_USER_READ |
						   GNOME_VFS_PERM_USER_WRITE |
						   GNOME_VFS_PERM_GROUP_READ |
						   GNOME_VFS_PERM_GROUP_WRITE |
						   GNOME_VFS_PERM_OTHER_READ );
				}
				g_free( dir );
				g_free( name );
			}
		}
		node = node->next;
	}
}

static void screem_site_check_template( ScreemPage *page, ScreemSite *site )
{
	gchar *tag;
	gchar *path;
	GSList *blocks;

	tag = NULL;

	blocks = NULL;
	if( ( ! screem_site_get_fake_flag( site ) ) &&
	    screem_page_is_template( page, &tag, &path, &blocks ) ) {
		const GList *pages;
				
		for( pages = screem_site_get_pages( site );
			pages; pages = pages->next ) {
			ScreemPage *p;
			
			p = SCREEM_PAGE( pages->data );
			if( p != page ) {
				screem_markup_update_from_template( site, page, p, tag, blocks );
			}
		}
		g_slist_foreach( blocks, (GFunc)g_free, NULL );
		g_slist_free( blocks );
	}		
}

static gboolean screem_site_manage_present( ScreemSite *site,
					    const gchar *path,
					    GtkListStore *store )
{
	/* we want a full URI as the path */
	GnomeVFSURI *uri;
	gchar *uripath;
	GtkTreeIter *it;
	gboolean ret;

	g_return_val_if_fail( screem_site_get_fake_flag( site ) != TRUE, FALSE );

	uri = gnome_vfs_uri_new( path );
	uripath = gnome_vfs_uri_to_string( uri, 0 );
	gnome_vfs_uri_unref( uri );

	it = screem_support_find_in_list( store, 0, uripath );
	ret = FALSE;
	if( it ) {
		gtk_tree_iter_free( it );
		ret = TRUE;
	}
	g_free( uripath );
	
	return ret;
}

static void screem_site_manage_add( ScreemSite *site, const gchar *pattern,
				    GtkListStore *store )
{
	GtkTreeIter *it;

	it = screem_support_find_in_list( store, 0, pattern );
	if( ! it ) {
		GtkTreeIter addit;

		gtk_list_store_append( store, &addit );
		gtk_list_store_set( store, &addit,
				    0, pattern,
				    1, TRUE, -1 );
		
		/* update the view model for the site */
		screem_site_view_update( site );
		
		screem_site_save( site );
	} else {
		gtk_tree_iter_free( it );
	}
}

static void screem_site_manage_remove( ScreemSite *site, const gchar *pattern,
				    	GtkListStore *store )
{
	GtkTreeIter *it;

	it = screem_support_find_in_list( store, 0, pattern );
	if( it ) {
		gtk_list_store_remove( store, it );
		gtk_tree_iter_free( it );

		/* update the view model for the site */
		screem_site_view_update( site );

		screem_site_save( site );
	}
}

static GList *screem_site_manage_get( ScreemSite *site,
					GtkListStore *store )
{
	GList *ret;
	GtkTreeIter it;
	gchar *val;
	
	ret = NULL;

	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( store ), &it ) ) {
		do {
			gtk_tree_model_get( GTK_TREE_MODEL( store ), &it, 
					    0, &val, -1 );
			ret = g_list_prepend( ret, val );
		} while( gtk_tree_model_iter_next( GTK_TREE_MODEL( store ),
				 		   &it ) );
	}
	ret = g_list_reverse( ret );

	return ret;
}

/* G Object stuff */
#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_site_class_init( ScreemSiteClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->get_property = screem_site_get_prop;
	object_class->set_property = screem_site_set_prop;
	object_class->finalize = screem_site_finalize;

	g_object_class_install_property(object_class,
					PROP_APP,
					g_param_spec_object("app",
							    "Application",
							    "The Application",
							    G_TYPE_OBJECT,
							    G_PARAM_READWRITE |
							    G_PARAM_CONSTRUCT)
					);
}

static void screem_site_init( ScreemSite *site )
{
	site->private = g_new0( ScreemSitePrivate, 1 );

	site->private->documents = g_hash_table_new( g_str_hash, g_str_equal );

	site->private->tasks = G_OBJECT( screem_todo_new( site ) );

	/* all new sites are fake ones, we change this when we
	   load / create a site */
	site->private->fake_site = TRUE;
	screem_site_set_name( site, _("Individual Files") );

	screem_site_view_initialise( site );

	site->private->excludes = gtk_list_store_new( 2, 
						      G_TYPE_STRING,
						      G_TYPE_BOOLEAN );
	site->private->ignores = gtk_list_store_new( 2, 
						     G_TYPE_STRING,
						     G_TYPE_BOOLEAN );
	site->private->asciis = gtk_list_store_new( 2, 
						    G_TYPE_STRING,
						    G_TYPE_BOOLEAN );
}

static void screem_site_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec )
{
	ScreemSite *site;

	site = SCREEM_SITE( object );

	switch( prop_id ) {
		case PROP_APP:
			site->private->application = g_value_get_object( value );
			break;
		default:
			break;
	}
}

static void screem_site_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec )
{
	ScreemSite *site;

	site = SCREEM_SITE( object );

	switch( prop_id ) {
		case PROP_APP:
			g_value_set_object( value, site->private->application );
			break;
		default:
			break;
	}
}

static void screem_site_finalize( GObject *object )
{
	ScreemSite *site;
	ScreemSitePrivate *private;

	site = SCREEM_SITE( object );

	private = site->private;

	screem_site_purge( site );
	g_object_unref( screem_site_get_todo( site ) );

	screem_site_view_destroy( site );

	g_free( private->name );
	g_free( private->pathname );
	g_free( private->remote_url );
	g_free( private->remote_path );
	g_free( private->remote_user );
	g_free( private->remote_pass );
	g_free( private->http_url );
	g_free( private->cvs_root );
	g_free( private->template_path );

	/* table already cleared by screem_site_purge() */
	g_hash_table_destroy( private->documents );

	g_object_unref( private->excludes );
	g_object_unref( private->ignores );
	g_object_unref( private->asciis );

	if( private->auto_open ) {
		g_list_foreach( private->auto_open, (GFunc)g_free, NULL );
		g_list_free( private->auto_open );
	}

	g_free( private );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}



GType screem_site_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSiteClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_site_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSite ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_site_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemSite",
					       &info, 0 );
	}

	return type;
}
