/*  Screem:  support.c,
 *  some useful functions
 *
 *  Copyright (C) 1999, 2000  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <errno.h>
#include <fnmatch.h>
#include <unistd.h>

#include <glib/gi18n.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-exec.h>

#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs.h>

#include <libgnomeui/libgnomeui.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gtksourceview/gtksourcelanguagesmanager.h>

#include <glib/ghash.h>

#include <gconf/gconf-client.h>

#include <stdio.h>
#include <string.h>

#include "fileops.h"

#include "support.h"
#include "readtags.h"

#include "screem-application.h"

/**
 * change_state:
 *
 * toggles widget sensitivity
 *
 * return values: none
 */
void change_state( GtkWidget *widget, gpointer data )
{
        gtk_widget_set_sensitive( widget, !GTK_WIDGET_IS_SENSITIVE( widget ) );
}

void change_visible( GtkWidget *widget, gpointer data )
{
	if( GTK_WIDGET_VISIBLE( widget ) ) {
		gtk_widget_hide( widget );
	} else {
		gtk_widget_show( widget );
	}
}

gchar *convert_keysym_state_to_string( guint keysym, guint state )
{
        gchar *key;
	GString *str;

        if( keysym == GDK_VoidSymbol || keysym == GDK_BackSpace ||
	    keysym == GDK_Delete || keysym == 0 ) {
                return NULL;
	}

        gdk_error_trap_push();
        key = gdk_keyval_name( keysym );
        gdk_error_trap_pop();
        if( ! key )
		return NULL;

	str = g_string_new( NULL );

        if( state & GDK_CONTROL_MASK ) {
		g_string_append( str, "Control-" );
	}
        if( state & GDK_LOCK_MASK ) {
		g_string_append( str, "Lock-" );
        }
        if( state & GDK_SHIFT_MASK ) {
		g_string_append( str, "Shift-" );
        }
        if( state & GDK_MOD1_MASK ) {
		g_string_append( str, "Mod1-" );
        }
        if( state & GDK_MOD2_MASK ) {
		g_string_append( str, "Mod2-" );
        }
        if( state & GDK_MOD3_MASK ) {
		g_string_append( str, "Mod3-" );
        }
        if( state & GDK_MOD4_MASK ) {
		g_string_append( str, "Mod4-" );
        }
        if( state & GDK_MOD5_MASK ) {
		g_string_append( str, "Mod5-" );
        }

	if( str->len ) {
		g_string_append( str, key );
	}

	key = str->str;
	g_string_free( str, FALSE );

	return key;
}

gchar *screem_support_ctags( const gchar *dirname,
			     const gchar *word,
			     gchar **pattern )
{
	gchar *ret;
	tagFile *file;
	tagFileInfo *info;
	tagEntry *entry;

	gchar *tagfile;

	if( ! word ) {
		return NULL;
	}
	
	info = g_new0( tagFileInfo, 1 );
	entry = g_new0( tagEntry, 1 );

	ret = NULL;

	tagfile = g_strconcat( dirname, G_DIR_SEPARATOR_S, "tags", NULL );
	file = tagsOpen( tagfile, info );
	if( ! file ) {
		g_free( tagfile );
		tagfile = g_strconcat( dirname, G_DIR_SEPARATOR_S, "TAGS",
				       NULL );
		file = tagsOpen( tagfile, info );
	}
	g_free( tagfile );

	if( file ) {
		if(tagsFind(file, entry, word, 
			    TAG_OBSERVECASE | TAG_FULLMATCH) == TagSuccess) {
			ret = g_strconcat( dirname, G_DIR_SEPARATOR_S,
					   entry->file, NULL );
			if( pattern ) {
				if( entry->address.pattern ) {
					*pattern = screem_support_escape_regexp( entry->address.pattern );				
				} else {
					*pattern = NULL;
				}
			}
		}
	}
	
	g_free( entry );
	g_free( info );

	return ret;
}

/* process regexp, strip leading and trailing '/' characters,
   escape ( ) { } $ * + ? */
gchar *screem_support_escape_regexp( const gchar *pattern )
{
	gchar *ret;
	GString *temp;

	g_return_val_if_fail( pattern != NULL, NULL );
	g_return_val_if_fail( *pattern == '/', NULL );

	temp = g_string_new( NULL );
	
	pattern ++;
	while( *pattern != '\0' &&
	       ( *pattern != '/' || *( pattern - 1 ) == '\\' ) ) {
		gchar p;
		gchar c;

		p = *( pattern - 1 );
		c = *pattern;

		if( p != '\\' &&
		    ( c == '(' || c == ')' || c == '{' || c == '}' ||
		      c == '*' || c == '+' || c == '?' ) ) {
			g_string_append_c( temp, '\\' );
			g_string_append_c( temp, c );
		} else if( p != '\\' && c == '$' && *(pattern+1) != '/' ) {
			g_string_append_c( temp, '\\' );
			g_string_append_c( temp, c );
		} else {
			g_string_append_c( temp, c );
		}

		pattern ++;
	}

	if( temp->len ) {
		ret = g_strdup( temp->str );
	} else {
		ret = NULL;
	}
	g_string_free( temp, TRUE );
	

	return ret;
}


GtkTreeIter *screem_support_find_in_list( GtkListStore *store, 
		guint column, const gchar *str )
{
	GtkTreeIter it;
	GtkTreeIter *ret;
	gchar *val;
	
	ret = NULL;

	g_assert( gtk_tree_model_get_column_type( GTK_TREE_MODEL( store ),
						  column ) == G_TYPE_STRING );
	
	if( gtk_tree_model_get_iter_first( GTK_TREE_MODEL( store ), &it ) ) {
		do {
			gtk_tree_model_get( GTK_TREE_MODEL( store ), &it, 
					    column, &val, -1 );

			if( val && ! strncmp( "regex:", val, strlen( "regex:" ) ) )  {
				/* regexp pattern */
				g_warning( "regex: pattern match not supported yet\n" );
			} else if( val && ! strncmp( "glob:", val, strlen( "glob:" ) ) ) {
				/* glob */
				if( fnmatch( val + strlen( "glob:" ), str, 0 ) == 0 ) {
					ret = gtk_tree_iter_copy( &it );
				}
			} else 	if( val && ! strcmp( val, str ) ) {
				/* string pattern */
				
				ret = gtk_tree_iter_copy( &it );
			}
			g_free( val );
		} while( ( ! ret ) && 
			 gtk_tree_model_iter_next( GTK_TREE_MODEL( store ),
				 		   &it ) );
	}
	
	return ret;
}

gchar *screem_support_charset_convert( const gchar *data, guint len,
		gchar **data_charset )
{
#define BOM (gunichar2)0xFEFF
#define ANTIBOM (gunichar2)0xFFFE
	
	const gunichar2 *utf16data;
	GConfClient *client;
	const gchar *end;
	gchar *ret;
	gchar *default_charset;
	
	client = gconf_client_get_default();
	default_charset = gconf_client_get_string( client,
			"/apps/screem/editor/default_charset",
			NULL );
	ret = NULL;
	utf16data = (const gunichar2*)data;
	
	if( data_charset ) {
		*data_charset = NULL;
	}
	
	if( ! data ) {

	} else if( utf16data[ 0 ] == BOM || utf16data[ 0 ] == ANTIBOM ) {
		/* UTF-16 LE / UTF-16 BE,
		 * code from ximian bugzilla attach_id=7551 */
		gunichar2 *it;
		gchar *temp;
		
		if( utf16data[ 0 ] == ANTIBOM ) {
			/* WARNING: should really be changing
			 * a const string */
			for( it = utf16data; *it != '\0'; it ++ ) {
				*it = GUINT16_SWAP_LE_BE( *it );
			}
		}
		if( utf16data[ 0 ] == BOM ) {
			utf16data ++;
			len -= 2;
		}
		temp = g_utf16_to_utf8( utf16data, 
				len / 2, NULL, NULL, NULL );
		if( temp ) {
			ret = temp;
			if( data_charset ) {
				*data_charset = g_strdup( "UTF-16" );
			}
		}
		
	} else if( ! g_utf8_validate( data, strlen( data ), &end ) ) {
		gchar *temp = NULL;
		const gchar *charset;
		gboolean utf8;

		/* FIXME: perform charset detection based on markup 
		 * rules,
		 * e.g. meta, <?xml?> etc. 
		 *
		 * can't use screem_markup_get_charset() here as
		 * it only deals with utf-8 input
		 * */
		
		utf8 = g_get_charset( &charset );

		if( ! utf8 ) {
			temp = screem_support_charset_convert_to( data,
					"UTF-8", charset );
			if( temp && data_charset ) {
				*data_charset = g_strdup( charset );
			}
		}
		if( ( ! temp ) && default_charset ) {
			temp = screem_support_charset_convert_to( data,
					"UTF-8", default_charset );
			if( temp && data_charset ) {
				*data_charset = g_strdup( default_charset );
			}
		}
		if( ( ! temp ) &&
			g_ascii_strcasecmp( charset, "ISO-8859-1" ) ) { 
			temp = screem_support_charset_convert_to( data,
					"UTF-8", "ISO-8859-1" );
			if( temp && data_charset ) {
				*data_charset = g_strdup( "ISO-8859-1" );
			}
		}

		ret = temp;
	} else  {
		ret = g_strdup( data );
		if( data_charset ) {
			*data_charset = g_strdup( "UTF-8" );
		}
	}
	g_free( default_charset );
	g_object_unref( client );
	
	return ret;
}

gchar *screem_support_charset_convert_to( const gchar *data, 
		const gchar *charset, const gchar *from )
{
	gchar *ret;

	ret = NULL;
	
	if( from && charset && g_strcasecmp( from, charset ) ) {
		/* need to convert from UTF-8 to charset */
		gchar *temp;
		guint len;
		
		temp = g_convert( data, strlen( data ), charset, 
				  from, 
				  NULL, &len, NULL );

		ret = temp;
	}

	return ret;
}

/* HACK due to the huge annoyance of gnome-vfs mime data
 * determining that anything starting <?xml  is text/xml
 * no matter what any *.mime files say about the extension */
gchar *screem_get_mime_type( const gchar *filename, gboolean fast )
{
	GnomeVFSFileInfo *info;
	GnomeVFSFileInfoOptions options;
	const gchar *type;
	gchar *ret;
	
	options = GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
	if( fast ) {
		options |= GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE;
	} else {
		
		options |= GNOME_VFS_FILE_INFO_FORCE_SLOW_MIME_TYPE;
	}
	
	info = gnome_vfs_file_info_new();
	type = NULL;
	if( gnome_vfs_get_file_info( filename, info,
				options ) == GNOME_VFS_OK ) {
		type = info->mime_type;
	}
	ret = screem_get_mime_type_override( filename, type );
	gnome_vfs_file_info_unref( info );
	
	return ret;
}

gchar *screem_get_mime_type_override( const gchar *filename,
					const gchar *type ) 
{
	const gchar *ext;
	gchar *ret;
	
	ret = NULL;
	if( type ) {
		ext = strrchr( filename, '.' );
		if( ! ext ) {
			ret = g_strdup( type );
		} else if( ! strcmp( ".screem", ext ) ) {
			ret = g_strdup("application/x-screem");
		} else if( ! strcmp( ".tagtree", ext ) ) { 
			ret = g_strdup( "application/x-screem-tag-tree" );
		} else if( ! strcmp( ".wml", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wml" );
		} else if( ! strcmp( ".wmls", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlscript" );
		} else if( ! strcmp( ".wmlb", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlb" );
		} else if( ! strcmp( ".wmlc", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlc" );
		} else if( ! strcmp( ".wmlsc", ext ) ) {
			ret = g_strdup( "text/vnd.wap.wmlscriptc" );
		} else {
			ret = g_strdup( type );
		}
	} else if( type ) {
		ret = g_strdup( type );
	}

	return ret;
}

const gchar *g_utf8_skip_space( const gchar *txt )
{
	gunichar c;

	g_return_val_if_fail( txt != NULL, NULL );
	
	while( *txt ) {
		c = g_utf8_get_char( txt );
		if( ! g_unichar_isspace( c ) ) {
			break;
		}
		txt = g_utf8_next_char( txt );
	}

	return txt;
}

const gchar *screem_utf8_skip_to_space( const gchar *txt )
{
	gunichar c;

	g_return_val_if_fail( txt != NULL, NULL );
	
	while( *txt ) {
		c = g_utf8_get_char( txt );
		if( g_unichar_isspace( c ) ) {
			break;
		}
		txt = g_utf8_next_char( txt );
	}

	return txt;
}



/* ripped out of gedit2 */
gchar* screem_escape_underlines( const gchar* text )
{
	GString *str;
	gint length;
	const gchar *p;
 	const gchar *end;

  	g_return_val_if_fail( text != NULL, NULL );

    	length = strlen( text );

	str = g_string_new( NULL );

  	p = text;
  	end = text + length;

  	while( p != end ) {
      		const gchar *next;
      		next = g_utf8_next_char( p );

		switch( *p ) {
       			case '_':
          			g_string_append( str, "__" );
          			break;
        		default:
          			g_string_append_len( str, p, next - p );
          			break;
        	}
      		p = next;
    	}

	return g_string_free( str, FALSE );
}

gboolean screem_execute_default_app( const gchar *uri )
{
	const gchar *mime;
	GnomeVFSMimeApplication *app;
	gchar *url;
	gchar *exstr;
	gchar *temp;

	mime = gnome_vfs_get_mime_type( uri );
	app = gnome_vfs_mime_get_default_application( mime );
	if( app ) {
		url = NULL;
		if( ! app->expects_uris ) {
			url = gnome_vfs_get_local_path_from_uri( uri );
		} else {
			/* FIXME check app supports the method uri
			   is using */
		}
		
		if( ! url ) {
			url = g_strdup( uri );
		}
		uri = strstr( app->command, "%s" );
		if( ! uri ) {
			exstr = g_strconcat( app->command, " \"", url, "\"", NULL );
		} else {
			temp = g_strndup( app->command, uri - app->command );
			exstr = g_strconcat( temp, url, 
					uri + strlen( "%s" ), NULL );
			g_free( temp );
		}
		if( app->requires_terminal ) {
			gnome_execute_terminal_shell( NULL, exstr );
		} else {
			gnome_execute_shell( NULL, exstr );
		}
		g_free( exstr );
	}

	return ( app != NULL );
}

void g_string_append_utf8_len( GString *str, const gchar *utf8,
				guint len )
{
	gunichar c;
	
	while( len ) {
		c = g_utf8_get_char( utf8 );
		g_string_append_unichar( str, c );
		utf8 = g_utf8_next_char( utf8 );
		len --;
	}
}

GtkPositionType screem_get_tab_position( void )
{
	GConfClient *client;
	static GConfEnumStringPair positions[] = {
		{ GTK_POS_LEFT, "left" },
		{ GTK_POS_RIGHT, "right" },
		{ GTK_POS_TOP, "top" },
		{ GTK_POS_BOTTOM, "bottom" },
		{ GTK_POS_TOP, "" },
		{ GTK_POS_TOP, NULL }
	};
	gchar *pos;
	gint position;

	client = gconf_client_get_default();
		
	pos = gconf_client_get_string( client,
			"/apps/screem/ui/tab_position",
			NULL );
	if( ! pos ) {
		pos = g_strdup( "top" );
	}
	gconf_string_to_enum( positions, pos, &position );
	g_free( pos );

	g_object_unref( client );

	return position;
}

void screem_set_cursor( GtkWidget *widget, GdkCursorType type )
{
	GdkDisplay *display;
	GdkCursor *cursor;
		
	cursor = NULL;
	display = NULL;
	if( widget ) {
		display = gtk_widget_get_display( GTK_WIDGET( widget ) );
	}
	if( type != GDK_LAST_CURSOR ) {
		if( display ) {
			cursor = gdk_cursor_new_for_display( display, type );
		} else {
			cursor = gdk_cursor_new( type );
		}
	}
	if( widget && widget->window ) {
		gdk_window_set_cursor( widget->window, cursor );
	}
	if( cursor ) {
		gdk_cursor_unref( cursor );
	}
	gdk_threads_leave(); 
	gdk_flush();
	gdk_threads_enter();
}


/**
 * screem_popup_menu_do_popup_modal, based off 
 * gnome_popup_menu_do_popup_modal
 */
void screem_popup_menu_do_popup_modal( GtkWidget *popup, 
		GtkMenuPositionFunc pos_func, gpointer pos_data,
		GdkEventButton *event, gpointer user_data, 
		GtkWidget *for_widget )
{
	guint id;
	guint button;
	guint32 timestamp;
	
	id = g_signal_connect( popup, "deactivate",
			       G_CALLBACK( gtk_main_quit ),
			       NULL );

	if( event ) {
		button = event->button;
		timestamp = event->time;
	} else {
		button = 0;
		timestamp = GDK_CURRENT_TIME;
	}
#ifdef HAVE_GTK_MULTIHEAD
	gtk_menu_set_screen( GTK_MENU( popup ), 
			gtk_widget_get_screen( for_widget ) );
#endif	
	
	gtk_menu_popup( GTK_MENU( popup ), 
			NULL, NULL, 
			pos_func, pos_data, button, timestamp );
	
	gtk_grab_add( popup );
	gtk_main();
	gtk_grab_remove( popup );

	g_signal_handler_disconnect( G_OBJECT( popup ), id );

}

static gboolean add_filter_type( gpointer key, gpointer value,
		gpointer data )
{
	gtk_file_filter_add_mime_type( GTK_FILE_FILTER( data ),
			(const gchar *)value );
	return FALSE;
}

static gboolean filter_mime_wildcard( const GtkFileFilterInfo *info,
		gpointer data )
{
	gboolean ret;

	ret = FALSE;
	if( info->contains & GTK_FILE_FILTER_MIME_TYPE ) {
		ret = g_pattern_match_string( (GPatternSpec *)data, 
				info->mime_type );
	}

	return ret;
}

static gboolean filter_mime_inherits( const GtkFileFilterInfo *info,
		gpointer data )
{
	gboolean ret;

	ret = FALSE;
	
	if( info->contains & GTK_FILE_FILTER_MIME_TYPE ) {
		ret = ! strcmp( (gchar*)data, info->mime_type );
		if( ! ret ) {
			ret = ( gnome_vfs_mime_type_get_equivalence( info->mime_type, (gchar*)data ) == GNOME_VFS_MIME_PARENT );
		}
	}

	return ret;
}

GtkFileFilter *screem_get_file_filter( const gchar *name )
{
	static GHashTable *filters = NULL;
	GtkSourceLanguagesManager* lm;
	GHashTable *features;
	GHashTable *feature;
	GtkFileFilter *filter = NULL;

	/* cleanup mode */
	if( ! name ) {
		if( filters ) {
			g_hash_table_destroy( filters );
			filters = NULL;
		}

		return NULL;
	}
	
	if( ! filters ) {
		filters = g_hash_table_new_full( g_str_hash,
				g_str_equal,
				(GDestroyNotify)g_free,
				(GDestroyNotify)g_object_unref );
	}
	
	filter = g_hash_table_lookup( filters, name );
	if( ! filter ) {
		lm = screem_application_load_syntax_tables();
		features = g_object_get_data( G_OBJECT( lm ),
				"features" );
		filter = gtk_file_filter_new();
		gtk_file_filter_set_name( filter, name );
	
		if( ! strcmp( _( "All Files" ), name ) ) {
			gtk_file_filter_add_pattern( filter, "*" );
		} else 	if( ! strcmp( _( "Markup Files" ), name ) ) {
			/* get markup types */
			feature = g_hash_table_lookup( features,
					"markup" );
			g_assert( feature );
			g_hash_table_foreach( feature,
					(GHFunc)add_filter_type, 
					filter );
			/* we also want to handle text/xml types,
			 * also add application/+xml types for
			 * those that don't correctly set what
			 * they inherit from */
			gtk_file_filter_add_custom( filter,
					GTK_FILE_FILTER_MIME_TYPE,
					(GtkFileFilterFunc)filter_mime_inherits,
					"text/xml", NULL ); 
			gtk_file_filter_add_custom( filter,
					GTK_FILE_FILTER_MIME_TYPE,
					(GtkFileFilterFunc)filter_mime_wildcard,
					g_pattern_spec_new( "application/*+xml" ), 
					(GDestroyNotify)g_pattern_spec_free);
		} else if( ! strcmp( _( "CSS Files" ), name ) ) {
			gtk_file_filter_add_mime_type( filter,
					"text/css" );
		} else if( ! strcmp( _( "Javascript Files" ), name ) ) {
			gtk_file_filter_add_mime_type( filter,
					"text/javascript" );
			gtk_file_filter_add_mime_type( filter,
					"text/x-javascript" );
		} else if( ! strcmp( _( "All Text Files" ), name ) ) {
			gtk_file_filter_add_custom( filter,
					GTK_FILE_FILTER_MIME_TYPE,
					(GtkFileFilterFunc)filter_mime_wildcard,
					g_pattern_spec_new( "text/*" ), 
					(GDestroyNotify)g_pattern_spec_free);
			gtk_file_filter_add_custom( filter,
					GTK_FILE_FILTER_MIME_TYPE,
					(GtkFileFilterFunc)filter_mime_inherits,
					"text/plain", NULL ); 
} else {
			/* unknown filter requested */
			g_assert( FALSE );
		}
		g_object_ref( filter );
		g_hash_table_insert( filters, g_strdup( name ),
				filter );
	} 

	return filter;
}

/* override a GnomeFileEntry widgets browser button, horrible
 * hack but it gets us the ability to place / size the dialog +
 * use non local volumes */
void screem_file_entry_browse( GnomeFileEntry *entry, gpointer data )
{
	gchar *title;
	gchar *dir;
	GtkFileChooserAction action;
	gboolean isdir;
	gchar *filename;
	GtkWidget *widget;
	
	g_signal_stop_emission_by_name( G_OBJECT( entry ), 
			"browse_clicked" );

	g_object_get( G_OBJECT( entry ),
			"browse_dialog_title", &title,
			"default_path", &dir,
			"filechooser_action", &action,
			"gtk_entry", &widget,
			"directory_entry", &isdir,
			NULL );
	
	if( isdir ) {
		switch( action ) {
			case GTK_FILE_CHOOSER_ACTION_SAVE:
				action = GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER;
				break;
			case GTK_FILE_CHOOSER_ACTION_OPEN:
				action = GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER;
				break;
			default:
				break;
		}
		
	}
	
	filename = screem_fileselect( title, NULL, action, dir, NULL ); 

	if( filename ) {
		gtk_entry_set_text( GTK_ENTRY( widget ), filename );
		g_free( filename );
	}
	
	g_free( title );
	g_free( dir );
	g_object_unref( G_OBJECT( widget ) );
}

gchar *screem_get_glade_path( void )
{
	GConfClient *client;
	gchar *glade_path;
	
	client = gconf_client_get_default();
	glade_path = gconf_client_get_string( client,
			"/apps/screem/general/glade_path", NULL );
	/* check for existance as well, if screem has been installed
	 * into another prefix and the old version erased the gconf
	 * key will point to the wrong place */
	if( ( ! glade_path ) || ! uri_exists( glade_path, NULL ) ) {
		glade_path = g_strconcat( GLADE_PATH, "/screem.glade", 
				NULL );
		gconf_client_set_string( client,
				"/apps/screem/general/glade_path",
				glade_path, NULL );
	}

	g_object_unref( client );

	return glade_path;
}

static gboolean find_accel( GtkAccelKey *key, GClosure *closure,
		GdkEventKey *event )
{
	gboolean ret;

	ret = ( key->accel_key == event->keyval );
	if( ret ) {
		ret = ( key->accel_mods & event->state );
	}

	return ret;
}

gboolean screem_find_accel( GtkAccelGroup *group, GdkEventKey *event )
{
	return ( gtk_accel_group_find( group, 
				(GtkAccelGroupFindFunc)find_accel, 
				event ) != NULL );
}

gint screem_mime_type_compare( gconstpointer a, gconstpointer b )
{
	const gchar *showa;
	const gchar *showb;
	
	showa = gnome_vfs_mime_get_description( a );
	showb = gnome_vfs_mime_get_description( b );

	if( ! showa ) {
		showa = a;
	}
	if( ! showb ) {
		showb = b;
	}

	return g_strcasecmp( showa, showb );
}

void screem_hig_alert( const gchar *stock_id,
		const gchar *primary, const gchar *secondary,
		GtkWidget *parent )
{
	GtkWidget *widget;
	GtkWidget *hbox;
	GtkWidget *icon;
	
	GtkWidget *label;
	gchar *txt;
	gchar *labeltxt;

	widget = gtk_dialog_new();
	gtk_dialog_set_has_separator( GTK_DIALOG( widget ), FALSE );
	gtk_window_set_role( GTK_WINDOW( widget ), 
			"screem_startup_error" );
	gtk_container_set_border_width( GTK_CONTAINER( widget ), 6 );
	
	gtk_box_set_spacing( GTK_BOX( GTK_DIALOG( widget )->vbox ), 
			12 );
	
	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_container_set_border_width( GTK_CONTAINER( hbox ), 6 );
	gtk_box_set_spacing( GTK_BOX( hbox ), 12 );
		
	gtk_container_add( GTK_CONTAINER( GTK_DIALOG( widget )->vbox ),
			hbox );

	gtk_dialog_add_button( GTK_DIALOG( widget ),
			GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
	
	txt = g_strconcat( "<span weight=\"bold\" size=\"larger\">",
			primary, "</span>", NULL );
	label = gtk_label_new( "" );
	labeltxt = g_strconcat( txt, "\n\n", secondary, NULL );
	gtk_label_set_markup( GTK_LABEL( label ), labeltxt );
	g_free( txt );
	g_free( labeltxt );
	gtk_label_set_selectable( GTK_LABEL( label ), TRUE );

	icon = gtk_image_new_from_stock( GTK_STOCK_DIALOG_ERROR,
			GTK_ICON_SIZE_DIALOG );
	gtk_misc_set_alignment( GTK_MISC( icon ), 0.0, 0.0 );
	gtk_box_pack_start_defaults( GTK_BOX( hbox ), icon );
	
	gtk_label_set_line_wrap( GTK_LABEL( label ), TRUE );
	gtk_misc_set_alignment( GTK_MISC( label ), 0.0, 0.0 );
	gtk_box_pack_start_defaults( GTK_BOX( hbox ), label );
	
	
	gtk_widget_show_all( widget );
	
	if( parent ) {
		gtk_window_set_transient_for( GTK_WINDOW( widget ),
				GTK_WINDOW( parent ) );
	}
	gtk_dialog_run( GTK_DIALOG( widget ) );
	gtk_widget_destroy( widget );
}

GtkListStore *screem_support_get_mime_types_store( void )
{
	GtkTreeModel *list;
	GSList *types;
	const GSList *langs;
	GtkSourceLanguagesManager *lm;
	GtkTreeIter it;
	
	list = GTK_TREE_MODEL( gtk_list_store_new( 2, 
				G_TYPE_STRING, G_TYPE_STRING ) );
	types = NULL;
	lm = screem_application_load_syntax_tables();
	langs = gtk_source_languages_manager_get_available_languages( lm );
	while( langs ) {
		GtkSourceLanguage *lang;
		GSList *tmp;
		lang = GTK_SOURCE_LANGUAGE( langs->data );
		tmp = gtk_source_language_get_mime_types( lang );
		tmp = g_slist_copy( tmp );
		if( types ) {
			types = g_slist_concat( types, tmp );
		} else {
			types = tmp;
		}
		langs = langs->next;
	}
	/* add plain text */
	types = g_slist_prepend( types, "text/plain" );
	types = g_slist_sort( types, (GCompareFunc)screem_mime_type_compare );
	for( langs = types; langs; langs = langs->next ) {
		const gchar *showtype;
		GtkTreeIter *fit;

		showtype = gnome_vfs_mime_get_description( langs->data );
		if( ! showtype ) {
			showtype = langs->data;
		}
			
		fit = screem_support_find_in_list( GTK_LIST_STORE( list ), 1, langs->data );
		if( ! fit ) {
			gtk_list_store_append( GTK_LIST_STORE( list ),
					&it );
			gtk_list_store_set( GTK_LIST_STORE( list ), 
					&it,
					0, showtype,
					1, langs->data,
					-1 );
		} else {
			gtk_tree_iter_free( fit );
		}
	}

	return GTK_LIST_STORE( list );
}

gchar *screem_gdk_color_to_string( GdkColor *colour )
{
	guint16 r;
	guint16 g;
	guint16 b;
		
	r = colour->red;
	g = colour->green;
	b = colour->blue;

	r = r >> 8;
	g = g >> 8;
	b = b >> 8;
	
	return g_strdup_printf( "#%.2x%.2x%.2x", r, g, b );
}

gchar* screem_escape_char( const gchar* text, gchar c )
{
	GString *str;
	gint length;
	const gchar *p;
 	const gchar *end;

  	g_return_val_if_fail( text != NULL, NULL );

    	length = strlen( text );

	str = g_string_new( NULL );

  	p = text;
  	end = text + length;

  	while( p != end ) {
      		const gchar *next;
      		next = g_utf8_next_char( p );

		if( *p == c ) {
			g_string_append_c( str, '_' );
		} else {
			g_string_append_len( str, p, next - p );
		}
	  	p = next;
    	}

	return g_string_free( str, FALSE );
}

