#ifndef SCREEM_PLUGIN_H
#define SCREEM_PLUGIN_H

#include <glib-object.h>

#include <libgnomevfs/gnome-vfs.h>

#include "screem-dtd.h"
#include "screem-page.h"
#include "screem-site.h"
#include "fileops.h"

#define SCREEM_TYPE_PLUGIN            (screem_plugin_get_type ())
#define SCREEM_PLUGIN(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), SCREEM_TYPE_PLUGIN, ScreemPlugin))
#define SCREEM_PLUGIN_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), SCREEM_TYPE_PLUGIN, ScreemPluginClass))
#define SCREEM_IS_PLUGIN(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), SCREEM_TYPE_PLUGIN))

typedef struct ScreemPluginPrivate ScreemPluginPrivate;

typedef struct ScreemPlugin ScreemPlugin;

#define SCREEM_PLUGIN_REQUIRED_VERSION 5

#define SCREEM_INSERT_MENU "/Insert"
#define SCREEM_INSERT_WIZARD_MENU "/Insert/Wizards"
#define SCREEM_WIZARD_TOOLBAR "/Wizards Toolbar"

struct ScreemPlugin {
	GObject parent;

	/* virtual functions */
	gboolean (*setup)( ScreemPlugin *plugin );
	
	ScreemPluginPrivate *priv;
};

typedef struct {
	GObjectClass parent_class;

} ScreemPluginClass;

GType screem_plugin_get_type( void );
ScreemPlugin* screem_plugin_new( const gchar *name );

gboolean screem_plugin_setup( ScreemPlugin *plugin );

/* UI setup functions */
gboolean screem_plugin_add_menu( ScreemPlugin *plugin,
		const gchar *path, const gchar *action,
		GError **error );
gboolean screem_plugin_add_toolbar( ScreemPlugin *plugin,
		const gchar *path, const gchar *action,
		GError **error );
gboolean screem_plugin_add_action( ScreemPlugin *plugin,
		const gchar *name, const gchar *label,
		const gchar *tip, const gchar *stock_id,
		GCallback activate, GError **error );
gboolean screem_plugin_add_gtk_action( ScreemPlugin *plugin,
		GtkAction *action, GError **error  );

/* util function for the usual case of 1 action + 
 * 1 menu item + 1 toolbar item */
gboolean screem_plugin_add_interface( ScreemPlugin *plugin,
		const gchar *name, const gchar *label,
		const gchar *tip, const gchar *stock_id,
		GCallback activate,
		GError **error );

/* session management */
void screem_plugin_restore_from_session( ScreemPlugin *plugin,
		GtkWidget *window );
void screem_plugin_store_in_session( ScreemPlugin *plugin,
		GtkWidget *window );

/* current open file info */
ScreemSite *screem_plugin_get_current_site( ScreemPlugin *plugin );
ScreemPage *screem_plugin_get_current_document( ScreemPlugin *plugin );
ScreemDTD *screem_plugin_get_current_dtd( ScreemPlugin *plugin );

/* text insertion */
guint screem_plugin_get_cursor_position( ScreemPlugin *plugin );
void screem_plugin_set_cursor_position( ScreemPlugin *plugin, guint pos );

void screem_plugin_insert( ScreemPlugin *plugin, gint pos,
		const gchar *text, guint length, gboolean indent );

void screem_plugin_insert_markup( ScreemPlugin *plugin,
		const gchar *open, const gchar *close,
		gboolean indent );

/* file related */
void screem_plugin_file_op( GnomeVFSMonitorEventType type,
		const gchar *uri, gpointer data );

/* messaging */
void screem_plugin_show_message( ScreemPlugin *plugin,
		const gchar *message );
void screem_plugin_show_error( ScreemPlugin *plugin, const gchar *error );

typedef struct ScreemPluginDetails {
	const gchar *name;
	const gchar *displayed_name;
	const gchar **authors;
	const gchar *description;
	const gchar *version;
	GModule *mod;
	ScreemPlugin* (*create)( void );
	gint api_version;
} ScreemPluginDetails;

#endif

