/*  Screem:  screem-page.c
 *
 *  The ScreemPage object
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>

#include <gconf/gconf-client.h>

#include <glib/gconvert.h>
#include <glib/gunicode.h>
#include <glib/gi18n.h>

#include <gtk/gtktreemodel.h>
#include <gtk/gtkliststore.h>

#include <gtksourceview/gtksourcebuffer.h>

#include "screem-page.h"
#include "screem-page-model.h"

#include "fileops.h"

/* yuk */
#include "pageUI.h"

/* yuk */
#include "screem-markup.h"

#include "screem-search.h"

/* yuk */
#include "support.h"

#include "screemmarshal.h"

enum {
	SAVING,
	SAVED,
	/* these 2 are fired from the ScreemPageModel */
	BUILDING,
	BUILT,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_APP,
	PROP_PATHNAME,
	PROP_CHANGED,
	PROP_MIMETYPE,
	PROP_OPEN,
	PROP_CHARSET,
	PROP_DTD
};

static guint screem_page_signals[ LAST_SIGNAL ] = { 0 };

static gboolean bookmark_store_find( GtkTreeModel *model, GtkTreePath *path,
				     GtkTreeIter *iter, gpointer data );

static void changed( GtkTextBuffer *buffer, ScreemPage *page );

static void screem_page_monitor_cb( GnomeVFSMonitorHandle *handle,
				    const gchar *monitor_uri,
				    const gchar *info_uri,
				    GnomeVFSMonitorEventType type,
				    gpointer data );

static gboolean screem_page_auto_save( ScreemPage *page );
static void screem_page_auto_save_notify( GConfClient *client,
					  guint cnxn_id,  GConfEntry *entry,
					  gpointer data );
static void screem_page_auto_save_interval_notify( GConfClient *client,
					  guint cnxn_id,  GConfEntry *entry,
					  gpointer data );

struct ScreemPagePrivate {
	ScreemPageModel *model;
	
	gchar *pathname;     /* pathname for the page */
	gchar *mime_type;
	gboolean changed;

	gboolean loaded;
	gboolean opened;

	GConfClient *client;
	gboolean highlight;  /* do we think highlighting is on? */
	gint highlight_notify;

	GHashTable *bookmarks;
	GtkListStore *bookmarkstore;

	GnomeVFSMonitorHandle *monitor;
	gboolean modified;
	GTimeVal time;

	gboolean compress;

	gboolean auto_save;
	guint auto_save_idle;
	guint auto_save_interval;
	guint auto_save_notify;
	guint auto_save_interval_notify;

};

void screem_page_undo( ScreemPage *page )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	if( gtk_source_buffer_can_undo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_undo( GTK_SOURCE_BUFFER( page ) );
	}
}

void screem_page_redo( ScreemPage *page )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	if( gtk_source_buffer_can_redo( GTK_SOURCE_BUFFER( page ) ) ) {
		gtk_source_buffer_redo( GTK_SOURCE_BUFFER( page ) );
	}
}

ScreemPage *screem_page_new( GObject *application )
{
	ScreemPage *page;
	GType type;

	type = screem_page_get_type();
	
	page = SCREEM_PAGE( g_object_new( type, "tag-table", NULL, "app", application, NULL ) );

	/* need to do this here so the initial mime type is
	 * correctly set otherwise we don't get highlighting.
	 * can't do it in screem_page_init() as the
	 * GtkSourceBuffer construction hasn't finished */
	screem_page_set_mime_type( page, page->private->mime_type );
	
	return page;
}

gboolean screem_page_is_loaded( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return ( page->private->loaded || 
		 page->private->pathname == NULL );
}

gboolean screem_page_is_open( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return page->private->opened;
}

void screem_page_set_open( ScreemPage *page, gboolean opened )
{
	g_object_set( G_OBJECT( page ), "open", opened, NULL );
}

static void screem_page_set_open_real( ScreemPage *page, gboolean opened )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	page->private->opened = opened;
}


void screem_page_set_pathname( ScreemPage *page, const gchar *path )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	g_object_set( G_OBJECT( page ), "pathname", path, NULL );
}

static void screem_page_set_pathname_real( ScreemPage *page, const gchar *path )
{	
	if( page->private->pathname && path != page->private->pathname ) {
		g_free( page->private->pathname );
	}

	if( page->private->mime_type ) {
		g_free( page->private->mime_type );
	}
	page->private->mime_type = NULL;
	
	if( path ) {
		page->private->mime_type = 
			screem_get_mime_type( path, FALSE );
	}
	if( ! page->private->mime_type ) {
		page->private->mime_type = gconf_client_get_string( page->private->client, "/apps/screem/default_mimetype", NULL );
		if( ! page->private->mime_type ) {
			page->private->mime_type = g_strdup( "text/html" );
		}
	}

	if( path != page->private->pathname ) {
		if( path ) {
			page->private->pathname = g_strdup( path );
		} else {
			page->private->pathname = NULL;
		}
	}
	
	if( page->private->monitor ) {
		gnome_vfs_monitor_cancel( page->private->monitor );
		page->private->monitor = NULL;
	}
	if( page->private->pathname ) {
		gnome_vfs_monitor_add( &page->private->monitor,
				       page->private->pathname,
				       GNOME_VFS_MONITOR_FILE,
				       screem_page_monitor_cb, page );
	}
}

const gchar *screem_page_get_pathname( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	return page->private->pathname;
}

gchar *screem_page_get_short_name( ScreemPage *page )
{
	const gchar *pathname;
	gchar *basename;
	gchar *temp;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );
	
	pathname = screem_page_get_pathname( page );

	if( ! pathname ) {
		pathname = _( "Untitled" );
	}
	basename = g_path_get_basename( pathname );
	temp = gnome_vfs_unescape_string_for_display( basename );
	g_free( basename );
	
	return temp;
}

gchar *screem_page_get_title_name( ScreemPage *page )
{
	const gchar *mime;
	const gchar *showtype;
	const gchar *pathname;
	gchar *tmppath;
	gchar *base;
	gchar *dir;
	gchar *temp;
	gchar *title;
	gchar *path;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), "Screem" );
	
	mime = screem_page_get_mime_type( page );
	
	showtype = NULL;
	if( mime ) {
		showtype = gnome_vfs_mime_get_description( mime );
	} else {
		showtype = "";
	}
	
	pathname = screem_page_get_pathname( page );
	
	tmppath = NULL;
	if( ! pathname ) {
		pathname = _( "Untitled" );
		base = g_strconcat( pathname, " ",
				showtype, NULL );
		dir = NULL;
	} else {
		if( g_str_has_prefix( pathname, "file://" ) ) {
			pathname += strlen( "file://" );
		}
		path = NULL;
		if( *pathname == '/' ) {
			dir = g_build_path( G_DIR_SEPARATOR_S,
				g_get_home_dir(), "", NULL );
			if( g_str_has_prefix( pathname, dir ) ) {
				path = g_build_path( G_DIR_SEPARATOR_S,
					"~", pathname + strlen( dir ), NULL );
				pathname = path; 
			}
			g_free( dir );
		}
		dir = g_path_get_dirname( pathname );
		base = g_path_get_basename( pathname );
		temp = gnome_vfs_unescape_string_for_display( base );
		g_free( base );
		base = temp;
			
		temp = gnome_vfs_unescape_string_for_display( dir );
		g_free( dir );
		dir = temp;

		g_free( path );
	}

	if( ! dir ) {
		title = g_strdup( base );
	} else {
		title = g_strdup_printf( "%s (%s)", base, dir );
	}

	g_free( tmppath );
	g_free( base );
	g_free( dir ); 

	return title;
}

void screem_page_set_mime_type( ScreemPage *page, const gchar *mime_type )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	g_object_set( G_OBJECT( page ), "mime-type", mime_type, NULL );
}

static void screem_page_set_mime_type_real( ScreemPage *page, const gchar *mime_type )
{
	gchar *type;
	
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	type = NULL;
	if( ! mime_type ) {
		type = gconf_client_get_string( page->private->client, "/apps/screem/default_mimetype", NULL );
		if( ! type ) {
			type = g_strdup( "text/html" );
		}
		mime_type = type;
	}
	
	if( page->private->mime_type ) {
		g_free( page->private->mime_type );
	}
	page->private->mime_type = g_strdup( mime_type );

	screem_page_model_force_check( page->private->model,
			FALSE );

	g_free( type );
}

const gchar *screem_page_get_mime_type( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	return page->private->mime_type;
}

gchar *screem_page_get_charset( ScreemPage *page )
{
	gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	ret = NULL;

	g_object_get( G_OBJECT( page->private->model ),
			"charset", &ret, NULL );
	
	return ret;
}

void screem_page_set_data( ScreemPage *page, const gchar *data )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	gtk_source_buffer_begin_not_undoable_action( GTK_SOURCE_BUFFER( page ) );

	if( ! data ) {
		data = "";
	}
	gtk_text_buffer_set_text( GTK_TEXT_BUFFER( page ), data, 
				strlen( data ) );
	
/*	screem_page_model_force_check( page->private->model );*/
		
	page->private->loaded = TRUE;
	gtk_source_buffer_end_not_undoable_action( GTK_SOURCE_BUFFER( page ) );
}

gchar *screem_page_get_data( ScreemPage *page )
{
	GtkTextIter it;
	GtkTextIter eit;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	gtk_text_buffer_get_start_iter( GTK_TEXT_BUFFER( page ),
					&it );
	gtk_text_buffer_get_end_iter( GTK_TEXT_BUFFER( page ),
					&eit );

	return gtk_text_buffer_get_text( GTK_TEXT_BUFFER( page ), 
					&it, &eit, TRUE );
}

void screem_page_set_changed( ScreemPage *page, gboolean flag )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	g_object_set( G_OBJECT( page ), "changed", flag, NULL );
}

static void screem_page_set_changed_real( ScreemPage *page, gboolean flag )
{
	ScreemPagePrivate *priv;
	guint interval;
	
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	priv = page->private;

	priv->changed = flag;

	if( ! flag ) {
		/* remove timer as we have nothing to save */
		if( priv->auto_save_idle ) {
			g_source_remove( priv->auto_save_idle );
			priv->auto_save_idle = 0;
		}
		g_get_current_time( &priv->time );
	} else if( priv->auto_save && ! priv->auto_save_idle ) {
		/* add timer if needed */
		interval = priv->auto_save_interval * 60 * 1000;
		priv->auto_save_idle = g_timeout_add( interval,
			(GSourceFunc)screem_page_auto_save, page );
	}
}

gboolean screem_page_get_changed( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return page->private->changed;
}

gboolean screem_page_load( ScreemPage *page, GError **error )
{
	ScreemPagePrivate *priv;
	const gchar *path;
	gboolean compress;
	gchar *charset;

	gchar *data;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	if( error ) {
		*error = NULL;
	}
	
	if( screem_page_is_loaded( page ) ) {
		return TRUE;
	}

	priv = page->private;
	path = screem_page_get_pathname( page );

	g_return_val_if_fail( path != NULL, FALSE );
	
	priv->compress = FALSE;
	data = load_file( path, &compress, &charset, error );
	
	if( data ) {
		screem_page_set_data( page, data );

		screem_page_set_changed( page, FALSE );

		priv->loaded = TRUE;
		priv->compress = compress;
	
		if( screem_page_model_is_default_charset( priv->model ) ) {
			/* set to charset we detected at load time */
			g_object_set( G_OBJECT( priv->model ), "charset", charset, NULL );
		}	
		g_free( charset );
		g_free( data );
	} else {
		g_print( "FAILED TO LOAD\n" );
	}

	page->private->modified = FALSE;

	return ( data != NULL );
}

gboolean screem_page_revert( ScreemPage *page, GError **error )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	page->private->loaded = FALSE;

	return screem_page_load( page, error );
}

gboolean screem_page_save( ScreemPage *page, GError **error )
{
	ScreemPagePrivate *priv;
	gchar *data;
	gboolean retval;

	gchar *charset;
	gchar *temp;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	if( error ) {
		*error = NULL;
	}
	
	if( ( ! screem_page_get_changed( page ) ) ||
	    ( ! screem_page_save_check( page ) ) ) {
		return TRUE;
	}

	priv = page->private;
	
	g_signal_emit( G_OBJECT( page ),
		       screem_page_signals[ SAVING ], 0 );

	screem_page_model_force_check( priv->model, TRUE );
	
	data = screem_page_get_data( page );

	charset = screem_page_get_charset( page );
	
	if( ! charset ) {
		GConfClient *client;

		client = gconf_client_get_default();

		charset = gconf_client_get_string( client,
						   "/apps/screem/editor/default_charset",
						   NULL );
		if( ! charset ) {
			g_get_charset( (const gchar**)&charset );
			charset = g_strdup( charset );
		}
		g_object_unref( G_OBJECT( client ) );
	}

	temp = screem_support_charset_convert_to( data, charset,
			"UTF-8" );
	if( temp ) {
		g_free( data );
		data = temp;
	}
	g_free( charset );

	/* stop monitoring while we write to the file ourselves */
	if( priv->monitor ) {
		gnome_vfs_monitor_cancel( priv->monitor );
	}
	priv->monitor = NULL;

	retval = save_file( screem_page_get_pathname( page ),
			    data, 
			    GNOME_VFS_PERM_USER_READ | 
			    GNOME_VFS_PERM_USER_WRITE |
			    GNOME_VFS_PERM_GROUP_READ |
			    GNOME_VFS_PERM_GROUP_WRITE |
			    GNOME_VFS_PERM_OTHER_READ,
			    priv->compress,
			    TRUE,
			    error );

	g_free( data );

	if( retval ) {
		g_signal_emit( G_OBJECT( page ),
			       screem_page_signals[ SAVED ], 0 );
		screem_page_set_changed( page, FALSE );
	}
	
	priv->modified = FALSE;

      	if( ! screem_page_get_mime_type( page ) ) {
		screem_page_set_pathname( page, 
					  screem_page_get_pathname( page ) );
	} else {
		gnome_vfs_monitor_add( &priv->monitor,
				       priv->pathname,
				       GNOME_VFS_MONITOR_FILE,
				       screem_page_monitor_cb, page );
	}
	return retval;
}

void screem_page_unload( ScreemPage *page )
{
	GtkTextIter it;
	GtkTextIter eit;
	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	g_return_if_fail( screem_page_is_open( page ) == FALSE );
		
	if( screem_page_is_loaded( page ) ) {
		page->private->loaded = FALSE;
		gtk_text_buffer_get_bounds( GTK_TEXT_BUFFER( page ), &it, &eit );
		gtk_text_buffer_delete( GTK_TEXT_BUFFER( page ), &it, &eit );
	}
}

glong screem_page_get_seconds_since_last_save_or_load( ScreemPage *page )
{
	GTimeVal current_time;
	ScreemPagePrivate *priv;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), -1); 
	
	priv = page->private;
	g_get_current_time(&current_time); 
	
	return( current_time.tv_sec - priv->time.tv_sec );
}


gboolean screem_page_is_file_page( const gchar *path )
{
	gchar *mime_type;
	gboolean ret;
	
	g_return_val_if_fail( path != NULL, FALSE );
	
	mime_type = screem_get_mime_type( path, FALSE );

	ret =  screem_page_is_mime_type_page( mime_type );
	g_free( mime_type );

	return ret;
}

gboolean screem_page_is_mime_type_page( const gchar *mime_type )
{
	gboolean is_page;
	const gchar *end;
	
	is_page = FALSE;
	
	if( mime_type ) {
		is_page = ! strncmp( mime_type, "text/", strlen( "text/" ) );
		is_page |= ! strcmp( mime_type, "application/x-php" );
		is_page |= ! strcmp( mime_type, "application/x-asp" );
		is_page |= ! strcmp( mime_type, "application/x-cgi" );
		is_page |= ! strcmp( mime_type, "application/x-screem-tag-tree" );
		if( ! is_page && ! strncmp( "application/", mime_type,
					strlen( "application/" ) ) ) {
			
			end = g_utf8_strrchr( mime_type, -1, '+'  );
			if( end ) {
				is_page |= ! strcmp( end, "+xml" );
			}
		}
		if( ! is_page ) {
			is_page = ( gnome_vfs_mime_type_get_equivalence( mime_type, "text/plain" ) == GNOME_VFS_MIME_PARENT );
		}
	}
	
	return is_page;
}


gboolean screem_page_get_modified( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	return page->private->modified;
}

GtkListStore *screem_page_get_bookmark_model( ScreemPage *page )
{
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), NULL );

	return page->private->bookmarkstore;
}

void screem_page_add_bookmark( ScreemPage *page, gint line, const gchar *name )
{
	ScreemPagePrivate *private;
	ScreemPageBookMark *bookmark;
	GHashTable *table;
	GtkTreeIter it;
	GtkTextIter tit;

	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	private = page->private;

	table = private->bookmarks;

	gtk_text_buffer_get_iter_at_line( GTK_TEXT_BUFFER( page ),
					  &tit, line - 1 );
		
	bookmark = g_hash_table_lookup( table, name );
	if( bookmark ) {
		/* already got a bookmark with this name, just update
		   the line */
		bookmark->line_number = line;

		gtk_tree_model_foreach( GTK_TREE_MODEL(private->bookmarkstore),
					bookmark_store_find, bookmark );	
		it = *(GtkTreeIter*)bookmark->iter;
		gtk_source_buffer_move_marker( GTK_SOURCE_BUFFER( page ),
						bookmark->marker,
						&tit );
	} else {

		bookmark = g_new0( ScreemPageBookMark, 1 );
		bookmark->line_number = line;
		bookmark->name = g_string_new( name );

		gtk_list_store_append( private->bookmarkstore, &it );

		g_hash_table_insert( table, bookmark->name->str, bookmark );
		
		bookmark->marker = 
			gtk_source_buffer_create_marker( GTK_SOURCE_BUFFER( page ), 
							 NULL,
							 BOOKMARK_MARKER, 
							 &tit );
	}

	gtk_list_store_set( private->bookmarkstore, &it,
			    0, bookmark->name->str,
			    1, bookmark->line_number,
			    2, bookmark, -1 );
	
}

void screem_page_remove_bookmark( ScreemPage *page, const gchar *name )
{
	ScreemPagePrivate *private;
	ScreemPageBookMark *bookmark;
	GHashTable *table;

	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	private = page->private;

	table = private->bookmarks;

	bookmark = g_hash_table_lookup( table, name );
	if( bookmark ) {
		/* found the bookmark we are after */
		GtkTreeIter it;

		g_hash_table_remove( table, name );

		gtk_source_buffer_delete_marker( GTK_SOURCE_BUFFER( page ),
						bookmark->marker );
		
		gtk_tree_model_foreach( GTK_TREE_MODEL(private->bookmarkstore),
					bookmark_store_find, bookmark );
		it = *(GtkTreeIter*)bookmark->iter;
		gtk_list_store_remove( private->bookmarkstore,
				       &it );

		g_string_free( bookmark->name, TRUE );
		g_free( bookmark );
	}
}

gint screem_page_get_bookmark( ScreemPage *page, const gchar *name )
{
	ScreemPagePrivate *private;
	ScreemPageBookMark *bookmark;
	GHashTable *table;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), -1 );

	private = page->private;

	table = private->bookmarks;

	bookmark = g_hash_table_lookup( table, name );
	if( bookmark ) {
		return bookmark->line_number;
	} else {
		return -1;
	}
}

static gboolean screem_page_clear_bookmarks( gpointer key, gpointer value, 
					     gpointer data )
{
	screem_page_remove_bookmark( SCREEM_PAGE( data ), key );

	return FALSE;
}

static gboolean bookmark_store_find( GtkTreeModel *model, GtkTreePath *path,
				     GtkTreeIter *iter, gpointer data )
{
	ScreemPageBookMark *bookmark;
	ScreemPageBookMark *bookmark2;
     	GValue value = {0,};

	bookmark = (ScreemPageBookMark*)data;
	
	gtk_tree_model_get_value( GTK_TREE_MODEL( model ),
				  iter, 2, &value );
	bookmark2 = g_value_get_pointer( &value );
	
	bookmark->iter = NULL;
	if( !strcmp( bookmark->name->str, bookmark2->name->str ) )
		bookmark->iter = iter;

	return ( bookmark->iter != NULL );
}

static void changed( GtkTextBuffer *buffer, ScreemPage *page )
{
	g_return_if_fail( SCREEM_IS_PAGE( page ) );

	screem_page_set_changed( page, TRUE );
}

static void screem_page_highlight_notify( GConfClient *client,
					  guint cnxn_id,
					  GConfEntry *entry,
					  gpointer data )
{
	ScreemPage *page;

	page = SCREEM_PAGE( data );

	if( entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state = gconf_value_get_bool( entry->value );

		page->private->highlight = state;
		gtk_source_buffer_set_highlight( GTK_SOURCE_BUFFER( page ),
						 state );
	}
}

static void screem_page_monitor_cb( GnomeVFSMonitorHandle *handle,
				    const gchar *monitor_uri,
				    const gchar *info_uri,
				    GnomeVFSMonitorEventType type,
				    gpointer data )
{
	ScreemPage *page;

	page = SCREEM_PAGE( data );

	switch( type ) {
	case GNOME_VFS_MONITOR_EVENT_CHANGED:
		page->private->modified = TRUE;
		break;
	case GNOME_VFS_MONITOR_EVENT_DELETED:
		/* FIXME: hmm, should alert the user really */
		break;
	case GNOME_VFS_MONITOR_EVENT_STARTEXECUTING:
		break;
	case GNOME_VFS_MONITOR_EVENT_STOPEXECUTING:
		break;
	case GNOME_VFS_MONITOR_EVENT_CREATED:
		page->private->modified = FALSE;
		break;
	case GNOME_VFS_MONITOR_EVENT_METADATA_CHANGED:
		break;
	default:
		break;
	}
}

gboolean screem_page_is_feature( ScreemPage *page,
				const gchar *feature )
{
	GtkSourceLanguagesManager *lm;
	GHashTable *table;
	const gchar *mime_type;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	mime_type = screem_page_get_mime_type( page );

	ret = FALSE;
	lm = screem_application_load_syntax_tables();
	if( lm ) {
		table = g_object_get_data( G_OBJECT( lm ), "features" );
		table = g_hash_table_lookup( table, feature );
		if( table && mime_type ) {
			ret = ( g_hash_table_lookup( table, mime_type ) != NULL );
		}
	}
	
	return ret;
}

gboolean screem_page_is_dynamic( ScreemPage *page )
{
	return screem_page_is_feature( page, "dynamic" );
}

gboolean screem_page_is_markup( ScreemPage *page )
{
	gboolean ret;

	ret = ( screem_page_is_feature( page, "markup" ) ||
		screem_page_is_xml( page ) );
	
	return ret;
}

gchar *screem_page_template_path( ScreemPage *page )
{
	gchar *text;
	guint len;
	gchar *name;
	gchar *ret;
	gchar *end;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );

	if( ! screem_page_is_markup( page ) ) {
		return NULL;
	}
	
	text = screem_page_get_data( page );
	name = find_text( text, 
			"<!-- #BeginTemplate[ \t]+\"[^\"]*\"[ \t]*-->",
			NULL, &len );
	ret = NULL;
	if( name ) {
		name += strlen( "<!-- #BeginTemplate \"" );
		end = strchr( name, '"' );
		ret = g_strndup( name, end - name );
	}
	g_free( text );

	return ret;
}

gboolean screem_page_is_template( ScreemPage *page, 
		const gchar *base,
		gchar **tag, gchar **path, GSList **blocks )
{
	const gchar *pathname;
	gchar *text;
	guint len;
	gchar *name;
	gboolean ret;
	gchar *tmp;
	gchar *dwcompat;

	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );
	g_return_val_if_fail( tag != NULL, FALSE );
	g_return_val_if_fail( path != NULL, FALSE );

	if( ! screem_page_is_markup( page ) ) {
		return FALSE;
	}
		
	text = screem_page_get_data( page );
	name = find_text( text, "<!-- #BeginTemplate[ \t]+\"[^\"]*\"[ \t]*-->",
			  NULL, &len );
	ret = ( name != NULL );
	
	dwcompat = NULL;
	if( ! ret ) {
		/* unable to find template header, Dreamweaver
		 * doesn't put this on the templates themselves,
		 * check page pathname to see if it is /Templates */
		GnomeVFSURI *uri;

		pathname = screem_page_get_pathname( page );
		pathname += strlen( base );
		uri = gnome_vfs_uri_new( pathname );

		if( uri ) {
			pathname = gnome_vfs_uri_get_path( uri );
		
			if( ! strncmp( "/Templates/", pathname,
					strlen( "/Templates/" ) ) ) {
				dwcompat = g_strconcat( "<!-- #BeginTemplate \"",
					pathname, "\" -->", NULL );
				len = strlen( dwcompat );
				name = dwcompat;
				ret = TRUE;
			}
		}
	}
	
	if( ret ) {
		gchar *end;
		*tag= g_strndup( name, len );
		name += strlen( "<!-- #BeginTemplate \"" );
		end = strchr( name, '"' );
	
		tmp = g_strndup( name, end - name );	
		if( ! strncmp( "/Templates/", name, strlen( "/Templates/" ) ) ) {
			*path = g_strconcat( base, tmp, NULL );
			g_free( tmp );
		} else {
			*path = tmp;
		}
		
		pathname = screem_page_get_pathname( page );
		if( ! gnome_vfs_uris_match( pathname, *path ) ) {
			/* page is not the master template, it just 
			 * uses it */
			g_free( *path );
			g_free( *tag );
			ret = FALSE;
		} else if( blocks ){
			/* build list of blocks */
			gchar *template_end;
		
			if( ! dwcompat ) {
				name = strstr( name, "-->" );
				g_assert( name );
				name += strlen( "-->" );
				template_end = strstr( name, "<!-- #EndTemplate -->" );
			} else {
				/* Dreamweaver compatability mode */
				name = strstr( text, "<HTML" );
				if( ! name ) {
					name = strstr( text, "<html" );
				}
				if( name ) {
					name = strchr( name, '>' );
					if( name ) {
						name += strlen( ">" );
					}
				}
				/* no <html> to bypass */
				if( ! name ) {
					name = text;
				}
				template_end = strstr( name, 
						"</HTML>" );
				if( ! template_end ) {
					template_end = strstr( name, 
							"</html>" );
				}
			}
		
			do {
				gchar *ed_name = NULL;

				end = strstr( name, "<!-- #BeginEditable" );
				if( end ) {
					/* get name of editable section */
					
					ed_name = find_text( end, "\"[^\"]*\"",
								NULL, &len );
					end = strstr( end, "-->" );
					if( ed_name > end ) {
						ed_name = g_strdup( "" );
					} else {
						ed_name = g_strndup( ed_name,
									len );
					}
				}
				if( ! end ) {
					if( template_end ) {
						*blocks = g_slist_prepend( *blocks, g_strndup( name, template_end - name ) );		
					} else {
						*blocks = g_slist_prepend( *blocks, g_strdup( name ) );
					}
				} else {
					end += strlen( "-->" );
					*blocks = g_slist_prepend( *blocks, g_strndup( name, end - name ) );
					*blocks = g_slist_prepend( *blocks, ed_name );
					end = strstr( end, "<!-- #EndEditable -->" );
				}
				name = end;
			} while( name );
			*blocks = g_slist_reverse( *blocks );
		}
	}
	g_free( dwcompat );
	g_free( text );
	
	return ret;
}

GtkTreeModel *screem_page_get_model( ScreemPage *page )
{
	return GTK_TREE_MODEL( page->private->model );
}

void screem_page_build_model( ScreemPage *page )
{
	screem_page_model_build_model( page->private->model );
}

void screem_page_ensure_model_built( ScreemPage *page )
{
	screem_page_model_ensure_built( page->private->model );
}

gboolean screem_page_select_context( ScreemPage *page,
				  guint pos,
				  guint *start, guint *end,
				  gboolean select_text )
{
	return screem_page_model_select_context( page->private->model,
			pos, start, end, select_text );
}

gboolean screem_page_select_parent_context( ScreemPage *page,
				  guint pos,
				  guint *start, guint *end,
				  gboolean select_text )
{
	return screem_page_model_select_parent_context( page->private->model,
			pos, start, end, select_text );
}

gboolean screem_page_select_content( ScreemPage *page,
				  guint pos,
				  guint *start, guint *end,
				  gboolean select_text )
{
	return screem_page_model_select_content( page->private->model,
			pos, start, end, select_text );
}

gchar *screem_page_query_context( ScreemPage *page,
		guint pos, gboolean query_text,
		gboolean build, 
		guint *depth, guint *start, guint *end,
		xmlNodePtr *node )
{
	return screem_page_model_query_context( page->private->model,
			pos, query_text, build, depth, start, end,
			node );
}

ScreemDTD *screem_page_get_dtd( ScreemPage *page )
{
	ScreemDTD *ret;

	ret = NULL;
	g_object_get( G_OBJECT( page->private->model ),
			"dtd", &ret, NULL );
	/* model will be holding a references, the model
	 * is kept around as long as the page, so
	 * release the reference added with g_object_get */
	if( ret ) {
		g_object_unref( ret );
	}
	
	return ret;
}

void screem_page_model_emit_building( ScreemPage *page )
{
	g_signal_emit( G_OBJECT( page ),
		       screem_page_signals[ BUILDING ], 0 );
}

void screem_page_model_emit_built( ScreemPage *page )
{
	g_signal_emit( G_OBJECT( page ),
		       screem_page_signals[ BUILT ], 0 );
}

/* FIXME: be nice if we did some caching here */
gboolean screem_page_is_xml( ScreemPage *page )
{
	ScreemPagePrivate *priv;
	gboolean ret;
	ScreemDTD *dtd;
	const gchar *mime;
	const gchar *pathname;
	const gchar *ext;
	gchar *id;
	gchar *data;
	const gchar *tmp;
	
	g_return_val_if_fail( SCREEM_IS_PAGE( page ), FALSE );
	
	priv = page->private;
	
	id = NULL;
	ret = TRUE;
	dtd = screem_page_get_dtd( page );
	mime = screem_page_get_mime_type( page );
	pathname = screem_page_get_pathname( page );
	
	ret = ! strcmp( "text/xml", mime );
	if( ! ret ) {
		ret = ( gnome_vfs_mime_type_get_equivalence( priv->mime_type, "text/xml" ) == GNOME_VFS_MIME_PARENT );
	}
	if( ! ret ) {
		tmp = strrchr( priv->mime_type, '+' );
		ret = ( tmp && ! strcmp( tmp, "+xml" ) );
	}
	if( ! ret ) {
		if( dtd ) {
			g_object_get( G_OBJECT( dtd ), 
					"public", &id, NULL );
		}
		if( id ) {
			ret = ( strstr( id, " XHTML " ) != NULL );
			ret |= ( strstr( id, " xhtml " ) != NULL );
			g_free( id );
		}
		if( pathname && ! ret ) {
			ext = strrchr( pathname, '.' );
			if( ext ) {
				ret = ! strcmp( ".xml", ext );
				ret |= ! strcmp( ".xhtml", ext );
			}
		}
		if( ! ret ) {
			data = screem_page_get_data( page );
			ret = ! strncmp( "<?xml", data, 
					strlen( "<?xml" ) );
			g_free( data );
		}
	}
	
	return ret;
}

void screem_page_set_lang( ScreemPage *page )
{
	ScreemPagePrivate *priv;
	GtkSourceLanguage *lang;
	GtkSourceLanguagesManager *lm;
	GtkTextTagTable *table;
	GtkTextTag *tag;

	g_return_if_fail( SCREEM_IS_PAGE( page ) );
	
	priv = page->private;
	
	lm = screem_application_load_syntax_tables();

	lang = gtk_source_languages_manager_get_language_from_mime_type( lm, priv->mime_type );

	if( ! lang && screem_page_is_xml( page ) ) {
		lang = gtk_source_languages_manager_get_language_from_mime_type( lm, "text/xml" );
	}

	/* iterate over languages again, this time checking for
	 * parent types, still need the above special case for
	 * text/xml though as we detect XML types in other methods
	 * than just the mime type equivalence */
	if( ! lang ) {
		GSList *langs;
		GSList *types;
		GSList *tmp;
		GnomeVFSMimeEquivalence equal;
		
		langs = gtk_source_languages_manager_get_available_languages( lm );
		while( langs && ! lang ) {
			types = gtk_source_language_get_mime_types( GTK_SOURCE_LANGUAGE( langs->data ) );
			for( tmp = types; tmp; tmp = tmp->next ) {

				equal = gnome_vfs_mime_type_get_equivalence( priv->mime_type, tmp->data );
				
				if( equal == GNOME_VFS_MIME_PARENT ) {
					break;
				}
			}
			g_slist_foreach( types, (GFunc)g_free, NULL );
			g_slist_free( types );
			if( tmp ) {
				lang = langs->data;
			}
			langs = langs->next;
		}
	}
	
	if( lang ) {
		screem_application_init_tag_styles( lang );
		gtk_source_buffer_set_language( GTK_SOURCE_BUFFER( page ), lang );
		table = gtk_text_buffer_get_tag_table( GTK_TEXT_BUFFER( page ) );
		if( ! gtk_text_tag_table_lookup( table,
					SCREEM_INVALID_MARKUP_TAG ) ) {
 
			tag = gtk_text_tag_new( SCREEM_INVALID_MARKUP_TAG );
			gtk_text_tag_table_add( table, tag );
			g_object_set( G_OBJECT( tag ),
/*				"background", "#ffffffff0000",
				"foreground", "#000000000000",*/
				"underline", PANGO_UNDERLINE_ERROR,
				NULL );
			g_object_unref( tag );
		} 
	}
}


static gboolean screem_page_auto_save( ScreemPage *page )
{
	ScreemPagePrivate *priv;
	GError *error;
	
	priv = page->private;
	
	priv->auto_save_idle = 0;
	
	error = NULL;
	if( ! screem_page_save( page, &error ) ) {
	
		if( error ) {
			g_error_free( error );	
		}
	}
	
	return FALSE;
}

static void screem_page_auto_save_notify( GConfClient *client,
					  guint cnxn_id,  GConfEntry *entry,
					  gpointer data )
{
	ScreemPage *page;
	ScreemPagePrivate *priv;

	page = SCREEM_PAGE( data );
	priv = page->private;

	if( entry->value->type == GCONF_VALUE_BOOL ) {
		gboolean state = gconf_value_get_bool( entry->value );

		priv->auto_save = state;
		
		if( priv->auto_save_idle && ! state ) {
			g_source_remove( priv->auto_save_idle );
			priv->auto_save_idle = 0;
		} else if( priv->changed && state && ! priv->auto_save_idle ) {
			screem_page_set_changed_real( page, priv->changed );
		}
	}
}

static void screem_page_auto_save_interval_notify( GConfClient *client,
					  guint cnxn_id,  GConfEntry *entry,
					  gpointer data )
{
	ScreemPage *page;
	ScreemPagePrivate *priv;

	page = SCREEM_PAGE( data );
	priv = page->private;

	if( entry->value->type == GCONF_VALUE_INT ) {
		gint interval = gconf_value_get_int( entry->value );

		if( interval != priv->auto_save_interval ) {
			priv->auto_save_interval = interval;		
		
			screem_page_set_changed_real( page, priv->changed );
		}
	}
}


/* G Object stuff */
G_DEFINE_TYPE( ScreemPage, screem_page, GTK_TYPE_SOURCE_BUFFER )

static void screem_page_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec );
static void screem_page_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec );
static void screem_page_finalize( GObject *object );


static void screem_page_class_init( ScreemPageClass *klass )
{
	GObjectClass *object_class;
	GParamSpec *pspec;
	
	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_page_finalize;
	object_class->get_property = screem_page_get_prop;
	object_class->set_property = screem_page_set_prop;


	pspec = g_param_spec_object( "app",
			"Application",
			"The Application",
			G_TYPE_OBJECT,
			G_PARAM_READWRITE | G_PARAM_CONSTRUCT );
	g_object_class_install_property( object_class, PROP_APP,
			pspec );

	pspec = g_param_spec_string( "pathname",
			"Page pathname",
			"The pathname of the page",
			"",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class, PROP_PATHNAME,
			pspec );

	pspec = g_param_spec_boolean( "changed",
			"changed",
			"If the page has changed",
			FALSE,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class, PROP_CHANGED,
			pspec );

	pspec = g_param_spec_string( "mime-type",
			"Page mime type",
			"The mime-type of the page",
			"",
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class, PROP_MIMETYPE,
			pspec );

	pspec = g_param_spec_boolean( "open",
			"open",
			"If the page is open",
			FALSE,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class, PROP_OPEN,
			pspec );

	pspec = g_param_spec_string( "charset",
			"Page character set",
			"The character set of the page",
			"",
			G_PARAM_READABLE );
	g_object_class_install_property( object_class, PROP_CHARSET,
			pspec );
	
	pspec = g_param_spec_object( "dtd",
			"DTD",
			"The ScreemDTD for the page",
			G_TYPE_OBJECT,
			G_PARAM_READWRITE );
	g_object_class_install_property( object_class, PROP_DTD,
			pspec );
	
	screem_page_signals[ SAVING ] =
		g_signal_new( "saving",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemPageClass, 
					       saving),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	screem_page_signals[ SAVED ] =
		g_signal_new( "saved",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemPageClass, 
					       saved),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	screem_page_signals[ BUILDING ] =
		g_signal_new( "building",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemPageClass, 
					       building ),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
	screem_page_signals[ BUILT ] =
		g_signal_new( "built",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemPageClass, 
					       built),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0 );
}

static void screem_page_init( ScreemPage *page )
{
	ScreemPagePrivate *priv;
	
	priv = page->private = g_new0( ScreemPagePrivate, 1 );

	priv->client = gconf_client_get_default();
	priv->highlight = gconf_client_get_bool( priv->client,
		"/apps/screem/editor/highlight", NULL );
	priv->highlight_notify = 
		gconf_client_notify_add( priv->client,
					 "/apps/screem/editor/highlight",
					 screem_page_highlight_notify,
					 page, NULL, NULL );
	
	priv->bookmarks = g_hash_table_new( g_str_hash, g_str_equal );

	/* NAME, LINE, BOOKMARK STRUCT */
	priv->bookmarkstore = gtk_list_store_new( 3, G_TYPE_STRING,
		G_TYPE_INT, G_TYPE_POINTER, NULL );	
	priv->mime_type = gconf_client_get_string( priv->client, 
		"/apps/screem/default_mimetype", NULL );
	if( ! priv->mime_type ) {
		priv->mime_type = g_strdup( "text/html" );
	}
	
	g_signal_connect( G_OBJECT( page ),
			  "changed", G_CALLBACK( changed ), page );

	gtk_text_buffer_add_selection_clipboard( GTK_TEXT_BUFFER( page ),
			gtk_clipboard_get( GDK_SELECTION_PRIMARY ) );

	priv->auto_save = gconf_client_get_bool( priv->client,
			"/apps/screem/general/auto_save", NULL );
	priv->auto_save_interval = gconf_client_get_int( priv->client,
			"/apps/screem/general/auto_save_interval", NULL );

	priv->auto_save_notify = gconf_client_notify_add( priv->client,
			"/apps/screem/general/auto_save",
			screem_page_auto_save_notify, page, NULL, NULL );
	priv->auto_save_interval_notify = gconf_client_notify_add( priv->client,
			"/apps/screem/general/auto_save_interval",
			screem_page_auto_save_interval_notify, page, NULL, NULL );

	g_get_current_time( &priv->time );
}

static void screem_page_set_prop( GObject *object, guint prop_id,
				  const GValue *value, GParamSpec *spec )
{
	ScreemPage *page;
	GObject *obj;

	page = SCREEM_PAGE( object );

	switch( prop_id ) {
		case PROP_APP:
			obj = g_value_get_object( value );
			page->private->model = screem_page_model_new( page, obj );
			gtk_source_buffer_set_highlight( GTK_SOURCE_BUFFER( page ), page->private->highlight );
			break;
		case PROP_PATHNAME:
			screem_page_set_pathname_real( page, g_value_get_string( value ) );
			break;
		case PROP_CHANGED:
			screem_page_set_changed_real( page, g_value_get_boolean( value ) );
			break;
		case PROP_MIMETYPE:
			screem_page_set_mime_type_real( page, g_value_get_string( value ) );
			break;
		case PROP_OPEN:
			screem_page_set_open_real( page, g_value_get_boolean( value ) );
			break;
		case PROP_DTD:
			break;
	}
}

static void screem_page_get_prop( GObject *object, guint prop_id,
				  GValue *value, GParamSpec *spec )
{
	ScreemPage *page;
	GObject *obj;
	gchar *str;
	const gchar *cstr;
	
	page = SCREEM_PAGE( object );

	switch( prop_id ) {
		case PROP_APP:
			g_object_get( G_OBJECT( page->private->model ),
					"application", &obj, NULL );
			g_value_set_object( value, obj );
			g_object_unref( obj );
			break;
		case PROP_PATHNAME:
			cstr = screem_page_get_pathname( page );
			if( cstr ) {
				g_value_set_string( value, cstr );
			}
			break;
		case PROP_CHANGED:
			g_value_set_boolean( value, screem_page_get_changed( page ) );
			break;
		case PROP_MIMETYPE:
			cstr = screem_page_get_mime_type( page );
			if( cstr ) {
				g_value_set_string( value, cstr );
			}
			break;
		case PROP_OPEN:
			g_value_set_boolean( value, screem_page_is_open( page ) );
			break;
		case PROP_CHARSET:
			g_object_get( G_OBJECT( page->private->model ),
					"charset", &str, NULL );
			if( str ) {
				g_value_set_string( value, str );
				g_free( str );
			}
			break;
		case PROP_DTD:
			g_object_get( G_OBJECT( page->private->model ),
					"dtd", &obj, NULL );
			g_value_set_object( value, obj );
			g_object_unref( obj );
			break;
		default:
			break;
	}
}


static void screem_page_finalize( GObject *object )
{
	ScreemPage *page;
	ScreemPagePrivate *priv;

	page = SCREEM_PAGE( object );
	priv = page->private;

	gconf_client_notify_remove( priv->client, 
				    priv->highlight_notify );
	gconf_client_notify_remove( priv->client,
					priv->auto_save_notify );
	gconf_client_notify_remove( priv->client,
					priv->auto_save_interval_notify );

	g_hash_table_foreach( page->private->bookmarks,
			      (GHFunc)screem_page_clear_bookmarks, page );
	g_hash_table_destroy( page->private->bookmarks );
	g_object_unref( page->private->bookmarkstore );

	g_object_unref( page->private->client );

	g_object_unref( page->private->model );

	g_free( page->private->pathname );
	g_free( page->private->mime_type );

	if( page->private->monitor ) {
		gnome_vfs_monitor_cancel( page->private->monitor );
	}

	g_free( page->private );

	G_OBJECT_CLASS( screem_page_parent_class )->finalize( object );
}

