#ifndef TAGCOLL_FILTER_H
#define TAGCOLL_FILTER_H

/** \file
 * Filter interface for a stream of tagged items
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/Consumer.h>

namespace Tagcoll
{

/**
 * Interface for injecting or removing packages from a stream of tagged items
 *
 * The filter will not perform any memory management on its consumer: the
 * caller must make sure that the lifetime of the Consumer is longer than
 * the one of the Filter.
 */
template< typename ITEM, typename TAG >
class Filter : public Consumer<ITEM, TAG>
{
protected:
	Consumer<ITEM, TAG>* consumer;

	virtual void consumeItemUntagged(const ITEM& item) { consumer->consume(item); }
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags) { consumer->consume(item, tags); }
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items) { consumer->consume(items); }
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<TAG>& tags) { consumer->consume(items, tags); }

public:
	/**
	 * Build an incomplete filter.
	 *
	 * This is useful as an intermediate step when constructing filters, but an
	 * incomplete filter should never be used: it will not even throw an
	 * expection: it will segfault.
	 *
	 * In doubt, use the other version of the constructor.
	 */
	Filter() : consumer(0) {}

	/**
	 * Build a filter passing the filtered stream to the given consumer
	 */
	Filter(Consumer<ITEM, TAG>& consumer) : consumer(&consumer) {}
	virtual ~Filter() {}

	/// Return true if the filter is not incomplete
	bool isComplete() { return consumer != 0; }

	/// Access the filter's consumer 
	virtual Consumer<ITEM, TAG>& getConsumer() const { return *consumer; }
	
	/// Set or change the filter's consumer
	virtual void setConsumer(Consumer<ITEM, TAG>& consumer) { this->consumer = &consumer; }
};

/**
 * Chains more filters together
 */
template< typename ITEM, typename TAG >
class FilterChain : public Filter<ITEM, TAG>
{
protected:
	Filter<ITEM, TAG>* last;

public:
	FilterChain<ITEM, TAG>() throw () : Filter<ITEM, TAG>(), last(0) {}
	FilterChain<ITEM, TAG>(Consumer<ITEM, TAG>& consumer) throw ()
		: Filter<ITEM, TAG>(consumer), last(0) {}
	virtual ~FilterChain<ITEM, TAG>() {}

	/// Set the consumer at the end of the chain
	void setConsumer(Consumer<ITEM, TAG>& c) throw ()
	{
		if (last)
			last->setConsumer(c);
		else
			this->consumer = &c;
	}

	/// Add a filter at the end of the chain
	void appendFilter(Filter<ITEM, TAG>& f) throw ()
	{
		if (last)
		{
			if (last->isComplete())
				f.setConsumer(last->getConsumer());
			last->setConsumer(f);
			last = &f;
		} else {
			if (this->isComplete())
				f.setConsumer(this->getConsumer());
			this->consumer = last = &f;
		}
	}
};

}

// vim:set ts=4 sw=4:
#endif
