#ifndef TAGCOLL_INPUTMERGER_H
#define TAGCOLL_INPUTMERGER_H

/** \file
 * Simple tagged collection.
 *
 * Also used for merging tags of items appearing multiple times in a stream of
 * tagged items
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/Collection.h>

#include <map>

namespace Tagcoll
{
template<typename, typename> class PatchList;

/**
 * Simple Collection.
 *
 * It can be used to merge input values: if an item is added multiple times,
 * its various tagsets are merged in a single one.
 *
 * It is also a full-featured collection, although not very optimized.
 */
template<class ITEM, class TAG>
class InputMerger : public Collection<ITEM, TAG>
{
protected:
	std::map< ITEM, OpSet<TAG> > coll;
	
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags);

	virtual OpSet<TAG> getTagsOfItem(const ITEM& item) const;
	virtual OpSet<ITEM> getItemsHavingTag(const TAG& tag) const;
	virtual OpSet<ITEM> getItemsHavingTags(const OpSet<TAG>& tags) const;


public:
	virtual ~InputMerger<ITEM, TAG>() {}

	bool hasItem(const ITEM& item) const { return coll.find(item) != coll.end(); }

	void output(Consumer<ITEM, TAG>& consumer) const;
	void outputHavingTags(const OpSet<TAG>& ts, Consumer<ITEM, TAG>& consumer) const;

	/**
	 * Send the merged data to a consumer, but reversed: the tag become items,
	 * and they are tagged with the items that had them
	 */
	void outputReversed(Consumer<TAG, ITEM>& consumer) const;

	void applyChange(const PatchList<ITEM, TAG>& change);

	virtual OpSet<ITEM> getTaggedItems() const
	{
		OpSet<ITEM> res;
		for (typename std::map< ITEM, OpSet<TAG> >::const_iterator i = coll.begin();
				i != coll.end(); i++)
			res += i->first;
		return res;
	}
	
	OpSet<TAG> getAllTags() const;

	OpSet<TAG> getCompanionTags(const OpSet<TAG>& ts) const;

	OpSet<ITEM> getRelatedItems(const OpSet<TAG>& ts, int maxdistance = 1) const;

	/**
	 * Count the number of items
	 */
	unsigned int itemCount() const;

	/**
	 * Empty the collection
	 */
	void clear();
};

};

// vim:set ts=4 sw=4:
#endif
