/*
 * std::set with operators overridden with set operations
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/OpSet.h>

#include <algorithm>

#include <string>

using namespace std;
using namespace Tagcoll;

//template class OpSet<string>;
//template class OpSet<int>;
//
//namespace Debtags { class Package; }
//template class OpSet<Debtags::Package*>;

template<class T>
bool OpSet<T>::contains(const OpSet<T>& ts) const
{
	typename OpSet<T>::const_iterator b = ts.begin();

	for (typename OpSet<T>::const_iterator a = begin(); a != end(); a++)
		if (b == ts.end())
			return true;
		else if (*a == *b)
			b++;
		else if (*b < *a)
			return false;

	return b == ts.end();
}

template<class T>
int OpSet<T>::distance(const OpSet<T>& ts) const
{
	int res = 0;
	int intCount = 0;

	typename OpSet<T>::const_iterator a = begin();
	typename OpSet<T>::const_iterator b = ts.begin();

	while (a != end() || b != ts.end())
		if ((b == ts.end()) || (a != end() && *a < *b))
		{
			res++;
			a++;
		}
		else if ((a == end()) || (b != ts.end() && *b < *a))
		{
			res++;
			b++;
		}
		else
		{
			a++;
			b++;
			intCount++;
		}
	
	return intCount ? res : -1;
}

template<class T>
OpSet<T> OpSet<T>::operator+(const T& item) const
{
	OpSet<T> res = *this;
	res.insert(item);
	return res;
}
	
template<class T>
OpSet<T> OpSet<T>::operator+(const OpSet<T>& ts) const
{
	OpSet<T> res;
	set_union(begin(), end(), ts.begin(), ts.end(), inserter(res, res.begin()));
	return res;
}

template<class T>
OpSet<T> OpSet<T>::operator-(const T& item) const
{
	OpSet<T> res = *this;
	res.erase(item);
	return res;
}
	
template<class T>
OpSet<T> OpSet<T>::operator-(const OpSet<T>& ts) const
{
	OpSet<T> res;
	set_difference(begin(), end(), ts.begin(), ts.end(), inserter(res, res.begin()));
	return res;
}

template<class T>
OpSet<T> OpSet<T>::operator^(const OpSet<T>& ts) const
{
	OpSet<T> res;
	set_intersection(begin(), end(), ts.begin(), ts.end(), inserter(res, res.begin()));
	return res;
}

template<class T>
OpSet<T>& OpSet<T>::operator+=(const T& item)
{
	insert(item);
	return *this;
}

template<class T>
OpSet<T>& OpSet<T>::operator+=(const OpSet<T>& ts)
{
	for (typename OpSet<T>::const_iterator t = ts.begin();
			t != ts.end(); t++)
		insert(*t);
	return *this;
}

template<class T>
OpSet<T>& OpSet<T>::operator-=(const T& item)
{
	erase(item);
	return *this;
}

template<class T>
OpSet<T>& OpSet<T>::operator-=(const OpSet<T>& ts)
{
	for (typename OpSet<T>::const_iterator t = ts.begin();
			t != ts.end(); t++)
		erase(*t);
	return *this;
}

template<class T>
OpSet<T>& OpSet<T>::operator^=(const OpSet<T>& ts)
{
	(*this) = (*this) ^ ts;
	return *this;
}

#ifndef INSTANTIATING_TEMPLATES
template class OpSet<string>;
#endif

#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {
using namespace tut_tagcoll;

struct tagcoll_opset_shar {
};
TESTGRP(tagcoll_opset);

template<> template<>
void to::test<1>()
{
	OpSet<int> test, test1, test2;
	gen_ensure(test.empty());
	gen_ensure(test.size() == 0);
	gen_ensure(test == test1);

	test += 1;
	gen_ensure(test.size() == 1);
	gen_ensure(test != test1);

	test1 += 1;
	gen_ensure(test == test1);

	test += 2;
	gen_ensure(test.size() == 2);

	test += 2;
	gen_ensure(test.size() == 2);

	test1 += test;
	gen_ensure(test.size() == 2);
	gen_ensure(test1.size() == 2);
	gen_ensure(test == test1);

	test1 += 3;
	test2 += 3;
	gen_ensure((test + test1) == test1);
	gen_ensure((test - test1) == OpSet<int>());
	gen_ensure((test1 - test) == test2);
	gen_ensure((test ^ test1) == test);
}

}

#endif
// vim:set ts=4 sw=4:
