#ifndef TAGCOLL_PATCHCOLLECTION_H
#define TAGCOLL_PATCHCOLLECTION_H

/** \file
 * Wrap a Collection, preserving modifications as patches
 */

/*
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Collection.h>
#include <tagcoll/Patches.h>

namespace Tagcoll
{

/**
 * Wraps a collection by intercepting all changes to it and preserving them as
 * a PatchList.
 */
template<class ITEM, class TAG>
class PatchCollection : public Collection<ITEM, TAG>
{
protected:
	const Collection<ITEM, TAG>& coll;
	PatchList<ITEM, TAG> changes;

	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags);

	virtual OpSet<ITEM> getItemsHavingTag(const TAG& tag) const;
	virtual OpSet<TAG> getTagsOfItem(const ITEM& item) const;


public:
	PatchCollection(const Collection<ITEM, TAG>& coll) : coll(coll) {}
	virtual ~PatchCollection() {}

	/**
	 * Get the changes that have been applied to this collection
	 */
	const PatchList<ITEM, TAG>& getChanges() const { return changes; }

	/**
	 * Set the changes list to a specific patch list
	 */
	void setChanges(const PatchList<ITEM, TAG>& changes);

    virtual bool hasTag(const TAG& tag) const;

	virtual OpSet<ITEM> getTaggedItems() const;
	virtual OpSet<TAG> getAllTags() const;

	virtual int getCardinality(const TAG& tag) const;

	virtual void applyChange(const PatchList<ITEM, TAG>& change);

	virtual void output(Consumer<ITEM, TAG>& consumer) const;
};

};

// vim:set ts=4 sw=4:
#endif
