#ifndef TAGCOLL_TDB_DISK_INDEX_H
#define TAGCOLL_TDB_DISK_INDEX_H

/** \file
 * Fast on-disk index for tag data
 */

/*
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Collection.h>
#include <tagcoll/Serializer.h>
#include <tagcoll/Exception.h>
#include <tagcoll/TDBFile.h>
#include <map>

#include <string>

namespace Tagcoll
{

/**
 * Full TaggedCollection implementation on top of a persistent on-disk TDB
 * database.
 *
 * It allows to efficiently query a collection without having to store it all
 * into memory.
 *
 * If used for heavy modifications, the performance is slower compared to other
 * in-memory collections.  If database writes are mainly used for populating
 * the index, then TDBIndexer should be used to create the index and
 * TDBDiskIndex to access it afterwards.
 */
template<class ITEM, class TAG>
class TDBDiskIndex : public Collection<ITEM, TAG>
{
protected:
	TDBFile pkgdb;
	TDBFile tagdb;
	mutable Converter<ITEM, std::string>& fromitem;
	mutable Converter<TAG, std::string>& fromtag;
	mutable Converter<std::string, ITEM>& toitem;
	mutable Converter<std::string, TAG>& totag;

	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags);
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<TAG>& tags);

	virtual OpSet<ITEM> getItemsHavingTag(const TAG& tag) const;
	virtual OpSet<TAG> getTagsOfItem(const ITEM& item) const;

public:
	/**
	 * Create a new TDBDiskIndex
	 *
	 * @param pkgidx
	 *   The file name of the package index
	 * @param tagidx
	 *   The file name of the tag index
	 * @param fromitem, fromtag, toitem, totag
	 *   The Converter-s used to convert ITEMS and TAGS to and from strings.
	 *   If 0 is passed, this TDBDiskIndex will only be able to work with
	 *   string items and string tags.
	 * @param write
	 *   Set to false if the index should be opened in read-only mode.  If
	 *   opened in read-only mode, all non-const methods of this class will
	 *   throw an exception if invoked.
	 *   It defaults to true.
	 */
	TDBDiskIndex(
			const std::string& pkgidx,
			const std::string& tagidx,
			Converter<ITEM, std::string>& fromitem,
			Converter<TAG, std::string>& fromtag,
			Converter<std::string, ITEM>& toitem,
			Converter<std::string, TAG>& totag,
			bool write = true);
	virtual ~TDBDiskIndex() {}

    virtual bool hasTag(const TAG& tag) const;

	virtual OpSet<ITEM> getTaggedItems() const;
	virtual OpSet<TAG> getAllTags() const;

	virtual int getCardinality(const TAG& tag) const;

	virtual void output(Consumer<ITEM, TAG>& consumer) const;

	virtual void applyChange(const PatchList<ITEM, TAG>& change);
};

};

// vim:set ts=4 sw=4:
#endif
