#ifndef TDB_INDEXER_H
#define TDB_INDEXER_H

/** \file
 * Fast index for tag data
 */

/*
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Collection.h>
#include <tagcoll/Serializer.h>
#include <tagcoll/Exception.h>
#include <map>

namespace Tagcoll
{

/**
 * In-memory collection whose representation is organised similarly to the
 * on-disk representation used by TDBDiskIndex.
 *
 * It can be used as a working collection, and it can also write its contents
 * into an on-disk index that can later be used by TDBDiskIndex.
 *
 * On-disk indexes produced by TDBIndexer are written in a single, optimised
 * run and tend to be more compact than the ones created by TDBDiskIndex.
 */
template<class ITEM, class TAG>
class TDBIndexer : public Collection<ITEM, TAG>
{
protected:
	std::map<ITEM, OpSet<TAG> > items;
	std::map<TAG, OpSet<ITEM> > tags;

	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags);
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<TAG>& tags);

	virtual OpSet<ITEM> getItemsHavingTag(const TAG& tag) const;
	virtual OpSet<TAG> getTagsOfItem(const ITEM& item) const;

public:
	virtual ~TDBIndexer() {}

    virtual bool hasItem(const ITEM& item) const { return items.find(item) != items.end(); }
    virtual bool hasTag(const TAG& tag) const { return tags.find(tag) != tags.end(); }
	virtual OpSet<ITEM> getTaggedItems() const;
	virtual OpSet<TAG> getAllTags() const;
	virtual void output(Consumer<ITEM, TAG>& consumer) const;
	virtual void applyChange(const PatchList<ITEM, TAG>& change);

	/**
	 * Write all collected informations to a disk index
	 *
	 * \param itemconv
	 *   Converter than can convert an ITEM to a string
	 * \param tagconv
	 *   Converter than can convert a TAG to a string
	 * \param pkgidx
	 *   File name for the package index to write
	 * \param tagidx
	 *   File name for the tag index to write
	 */
	void writeIndex(
			Converter<ITEM, std::string>& itemconv,
			Converter<TAG, std::string>& tagconv,
			const std::string& pkgidx,
			const std::string& tagidx) const;
};

};

// vim:set ts=4 sw=4:
#endif
