/*
 * Merge tags of items appearing multiple times in a stream of tagged items
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/InputMerger.h>
#include <tagcoll/Patches.h>

using namespace std;

namespace Tagcoll {

template<class T, class Tag>
void InputMerger<T, Tag>::consumeItem(const T& item, const OpSet<Tag>& tags)
{
	typename map< T, OpSet<Tag> >::iterator i = coll.find(item);
	if (i == coll.end())
		coll.insert(make_pair(item, tags));
	else
		i->second += tags;
}

template<class T, class Tag>
OpSet<Tag> InputMerger<T, Tag>::getTagsOfItem(const T& item) const
{
	typename map< T, OpSet<Tag> >::const_iterator i = coll.find(item);
	
	if (i == coll.end())
		return OpSet<Tag>();
	else
		return i->second;
}

template<class ITEM, class TAG>
OpSet<ITEM> InputMerger<ITEM, TAG>::getItemsHavingTag(const TAG& tag) const
{
	OpSet<ITEM> res;
	for (typename map< ITEM, OpSet<TAG> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (i->second.contains(tag))
			res += i->first;
	return res;
}

template<class ITEM, class TAG>
OpSet<ITEM> InputMerger<ITEM, TAG>::getItemsHavingTags(const OpSet<TAG>& tags) const
{
	OpSet<ITEM> res;
	for (typename map< ITEM, OpSet<TAG> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (i->second.contains(tags))
			res += i->first;
	return res;
}

template<class T, class Tag>
void InputMerger<T, Tag>::output(Consumer<T, Tag>& consumer) const
{
	for (typename map< T, OpSet<Tag> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (!i->second.empty())
			consumer.consume(i->first, i->second);
}

template<class T, class Tag>
void InputMerger<T, Tag>::outputReversed(Consumer<Tag, T>& consumer) const
{
	for (typename map< T, OpSet<Tag> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
	{
		OpSet<T> items;
		items += i->first;
		consumer.consume(i->second, items);
	}
}

template<class ITEM, class TAG>
void InputMerger<ITEM, TAG>::outputHavingTags(const OpSet<TAG>& ts, Consumer<ITEM, TAG>& consumer) const
{
	for (typename map< ITEM, OpSet<TAG> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (i->second.contains(ts))
			consumer.consume(i->first, i->second);
}

	


template<class T, class Tag>
void InputMerger<T, Tag>::applyChange(const PatchList<T, Tag>& change)
{
	for (typename PatchList<T, Tag>::const_iterator i = change.begin(); i != change.end(); i++)
	{
		typename map< T, OpSet<Tag> >::iterator it = coll.find(i->first);
		if (it == coll.end())
		{
			// If the item doesn't exist, create it
			coll.insert(make_pair(i->first, i->second.getAdded()));
		} else {
			it->second = i->second.apply(it->second);
		}
	}
}

template<class T, class Tag>
OpSet<Tag> InputMerger<T, Tag>::getAllTags() const
{
	OpSet<Tag> tags;

	for (typename map< T, OpSet<Tag> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		tags += i->second;
	
	return tags;
}

template<class T, class Tag>
OpSet<Tag> InputMerger<T, Tag>::getCompanionTags(const OpSet<Tag>& ts) const
{
	OpSet<Tag> tags;

	for (typename map< T, OpSet<Tag> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
		if (i->second.contains(ts))
			tags += i->second - ts;
	
	return tags;
}

template<class T, class Tag>
OpSet<T> InputMerger<T, Tag>::getRelatedItems(const OpSet<Tag>& tags, int maxdistance) const
{
	OpSet<T> res;

	for (typename map< T, OpSet<Tag> >::const_iterator i = coll.begin();
			i != coll.end(); i++)
	{
		int dist = tags.distance(i->second);
		if (dist >= 0 && dist <= maxdistance)
			res += i->first;
	}
	
	return res;
}

template<class T, class Tag>
unsigned int InputMerger<T, Tag>::itemCount() const
{
	return coll.size();
}

template<class T, class Tag>
void InputMerger<T, Tag>::clear()
{
	return coll.clear();
}

}


#ifndef INSTANTIATING_TEMPLATES
#include <string>

namespace Tagcoll {
template class InputMerger<std::string, std::string>;
}
#endif


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {
using namespace tut_tagcoll;

struct tagcoll_inputmerger_shar {
};
TESTGRP(tagcoll_inputmerger);

template<> template<>
void to::test<1>()
{
	InputMerger<string, string> coll;

	output_test_collection(coll);
	test_readonly_collection(coll);
}

template<> template<>
void to::test<2>()
{
	InputMerger<string, string> coll;

	test_collection(coll);
}

}

#endif

// vim:set ts=4 sw=4:
