#ifndef TAGCOLL_INT_DISK_INDEX_H
#define TAGCOLL_INT_DISK_INDEX_H

/** \file
 * Fast on-disk index for tag data
 */

/*
 * Copyright (C) 2006  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Collection.h>
#include <tagcoll/Serializer.h>
#include <tagcoll/Exception.h>
#include <tagcoll/IntIndex.h>

namespace Tagcoll
{

/**
 * Full TaggedCollection implementation on top of a persistent on-disk TDB
 * database.
 *
 * It allows to efficiently query a collection without having to store it all
 * into memory.
 *
 * If used for heavy modifications, the performance is slower compared to other
 * in-memory collections.  If database writes are mainly used for populating
 * the index, then TDBIndexer should be used to create the index and
 * TDBDiskIndex to access it afterwards.
 */
template<class ITEM, class TAG>
class IntDiskIndex : public ReadonlyCollection<ITEM, TAG>
{
protected:
	IntIndex pkgidx;
	IntIndex tagidx;
	const Converter<ITEM, int>* m_fromitem;
	const Converter<TAG, int>* m_fromtag;
	const Converter<int, ITEM>* m_toitem;
	const Converter<int, TAG>* m_totag;

	inline int fromitem(const ITEM& tag) const { return (*m_fromitem)(tag); }
	inline int fromtag(const TAG& tag) const { return (*m_fromtag)(tag); }
	inline ITEM toitem(const int& tag) const { return (*m_toitem)(tag); }
	inline TAG totag(const int& tag) const { return (*m_totag)(tag); }

	virtual OpSet<ITEM> getItemsHavingTag(const TAG& tag) const;
	virtual OpSet<ITEM> getItemsHavingTags(const OpSet<TAG>& tags) const;
	virtual OpSet<TAG> getTagsOfItem(const ITEM& item) const;
	virtual OpSet<TAG> getTagsOfItems(const OpSet<ITEM>& items) const;

public:
	/**
	 * Create a new IntDiskIndex
	 *
	 * @param filename
	 *   The file name of the package index
	 * @param tagidx
	 *   The file name of the tag index
	 * @param fromitem, fromtag, toitem, totag
	 *   The Converter-s used to convert ITEMS and TAGS to and from strings.
	 *   If 0 is passed, this TDBDiskIndex will only be able to work with
	 *   string items and string tags.
	 * @param write
	 *   Set to false if the index should be opened in read-only mode.  If
	 *   opened in read-only mode, all non-const methods of this class will
	 *   throw an exception if invoked.
	 *   It defaults to true.
	 */
	IntDiskIndex()
		:   m_fromitem(0), m_fromtag(0), m_toitem(0), m_totag(0) {}
	IntDiskIndex(
			const MasterMMapIndex& master,
			int pkgindex, int tagindex,
			const Converter<ITEM, int>* fromitem,
			const Converter<TAG, int>* fromtag,
			const Converter<int, ITEM>* toitem,
			const Converter<int, TAG>* totag)
		:   pkgidx(master, pkgindex), tagidx(master, tagindex),
			m_fromitem(fromitem), m_fromtag(fromtag),
			m_toitem(toitem), m_totag(totag) {}
	virtual ~IntDiskIndex() {}

	void init(const MasterMMapIndex& master, int pkgindex, int tagindex,
			const Converter<ITEM, int>* fromitem,
			const Converter<TAG, int>* fromtag,
			const Converter<int, ITEM>* toitem,
			const Converter<int, TAG>* totag)
	{
		m_fromitem = fromitem;
		m_fromtag = fromtag;
		m_toitem = toitem;
		m_totag = totag;
		pkgidx.init(master, pkgindex);
		tagidx.init(master, tagindex);
	}

	void init(const MasterMMapIndex& master, int pkgindex, int tagindex)
	{
		pkgidx.init(master, pkgindex);
		tagidx.init(master, tagindex);
	}

    virtual bool hasTag(const TAG& tag) const
	{
		return tagidx.size(fromtag(tag)) > 0;
	}

	virtual OpSet<ITEM> getTaggedItems() const;

	virtual OpSet<TAG> getAllTags() const;

	virtual int getCardinality(const TAG& tag) const
	{
		return tagidx.size(fromtag(tag));
	}

	virtual OpSet<TAG> getCompanionTags(const OpSet<TAG>& tags) const;

	virtual void output(Consumer<ITEM, TAG>& consumer) const;
};

template<class ITEM, class TAG>
class IntDiskIndexer : public Consumer<ITEM, TAG>
{
protected:
	IntIndexer pkgidx;
	IntIndexer tagidx;
	const Converter<ITEM, int>& fromitem;
	const Converter<TAG, int>& fromtag;

	virtual void consumeItemUntagged(const ITEM&) {}
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags);

public:
	IntDiskIndexer(
			const Converter<ITEM, int>& fromitem,
			const Converter<TAG, int>& fromtag);
	virtual ~IntDiskIndexer() {}

	const MMapIndexer& pkgIndexer() const { return pkgidx; }
	const MMapIndexer& tagIndexer() const { return tagidx; }
};


};

// vim:set ts=4 sw=4:
#endif
