#ifndef PARSER_BASE_H
#define PARSER_BASE_H

/** \file
 * Base class for parsers
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <string>
#include <tagcoll/Exception.h>

namespace Tagcoll {

class ParserInput;

/**
 * Base exception for parser errors
 */
class ParserException: public ContextException
{
protected:
	std::string _file;
	int _line;
	
public:
	ParserException(const ParserInput& input, const std::string& message) throw ();
	ParserException(const std::string& file, int line, const std::string& message) throw ()
		: ContextException(message), _file(file), _line(line) {}
	ParserException(int line, const std::string& message) throw ()
		: ContextException(message), _line(line) {}
	ParserException(const std::string& message) throw ()
		: ContextException(message), _line(-1) {}
	virtual ~ParserException() throw () {}

	int line() const throw () { return _line; }
	int line(int line) throw () { return _line = line; }

	const std::string& file() const throw () { return _file; }
	std::string file() throw () { return _file; }
	std::string file(const std::string file) throw () { return _file = file; }

	virtual const char* type() const throw ()
	{
		return "ParserException";
	}

	virtual std::string desc() const throw ();
};

/**
 * Exception thrown in case of problems accessing the input of the parser
 */
class ParserInputException : public ParserException
{
public:
	ParserInputException(const ParserInput& input, const std::string& message) throw ()
		: ParserException(input, message) {}
	virtual ~ParserInputException() throw () {}

	virtual const char* type() const throw ()
	{
		return "ParserInputException";
	}

};

/**
 * Generic interface for parser input readers.
 *
 * It encapsulates and hides the reading machinery.  It can be implemented as a
 * file read, a stream read, a decompressing file read, a network read or
 * whatever else is needed.
 */
class ParserInput
{
public:
	static const int Eof = -1;

	ParserInput() {}
	virtual ~ParserInput() {}

	virtual const std::string& fileName() const = 0;
	virtual int lineNumber() const = 0;
	virtual int nextChar() throw (ParserInputException) = 0;
	virtual void pushChar(int c) throw (ParserInputException) = 0;
};

}

// vim:set ts=4 sw=4:
#endif
