/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

function PrefManager (branchName)
{
    var prefManager = this;
    
    function pm_observe (prefService, topic, prefName)
    {
        prefManager.dirtyPrefs[prefName] = true;
    };

    const PREF_CTRID = "@mozilla.org/preferences-service;1";
    const nsIPrefService = Components.interfaces.nsIPrefService;
    const nsIPrefBranch = Components.interfaces.nsIPrefBranch;
    const nsIPrefBranchInternal = Components.interfaces.nsIPrefBranchInternal;

    this.prefService =
        Components.classes[PREF_CTRID].getService(nsIPrefService);
    this.prefBranch = this.prefService.getBranch(branchName);
    this.prefNames = new Array();
    this.dirtyPrefs = new Object();
    this.prefs = new Object();
    this.observer = { observe: pm_observe };

    this.prefBranchInternal =
        this.prefBranch.QueryInterface(nsIPrefBranchInternal);
    this.prefBranchInternal.addObserver("", this.observer, false);
}


PrefManager.prototype.destroy =
function pm_destroy()
{
    this.prefBranchInternal.removeObserver("", this.observer);
}

PrefManager.prototype.listPrefs =
function pm_listprefs (prefix)
{
    var list = new Array();
    var names = this.prefNames;
    for (var i = 0; i < names.length; ++i)
    {
        if (!prefix || names[i].indexOf(prefix) == 0)
            list.push (names[i]);
    }

    return list;
}

PrefManager.prototype.readPrefs =
function pm_readprefs ()
{
    const nsIPrefBranch = Components.interfaces.nsIPrefBranch;

    var list = this.prefBranch.getChildList("", {});
    for (var i = 0; i < list.length; ++i)
    {
        if (!(list[i] in this))
        {
            var type = this.prefBranch.getPrefType (list[i]);
            var defaultValue;
            
            switch (type)
            {
                case nsIPrefBranch.PREF_INT:
                    defaultValue = 0;
                    break;
                
                case nsIPrefBranch.PREF_BOOL:
                    defaultValue = false;
                    break;

                default:
                    defaultValue = "";
            }
            
            this.addPref(list[i], defaultValue);
        }
    }
}

PrefManager.prototype.addPrefs =
function pm_addprefs (prefSpecs)
{
    for (var i = 0; i < prefSpecs.length; ++i)
        this.addPref(prefSpecs[i][0], prefSpecs[i][1]);
}
        
PrefManager.prototype.addPref =
function pm_addpref (prefName, defaultValue)
{
    var realValue;
    
    var prefManager = this;
    
    function prefGetter ()
    {
        //dd ("getter for ``" + prefName + "''");
        //dd ("default: " + defaultValue);
        //dd ("real: " + realValue);

        if (typeof realValue == "undefined" ||
            prefName in prefManager.dirtyPrefs)
        {
            try
            {
                if (typeof defaultValue == "boolean")
                {
                    realValue = prefManager.prefBranch.getBoolPref(prefName);
                }
                else if (typeof defaultValue == "number")
                {
                    realValue = prefManager.prefBranch.getIntPref(prefName);
                }
                else if (defaultValue instanceof Array)
                {
                    realValue = prefManager.prefBranch.getCharPref(prefName);
                    realValue = realValue.split(/s*;\s*/);
                    for (i = 0; i < realValue.length; ++i)
                        realValue[i] = unencode(realValue[i]);
                }
                else if (typeof defaultValue == "string")
                {
                    realValue = prefManager.prefBranch.getCharPref(prefName);
                }
                else
                {
                    realValue = defaultValue;
                }
            }
            catch (ex)
            {
                //dd ("caught exception reading pref ``" + prefName + "''\n" +
                //ex);
                realValue = defaultValue;
            }
        }

        delete prefManager.dirtyPrefs[prefName];
        return realValue;
    }
    
    function prefSetter (value)
    {
        try
        {
            if (typeof defaultValue == "boolean")
            {
                prefManager.prefBranch.setBoolPref(prefName, value);
            }
            else if (typeof defaultValue == "number")
            {
                prefManager.prefBranch.setIntPref(prefName, value);
            }
            else if (defaultValue instanceof Array)
            {
                var ary = new Array();
                for (i = 0; i < value.length; ++i)
                    ary[i] = encode(value[i]);
                
                prefManager.prefBranch.setCharPref(prefName, ary.join("; "));
            }
            else
            {
                prefManager.prefBranch.setCharPref(prefName, value);
            }

            prefManager.prefService.savePrefFile(null);
        }
        catch (ex)
        {
            dd ("caught exception writing pref ``" + prefName + "''\n" + ex);
        }

        return value;
    };
    
    if (!arrayContains(prefManager.prefNames, prefName))
        prefManager.prefNames.push(prefName);

    prefManager.prefNames.sort();
    prefManager.prefs.__defineGetter__(prefName, prefGetter);
    prefManager.prefs.__defineSetter__(prefName, prefSetter);
}
