/* ustr.c: 
 *
 ****************************************************************
 * Copyright (C) 2004 Tom Lord
 * 
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "hackerlab/bugs/panic.h"
#include "hackerlab/mem/mem.h"
#include "hackerlab/strings/ustr.h"



ssize_t 
ustr_length_in_encoding (enum uni_encoding_scheme enc,
                         uni_string str,
                         enum uni_encoding_scheme str_enc)
{
  size_t answer = 0;
  size_t pos = 0;

#define LENGTH_IN_ENCODING(A, A_TYPE, B, B_TYPE) \
 case UNI_ENCODING_PAIR(uni_ ## A, uni_ ## B): \
  { \
    while (1) \
      { \
        t_unicode c; \
        c = uni_ ## B ## _scan ((B_TYPE *)str, &pos, pos + 100); \
        invariant (c != 0xffff); \
        if (!c) \
          break; \
        else \
          answer += uni_length_in_ ## A (c); \
      } \
  }

  switch (UNI_ENCODING_PAIR (enc, str_enc))
    {
    default:
      panic ("unhandled encoding case in ustr_length_in_encoding");
      break;
    UNI_EVERY_NATIVE_ENCODING_PAIR (LENGTH_IN_ENCODING)
    }

  return answer;
}


ssize_t 
ustr_length_in_encoding_n (enum uni_encoding_scheme enc,
                           uni_string str,
                           enum uni_encoding_scheme str_enc,
                           size_t len)
{
  size_t answer = 0;
  size_t pos = 0;

#define LENGTH_IN_ENCODING_N(A, A_TYPE, B, B_TYPE) \
 case UNI_ENCODING_PAIR(uni_ ## A, uni_ ## B): \
  { \
    while (pos < len) \
      { \
        t_unicode c; \
        c = uni_ ## B ## _scan ((B_TYPE *)str, &pos, len); \
        if (c == 0xffff) \
          return -1; \
        answer += uni_length_in_ ## A (c); \
      } \
  }

  switch (UNI_ENCODING_PAIR (enc, str_enc))
    {
    default:
      panic ("unhandled encoding case in ustr_length_in_encoding_n");
      break;
      UNI_EVERY_NATIVE_ENCODING_PAIR (LENGTH_IN_ENCODING_N)
    }

  return answer;
}





void
ustr_copy (uni_string dest,
           enum uni_encoding_scheme enc,
           uni_string source,
           enum uni_encoding_scheme source_enc)
{
  size_t out_pos = 0;
  size_t in_pos = 0;

#define USTR_COPY(A, A_TYPE, B, B_TYPE) \
  case UNI_ENCODING_PAIR(uni_ ## A, uni_ ## B): \
    { \
      while (1) \
        { \
          t_unicode c; \
          c = uni_ ## B ## _scan ((B_TYPE *)source, &in_pos, in_pos + 100); \
          invariant (c != 0xffff); \
          uni_ ## A ## _put ((A_TYPE *)dest, &out_pos, out_pos + 100, c); \
          if (!c) \
            break; \
        } \
    }

  switch (UNI_ENCODING_PAIR (enc, source_enc))
    {
    default:
      panic ("unhandled encoding case in ustr_copy");

      UNI_EVERY_NATIVE_ENCODING_PAIR (USTR_COPY);
    }
}


void
ustr_copy_n (uni_string dest,
             enum uni_encoding_scheme enc,
             uni_string source,
             enum uni_encoding_scheme source_enc,
             size_t in_len)
{
  size_t out_pos = 0;
  size_t in_pos = 0;

#define USTR_COPY_N(A, A_TYPE, B, B_TYPE) \
  case UNI_ENCODING_PAIR(uni_ ## A, uni_ ## B): \
    { \
      while (in_pos < in_len) \
        { \
          t_unicode c; \
          c = uni_ ## B ## _scan ((B_TYPE *)source, &in_pos, in_len); \
          invariant (c != 0xffff); \
          uni_ ## A ## _put ((A_TYPE *)dest, &out_pos, out_pos + 100, c); \
          if (!c) \
            break; \
        } \
    }

  switch (UNI_ENCODING_PAIR (enc, source_enc))
    {
    default:
      panic ("unhandled encoding case in ustr_copy_n");

      UNI_EVERY_NATIVE_ENCODING_PAIR (USTR_COPY_N);
    }
}




uni_string
ustr_save (alloc_limits limits,
           enum uni_encoding_scheme enc,
           uni_string source,
           enum uni_encoding_scheme source_enc)
{
  ssize_t code_len;
  size_t code_unit_size ;
  size_t byte_len;
  t_uchar * answer = 0;

  code_len = ustr_length_in_encoding (enc, source, source_enc);
  invariant (code_len >= 0);
  code_unit_size = uni_code_unit_size (enc);
  byte_len = (1 + code_len) * code_unit_size;

  answer = lim_malloc (limits, byte_len);
  if (!answer)
    return 0;

  ustr_copy ((uni_string)answer, enc, source, source_enc);

  return (uni_string)answer;
}


uni_string
ustr_save_n (alloc_limits limits,
             enum uni_encoding_scheme enc,
             uni_string source,
             enum uni_encoding_scheme source_enc,
             size_t length)
{
  ssize_t code_len;
  size_t code_unit_size ;
  size_t byte_len;
  t_uchar * answer = 0;

  code_len = ustr_length_in_encoding_n (enc, source, source_enc, length);
  invariant (code_len >= 0);
  code_unit_size = uni_code_unit_size (enc);
  byte_len = (1 + code_len) * code_unit_size;

  answer = lim_malloc (limits, byte_len);
  if (!answer)
    return 0;

  ustr_copy_n ((uni_string)answer, enc, source, source_enc, length);

  /* terminating 0
   */
  mem_set0 (answer + byte_len - code_unit_size, code_unit_size);

  return (uni_string)answer;
}







/* tag: Tom Lord Thu Jan  1 13:54:53 2004 (ustr.c)
 */
