
#include <gio/gio.h>
#include <libdbustest/dbus-test.h>

/* Starts up the provider with a specified config file and waits until
   it is available before returning */
DbusTestService *
start_with_config (const gchar * config)
{
	DbusTestService * service = dbus_test_service_new(NULL);

	/* GeoIP */
	DbusTestProcess * geoip = dbus_test_process_new(UBUNTU_GEOIP_PATH);
	dbus_test_process_append_param(geoip, "--test-data");
	dbus_test_process_append_param(geoip, config);
	dbus_test_service_add_task(service, DBUS_TEST_TASK(geoip));
	g_object_unref(G_OBJECT(geoip));

	/* Dummy */
	DbusTestTask * dummy = dbus_test_task_new();
	dbus_test_task_set_wait_for(dummy, "org.freedesktop.Geoclue.Providers.UbuntuGeoIP");
	dbus_test_service_add_task(service, dummy);
	g_object_unref(G_OBJECT(dummy));

	/* Get everything running */
	dbus_test_service_start_tasks(service);

	GDBusConnection * session = g_bus_get_sync(G_BUS_TYPE_SESSION, NULL, NULL);
	g_dbus_connection_set_exit_on_close(session, FALSE);

	int i = 0;
	for (i = 0; i < 100; i++) {
		GVariant * retval = g_dbus_connection_call_sync(session,
		                                                "org.freedesktop.Geoclue.Providers.UbuntuGeoIP",
		                                                "/org/freedesktop/Geoclue/Providers/UbuntuGeoIP",
		                                                "org.freedesktop.Geoclue",
		                                                "GetStatus",
		                                                NULL, /* params */
		                                                G_VARIANT_TYPE("(i)"),
		                                                G_DBUS_CALL_FLAGS_NONE,
		                                                -1,
		                                                NULL,
		                                                NULL);

		if (retval == NULL) {
			continue;
		}

		GVariant * child = g_variant_get_child_value(retval, 0);
		gint status = g_variant_get_int32(child);

		g_variant_unref(child);
		g_variant_unref(retval);

		if (status == 3) {
			break;
		}
	}

	g_object_unref(session);

	if (i == 100) {
		g_warning("Unable to get an available status");
		g_clear_object(&service);
	}

	return service;
}

/* Gets the position data */
static void
get_position (gdouble * lat, gdouble * lon, gdouble * alt)
{
	GError * error = NULL;
	GDBusConnection * session = g_bus_get_sync(G_BUS_TYPE_SESSION, NULL, NULL);
	g_dbus_connection_set_exit_on_close(session, FALSE);

	GVariant * retval = g_dbus_connection_call_sync(session,
	                                                "org.freedesktop.Geoclue.Providers.UbuntuGeoIP",
	                                                "/org/freedesktop/Geoclue/Providers/UbuntuGeoIP",
	                                                "org.freedesktop.Geoclue.Position",
	                                                "GetPosition",
	                                                NULL, /* params */
	                                                G_VARIANT_TYPE("(iiddd(idd))"),
	                                                G_DBUS_CALL_FLAGS_NONE,
	                                                -1,
	                                                NULL,
	                                                NULL);

	if (error != NULL) {
		g_debug("GetAddress Error: %s", error->message);
		g_error_free(error);
	}

	g_assert(retval != NULL);

	g_object_unref(G_OBJECT(session));

	GVariant * vlat = g_variant_get_child_value(retval, 2);
	GVariant * vlon = g_variant_get_child_value(retval, 3);
	GVariant * valt = g_variant_get_child_value(retval, 4);

	*lat = g_variant_get_double(vlat);
	*lon = g_variant_get_double(vlon);
	*alt = g_variant_get_double(valt);

	g_variant_unref(vlat);
	g_variant_unref(vlon);
	g_variant_unref(valt);
	g_variant_unref(retval);

	return;
}

/* Gets the address data */
static GHashTable *
get_address (void)
{
	GError * error = NULL;
	GDBusConnection * session = g_bus_get_sync(G_BUS_TYPE_SESSION, NULL, NULL);
	g_dbus_connection_set_exit_on_close(session, FALSE);

	GVariant * retval = g_dbus_connection_call_sync(session,
	                                                "org.freedesktop.Geoclue.Providers.UbuntuGeoIP",
	                                                "/org/freedesktop/Geoclue/Providers/UbuntuGeoIP",
	                                                "org.freedesktop.Geoclue.Address",
	                                                "GetAddress",
	                                                NULL, /* params */
	                                                G_VARIANT_TYPE("(ia{ss}(idd))"),
	                                                G_DBUS_CALL_FLAGS_NONE,
	                                                -1,
	                                                NULL,
	                                                &error);
	if (error != NULL) {
		g_debug("GetAddress Error: %s", error->message);
		g_error_free(error);
	}

	g_assert(retval != NULL);

	g_object_unref(G_OBJECT(session));

	GHashTable * hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
	GVariant * vdata = g_variant_get_child_value(retval, 1);

	gchar * key = NULL;
	gchar * value = NULL;
	GVariantIter iter;
	g_variant_iter_init(&iter, vdata);
	while (g_variant_iter_loop(&iter, "{ss}", &key, &value)) {
		g_hash_table_insert(hash, g_strdup(key), g_strdup(value));
	}

	g_variant_unref(vdata);
	g_variant_unref(retval);

	return hash;
}

/* Get the position accuracy */
static gint
get_position_accuracy (void)
{
	GError * error = NULL;
	GDBusConnection * session = g_bus_get_sync(G_BUS_TYPE_SESSION, NULL, NULL);
	g_dbus_connection_set_exit_on_close(session, FALSE);

	GVariant * retval = g_dbus_connection_call_sync(session,
	                                                "org.freedesktop.Geoclue.Providers.UbuntuGeoIP",
	                                                "/org/freedesktop/Geoclue/Providers/UbuntuGeoIP",
	                                                "org.freedesktop.Geoclue.Position",
	                                                "GetPosition",
	                                                NULL, /* params */
	                                                G_VARIANT_TYPE("(iiddd(idd))"),
	                                                G_DBUS_CALL_FLAGS_NONE,
	                                                -1,
	                                                NULL,
	                                                NULL);

	if (error != NULL) {
		g_debug("GetAddress Error: %s", error->message);
		g_error_free(error);
	}

	g_assert(retval != NULL);

	g_object_unref(G_OBJECT(session));

	GVariant * vacc_array = g_variant_get_child_value(retval, 5);
	GVariant * vacc_value = g_variant_get_child_value(vacc_array, 0);

	gint accuracy = g_variant_get_int32(vacc_value);

	g_variant_unref(vacc_value);
	g_variant_unref(vacc_array);
	g_variant_unref(retval);

	return accuracy;
}

/* Get the address accuracy */
static gint
get_address_accuracy (void)
{
	GError * error = NULL;
	GDBusConnection * session = g_bus_get_sync(G_BUS_TYPE_SESSION, NULL, NULL);
	g_dbus_connection_set_exit_on_close(session, FALSE);

	GVariant * retval = g_dbus_connection_call_sync(session,
	                                                "org.freedesktop.Geoclue.Providers.UbuntuGeoIP",
	                                                "/org/freedesktop/Geoclue/Providers/UbuntuGeoIP",
	                                                "org.freedesktop.Geoclue.Address",
	                                                "GetAddress",
	                                                NULL, /* params */
	                                                G_VARIANT_TYPE("(ia{ss}(idd))"),
	                                                G_DBUS_CALL_FLAGS_NONE,
	                                                -1,
	                                                NULL,
	                                                &error);
	if (error != NULL) {
		g_debug("GetAddress Error: %s", error->message);
		g_error_free(error);
	}

	g_assert(retval != NULL);

	g_object_unref(G_OBJECT(session));

	GVariant * vacc_array = g_variant_get_child_value(retval, 2);
	GVariant * vacc_value = g_variant_get_child_value(vacc_array, 0);

	gint accuracy = g_variant_get_int32(vacc_value);

	g_variant_unref(vacc_value);
	g_variant_unref(vacc_array);
	g_variant_unref(retval);

	return accuracy;
}

/* Test to ensure we can start up the process and things are good */
static void
test_geoip_start (void)
{
	DbusTestService * service = start_with_config(CANONICAL_DC_PATH);
	g_assert(service != NULL);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the position for the Canonical DC */
static void
test_geoip_canonical_position (void)
{
	DbusTestService * service = start_with_config(CANONICAL_DC_PATH);
	g_assert(service != NULL);

	gdouble lat = 0.0, lon = 0.0, alt = 0.0;
	get_position(&lat, &lon, &alt);

	g_assert(lat == 54.0);
	g_assert(lon == -2.0);
	g_assert(alt == 0.0);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the position for the Canonical DC */
static void
test_geoip_canonical_position_accuracy (void)
{
	DbusTestService * service = start_with_config(CANONICAL_DC_PATH);
	g_assert(service != NULL);

	g_assert(get_position_accuracy() == 1);

	g_object_unref(G_OBJECT(service));
	return;
}


/* Ensure the address for the Canonical DC */
static void
test_geoip_canonical_address (void)
{
	DbusTestService * service = start_with_config(CANONICAL_DC_PATH);
	g_assert(service != NULL);

	GHashTable * data = get_address();

	g_assert(g_hash_table_contains(data, "countrycode"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "countrycode"), "GB") == 0);

	g_assert(g_hash_table_contains(data, "country"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "country"), "United Kingdom") == 0);

	g_assert(!g_hash_table_contains(data, "locality"));
	g_assert(!g_hash_table_contains(data, "region"));

	g_assert(g_hash_table_contains(data, "timezone"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "timezone"), "Europe/London") == 0);

	g_hash_table_unref(data);
	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the address accuracy for the Canonical DC */
static void
test_geoip_canonical_address_accuracy (void)
{
	DbusTestService * service = start_with_config(CANONICAL_DC_PATH);
	g_assert(service != NULL);

	g_assert(get_address_accuracy() == 1);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the position for Texas */
static void
test_geoip_cableone_position (void)
{
	DbusTestService * service = start_with_config(CABLE_ONE_PATH);
	g_assert(service != NULL);

	gdouble lat = 0.0, lon = 0.0, alt = 0.0;
	get_position(&lat, &lon, &alt);

	g_assert(lat == 33.7762);
	g_assert(lon == -96.6067);
	g_assert(alt == 0.0);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the position accuracy for Texas */
static void
test_geoip_cableone_position_accuracy (void)
{
	DbusTestService * service = start_with_config(CABLE_ONE_PATH);
	g_assert(service != NULL);

	g_assert(get_position_accuracy() == 3);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the address for Texas */
static void
test_geoip_cableone_address (void)
{
	DbusTestService * service = start_with_config(CABLE_ONE_PATH);
	g_assert(service != NULL);

	GHashTable * data = get_address();

	g_assert(g_hash_table_contains(data, "countrycode"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "countrycode"), "US") == 0);

	g_assert(g_hash_table_contains(data, "country"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "country"), "United States") == 0);

	g_assert(g_hash_table_contains(data, "locality"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "locality"), "Denison") == 0);

	g_assert(g_hash_table_contains(data, "region"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "region"), "Texas") == 0);

	g_assert(g_hash_table_contains(data, "timezone"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "timezone"), "America/Chicago") == 0);

	g_hash_table_unref(data);
	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the address accuracy for Texas */
static void
test_geoip_cableone_address_accuracy (void)
{
	DbusTestService * service = start_with_config(CABLE_ONE_PATH);
	g_assert(service != NULL);

	g_assert(get_address_accuracy() == 3);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the position for Buenos Aires */
static void
test_geoip_buenosaires_position (void)
{
	DbusTestService * service = start_with_config(BUENOS_AIRES_PATH);
	g_assert(service != NULL);

	gdouble lat = 0.0, lon = 0.0, alt = 0.0;
	get_position(&lat, &lon, &alt);

	g_assert(lat == -34.5875);
	g_assert(lon == -58.6725);
	g_assert(alt == 0.0);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the position for Buenos Aires */
static void
test_geoip_buenosaires_position_accuracy (void)
{
	DbusTestService * service = start_with_config(BUENOS_AIRES_PATH);
	g_assert(service != NULL);

	g_assert(get_position_accuracy() == 3);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the address for Buenos Aires */
static void
test_geoip_buenosaires_address (void)
{
	DbusTestService * service = start_with_config(BUENOS_AIRES_PATH);
	g_assert(service != NULL);

	GHashTable * data = get_address();

	g_assert(g_hash_table_contains(data, "countrycode"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "countrycode"), "AR") == 0);

	g_assert(g_hash_table_contains(data, "country"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "country"), "Argentina") == 0);

	g_assert(g_hash_table_contains(data, "locality"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "locality"), "Buenos Aires") == 0);

	g_assert(g_hash_table_contains(data, "region"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "region"), "Distrito Federal") == 0);

	g_assert(g_hash_table_contains(data, "timezone"));
	g_assert(g_strcmp0((gchar *)g_hash_table_lookup(data, "timezone"), "America/Argentina/Buenos_Aires") == 0);

	g_hash_table_unref(data);
	g_object_unref(G_OBJECT(service));
	return;
}

/* Ensure the address accuracy for Buenos Aires */
static void
test_geoip_buenosaires_address_accuracy (void)
{
	DbusTestService * service = start_with_config(BUENOS_AIRES_PATH);
	g_assert(service != NULL);

	g_assert(get_address_accuracy() == 3);

	g_object_unref(G_OBJECT(service));
	return;
}

/* Pull it all together */
gint
main (gint argc, gchar * argv[])
{
	g_type_init();
	g_test_init(&argc, &argv, NULL);

	/* Tests */
	g_test_add_func("/geoip/start",                   test_geoip_start);
	g_test_add_func("/geoip/canonical/position",      test_geoip_canonical_position);
	g_test_add_func("/geoip/canonical/position_acc",  test_geoip_canonical_position_accuracy);
	g_test_add_func("/geoip/canonical/address",       test_geoip_canonical_address);
	g_test_add_func("/geoip/canonical/address_acc",   test_geoip_canonical_address_accuracy);
	g_test_add_func("/geoip/cableone/position",       test_geoip_cableone_position);
	g_test_add_func("/geoip/cableone/position_acc",   test_geoip_cableone_position_accuracy);
	g_test_add_func("/geoip/cableone/address",        test_geoip_cableone_address);
	g_test_add_func("/geoip/cableone/address_acc",    test_geoip_cableone_address_accuracy);
	g_test_add_func("/geoip/buenosaires/position",    test_geoip_buenosaires_position);
	g_test_add_func("/geoip/buenosaires/position_acc",test_geoip_buenosaires_position_accuracy);
	g_test_add_func("/geoip/buenosaires/address",     test_geoip_buenosaires_address);
	g_test_add_func("/geoip/buenosaires/address_acc", test_geoip_buenosaires_address_accuracy);

	return g_test_run();
}
