#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
"""tests for the syncdaemon config module """
from __future__ import with_statement

import os
from ConfigParser import ConfigParser

from ubuntuone.syncdaemon import (
    config,
)
from contrib.testing.testcase import (
    BaseTwistedTestCase,
)


class TestConfigBasic(BaseTwistedTestCase):
    """Basic _Config object tests"""

    def setUp(self):
        BaseTwistedTestCase.setUp(self)
        self.test_root = self.mktemp()

    def tearDown(self):
        """cleanup everything, also the config module globals"""
        BaseTwistedTestCase.tearDown(self)
        config._user_config = None

    def assertThrottlingSection(self, expected, current, on, read, write):
        """Assert for equality two ConfigParser and against the on, read and
        write args
        """
        self.assertEquals(expected.getboolean(config.THROTTLING, 'on'), on)
        self.assertEquals(expected.getint(config.THROTTLING, 'read_limit'),
                          read)
        self.assertEquals(expected.getint(config.THROTTLING, 'write_limit'),
                          write)
        self.assertEquals(expected.getboolean(config.THROTTLING, 'on'),
                          current.get_throttling())
        self.assertEquals(expected.getint(config.THROTTLING, 'read_limit'),
                          current.get_throttling_read_limit())
        self.assertEquals(expected.getint(config.THROTTLING, 'write_limit'),
                          current.get_throttling_write_limit())

    def test_load_empty(self):
        """test loading the a non-existent config file"""
        conf_file = os.path.join(self.test_root, 'test_missing_config.conf')
        # create the config object with an empty config file
        conf = config._Config(conf_file)
        # lie a bit and load the defaults, as we can't access the data
        # dir to get the current config, and a test shouldn't depend on
        # external data, just fake all the defaults
        default_conf = os.path.join(self.test_root, 'defaults.conf')
        with open(default_conf, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on.default = False\n')
            fp.write('read_limit.default = -1\n')
            fp.write('write_limit.default = -1\n')
        conf.default = config._Config._load_defaults([default_conf])
        self.assertEquals(False, conf.get_throttling())
        self.assertEquals(None, conf.get_throttling_read_limit())
        self.assertEquals(None, conf.get_throttling_write_limit())

    def test_load_basic(self):
        """test loading the config file containing only the throttling values"""
        conf_file = os.path.join(self.test_root, 'test_load_config.conf')
        # write some throttling values to the config file
        with open(conf_file, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = True\n')
            fp.write('read_limit = 1000\n')
            fp.write('write_limit = 200\n')
        conf = config._Config(conf_file)
        self.assertEquals(True, conf.get_throttling())
        self.assertEquals(1000, conf.get_throttling_read_limit())
        self.assertEquals(200, conf.get_throttling_write_limit())

    def test_load_extra_data(self):
        """test loading the a config file with other sections too"""
        conf_file = os.path.join(self.test_root, 'test_load_extra_config.conf')
        # write some throttling values to the config file
        with open(conf_file, 'w') as fp:
            fp.write('[__main__]\n')
            fp.write('log_level = INFO\n')
            fp.write('disable_ssl_verify = True\n')
            fp.write('\n')
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = True\n')
            fp.write('read_limit = 1000\n')
            fp.write('write_limit = 200\n')
        conf = config._Config(conf_file)
        self.assertEquals(True, conf.get_throttling())
        self.assertEquals(1000, conf.get_throttling_read_limit())
        self.assertEquals(200, conf.get_throttling_write_limit())

    def test_write_new(self):
        """test writing the throttling section to a new config file"""
        conf_file = os.path.join(self.test_root, 'test_write_new_config.conf')
        self.assertFalse(os.path.exists(conf_file))
        conf = config._Config(conf_file)
        conf.set_throttling(True)
        conf.set_throttling_read_limit(1000)
        conf.set_throttling_write_limit(100)
        conf.save()
        # load the config in a barebone ConfigParser and check
        conf_1 = ConfigParser()
        conf_1.read(conf_file)
        self.assertThrottlingSection(conf_1, conf, True, 1000, 100)

    def test_write_existing(self):
        """test writing the throttling section to a existing config file"""
        conf_file = os.path.join(self.test_root,
                                 'test_write_existing_config.conf')
        # write some throttling values to the config file
        with open(conf_file, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = False\n')
            fp.write('read_limit = 1000\n')
            fp.write('write_limit = 100\n')
        self.assertTrue(os.path.exists(conf_file))
        conf = config._Config(conf_file)
        conf.set_throttling(True)
        conf.set_throttling_read_limit(2000)
        conf.set_throttling_write_limit(200)
        conf.save()
        # load the config in a barebone ConfigParser and check
        conf_1 = ConfigParser()
        conf_1.read(conf_file)
        self.assertThrottlingSection(conf_1, conf, True, 2000, 200)

    def test_write_extra(self):
        """test writing the throttling section back to the config file,
        including extra sections
        """
        conf_file = os.path.join(self.test_root, 'test_write_extra_config.conf')
        # write some throttling values to the config file
        with open(conf_file, 'w') as fp:
            fp.write('[__main__]\n')
            fp.write('log_level = INFO\n')
            fp.write('disable_ssl_verify = True\n')
            fp.write('\n')
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = False\n')
            fp.write('read_limit = 2000\n')
            fp.write('write_limit = 200\n')
        self.assertTrue(os.path.exists(conf_file))
        conf = config._Config(conf_file)
        conf.set_throttling(True)
        conf.set_throttling_read_limit(3000)
        conf.set_throttling_write_limit(300)
        conf.save()
        # load the config in a barebone ConfigParser and check
        conf_1 = ConfigParser()
        conf_1.read(conf_file)
        self.assertThrottlingSection(conf_1, conf, True, 3000, 300)
        self.assertEquals(conf_1.get('__main__', 'log_level'),
                          conf.get('__main__', 'log_level'))
        self.assertEquals(conf_1.getboolean('__main__', 'disable_ssl_verify'),
                          conf.getboolean('__main__', 'disable_ssl_verify'))

    def test_write_existing_partial(self):
        """test writing a partially updated throttling section
        to a existing config file
        """
        conf_file = os.path.join(self.test_root,
                                 'test_write_existing_config.conf')
        # write some throttling values to the config file
        with open(conf_file, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = True\n')
            fp.write('read_limit = 1000\n')
            fp.write('write_limit = 100\n')
        self.assertTrue(os.path.exists(conf_file))
        conf = config._Config(conf_file)
        conf.set_throttling(False)
        conf.save()
        # load the config in a barebone ConfigParser and check
        conf_1 = ConfigParser()
        conf_1.read(conf_file)
        self.assertThrottlingSection(conf_1, conf, False, 1000, 100)


    def test_load_negative_limits(self):
        """test loading the config file with negative read/write limits"""
        conf_file = os.path.join(self.test_root, 'test_load_config.conf')
        # write some throttling values to the config file
        with open(conf_file, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = True\n')
            fp.write('read_limit = -1\n')
            fp.write('write_limit = -1\n')
        conf = config._Config(conf_file)
        self.assertEquals(True, conf.get_throttling())
        self.assertEquals(None, conf.get_throttling_read_limit())
        self.assertEquals(None, conf.get_throttling_write_limit())


    def test_load_partial_config(self):
        """test loading a partial config file and fallback to defaults"""
        conf_file = os.path.join(self.test_root, 'test_load_config.conf')
        # write some throttling values to the config file
        with open(conf_file, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on = True\n')
            fp.write('read_limit = 1\n')
        conf = config._Config(conf_file)
        # lie a bit and load the defaults, as we can't access the data
        # dir to get the current config, and a test shouldn't depend on
        # external data, just fake all the defaults
        default_conf = os.path.join(self.test_root, 'defaults.conf')
        with open(default_conf, 'w') as fp:
            fp.write('[bandwidth_throttling]\n')
            fp.write('on.default = False\n')
            fp.write('read_limit.default = -1\n')
            fp.write('write_limit.default = -1\n')
        conf.default = config._Config._load_defaults([default_conf])
        self.assertEquals(True, conf.get_throttling())
        self.assertEquals(1, conf.get_throttling_read_limit())
        self.assertEquals(None, conf.get_throttling_write_limit())

