/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "syncdaemon-transfer-info.h"

G_DEFINE_TYPE(SyncdaemonTransferInfo, syncdaemon_transfer_info, G_TYPE_OBJECT)

struct _SyncdaemonTransferInfoPrivate {
	gchar *path;
	gchar *share_id;
	gchar *node_id;
	glong bytes_transferred;
	glong total_size;
};

static void
syncdaemon_transfer_info_finalize (GObject *object)
{
	SyncdaemonTransferInfo *tinfo = SYNCDAEMON_TRANSFER_INFO (object);

	if (tinfo->priv != NULL) {
		if (tinfo->priv->path != NULL)
			g_free (tinfo->priv->path);

		if (tinfo->priv->share_id != NULL)
			g_free (tinfo->priv->share_id);

		if (tinfo->priv->node_id != NULL)
			g_free (tinfo->priv->node_id);

		g_free (tinfo->priv);
	}

	G_OBJECT_CLASS (syncdaemon_transfer_info_parent_class)->finalize (object);
}

static void
syncdaemon_transfer_info_class_init (SyncdaemonTransferInfoClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_transfer_info_finalize;
}

static void
syncdaemon_transfer_info_init (SyncdaemonTransferInfo *tinfo)
{
	tinfo->priv = g_new0 (SyncdaemonTransferInfoPrivate, 1);
}

/**
 * syncdaemon_transfer_info_new:
 * @path: Local path of the file being transferred.
 *
 * Return value: A new #SyncdaemonTransferInfo object.
 */
SyncdaemonTransferInfo *
syncdaemon_transfer_info_new (const gchar *path)
{
	SyncdaemonTransferInfo *tinfo;

	tinfo = g_object_new (SYNCDAEMON_TYPE_TRANSFER_INFO, NULL);
	if (path != NULL)
		syncdaemon_transfer_info_set_path (tinfo, path);

	return tinfo;
}

/**
 * syncdaemon_transfer_info_new_from_hash_table:
 */
SyncdaemonTransferInfo *
syncdaemon_transfer_info_new_from_hash_table (GHashTable *hash)
{
	SyncdaemonTransferInfo *tinfo;

	tinfo = g_object_new (SYNCDAEMON_TYPE_TRANSFER_INFO, NULL);
	if (hash != NULL) {
		gchar *str;

		syncdaemon_transfer_info_set_path (tinfo, g_hash_table_lookup (hash, "path"));
		syncdaemon_transfer_info_set_share_id (tinfo, g_hash_table_lookup (hash, "share_id"));
		syncdaemon_transfer_info_set_node_id (tinfo, g_hash_table_lookup (hash, "node_id"));

		str = g_hash_table_lookup (hash, "n_bytes_read");
		if (str != NULL)
			syncdaemon_transfer_info_set_bytes_transferred (tinfo, atol (str));

		str = g_hash_table_lookup (hash, "deflated_size");
		if (str != NULL)
			syncdaemon_transfer_info_set_total_size (tinfo, atol (str));
	}

	return tinfo;
}

/**
 * syncdaemon_transfer_info_get_path:
 */
const gchar *
syncdaemon_transfer_info_get_path (SyncdaemonTransferInfo *tinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo), NULL);

	return (const gchar *) tinfo->priv->path;
}

/**
 * syncdaemon_transfer_info_set_path:
 */
void
syncdaemon_transfer_info_set_path (SyncdaemonTransferInfo *tinfo, const gchar *path)
{
	g_return_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));

	if (tinfo->priv->path != NULL)
		g_free (tinfo->priv->path);

	tinfo->priv->path = g_strdup (path);
}

/**
 * syncdaemon_transfer_info_get_share_id:
 */
const gchar *
syncdaemon_transfer_info_get_share_id (SyncdaemonTransferInfo *tinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo), NULL);

	return (const gchar *) tinfo->priv->share_id;
}

/**
 * syncdaemon_transfer_info_set_share_id:
 */
void
syncdaemon_transfer_info_set_share_id (SyncdaemonTransferInfo *tinfo, const gchar *share_id)
{
	g_return_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));

	if (tinfo->priv->share_id != NULL)
		g_free (tinfo->priv->share_id);

	tinfo->priv->share_id = g_strdup (share_id);
}

/**
 * syncdaemon_transfer_info_get_path:
 */
const gchar *
syncdaemon_transfer_info_get_node_id (SyncdaemonTransferInfo *tinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo), NULL);

	return (const gchar *) tinfo->priv->node_id;
}

/**
 * syncdaemon_transfer_info_set_node_id:
 */
void
syncdaemon_transfer_info_set_node_id (SyncdaemonTransferInfo *tinfo, const gchar *node_id)
{
	g_return_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));

	if (tinfo->priv->node_id != NULL)
		g_free (tinfo->priv->node_id);

	tinfo->priv->node_id = g_strdup (node_id);
}

/**
 * syncdaemon_transfer_info_get_bytes_transferred:
 */
glong
syncdaemon_transfer_info_get_bytes_transferred (SyncdaemonTransferInfo *tinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo), 0);

	return tinfo->priv->bytes_transferred;
}

/**
 * syncdaemon_transfer_info_set_bytes_transferred:
 */
void
syncdaemon_transfer_info_set_bytes_transferred (SyncdaemonTransferInfo *tinfo, glong bytes)
{
	g_return_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));

	tinfo->priv->bytes_transferred = bytes;
}

/**
 * syncdaemon_transfer_info_get_total_size:
 */
glong
syncdaemon_transfer_info_get_total_size (SyncdaemonTransferInfo *tinfo)
{
	g_return_val_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo), 0);

	return tinfo->priv->total_size;
}

/**
 * syncdaemon_transfer_info_set_total_size:
 */
void
syncdaemon_transfer_info_set_total_size (SyncdaemonTransferInfo *tinfo, glong bytes)
{
	g_return_if_fail (SYNCDAEMON_IS_TRANSFER_INFO (tinfo));

	tinfo->priv->total_size = bytes;
}
